## ----setup, include=FALSE-----------------------------------------------------------------------------------------------------------------------------------------------------------------------------
knitr::opts_chunk$set(echo = TRUE, eval=T)
options(width = 200, digits= 4)

me_ok <- requireNamespace("marginaleffects", quietly = TRUE) &&
  requireNamespace("sandwich", quietly = TRUE)
set.seed(1000)

## -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
library("cobalt")
data("lalonde", package = "cobalt")
head(lalonde)

## -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
bal.tab(treat ~ age + educ + race + married + nodegree + re74 + re75,
        data = lalonde, estimand = "ATT", thresholds = c(m = .05))

## -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
library("WeightIt")
W.out <- weightit(treat ~ age + educ + race + married + nodegree + re74 + re75,
                  data = lalonde, estimand = "ATT", method = "glm")
W.out #print the output

## -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
summary(W.out)

## -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
bal.tab(W.out, stats = c("m", "v"), thresholds = c(m = .05))

## -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
W.out <- weightit(treat ~ age + educ + race + married + nodegree + re74 + re75,
                  data = lalonde, estimand = "ATT", method = "ebal")
summary(W.out)

## -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
bal.tab(W.out, stats = c("m", "v"), thresholds = c(m = .05))

## ---- message=FALSE, eval = me_ok---------------------------------------------------------------------------------------------------------------------------------------------------------------------
# Attach weights to dataset
lalonde$weights <- W.out$weights

# Fit outcome model
fit <- lm(re78 ~ treat * (age + educ + race + married +
                            nodegree + re74 + re75),
          data = lalonde, weights = weights)

# G-computation for the treatment effect
library("marginaleffects")
avg_comparisons(fit, variables = "treat",
                vcov = "HC3",
                newdata = subset(lalonde, treat == 1),
                wts = "weights")

## -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
data("msmdata")
head(msmdata)

## -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
library("cobalt") #if not already attached
bal.tab(list(A_1 ~ X1_0 + X2_0,
             A_2 ~ X1_1 + X2_1 +
               A_1 + X1_0 + X2_0,
             A_3 ~ X1_2 + X2_2 +
               A_2 + X1_1 + X2_1 +
               A_1 + X1_0 + X2_0),
        data = msmdata, stats = c("m", "ks"),
        which.time = .all)

## -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
Wmsm.out <- weightitMSM(list(A_1 ~ X1_0 + X2_0,
                             A_2 ~ X1_1 + X2_1 +
                               A_1 + X1_0 + X2_0,
                             A_3 ~ X1_2 + X2_2 +
                               A_2 + X1_1 + X2_1 +
                               A_1 + X1_0 + X2_0),
                        data = msmdata, method = "glm",
                        stabilize = TRUE)
Wmsm.out

## -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
summary(Wmsm.out)

## -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
bal.tab(Wmsm.out, stats = c("m", "ks"), which.time = .none)

## -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
# Attach weights to dataset
msmdata$weights <- Wmsm.out$weights

# Fit outcome model
fit <- glm(Y_B ~ A_1 * A_2 * A_3 * (X1_0 + X2_0),
           data = msmdata, weights = weights,
           family = quasibinomial)


## ---- eval = me_ok------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
library("marginaleffects")
(p <- avg_predictions(fit,
                      vcov = "HC3",
                      newdata = datagridcf(A_1 = 0:1, A_2 = 0:1, A_3 = 0:1),
                      by = c("A_1", "A_2", "A_3"),
                      wts = "weights",
                      type = "response"))

## ---- eval = me_ok------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
hypotheses(p, "b8 - b1 = 0")

