## ----include = FALSE----------------------------------------------------------
source("common.R")

## -----------------------------------------------------------------------------
library(orderly)

## -----------------------------------------------------------------------------
path <- tempfile() # we'll use a temporary directory here - see note below
orderly_init(path)

## ----echo = FALSE-------------------------------------------------------------
dir_tree(path, all = TRUE)

## ----echo = FALSE, results = "asis"-------------------------------------------
json_output(readLines(file.path(path, "orderly_config.json")))

## ----include = FALSE----------------------------------------------------------
fs::dir_create(file.path(path, "src", "incoming_data"))
write.csv(data.frame(x = 1:10, y = 1:10 + rnorm(10)),
          file.path(path, "src", "incoming_data", "data.csv"),
          row.names = FALSE)
writeLines(c(
  'd <- read.csv("data.csv")',
  "d$z <- resid(lm(y ~ x, d))",
  'saveRDS(d, "data.rds")'),
  file.path(path, "src", "incoming_data", "incoming_data.R"))

## ----echo = FALSE-------------------------------------------------------------
dir_tree(path, all = FALSE)

## ----echo = FALSE, results = "asis"-------------------------------------------
r_output(readLines(file.path(path, "src/incoming_data/incoming_data.R")))

## ----orderly_root = path------------------------------------------------------
id <- orderly_run("incoming_data")
id

## ----echo = FALSE-------------------------------------------------------------
dir_tree(path, all = FALSE)

## ----orderly_root = path------------------------------------------------------
dest <- tempfile()
fs::dir_create(dest)
orderly_copy_files(id, files = c("final.rds" = "data.rds"),
                   dest = dest)

## ----include = FALSE----------------------------------------------------------
fs::dir_create(file.path(path, "src", "analysis"))
writeLines(c(
  'orderly_dependency("incoming_data", "latest()",',
  '                   c("incoming.rds" = "data.rds"))',
  'd <- readRDS("incoming.rds")',
  'png("analysis.png")',
  "plot(y ~ x, d)",
  "dev.off()"),
  file.path(path, "src", "analysis", "analysis.R"))

## ----echo = FALSE-------------------------------------------------------------
dir_tree(file.path(path), "src")

## ----echo = FALSE, results = "asis"-------------------------------------------
r_output(readLines(file.path(path, "src/analysis/analysis.R")))

## ----orderly_root = path------------------------------------------------------
id <- orderly_run("analysis")

## ----include = FALSE----------------------------------------------------------
code_data <- readLines(
  file.path(path, "src", "incoming_data", "incoming_data.R"))
writeLines(c(
  "orderly_strict_mode()",
  'orderly_resource("data.csv")',
  'orderly_artefact(description = "Processed data", "data.rds")',
  "",
  code_data),
  file.path(path, "src", "incoming_data", "incoming_data.R"))

## ----echo = FALSE, results = "asis"-------------------------------------------
r_output(readLines(file.path(path, "src/incoming_data/incoming_data.R")))

## ----orderly_root = path------------------------------------------------------
id <- orderly_run("incoming_data")

## ----include = FALSE----------------------------------------------------------
fs::dir_create(file.path(path, "src", "random"))
writeLines(c(
  "pars <- orderly_parameters(n_samples = 10)",
  "x <- seq_len(pars$n_samples)",
  "d <- data.frame(x = x, y = x + rnorm(pars$n_samples))",
  'saveRDS(d, "data.rds")'),
  file.path(path, "src", "random", "random.R"))

## ----echo = FALSE, results = "asis"-------------------------------------------
r_output(readLines(file.path(path, "src/random/random.R")))

## ----orderly_root = path------------------------------------------------------
id <- orderly_run("random", list(n_samples = 15))

## ----orderly_root = path------------------------------------------------------
orderly_copy_files(id, files = c("random.rds" = "data.rds"),
                   dest = dest)
readRDS(file.path(dest, "random.rds"))

## ----orderly_root = path------------------------------------------------------
orderly_search('latest(name == "random")')

## ----orderly_root = path------------------------------------------------------
orderly_search('latest(name == "random" && parameter:n_samples > 10)')

## ----echo = FALSE-------------------------------------------------------------
fs::dir_create(file.path(path, "shared"))
write.csv(data.frame(x = 1:10, y = runif(10)),
          file.path(path, "shared/data.csv"))
dir_tree(path)

## ----echo = FALSE, results = "asis"-------------------------------------------
fs::dir_create(file.path(path, "src", "use_shared"))
writeLines(c(
  'orderly_shared_resource("data.csv")',
  'orderly_artefact(description = "analysis", "analysis.png")',
  "",
  'd <- read.csv("data.csv")',
  'png("analysis.png")',
  "plot(y ~ x, d)",
  "dev.off()"),
  file.path(path, "src", "use_shared", "use_shared.R"))
r_output(readLines(file.path(path, "src/use_shared/use_shared.R")))

## ----orderly_root = path------------------------------------------------------
id <- orderly_run("use_shared")

## ----echo = FALSE-------------------------------------------------------------
dir_tree(path, "archive/use_shared")

## ----include = FALSE----------------------------------------------------------
withr::with_dir(file.path(path, "src/incoming_data"),
                sys.source("incoming_data.R", new.env(parent = .GlobalEnv)))

## ----orderly_root = path------------------------------------------------------
orderly_cleanup_status("incoming_data")

## ----orderly_root = path------------------------------------------------------
orderly_cleanup("incoming_data")

## ----orderly_root = path------------------------------------------------------
orderly_gitignore_update("incoming_data")

## ----orderly_root = path------------------------------------------------------
orderly_metadata_extract(
  name = "incoming_data",
  extract = c(time = "time.start"))

## ----include = FALSE, orderly_root = path-------------------------------------
id_latest <- orderly_search("latest", name = "incoming_data")
fs::dir_delete(file.path(path, "archive", "incoming_data", id_latest))

## ----error = TRUE, orderly_root = path----------------------------------------
try({
orderly_run("analysis")
})

## ----echo = FALSE, results = "asis"-------------------------------------------
r_output(
  sprintf('orderly_validate_archive("%s", action = "orphan")',
          id_latest))

## ----orderly_root = path------------------------------------------------------
orderly_validate_archive(action = "orphan")

## ----orderly_root = path------------------------------------------------------
orderly_prune_orphans()

## ----echo = FALSE-------------------------------------------------------------
dir_tree(path, ".outpack", all = TRUE)

## ----echo = FALSE, orderly_root = path----------------------------------------
orderly_config_set(core.use_file_store = TRUE)
dir_tree(path, ".outpack", all = TRUE)

