% Generated by roxygen2 (4.0.2): do not edit by hand
\name{sliding_window}
\alias{sliding_window}
\title{Expand time data series with a sliding window}
\usage{
sliding_window(time.data, win.h.size = NULL, win.h.weight = NULL,
  win.d.size = NULL, win.d.weight = NULL)
}
\arguments{
\item{time.data}{Time series data formatted with \code{\link{format_timedata}}}

\item{win.h.size}{Vector to determine the size of the window for adjacent time steps}

\item{win.h.weight}{Weights associated with adjacent hours}

\item{win.d.size}{Vector to determine the size of the window for adjacent days}

\item{win.d.weight}{Weights associated with adjacent days}
}
\value{
An expanded data frame with the same format as \code{data.time}, but with a sliding window applied to the
variable generation columns
}
\description{
This function allows to expand the time data series to include adjacent time steps and days, using a sliding window.
The procedure is used to shift the variable generation time series, so calculations are not limited to the observed
load and variable generation pairs. The objective is to further explore how the variable behaviour of these generators
affect reliability calculations.
}
\details{
The sliding window can include adjacent time steps (typically adjacent hours) and/or adjacent days. Table 1
shows and example, with the current hour being hour 6 in day 5. If an hour is withing the extent of the sliding window
it is marked with an x.

Table 1: Example of sliding window

\tabular{lcccccccccc}{
     \tab H1 \tab H2 \tab H3 \tab H4 \tab H5 \tab H6 \tab H7 \tab H8 \tab H9 \tab H10\cr
    Day 1 \tab \tab \tab \tab \tab \tab \tab \tab \tab \tab \cr
    Day 2 \tab \tab \tab \tab \tab \tab \tab \tab \tab \tab \cr
    Day 3 \tab \tab \tab x \tab x \tab x \tab x \tab x \tab x \tab x \tab \cr
    Day 4 \tab \tab \tab x \tab x \tab x \tab x \tab x \tab x \tab x \tab \cr
    Day 5 \tab \tab \tab x \tab x \tab x \tab Current \tab x \tab x \tab x \tab \cr
    Day 6 \tab \tab \tab x \tab x \tab x \tab x \tab x \tab x \tab x \tab \cr
    Day 7 \tab \tab \tab x \tab x \tab x \tab x \tab x \tab x \tab x \tab \cr
    Day 8 \tab \tab \tab \tab \tab \tab \tab \tab \tab \tab \cr
    Day 9 \tab \tab \tab \tab \tab \tab \tab \tab \tab \tab
}

\code{win.h.size} determines how many adjacent time steps are included in the sliding window. By default, the window
only includes the current hour. If specified, \code{win.h.size} must be a numeric vector with 2 values.

Similarly, \code{win.d.size} determines how many adjacent days are used in the window and defaults to the current day.
If defined, it must be a vector with 2 integers.

For example, to reproduce the window in Table 1 set:
\itemize{
   \item{\code{win.h.size = c(-3, 3)} }
   \item{\code{win.d.size = c(-2, 2)} }
}

By default, all the data points in the sliding window are given the same probability in the calculations. It is possible
to provide different weights to adjacent hours and adjacent days. This is achieved by using the \code{wind.h.weight} and
\code{wind.d.weight} parameters, respectively. The lengths of these needs to be consistent with the corresponding sizes
parameters. Table 2 shows an example of how the final weights are calculated, by
setting:

\itemize{
    \item{\code{wind.h.weight = c(0.1, 0.1, 0.2, 0.2, 0.2, 0.1, 0.1)} }
    \item{\code{wind.d.weight = c(0.1, 0.2, 0.4, 0.2 0.1)} }
}

Figure 2: Example of sliding window with different weights

\tabular{rccccccccccc}{
     \tab H1 \tab H2 \tab H3 \tab H4 \tab H5 \tab H6 \tab H7 \tab H8 \tab H9 \tab H10 \tab Day weight \cr
    Day 1 \tab \tab \tab \tab \tab \tab \tab \tab \tab \tab \cr
    Day 2 \tab \tab \tab \tab \tab \tab \tab \tab \tab \tab \cr
    Day 3 \tab \tab \tab 0.01 \tab 0.01 \tab 0.02 \tab 0.02 \tab 0.02 \tab 0.01 \tab 0.01 \tab \tab 0.1\cr
    Day 4 \tab \tab \tab 0.02 \tab 0.02 \tab 0.04 \tab 0.04 \tab 0.04 \tab 0.02 \tab 0.02 \tab \tab 0.2\cr
    Day 5 \tab \tab \tab 0.04 \tab 0.04 \tab 0.08 \tab \bold{0.08} \tab 0.08 \tab 0.04 \tab 0.04 \tab \tab \bold{0.4}\cr
    Day 6 \tab \tab \tab 0.02 \tab 0.02 \tab 0.04 \tab 0.04 \tab 0.04 \tab 0.02 \tab 0.02 \tab \tab 0.2\cr
    Day 7 \tab \tab \tab 0.01 \tab 0.01 \tab 0.02 \tab 0.02 \tab 0.02 \tab 0.01 \tab 0.01 \tab \tab 0.1\cr
    Day 8 \tab \tab \tab \tab \tab \tab \tab \tab \tab \tab \tab \cr
    Day 9 \tab \tab \tab \tab \tab \tab \tab \tab \tab \tab \tab \cr
     \tab \tab \tab \tab \tab \tab \tab \tab \tab \tab \tab \cr
    Hour weight \tab \tab \tab 0.1 \tab 0.1 \tab 0.2 \tab \bold{0.2} \tab 0.2 \tab 0.1 \tab 0.1 \tab \tab
}

Use of a sliding window is currently an open area of research. The authors suggest to use it as a sensitivitiy tool,
especially when only a few years of data is available. Special care should be taken with solar data. Because of its daily
pattern, a window with too many adjacent hours could return erroneous results.

A vignette in the package presents some additional examples and graphs for the use of this function. It can
be accessed with \code{browseVignettes("sliding_window")}.
}
\examples{
# Create time data object
tdata <- data.frame(Area = c(rep("A", 48), rep("B", 48)),
                    Time = 1:48,
                    Load = c(runif(48, 200, 250), runif(48, 400, 450)),
                    Wind = c(runif(48, 20, 25), runif(48, 40, 45)))
td <- format_timedata(tdata)
head(td)

# If no data is provided, results remain intact
td2 <- sliding_window(td)

# Expand data for adjacent time steps (with equal and different weights)
td3 <- sliding_window(td, win.h.size = c(-1, 1))
td4 <- sliding_window(td, win.h.size = c(-1, 1), win.h.weight = c(1, 2, 1))

# Expand data for ajdacent days
td5 <- sliding_window(td, win.d.size = c(0, 1))

# Expand data for both adjacent times and days
td6 <- sliding_window(td, win.h.size = c(-1, 1), win.d.size = c(0, 1))
}
\seealso{
\code{\link{format_timedata}} to create the \code{time.data} object

This function is called from \code{\link{calculate_metrics}} and \code{\link{calculate_elcc}}
}

