% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/AgeC14_Computation.R
\name{AgeC14_Computation}
\alias{AgeC14_Computation}
\title{Bayesian analysis for C-14 age estimations of various samples}
\usage{
AgeC14_Computation(
  Data_C14Cal,
  Data_SigmaC14Cal,
  SampleNames,
  Nb_sample,
  PriorAge = rep(c(10, 50), Nb_sample),
  SavePdf = FALSE,
  OutputFileName = c("MCMCplot", "HPD_CalC-14Curve", "summary"),
  OutputFilePath = c(""),
  SaveEstimates = FALSE,
  OutputTableName = c("DATA"),
  OutputTablePath = c(""),
  StratiConstraints = c(),
  sepSC = c(","),
  Model = c("full"),
  CalibrationCurve = c("IntCal20"),
  Iter = 50000,
  t = 5,
  n.chains = 3,
  quiet = FALSE,
  roundingOfValue = 3
)
}
\arguments{
\item{Data_C14Cal}{\link{numeric} (\strong{required}): corresponding to C-14 age estimate.}

\item{Data_SigmaC14Cal}{\link{numeric} (\strong{required}): corresponding to the error of C-14 age estimates.}

\item{SampleNames}{\link{character} (\strong{required}): names of sample. The length of this vector is equal to \code{Nb_sample}.}

\item{Nb_sample}{\link{integer}: number of samples.}

\item{PriorAge}{\link{numeric} (with default): lower and upper bounds for age parameter of each sample in years (not in ka).
Note that, \code{length(PriorAge) == 2 * Nb_sample}
and \code{PriorAge[2i-1,2i]} corresponds to the lower and upper bounds of sample whose number ID is equal to \code{i}.}

\item{SavePdf}{\link{logical} (with default): if TRUE save graphs in pdf file named \code{OutputFileName} in folder \code{OutputFilePath}.}

\item{OutputFileName}{\link{character} (with default): name of the pdf file that will be generated by the function if \code{SavePd=TRUE},
\verb{length(OutputFileName) = =3}, see \bold{PLOT OUTPUT} in \bold{Value} section for more informations.}

\item{OutputFilePath}{\link{character} (with default): path to the pdf file that will be generated by the function if \code{SavePdf=TRUE}.
If it is not equal to "", it must be terminated by "/".}

\item{SaveEstimates}{\link{logical} (with default): if TRUE save Bayes' estimates, credible interval at level 68\% and 95\% and
the result of the Gelman and Rubin test of convergence, in a csv table named \code{OutputFileName} in folder \code{OutputFilePath}.}

\item{OutputTableName}{\link{logical} (with default): name of the table that will be generated by the function if \code{SaveEstimates=TRUE}.}

\item{OutputTablePath}{character (with default): path to the table that will be generated by the function if \code{SaveEstimates=TRUE}.
If it is not equal to "", it must be terminated by "/".}

\item{StratiConstraints}{\link{numeric} matrix or character(with default): input object for the stratigraphic relation between samples.
If there is stratigraphic relation between samples see the details section for instructions regarding how to correctly fill \code{StratiConstraints};
the user can refer to a matrix (numeric matrix) or to a csv file (character).
If there is no stratigraphic relation default value is suitable.}

\item{sepSC}{\link{character} (with default): if \code{StratiConstraints} is character,
indicate column separator in \code{StratiConstraints} csv file.}

\item{Model}{\link{character} (with default): if \bold{"full"}, error on estimate calibration curve is taken account.
If \bold{"naive"} this error is not taken account in the age estimate.}

\item{CalibrationCurve}{\link{character} (with default): calibration curve chosen. Allowed inputs are
\itemize{
\item \bold{"Intcal13"} or \bold{"Intcal13"} for Northern Hemisphere atmospheric radiocarbon calibration curve,
\item \bold{"Marine13"} or \bold{"Marine13"} for Marine radiocarbon calibration curve,
\item \bold{"SHCal13"} or \bold{"SHCal20"} for Southern Hemisphere atmospheric radiocarbon calibration curve
\item \bold{a csv file, with tree columns, the first column is dedicated to \code{"Cal.BP"}, the second to \code{"XC-14.age"}, the third to \code{"Error"}.
The decimal of this file must be a dot, and the separator must be a comma. }
}}

\item{Iter}{\link{integer} (with default): number of iterations for the MCMC computation (for more information see \link[rjags:jags.model]{rjags::jags.model}).}

\item{t}{integer (with default): 1 every \code{t} iterations of the MCMC is considered for sampling the posterior distribution
(for more information see \link[rjags:jags.model]{rjags::jags.model}.}

\item{n.chains}{\link{integer} (with default): number of independent chains for the model (for more information see \link[rjags:jags.model]{rjags::jags.model}.}

\item{quiet}{\link{logical} (with default): enables/disables \link{rjags} messages}

\item{roundingOfValue}{\link{integer} (with default):  Integer indicating the number of decimal places to be used, default set to 3.}
}
\value{
\bold{NUMERICAL OUTPUT}

\enumerate{
\item \bold{A list containing the following objects:}
\itemize{
\item \bold{Sampling}: that corresponds to a sample of the posterior distributions of the age parameters;
\item \bold{Outlier}: stating the names of samples that are considered as outliers;
\item \bold{Model}: stating which model was chosen (\code{"full"} or \code{"naive"});
\item \bold{CalibrationCurve}: stating which radiocarbon calibration curve was chosen;
\item \bold{PriorAge}: stating the priors used for the age parameter;
\item \bold{StratiConstraints}: stating the stratigraphic relations between samples considered in the model.
}
\item\bold{The Gelman and Rubin test of convergency}: print the result of the Gelman and Rubin test of convergence for the age estimate for each sample.
A result close to one is expected.\cr
In addition, the user must visually assess the convergence of the trajectories by looking at the graph
generated by the function (see \bold{PLOT OUTPUT} for more informations).\cr
If both convergences (Gelman and Rubin test and plot checking) are satisfactory,
the user can consider the estimates as valid.
Otherwise, the user may try increasing the number of MCMC iterations (\code{Iter})
or being more precise if it is possible on the \code{PriorAge} parameter to reach convergence.
\item \bold{Credible intervals and Bayes estimates}: prints the Bayes' estimates, the credible intervals at 95\% and 68\% for
the age parameters for each sample.
}

\bold{PLOT OUTPUT}

\enumerate{
\item\bold{MCMC trajectories}: A graph with the MCMC trajectories and posterior distributions of the age parameter is displayed. \cr
On each line, the plot on the left represents the MCMC trajectories, and the one on the right the posterior distribution of the parameter.
\item \bold{Summary of sample age estimates}: plot credible intervals and Bayes' estimate of each sample age on one graph.
}

To give the results in a publication, we recommend to give the Bayes' estimate of the parameters as well as the credible interval at 95\% or 68\%.
}
\description{
This function calibrates the C-14 age of samples to get an age (in ka).
The user can choose one of the following radiocarbon calibration curve:
Northern or Southern Hemisphere or marine atmospheric. It must be the same curve for all samples.
}
\details{
\strong{How to fill \code{StratiConstraints}?}

If there is stratigraphic relations between samples, \emph{C-14 age in \code{Data_C14Cal} must be ordered by order of increasing ages}.

The user can fill the \code{StratiConstraints} matrix as follow.
\enumerate{
\item \strong{Size of the matrix}: row number of \code{StratiConstraints} matrix is equal to \code{Nb_sample+1},
and column number is equal to \code{Nb_sample}.
\item \strong{First line of the matrix}:
for all \verb{i in \{1,...,Nb_Sample\}}, \code{StratiConstraints[1,i]=1} that means the lower bound
of the sample age (given in \code{PriorAge[2i-1]})
for the sample whose number ID is equal to \code{i}, is taken into account.
\item \strong{Sample relations}: for all  \verb{j in \{2,...,Nb_Sample+1\}} and all \verb{i in \{j,...,Nb_Sample\}},
\code{StratiConstraints[j,i]=1} if sample age whose number ID is equal to \code{j-1} is lower than
sample age whose number ID is equal to \code{i}. Otherwise, \code{StratiConstraints[j,i]=0}.
}

Note that \verb{StratiConstraints_\{2:Nb_sample+1,1:Nb_sample\}} is a upper triangular matrix.

The user can also use \code{SCMatrix} or \link{SC_Ordered} (if all samples are ordered) functions
to construct the \code{StratiConstraints} matrix.

The user can also refer to a .csv file that contains the relation between samples as defined above.
The user must take care about the separator used in the csv file using the argument \code{sepSC}.\cr

** More precision on \code{Model} **\cr

We propose two models "full" or "naive". If \code{Model = 'full'} that means
measurement error and error on calibration curve are taken account in
the Bayesian model; if \code{Model = "naive"} that means only error on measurement
are taken account in the mode.

More precisely, the model considered here, as the one developed by Christen, JA (1994),
assume multiplicative effect of errors to address the problem of outliers.
In addition, to not penalise variables that are not outliers and damage theirs estimation,
we introduce a structure of mixture, that means only variable that are considered
as outlier have in addition a multiplicative error.
}
\note{
Please note that the initial values for all MCMC are currently all the same for all chains since we rely on the automatic
initial value generation of JAGS. This is not optimal and will be changed in future. However, it does not affect the quality
of the age estimates if the chains have converged.
}
\examples{
## Load data
data(DATA_C14,envir = environment())
C14Cal <- DATA_C14$C14[,1]
SigmaC14Cal <- DATA_C14$C14[,2]
Names <- DATA_C14$Names
nb_sample <- length(Names)

## Age computation of samples without stratigraphic relations
Age <- AgeC14_Computation(
 Data_C14Cal = C14Cal,
 Data_SigmaC14Cal = SigmaC14Cal,
 SampleNames = Names,
 Nb_sample = nb_sample,
 PriorAge = rep(c(20,60),nb_sample),
 Iter = 500,
 quiet = TRUE,
 roundingOfValue = 3)

}
\references{
Christen, JA (1994). Summarizing a set of radiocarbon determinations: a robust approach. Applied Statistics, 489-503.

Reimer PJ, Bard E, Bayliss A, Beck JW, Blackwell PC, Bronl Ramsey C, Buck CE, Cheng H, Edwards RL, Friedrich M,
Grootes PM, Guilderson TP, Haflidason H, Hajdas I, Hatte C, Heaton TJ, Hoffmann DL, Hogg AG, Hughen KA, Kaiser KF, Kromer B,
Manning SW, Niu M, Reimer RW, Richards DA, Scott EM, Southon JR, Staff RA, Turney CSM, van der Plicht J. 2013.
IntCal13 ans Marine13 radiocarbon age calibration curves 0-50000 years cal BP. Radiocarbon 55(4)=1869-1887.

Hogg AG, Hua Q, Blackwell PG, Niu M, Buck CE, Guilderson TP, Heaton TJ, Palmer JG, Reimer PJ, Reimer RW, Turney CSM, Zimmerman SRH.
2013. SHCal13 Southern Hemisphere calibration, 0-50000 years cal. BP Radiocarbon 55(4):1889-1903
}
\seealso{
\link{rjags}, \link{plot_MCMC}, \link{SCMatrix}, \link{plot_Ages}
}
\author{
Claire Christophe, Anne Philippe, Guillaume Guérin, Sebastian Kreutzer
, RLum Developer Team}
