% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/glm_random.R
\name{glm.random.a0}
\alias{glm.random.a0}
\title{Model fitting for generalized linear models with random a0}
\usage{
glm.random.a0(
  data.type,
  data.link,
  y,
  x,
  n = 1,
  borrow.treat = FALSE,
  historical,
  prior.beta.var = rep(10, 50),
  prior.a0.shape1 = rep(1, 10),
  prior.a0.shape2 = rep(1, 10),
  a0.coefficients,
  lower.limits = NULL,
  upper.limits = NULL,
  slice.widths = rep(0.1, 50),
  nMC = 10000,
  nBI = 250
)
}
\arguments{
\item{data.type}{Character string specifying the type of response. The options are "Normal", "Bernoulli", "Binomial", "Poisson" and "Exponential".}

\item{data.link}{Character string specifying the link function. The options are "Logistic", "Probit", "Log", "Identity-Positive", "Identity-Probability" and "Complementary Log-Log". Does not apply if \code{data.type} is "Normal".}

\item{y}{Vector of responses.}

\item{x}{Matrix of covariates. The first column should be the treatment indicator with 1 indicating treatment group. The number of rows should equal the length of the response vector \code{y}.}

\item{n}{(For binomial data only) vector of integers specifying the number of subjects who have a particular value of the covariate vector. If the data is binary and all covariates are discrete, collapsing Bernoulli data into a binomial structure can make the slice sampler much faster.
The sum of \code{n} should be equal to \code{data.size}. The length of \code{n} should be equal to the number of rows of \code{x0}.}

\item{borrow.treat}{Logical value indicating whether the historical information is used to inform the treatment effect parameter. The default value is FALSE. If TRUE, the first column of the historical covariate matrix must be the treatment indicator.
If FALSE, the historical covariate matrix must NOT have the treatment indicator, since the historical data is assumed to be from the control group only.}

\item{historical}{List of historical dataset(s). East historical dataset is stored in a list which contains two \emph{named} elements: \code{y0} and \code{x0}.
\itemize{
\item \code{y0} is a vector of responses.
\item \code{x0} is a matrix of covariates. If \code{borrow.treat} is FALSE (the default), \code{x0} should NOT have the treatment indicator. Apart from missing the treatment indicator, \code{x0} should have the same set of covariates in the same order as \code{x}.
If \code{borrow.treat} is TRUE, \code{x0} should have the same set of covariates in the same order as \code{x}, where the first column of \code{x0} must be the treatment indicator.
}
For binomial data, an additional element \code{n0} is required.
\itemize{
\item \code{n0} is vector of integers specifying the number of subjects who have a particular value of the covariate vector.
The length of \code{n0} should be equal to the number of rows of \code{x0}.
}}

\item{prior.beta.var}{Vector of variances of the independent normal initial priors on \eqn{\beta} with mean zero. The length of the vector should be equal to the length of \eqn{\beta}. The default variance is 10.}

\item{prior.a0.shape1}{Vector of the first shape parameters of the independent beta priors for \eqn{a_0}. The length of the vector should be equal to the number of historical datasets. The default is a vector of one's.}

\item{prior.a0.shape2}{Vector of the second shape parameters of the independent beta priors for \eqn{a_0}. The length of the vector should be equal to the number of historical datasets. The default is a vector of one's.}

\item{a0.coefficients}{Vector of coefficients for \eqn{a_0} returned by the function \code{\link{normalizing.constant}}. This is necessary for estimating the normalizing constant for the normalized power prior. Does not apply if \code{data.type} is "Normal".}

\item{lower.limits}{Vector of lower limits for parameters to be used by the slice sampler. If \code{data.type} is "Normal", slice sampling is used for \eqn{a_0}, and the length of the vector should be equal to the number of historical datasets.
For all other data types, slice sampling is used for \eqn{\beta} and \eqn{a_0}. The first P+1 elements apply to the sampling of \eqn{\beta} and the rest apply to the sampling of \eqn{a_0}.
The length of the vector should be equal to the sum of the total number of parameters (i.e. P+1 where P is the number of covariates) and the number of historical datasets.
The default is -100 for \eqn{\beta} and 0 for \eqn{a_0} (may not be appropriate for all situations).}

\item{upper.limits}{Vector of upper limits for parameters to be used by the slice sampler. If \code{data.type} is "Normal", slice sampling is used for \eqn{a_0}, and the length of the vector should be equal to the number of historical datasets.
For all other data types, slice sampling is used for \eqn{\beta} and \eqn{a_0}. The first P+1 elements apply to the sampling of \eqn{\beta} and the rest apply to the sampling of \eqn{a_0}.
The length of the vector should be equal to the sum of the total number of parameters (i.e. P+1 where P is the number of covariates) and the number of historical datasets.
The default is 100 for \eqn{\beta} and 1 for \eqn{a_0}  (may not be appropriate for all situations).}

\item{slice.widths}{Vector of initial slice widths used by the slice sampler. If \code{data.type} is "Normal", slice sampling is used for \eqn{a_0}, and the length of the vector should be equal to the number of historical datasets.
For all other data types, slice sampling is used for \eqn{\beta} and \eqn{a_0}. The first P+1 elements apply to the sampling of \eqn{\beta} and the rest apply to the sampling of \eqn{a_0}.
The length of the vector should be equal to the sum of the total number of parameters (i.e. P+1 where P is the number of covariates) and the number of historical datasets.
The default is 0.1 for all parameter (may not be appropriate for all situations).}

\item{nMC}{Number of iterations (excluding burn-in samples) for the slice sampler or Gibbs sampler. The default is 10,000.}

\item{nBI}{Number of burn-in samples for the slice sampler or Gibbs sampler. The default is 250.}
}
\value{
The function returns a S3 object with a \code{summary} method. If \code{data.type} is "Normal", posterior samples of \eqn{\beta}, \eqn{\tau} and \eqn{a_0} are returned.
For all other data types, posterior samples of \eqn{\beta} and \eqn{a_0} are returned.
The first column of the matrix of posterior samples of \eqn{\beta} contains posterior samples of the intercept.
The second column contains posterior samples of \eqn{\beta_1}, the parameter for the treatment indicator.
}
\description{
Model fitting using normalized power priors for generalized linear models with random \eqn{a_0}
}
\details{
The user should use the function \code{\link{normalizing.constant}} to obtain \code{a0.coefficients} (does not apply if \code{data.type} is "Normal").

If \code{data.type} is "Normal", the response \eqn{y_i} is assumed to follow \eqn{N(x_i'\beta, \tau^{-1})} where \eqn{x_i} is the vector of covariates for subject \eqn{i}.
Historical datasets are assumed to have the same precision parameter as the current dataset for computational simplicity.
The initial prior for \eqn{\tau} is the Jeffery's prior, \eqn{\tau^{-1}}.
Independent normal priors with mean zero and variance \code{prior.beta.var} are used for \eqn{\beta} to ensure the propriety of the normalized power prior. Posterior samples for \eqn{\beta} and \eqn{\tau} are obtained through Gibbs sampling.
Independent beta(\code{prior.a0.shape1}, \code{prior.a0.shape1}) priors are used for \eqn{a_0}. Posterior samples for \eqn{a_0} are obtained through slice sampling.

For all other data types, posterior samples are obtained through slice sampling.
The default lower limits are -100 for \eqn{\beta} and 0 for \eqn{a_0}. The default upper limits
for the parameters are 100 for \eqn{\beta} and 1 for \eqn{a_0}. The default slice widths for the parameters are 0.1.
The defaults may not be appropriate for all situations, and the user can specify the appropriate limits
and slice width for each parameter.
}
\examples{

data.type <- "Bernoulli"
data.link <- "Logistic"

# Simulate current data
set.seed(1)
p <- 3
n_total <- 100
y <- rbinom(n_total,size=1,prob=0.6)
# The first column of x is the treatment indicator.
x <- cbind(rbinom(n_total,size=1,prob=0.5),
           matrix(rnorm(p*n_total),ncol=p,nrow=n_total))

# Simulate two historical datasets
# Note that x0 does not have the treatment indicator
historical <- list(list(y0=rbinom(n_total,size=1,prob=0.2),
                        x0=matrix(rnorm(p*n_total),ncol=p,nrow=n_total)),
                   list(y0=rbinom(n_total, size=1, prob=0.5),
                        x0=matrix(rnorm(p*n_total),ncol=p,nrow=n_total)))

# Please see function "normalizing.constant" for how to obtain a0.coefficients
# Here, suppose one-degree polynomial regression is chosen by the "normalizing.constant"
# function. The coefficients are obtained for the intercept, a0_1 and a0_2.
a0.coefficients <- c(1, 0.5, -1)

# Set parameters of the slice sampler
# The dimension is the number of columns of x plus 1 (intercept)
# plus the number of historical datasets
lower.limits <- c(rep(-100, 5), rep(0, 2))
upper.limits <- c(rep(100, 5), rep(1, 2))
slice.widths <- rep(0.1, 7)

nMC <- 500 # nMC should be larger in practice
nBI <- 100
result <- glm.random.a0(data.type=data.type, data.link=data.link, y=y, x=x,
                        historical=historical, a0.coefficients=a0.coefficients,
                        lower.limits=lower.limits, upper.limits=upper.limits,
                        slice.widths=slice.widths, nMC=nMC, nBI=nBI)
summary(result)

}
\references{
Neal, Radford M. Slice sampling. Ann. Statist. 31 (2003), no. 3, 705--767.
}
\seealso{
\code{\link{normalizing.constant}} and \code{\link{power.glm.random.a0}}
}
