#' print.CAISEr
#'
#' S3 method for printing _CAISEr_ objects (the output of
#' [run_experiment()]).
#'
#' @param x list object of class _CAISEr_
#'          (generated by [run_experiment()])
#' @param ... other parameters to be passed down to specific
#'            summary functions (currently unused)
#' @param digits the minimum number of significant digits to be used.
#'               See [print.default()].
#' @param right logical, indicating whether or not strings should be
#'              right-aligned.
#' @param breakrows logical, indicating whether to "widen" the output table by
#'                  placing the bottom half to the right of the top half.
#'
#' @examples
#' # Example using dummy algorithms and instances. See ?dummyalgo for details.
#' # In this case all instances are the same, so we expect all cases to return
#' # a percent difference of approx. phi.j = 1.0 and sample sizes of
#' # approx. n1 = 31, n2 = 87
#' algorithm1 <- list(FUN = "dummyalgo", alias = "algo1",
#'                    distribution.fun = "rnorm",
#'                    distribution.pars = list(mean = 10, sd = 1))
#' algorithm2 <- list(FUN = "dummyalgo", alias = "algo2",
#'                    distribution.fun = "rnorm",
#'                    distribution.pars = list(mean = 20, sd = 4))
#' algolist <- list(algorithm1, algorithm2)
#' instlist <- vector(100, mode = "list")
#' for (i in 1:100) instlist[[i]] <- list(FUN = "dummyinstance",
#'                                        alias = paste0("Inst. ", i))
#'
#' out <- run_experiment(Instance.list = instlist,
#'                       Algorithm.list = algolist,
#'                       power = 0.8,
#'                       d = 1,
#'                       sig.level = 0.01,
#'                       se.max = 0.05,
#'                       dif = "perc",
#'                       nmax   = 200,
#'                       seed   = 1234)
#' out
#'
#' @method print CAISEr
#' @export
#'
print.CAISEr <- function(x, ...,
                         digits = 6,
                         right = TRUE,
                         breakrows = FALSE)
{

  # Error checking
  assertthat::assert_that("CAISEr" %in% class(x))

  # ===========================================================================

  my.table <- x$data.summary

  if (breakrows){
    ninst      <- nrow(my.table)
    breakpoint <- ceiling(ninst / 2)
    tophalf    <- my.table[1:breakpoint, ]
    bottomhalf <- my.table[(breakpoint + 1):ninst, ]
    if(nrow(tophalf) > nrow(bottomhalf)) bottomhalf <- rbind(bottomhalf,
                                                             rep(NA, 5))
    my.table   <- cbind(tophalf,
                        `|` = rep("|", breakpoint),
                        bottomhalf)
  }

  # Print summary
  cat("#====================================")
  cat("\n Summary table of CAISEr object\n")
  print.data.frame(my.table,
                   digits = digits, right = right,
                   quote = FALSE, row.names = FALSE)
  cat("\n#====================================")
  invisible(my.table)
}
