\name{DunnTest}
\alias{DunnTest}
\alias{DunnTest.default}
\alias{DunnTest.formula}
\alias{print.DunnTest}
\title{Dunn's Test of Multiple Comparisons}
\description{
  Performs Dunn's test of multiple comparisons using rank sums.
}
\usage{
DunnTest(x, \dots)

\method{DunnTest}{default}(x, g,
         method = c("holm", "hochberg", "hommel", "bonferroni", "BH",
                    "BY", "fdr", "none"),
         alternative = c("two.sided", "less", "greater"),
         out.list = TRUE, \dots)

\method{DunnTest}{formula}(formula, data, subset, na.action, \dots)

\method{print}{DunnTest}(x, digits = getOption("digits"), ...)

}
\arguments{
  \item{x}{a numeric vector of data values, or a list of numeric data
    vectors.}
  \item{g}{a vector or factor object giving the group for the
    corresponding elements of \code{x}.  Ignored if \code{x} is a
    list.}
  \item{method}{the method for adjusting p-values for multiple comparisons. The function is calling \code{\link{p.adjust}} and this parameter is directly passed through.}
  \item{alternative}{a character string specifying the alternative hypothesis, must be one of \code{"two.sided"} (default), \code{"greater"} or \code{"less"}. You can specify just the initial letter.}
  \item{out.list}{logical, indicating if the results should be printed in list mode or as a square matrix. Default is list (TRUE).}
  \item{formula}{a formula of the form \code{lhs ~ rhs} where \code{lhs}
    gives the data values and \code{rhs} the corresponding groups.}
  \item{data}{an optional matrix or data frame (or similar: see
    \code{\link{model.frame}}) containing the variables in the
    formula \code{formula}.  By default the variables are taken from
    \code{environment(formula)}.}
  \item{subset}{an optional vector specifying a subset of observations
    to be used.}
  \item{na.action}{a function which indicates what should happen when
    the data contain \code{NA}s.  Defaults to
    \code{getOption("na.action")}.}
  \item{digits}{controls the number of digits to print.
}

  \item{\dots}{further arguments to be passed to or from methods.}
}
\details{
  \code{DunnTest} performs the post hoc pairwise multiple comparisons procedure appropriate to follow the rejection of a Kruskal-Wallis test. The Kruskal-Wallis test, being a non-parametric analog of the one-way ANOVA, is an omnibus test of the null hypothesis that none of k  groups stochastically dominate one another.
Dunn's test is constructed in part by summing jointly ranked data. The rank sum test, itself a non-parametric analog of the unpaired t-test, is possibly intuitive, but inappropriate as a post hoc pairwise test, because (1) it fails to retain the dependent ranking that produced the Kruskal-Wallis test statistic, and (2) it does not incorporate the pooled variance estimate implied by the null hypothesis of the Kruskal-Wallis test.

  If \code{x} is a list, its elements are taken as the samples to be
  compared, and hence have to be numeric data vectors.  In this case,
  \code{g} is ignored, and one can simply use \code{DunnTest(x)}
  to perform the test.  If the samples are not yet contained in a
  list, use \code{DunnTest(list(x, ...))}.

  Otherwise, \code{x} must be a numeric data vector, and \code{g} must
  be a vector or factor object of the same length as \code{x} giving
  the group for the corresponding elements of \code{x}.
}
\value{
  A list with class \code{"DunnTest"} containing the following components:
  \item{res}{an array containing the mean rank differencens and the according p-values}
}
\author{Andri Signorell <andri@signorell.net>, the interface is based on R-Core code}

\references{
Dunn, O. J. (1961) Multiple comparisons among means \emph{Journal of the American Statistical Association}, 56(293):52-64.

Dunn, O. J. (1964) Multiple comparisons using rank sums \emph{Technometrics}, 6(3):241-252.

}
\seealso{
  \code{\link{kruskal.test}}, \code{\link{wilcox.test}}, \code{\link{p.adjust}}
}
\examples{
## Hollander & Wolfe (1973), 116.
## Mucociliary efficiency from the rate of removal of dust in normal
##  subjects, subjects with obstructive airway disease, and subjects
##  with asbestosis.
x <- c(2.9, 3.0, 2.5, 2.6, 3.2) # normal subjects
y <- c(3.8, 2.7, 4.0, 2.4)      # with obstructive airway disease
z <- c(2.8, 3.4, 3.7, 2.2, 2.0) # with asbestosis
DunnTest(list(x, y, z))

## Equivalently,
x <- c(x, y, z)
g <- factor(rep(1:3, c(5, 4, 5)),
            labels = c("Normal subjects",
                       "Subjects with obstructive airway disease",
                       "Subjects with asbestosis"))

# do the kruskal.test first
kruskal.test(x, g)

# ...and the pairwise test afterwards
DunnTest(x, g)

## Formula interface.
boxplot(Ozone ~ Month, data = airquality)
DunnTest(Ozone ~ Month, data = airquality)
}
\keyword{htest}
