% Generated by roxygen2 (4.0.2): do not edit by hand
\name{plot.netdx}
\alias{plot.netdx}
\title{Plot Dynamic Network Model Diagnostics}
\usage{
\method{plot}{netdx}(x, type = "formation", method = "l", sim, stats,
  sim.lines = TRUE, sim.col, sim.lwd, sim.lty = 1, mean.line = TRUE,
  mean.smooth = FALSE, mean.col, mean.lwd = 2, mean.lty = 1,
  qnts = FALSE, qnts.col, qnts.alpha, targ.line = TRUE, targ.col,
  targ.lwd = 2, targ.lty = 2, plots.joined, plot.leg = TRUE, ...)
}
\arguments{
\item{x}{An \code{EpiModel} object of class \code{netdx}.}

\item{type}{Plot type, with options of \code{"formation"} for network
model formation statistics, \code{"duration"} for dissolution model
statistics for average edge duration, or \code{"dissolution"} for
dissolution model statistics for proportion of ties dissolved per time
step.}

\item{method}{Plot method, with options of \code{"l"} for line plots and
\code{"b"} for boxplots.}

\item{sim}{Vector of simulation numbers to plot, with the default to plot
all simulations.}

\item{stats}{Network statistics to plot, among those specified in the call
to \code{\link{netdx}}, with the default to plot all statistics
contained in the object.}

\item{sim.lines}{If \code{TRUE}, plot individual simulation lines.}

\item{sim.col}{Vector of standard R colors for individual simulation lines,
with default colors based on \code{RColorBrewer} color palettes.}

\item{sim.lwd}{Line width for individual simulation lines, with defaults based
on number of simulations (more simulations results in thinner lines).}

\item{sim.lty}{Line type for the individual simulation lines.}

\item{mean.line}{If \code{TRUE}, plot mean of simulations across time.}

\item{mean.smooth}{If \code{TRUE}, use a lowess smoother on the mean line.}

\item{mean.col}{A vector of any standard R color format for mean lines.}

\item{mean.lwd}{Line width for mean lines.}

\item{mean.lty}{Line type for mean lines.}

\item{qnts}{If numeric, plot polygon of simulation quantiles based on the
range implied by the argument (0.5 is the middle 50% of the data).
If \code{FALSE}, suppress polygon from plot.}

\item{qnts.col}{Vector of any standard R color format for polygons.}

\item{qnts.alpha}{Transparency level for quantile polygons, where 0 =
transparent and 1 = opaque (see \code{\link{transco}}).}

\item{targ.line}{If \code{TRUE}, plot target or expected value line for
the statistic of interest.}

\item{targ.col}{Vector of standard R colors for target statistic lines, with
default colors based on \code{RColorBrewer} color palettes.}

\item{targ.lwd}{Line Width for the line showing the target statistic values.}

\item{targ.lty}{Line Type for the line showing the target statistic values.}

\item{plots.joined}{If \code{TRUE}, combine all target statistics in one plot,
otherwise use one plot window per target statistic.}

\item{plot.leg}{If \code{TRUE}, show legend (only if \code{plots.joined=TRUE})}

\item{...}{additional arguments passed to main plot windows.}
}
\description{
Plots dynamic network model diagnostics calculated in
             \code{netdx}.
}
\details{
The plot function for \code{netdx} objects will generate plots of two types of
model diagnostic statistics that run as part of the diagnostic tools within
that function. The \code{formation} plot shows the summary statistics
requested in \code{nwstats.formula}, where the default includes those
statistics in the network model formation formula specified in the original
call to \code{\link{netest}}.

The \code{duration} plot shows the average age of existing edges at each time
step, up until the maximum time step requested. This is calculated with the
\code{\link{edgelist_meanage}} function. The age is used as an estimator of
the average duration of edges in the equilibrium state.

The \code{dissolution} plot shows the proportion of the extant ties that are
dissolved at each time step, up until the maximum time step requested.
Typically the proportion of ties that are dissolved is the reciprocal of the
mean relational duration. This plot thus contains similar information to that
in the duration plot, but should reach its expected value more quickly, since
it is not subject to censoring.

The \code{plots.joined} argument will control whether the statistics
in the \code{formation} plot are joined in one plot or plotted separately.
The default is based on the number of network statistics requested. The
layout of the separate plots within the larger plot window is also based on
the number of statistics.
}
\examples{
\dontrun{
# Network initialization and model parameterization
nw <- network.initialize(100, directed = FALSE)
nw <- set.vertex.attribute(nw, "sex", rbinom(100, 1, 0.5))
formation <- ~ edges + nodematch("sex")
dissolution <- ~ offset(edges)
target.stats <- c(50, 40)
coef.diss <- dissolution_coefs(dissolution, duration = 50)

# Estimate the model
est <- netest(nw, formation, dissolution,
              target.stats, coef.diss, verbose = FALSE)

# Static diagnostics
dx1 <- netdx(est, nsims = 1e4, dynamic = FALSE,
             nwstats.formula = ~ edges + meandeg + concurrent +
                                 nodefactor("sex", base = 0) +
                                 nodematch("sex"))
dx1

# Only formation diagnostics are available to plot
plot(dx1, stats = "edges")
plot(dx1, stats = "edges", method = "b", col = "seagreen3")
plot(dx1, stats = c("nodefactor.sex.0", "nodefactor.sex.1"), method = "b")

# Dynamic diagnostics
dx2 <- netdx(est, nsims = 10, nsteps = 500,
             nwstats.formula = ~ edges + meandeg + concurrent +
                                 nodefactor("sex", base = 0) +
                                 nodematch("sex"))
dx2

# Formation statistics plots, joined and separate
plot(dx2)
plot(dx2, type = "formation", plots.joined = TRUE)
plot(dx2, type = "formation", sim = 1, plots.joined = TRUE)
plot(dx2, type = "formation", plots.joined = FALSE,
     stats = c("edges", "concurrent"))
plot(dx2, type = "formation", stats = "nodefactor.sex.0",
     sim = 1, sim.lwd = 5, sim.col = "darkmagenta")

plot(dx2, method = "b", col = "bisque")
plot(dx2, method = "b", stats = "meandeg", col = "dodgerblue")

# Duration statistics plot
plot(dx2, type = "duration")
plot(dx2, type = "duration", sim = 10,
     sim.col = "steelblue", sim.lwd = 3,
     targ.lty = 1, targ.lwd = 0.5)

# Dissolution statistics plot
plot(dx2, type = "dissolution")
plot(dx2, type = "dissolution", method = "b", col = "pink1")
}
}
\seealso{
\code{\link{netdx}}
}
\keyword{plot}

