
#' Function to control estimation of smooth offset
#' 
#' @param k_min maximal number of k in s()
#' @param rule which rule to use in approx() of the response before calculating the 
#' global mean, rule=1 means no extrapolation, rule=2 means to extrapolate the 
#' closest non-missing value, see \code{\link[stats]{approx}}
#' @param silent print error messages of model fit?
#' @param cyclic defaults to FALSE, if TRUE cyclic splines are used
#' @param knots arguments knots passed to \code{\link[mgcv]{gam}}

#' @export 
o_control <- function(k_min=20, rule=2, silent=TRUE, cyclic=FALSE, knots=NULL) { 
  RET <- list(k_min=k_min, rule=rule, silent=silent, cyclic=cyclic, knots=knots)
  class(RET) <- c("offset_control")
  RET
}



#' Function to truncate time in functional data 
#' 
#' @param funVar names of functional variables that should be truncated
#' @param time name of time variable
#' @param newtime new time vector that should be used. Must be part of the old time-line.
#' @param data list containing all the data
#' @note All variables that are not part if \code{funVar}, or \code{time}
#' are simply copied into the new data list
#' @return A list with the data containing all variables of the original dataset
#' with the variables of \code{funVar} truncated according to \code{newtime}.
#' @examples
#' if(require(fda)){
#'   dat <- fda::growth
#'   dat$hgtm <- t(dat$hgtm[,1:10])
#'   dat$hgtf <- t(dat$hgtf[,1:10])
#'   
#'   ## only use time-points 1:16 of variable age
#'   datTr <- truncateTime(funVar=c("hgtm","hgtf"), time="age", newtime=1:16, data=dat)
#'   
#'   \dontrun{
#'   par(mfrow=c(1,2))
#'   with(dat, funplot(age, hgtm, main="Original data"))
#'   with(datTr, funplot(age, hgtm, main="Yearly data"))
#'   par(mfrow=c(1,1))   
#'   }
#' }
#' @export 
truncateTime <- function(funVar, time, newtime, data){
  
  stopifnot(c(funVar, time) %in% names(data))
  stopifnot(newtime %in% data[[time]])
  
  ret <- data
  ret[[time]] <- newtime
  
  for(i in 1:length(funVar)){
    ret[[funVar[i]]] <- ret[[funVar[i]]][ , data[[time]] %in% newtime] 
  }  
  rm(data)
  return(ret)  
}

#' Plot functional data with linear interpolation of missing values 
#' 
#' @param x optional, time-vector for plotting 
#' @param y matrix of functional data with functions in rows and measured times in columns
#' @param id defaults to NULL for y matrix, is id-variables for y in long format
#' @param rug logical. Should rugs be plotted? Defaults to TRUE.
#' @param ... further arguments passed to \code{\link[graphics]{matplot}}.
#' 
#' @details All observations are marked by a small cross (\code{pch=3}).
#' Missing values are imputed by linear interpolation. Parts that are
#' interpolated are plotted by dotted lines, parts with non-missing values as solid lines.
#' @examples 
#' \dontrun{
#' ### examples for regular data in wide format
#' data(viscosity)
#' with(viscosity, funplot(timeAll, visAll, pch=20))
#' if(require(fda)){
#'   with(fda::growth, funplot(age, t(hgtm)))
#' }
#' }
#' @export
funplot <- function(x, y, id=NULL, rug=TRUE, ...){
  
  ### Get further arguments passed to the matplot-functions
  dots <- list(...)
  
  getArguments <- function(x, dots=dots){
    if(any(names(dots) %in% names(x))){
      dots[names(dots) %in% names(x)]
    }else list()
  }
  
  plotWithArgs <- function(plotFun, args, myargs){        
    args <- c(myargs[!names(myargs) %in% names(args)], args)        
    do.call(plotFun, args)            
  }
  
  argsMatplot  <- getArguments(x=c(formals(graphics::matplot), par(), main="", sub=""), dots=dots)
  argsPlot <- getArguments(x=c(formals(graphics::plot.default), par()), dots=dots)
  
  
  if(is.null(id)){
    
    # Deal with missing values: interpolate data
    if (missing(x)) {
      if (missing(y)) 
        stop("must specify at least one of 'x' and 'y'")
      else{
        x <- seq_len(NCOL(y))
        xlabel <- "index"
      } 
    } else xlabel <- deparse(substitute(x))
    
    ylabel <- if (!missing(y)) 
      deparse(substitute(y))
    
    time <- x
    
    stopifnot(length(x) == ncol(y))
    
    # Checke weather there are at least two values per row for the interpolation
    atLeast2values <- apply(y, 1, function(x) sum(is.na(x)) < length(x)-1 )
    if(any(!atLeast2values)) warning(sum(!atLeast2values), " rows contain less than 2 non-missing values.")  
    
    # Linear interpolation per row
    yint <- matrix(NA, ncol=ncol(y), nrow=nrow(y))
    yint[atLeast2values, ] <- t(apply(y[atLeast2values, ], 1, function(x) approx(time, x, xout=time)$y))
    
    # Plot the observed points
    plotWithArgs(matplot, args=argsMatplot, 
                 myargs=list(x=time, y=t(y), xlab=xlabel, ylab=ylabel, type="p", pch=3) )
    
    # Plot solid lines for parts of the function without missing values
    plotWithArgs(matplot, args=argsMatplot, 
                 myargs=list(x=time, y=t(y), type="l", lty=1, add=TRUE) )
    
    # Plot dotted lines for parts of the function without missing values
    plotWithArgs(matplot, args=argsMatplot, 
                 myargs=list(x=time, y=t(yint), type="l", lty=3, add=TRUE) )
    
    if(rug) rug(time, 0.01)
  
  }else{
    
    stopifnot(length(x)==length(y) & length(y)==length(id))
    
    idOrig <- id
    for(i in 1:length(unique(idOrig))){
      id[idOrig==unique(idOrig)[i]] <- i
    }
    
    xlabel <- deparse(substitute(x))
    ylabel <- deparse(substitute(y))
    
    # there should be no mising values in long format
    temp <- data.frame(id, y, x) # dim(temp)
    temp <- na.omit(temp)   
    # order values of temp
    temp <- temp[order(temp$id, temp$x),] 
    id <- temp$id
    x <- temp$x
    y <- temp$y
    rm(temp)
    
    # Plot the observed points
    if(!"add" %in% names(dots)){
      if(is.null(argsPlot$ylim)) argsPlot$ylim <- range(y, na.rm=TRUE) 
      plotWithArgs(plot, args=argsPlot, 
                   myargs=list(x=x[id==1], y=y[id==1], xlab=xlabel, ylab=ylabel, type="p", pch=3,
                               ylim=range(y, na.rm=TRUE), xlim=range(x, na.rm=TRUE)) )
    }

    if("col" %in% names(dots)){
      col <- dots$col
      argsPlot$col <- NULL
    }else{
      col <- 1:6
    }
    
    if(length(col)<length(unique(id))){
      col <- rep(col, l=length(unique(id)))
    }
    
    for(i in unique(id)){
      plotWithArgs(points, args=argsPlot, 
                   myargs=list(x=x[id==i], y=y[id==i], xlab=xlabel, ylab=ylabel, type="p", pch=3,
                               col=col[i]) )
      plotWithArgs(lines, args=argsPlot, 
                   myargs=list(x=x[id==i], y=y[id==i], xlab=xlabel, ylab=ylabel, col=col[i]) )
    }
    
    if(rug) rug(x, 0.01)
    
    
  }
  
#   matplot(time, t(y), xlab=xlabel, ylab="", type="p", pch=3)
#   matplot(time, t(y), type="l", pch=1, lty=1, add=TRUE)
#   matplot(time, t(yint), type="l", pch=1, lty=3, add=TRUE)  
}



#####################################################################################

#' @rdname plot.FDboost
#' @export
#' 
### function to plot the observed response and the predicted values of a model
plotPredicted <- function(x, subset=NULL, posLegend="topleft", lwdObs=1, lwdPred=1, ...){
  
  stopifnot("FDboost" %in% class(x))
  
  if(is.null(x$id)){
    if(is.null(subset)) subset <- 1:x$ydim[1]
    response <- matrix(x$response, nrow=x$ydim[1], ncol=x$ydim[2])[subset, , drop=FALSE] 
    pred <- fitted(x)[subset, , drop=FALSE]
    pred[is.na(response)] <- NA
  }else{
    if(is.null(subset)) subset <- unique(x$id)
    response <- x$response[x$id %in% subset] 
    pred <- fitted(x)[x$id %in% subset]
    pred[is.na(response)] <- NA
  }

  ylim <- range(response, pred, na.rm = TRUE)
  
  if(length(x$yind)>1){
    # Observed values
    funplot(x$yind, response, id=x$id, pch=1, ylim=ylim, lty=3, 
            ylab=x$yname, xlab=attr(x$yind, "nameyind"), lwd=lwdObs, ...)
    funplot(x$yind, pred, id=x$id, pch=2, lwd=lwdPred, add=TRUE, ...)
    # predicted values
    legend(posLegend, legend=c("observed","predicted"), col=1, pch=1:2)  
  }else{
    plot(response, pred, ylab="predicted", xlab="observed", ...)
    abline(0,1)
  }  
}


#####################################################################################

#' @rdname plot.FDboost
#' @export
#' 
### function to plot the residuals
plotResiduals <- function(x, subset=NULL, posLegend="topleft", ...){
  
  stopifnot("FDboost" %in% class(x))
  
  if(is.null(x$id)){
    if(is.null(subset)) subset <- 1:x$ydim[1]
    response <- matrix(x$response, nrow=x$ydim[1], ncol=x$ydim[2])[subset, , drop=FALSE] 
    pred <- fitted(x)[subset, , drop=FALSE]
    pred[is.na(response)] <- NA
  }else{
    if(is.null(subset)) subset <- unique(x$id)
    response <- x$response[x$id %in% subset] 
    pred <- fitted(x)[x$id %in% subset]
    pred[is.na(response)] <- NA
  }
  
  # Observed - predicted values
  if(length(x$yind)>1){
    funplot(x$yind, response-pred, id=x$id, ylab=x$yname, xlab=attr(x$yind, "nameyind"), ...) 
  }else{
    plot(response, response-pred, ylab="residuals", xlab="observed", ...)
    #abline(h=0)
  }

}


#####################################################################################
### Goodness of fit

# function to get y, yhat and time
getYYhatTime <- function(object, breaks=object$yind){
  
  y <- matrix(object$response, nrow=object$ydim[1], ncol=object$ydim[2]) 
  time <- object$yind
  
  ### use the original time variable
  if(all(breaks==object$yind)){ 
    yhat <- matrix(object$fitted(), nrow=object$ydim[1], ncol=object$ydim[2]) 
  }else{ ### use a time variables according to breaks
    if(length(breaks)==1){ # length of equidistant time-points
      time <- seq( min(object$yind), max(object$yind), l=breaks)      
    }else{ # time-points ot be evaluated
      time <- breaks
    }
    # Interpolate observed values
    yInter <- t(apply(y, 1, function(x) approx(object$yind, x, xout=time)$y))
    # Get dataframe to predict values at time
    newdata <- list()
    for(j in 1:length(object$baselearner)){
      datVarj <- object$baselearner[[j]]$get_data()
      if(grepl("bconcurrent", names(object$baselearner)[j])){
        datVarj <- t(apply(datVarj[[1]], 1, function(x) approx(object$yind, x, xout=time)$y))
        datVarj <- list(datVarj)
      } 
      names(datVarj) <- names(object$baselearner[[j]]$get_data())
      newdata <- c(newdata, datVarj)
    }
    newdata[[attr(object$yind, "nameyind")]] <- time 
    yhatInter <- predict(object, newdata=newdata)
    
    y <- yInter
    yhat <- yhatInter
  }
  
  return(list(y=y, yhat=yhat, time=time))
}



#' Functional R-squared
#' 
#' Calculates the functional R-squared for a fitted FDboost-object
#' 
#' @param object fitted FDboost-object
#' @param overTime per default the functional R-squared is calculated over time
#' if \code{overTime=FALSE}, the R-squared is calculated per curve
#' @param breaks an optional vector or number giving the time-points at which the model is evaluated.
#' Can be specified as number of equidistant time-points or as vector of time-points.
#' Defaults to the index of the response in the model.
#' @param global logical. defaults to \code{FALSE}, 
#' if TRUE the global R-squared like in a normal linear model is calculated
#' @param ... currently not used
#' 
#' @note \code{breaks} cannot be changed in the case the \code{bsignal()} 
#' is used over the same domain
#' as the response! In that case you would have to rename the index of the response or that 
#' of the covariates.
#' 
#' @details \code{breaks} should be set to some grid, if there are many
#' missing values or time-points with very few observations in the dataset.
#' Otherwise at these points of t the variance will be almost 0 
#' (or even 0 if there is only one observation at a time-point),
#' and then the prediction by the local means \eqn{\mu(t)} is locally very good.
#' The observations are interpolated linearly if necessary.
#' 
#' Formula to calculate R-squared over time, \code{overTime=TRUE}: \cr
#' \eqn{R^2(t) = 1 - \sum_{i}( Y_i(t) - \hat{Y}_i(t))^2 /  \sum_{i}( Y_i(t) - \bar{Y}(t) )^2 } 
#' 
#' Formula to calculate R-squared over subjects, \code{overTime=FALSE}: \cr
#' \eqn{R^2_i = 1 - \int (Y_i(t) - \hat{Y}_i(t))^2 dt /  \int (Y_i(t) - \bar{Y}_i )^2 dt }
#' 
#' @references Ramsay, J., Silverman, B. (2006). Functional data analysis. 
#' Wiley Online Library. chapter 16.3
#' 
#' @return Returns a vector with the calculated R-squared and some extra information in attributes.
#' 
#' @export
funRsquared <- function(object, overTime=TRUE, breaks=object$yind, global=FALSE, ...){
  
  if(length(object$yind)<2 | !is.null(object$id)){
    y <- object$response
    yhat <- object$fitted()
    time <- object$yind
    id <- object$id
    if(is.null(id)) id <- 1:length(y)
    if(overTime & !global) {
      overTime <- FALSE
      message("For scalar or irregualr response the functional R-squared cannot be computed over time.")
    }
    if(length(object$yind)<2) global <- TRUE
  }else{
    # Get y, yhat and time of the model fit
    temp <- getYYhatTime(object=object, breaks=breaks)
    y <- temp$y
    yhat <- temp$yhat
    time <- temp$time
    
    stopifnot(dim(y)==dim(yhat))
    stopifnot(dim(y)[2]==length(time))
    
  }
  
  if(global){
    ret <- 1 - ( sum((y-yhat)^2, na.rm=TRUE)  / sum( (y-mean(y, na.rm=TRUE))^2, na.rm=TRUE) )
    attr(ret, "name") <- "global R-squared"
    return(ret)
  }
    
  ### for each time-point t 
  if(overTime){ 
    # Mean function over time (matrix containing the mean in each t in the whole column)
    mut <- matrix(colMeans(y, na.rm=TRUE), nrow=nrow(y), ncol=ncol(y), byrow=TRUE)
    
    # numerator cannot be 0
    num <- colSums((y - mut)^2, na.rm=TRUE)
    num[round(num, 2)==0] <- NA
    ret <- 1 - ( colSums((y - yhat)^2, na.rm=TRUE) / num ) # over t 
    
    # Set values of R-squared to NA if too many values are missing
    ret[apply(y, 2, function(x) sum(is.na(x))>0.5*length(x) )] <- NA
    
    attr(ret, "name") <- "R-squared over time"
    attr(ret, "time") <- time
    attr(ret, "missings") <- apply(y, 2, function(x) sum(is.na(x))/length(x) )
    
  }else{ ### for each subject i
    if(length(object$yind)<2 | !is.null(object$id)){
      # Mean for each subject
      mut <- tapply(y, id, mean, na.rm=TRUE  )[id]
      # numerator cannot be 0
      num <- tapply((y - mut)^2, id, mean, na.rm=TRUE )[id]
      num[round(num, 2)==0] <- NA 
      ret <- 1 - tapply((y - yhat)^2 /num, id, mean, na.rm=TRUE  )
      attr(ret, "name") <- "MSE over subjects"              
    }else{
      # Mean for each subject
      mut <- matrix(rowMeans(y, na.rm=TRUE), nrow=nrow(y), ncol=ncol(y), byrow=TRUE)
      # numerator cannot be 0
      num <- rowSums((y - mut)^2, na.rm=TRUE)
      num[round(num, 2)==0] <- NA    
      ret <- 1 - ( rowSums((y - yhat)^2, na.rm=TRUE) /  num ) # over subjects
      
      attr(ret, "name") <- "R-squared over subjects"
      attr(ret, "missings") <- apply(y, 1, function(x) sum(is.na(x))/length(x)) 
    }
  }
  return(ret)
}


# R2t <- funRsqrt(mod)
# plot(attr(R2t, "time"), R2t, ylim=c(0,1), type="b")
# points(attr(R2t, "time"), attr(R2t, "missings"), col=2, type="b")
# abline(h=c(0, 1))
# 
# R2i <- funRsqrt(mod, overTime=FALSE)
# plot(R2i, type="b", ylim=c(0,1))
# points(attr(R2i, "missings"), col=2, type="b")
# abline(h=c(0, 1))

#' Functional MSE
#' 
#' Calculates the functional MSE for a fitted FDboost-object
#' 
#' @param object fitted FDboost-object
#' @param overTime per default the functional R-squared is calculated over time
#' if \code{overTime=FALSE}, the R-squared is calculated per curve
#' @param breaks an optional vector or number giving the time-points at which the model is evaluated.
#' Can be specified as number of equidistant time-points or as vector of time-points.
#' Defaults to the index of the response in the model.
#' @param global logical. defaults to \code{FALSE}, 
#' if TRUE the global R-squared like in a normal linear model is calculated
#' @param relative logical. defaults to \code{FALSE}. If \code{TRUE} the MSE is standardized
#' by the global variance of the response \cr
#' \eqn{ n^{-1} \int  \sum_i (Y_i(t) - \bar{Y})^2 dt \approx  G^{-1} n^{-1} \sum_g \sum_i (Y_i(t_g) - \bar{Y})^2 } 
#' @param root take the square root of the MSE
#' @param ... currently not used
#' 
#' @note \code{breaks} cannot be changed in the case the \code{bsignal()} 
#' is used over the same domain
#' as the response! In that case you would have to rename the index of the response or that 
#' of the covariates.
#' 
#' @details 
#' Formula to calculate MSE over time, \code{overTime=TRUE}: \cr
#' \eqn{ MSE(t) = n^{-1} \sum_i (Y_i(t) - \hat{Y}_i(t))^2 } 
#' 
#' Formula to calculate MSE over subjects, \code{overTime=FALSE}: \cr
#' \eqn{ MSE_i = \int (Y_i(t) - \hat{Y}_i(t))^2 dt  \approx G^{-1} \sum_g (Y_i(t_g) - \hat{Y}_i(t_g))^2}
#' 
#' @return Returns a vector with the calculated MSE and some extra information in attributes.
#' 
#' @export
funMSE <- function(object, overTime=TRUE, breaks=object$yind, global=FALSE, 
                   relative=FALSE, root=FALSE, ...){
  
  if(length(object$yind)<2 | !is.null(object$id)){
    y <- object$response
    yhat <- object$fitted()
    time <- object$yind
    id <- object$id
    if(is.null(id)) id <- 1:length(y)
    if(overTime & !global) {
      overTime <- FALSE
      message("For scalar or irregualr response the functional MSE cannot be computed over time.")
    }
  }else{
    # Get y, yhat and time of the model fit
    temp <- getYYhatTime(object=object, breaks=breaks)
    y <- temp$y
    yhat <- temp$yhat
    time <- temp$time
  }
  
  if(global){
    ret <- mean((y-yhat)^2, na.rm=TRUE)
    attr(ret, "name") <- "global MSE"
  }else{
    ### for each time-point t 
    if(overTime){     
      ret <- colMeans((y - yhat)^2, na.rm=TRUE)   
      attr(ret, "name") <- "MSE over time"
      attr(ret, "time") <- time
      attr(ret, "missings") <- apply(y, 2, function(x) sum(is.na(x))/length(x))     
    }else{ 
      ### for each subject i
      if(length(object$yind)<2 | !is.null(object$id)){
        ret <- tapply((y - yhat)^2, id, mean, na.rm=TRUE  )
        attr(ret, "name") <- "MSE over subjects"              
      }else{
        ret <- rowMeans((y - yhat)^2, na.rm=TRUE)
        attr(ret, "name") <- "MSE over subjects"      
        attr(ret, "missings") <- apply(y, 1, function(x) sum(is.na(x))/length(x))
      }
    }    
  }
  
  if(relative){
    variY <- mean( (y - mean(y, na.rm=TRUE))^2, na.rm=TRUE ) # global variance of y
    ret <- ret / variY
    attr(ret, "name") <- paste("relative", attr(ret, "name"))
  }
  
  if(root){
    ret <- sqrt(ret)
    attr(ret, "name") <- paste("root", attr(ret, "name"))
  }
  
  return(ret)
}


#' Functional MRD
#' 
#' Calculates the functional MRD for a fitted FDboost-object
#' 
#' @param object fitted FDboost-object with regular response
#' @param overTime per default the functional MRD is calculated over time
#' if \code{overTime=FALSE}, the MRD is calculated per curve
#' @param breaks an optional vector or number giving the time-points at which the model is evaluated.
#' Can be specified as number of equidistant time-points or as vector of time-points.
#' Defaults to the index of the response in the model.
#' @param global logical. defaults to \code{FALSE}, 
#' if TRUE the global MRD like in a normal linear model is calculated
#' @param ... currently not used
#' 
#' @note \code{breaks} cannot be changed in the case the \code{bsignal()} 
#' is used over the same domain
#' as the response! In that case you would have to rename the index of the response or that 
#' of the covariates.
#' 
#' @details 
#' Formula to calculate MRD over time, \code{overTime=TRUE}: \cr
#' \eqn{ MRD(t) = n^{-1} \sum_i |(Y_i(t) - \hat{Y}_i(t))^2|/|Y_i(t)| } 
#' 
#' Formula to calculate MRD over subjects, \code{overTime=FALSE}: \cr
#' \eqn{ MRD_i = \int |(Y_i(t) - \hat{Y}_i(t))^2|/|Y_i(t)| dt  \approx G^{-1} \sum_g |(Y_i(t_g) - \hat{Y}_i(t_g))^2| / |Y_i(t)|}
#' 
#' @return Returns a vector with the calculated MRD and some extra information in attributes.
#' 
#' @export
funMRD <- function(object, overTime=TRUE, breaks=object$yind, global=FALSE,  ...){
  
  if(length(object$yind)<2 | !is.null(object$id)){
    y <- object$response
    yhat <- object$fitted()
    time <- object$yind
    id <- object$id
    if(is.null(id)) id <- 1:length(y)
    if(overTime & !global) {
      overTime <- FALSE
      message("For scalar or irregualr response the functional MRD cannot be computed over time.")
    }
  }else{
    # Get y, yhat and time of the model fit
    temp <- getYYhatTime(object=object, breaks=breaks)
    y <- temp$y
    yhat <- temp$yhat
    time <- temp$time
  }

  # You cannot use observations that are 0, so set them to NA
  y1 <- y
  y1[ round(y1, 1) == 0 ] <- NA
  
  if(global){
    ret <- mean( abs((y1 - yhat) / y1), na.rm=TRUE  )
    attr(ret, "name") <- "global MRD"
  }else{
    ### for each time-point t 
    if(overTime){     
      ret <- colMeans( abs((y1 - yhat) / y1), na.rm=TRUE  )   
      attr(ret, "name") <- "MRD over time"
      attr(ret, "time") <- time
      attr(ret, "missings") <- apply(y, 2, function(x) sum(is.na(x))/length(x))     
    }else{ 
      ### for each subject i
      if(length(object$yind)<2 | !is.null(object$id)){
        ret <- tapply( abs((y1 - yhat) / y1), id, mean, na.rm=TRUE  )
        attr(ret, "name") <- "MRD over subjects"              
      }else{
        ret <- rowMeans( abs((y1 - yhat) / y1), na.rm=TRUE  )
        attr(ret, "name") <- "MRD over subjects"      
        attr(ret, "missings") <- apply(y, 1, function(x) sum(is.na(x))/length(x))
      }
    }    
  }
  
  return(ret)
}


