#' Save the chat history of a Generative AI model
#'
#' This function saves the chat history list of a Generative AI model as a CSV, JSON, or TXT file.
#'
#' @param history A list containing all the messages sent by the user and model in a chat session.
#' @param file.name A character string representing the name of the output file for the chat history.
#' @param format A character string representing the format of the output file for the chat history.
#' It can be either \code{"csv"}, \code{"json"}, or \code{"txt"}.
#'
#' @return If successful, a chat history file with a specific format is exported to the working directory.
#'
#' @details This function saves the chat history list of a Generative AI model as a CSV, JSON, or TXT file.
#' To save a chat history, the chat history list must follow the list formats generated by the
#' function \code{\link{chat.setup}}.
#'
#' @seealso
#' \href{https://genai.gd.edu.kg/r/documentation/}{GenAI - R Documentation}
#'
#' @examples
#' \dontrun{
#'  # Get available models
#'  models = available.models()
#'
#'  # Connect to the model, replace API_KEY with your api key
#'  google.model = connect.genai("google",
#'                               models$google$model[1],
#'                               models$google$version[1],
#'                               "API_KEY",
#'                               FALSE)
#'
#'  # Connect to the model, replace API_KEY with your api key
#'  openai.model = connect.genai("openai",
#'                               models$openai$model[1],
#'                               models$openai$version[1],
#'                               "API_KEY",
#'                               FALSE)
#'  # Setup an empty chat history
#'  google.history = chat.setup(google.model)
#'  openai.history = chat.setup(openai.model)
#'
#'  # Start chat
#'  temperature = 0.9
#'  prompt = "Write a story about a magic backpack in about 100 words."
#'  google.history = chat(google.model,
#'                        temperature,
#'                        google.history,
#'                        prompt)
#'
#'  # Get the most recent message generated by the model
#'  cat(chat.recent(google.model, google.history))
#'
#'  openai.history = chat(openai.model,
#'                        temperature,
#'                        openai.history,
#'                        prompt)
#'
#'  # Get the most recent message generated by the model
#'  cat(chat.recent(openai.model, openai.history))
#'
#'  prompt = "What is the word count of the story you just wrote?"
#'  google.history = chat(google.model,
#'                        temperature,
#'                        google.history,
#'                        prompt)
#'
#'  # Get the most recent message generated by the model
#'  cat(chat.recent(google.model, google.history))
#'
#'  openai.history = chat(openai.model,
#'                        temperature,
#'                        openai.history,
#'                        prompt)
#'
#'  # Get the most recent message generated by the model
#'  cat(chat.recent(openai.model, openai.history))
#'
#'  # Save chat history
#'  chat.save(google.history,
#'            "google_history",
#'            "json")
#'
#'  chat.save(openai.history,
#'            "openai_history",
#'            "json")
#' }
#'
#' @export
chat.save = function(history,
                     file.name,
                     format) {
  format = match.arg(format, c("csv", "txt", "json"), several.ok = FALSE)
  switch (format,
          csv = {
            sink(paste0(file.name, ".", format))
            print(history)
            sink()
          },
          txt = {
            sink(paste0(file.name, ".", format))
            print(history)
            sink()
          },
          json = {
            write(jsonlite::toJSON(history, pretty = TRUE),
                  paste0(file.name, ".", format))
          })
}
