# Functions for predicting study mean responses
# Author: Hugo Pedder
# Date created: 2018-09-10


#' Predict responses over time in a given population based on MBNMA time-course
#' models
#'
#' Used to predict responses over time for different treatments or to predict
#' the results of a new study. For MBNMA models that include consistency
#' relative effects on time-course parameters, this is calculated by combining
#' relative treatment effects with a given reference treatment response
#' (specific to the population of interest).
#'
#' @param object An S3 object of class `"mbnma"` generated by running
#'   a time-course MBNMA model
#' @param times A sequence of positive numbers indicating which time points to
#'   predict mean responses for
#' @param E0 An object to indicate the value(s) to use for the response at time = 0
#'   in the prediction. This can take a number of different formats depending
#'   on how it will be used/calculated. The default is `0` but this may lead
#'   to non-sensical predictions.
#'   * `numeric()` A single numeric value representing the deterministic response at time = 0,
#'   given.
#'   * `character()` A single string representing a stochastic distribution for the response
#'   at time = 0. This is specified as a random number generator
#'   (RNG) given as a string, and can take any RNG distribution for which a function exists
#'   in R. For example: `"rnorm(n, 7, 0.5)"`.
#' @param treats A character vector of treatment names or a numeric vector of treatment codes (as coded in `mbnma`)
#'   that indicate which treatments to calculate predictions for. If left `NULL``
#'   then predictions will be calculated for all treatments.
#' @param ref.resp An object to indicate the value(s) to use for the reference treatment response in MBNMA models
#'   in which the reference treatment response is not estimated within the model (i.e. those that model any time-
#'   course paramters using `pool="rel"`). This can take a number of different formats depending
#'   on how it will be used/calculated. There are two approaches for this:
#'
#'   1. The reference response can be estimated from a dataset of studies investigating the reference
#'   treatment using meta-analysis. This dataset could be a set of observational
#'   studies that are specific to the population on which to make
#'   predictions, or it could be a subset of the study arms within the MBNMA dataset
#'   that investigate the reference treatment. The data should be provided to `ref.resp` as a
#'   `data.frame()` containing the data in long format (one row per observation). See [ref.synth()]
#'
#'   2. Values for the reference treatment response can be assigned to different time-course parameters
#'   within the model that have been modelled using consistency relative effects (`pool="rel"`).
#'   These are given as a list, in which each named element corresponds to a time-course
#'   parameter modelled in `mbnma`. Their values can be either of the following:
#'   * `numeric()` A single numeric value representing the deterministic value of the time-course parameter in
#'   question in individuals given the reference treatment. `0` is used as the default, though this may produce
#'   nonsensical predictions as this typically assumes no effect of time on the reference treatment.
#'   * `character()` A single string representing a stochastic distribution for the value of the time-course
#'   parameter in question. This is specified as a random number generator (RNG) given as a string,
#'   and can take any RNG distribution for which a function exists in R. For example: `"rnorm(n, -3, 0.2)"`.

#' @param synth A character object that can take the value `"fixed"` or `"random"` that
#'   specifies the the type of pooling to use for synthesis of `ref.resp`. Using `"random"` rather
#'   than `"fixed"` for `synth` will result in wider 95\\% CrI for predictions.
#' @param ... Arguments to be sent to R2jags for synthesis of the network
#'   reference treatment effect (using [ref.synth()])
#'
#'
#' @return An S3 object of class `mb.predict` that contains the following
#'   elements:
#'   * `summary` A named list of data frames. Each data frame contains
#'   a summary of predicted responses at follow-up times specified in `times`
#'   for each treatment specified in `treats`
#'   * `pred.mat` A named list of
#'   matrices. Each matrix contains the MCMC results of predicted responses at
#'   follow-up times specified in `times` for each treatment specified in
#'   `treats`
#'
#' @details `ref.resp` only needs to be specified if `mbnma` has
#'   been estimated using consistency relative effects (`pool="rel"`) for
#'   any time-course parameters, as these inform the absolute values of the
#'   network reference treatment parameters which can then be added to the
#'   relative effects to calculate specific predictions.
#'
#' @examples
#' \donttest{
#' # Create an mb.network object from a dataset
#' network <- mb.network(osteopain)
#'
#' # Run an MBNMA model with an Emax time-course
#' emax <- mb.emax(network,
#'   emax=list(pool="rel", method="common"),
#'   et50=list(pool="const", method="common"),
#'   positive.scale=TRUE)
#'
#' # Predict responses using a stochastic baseline (E0) and a distribution for the
#' #network reference treatment
#' preds <- predict(emax, times=c(0:10),
#'   E0="rnorm(n, 7, 0.5)",
#'   ref.resp=list("emax"="rnorm(n, -0.5, 0.05)"))
#' summary(preds)
#'
#' # Predict responses using the original dataset to estimate the network reference
#' #treatment response
#' paindata.ref <- osteopain[osteopain$treatname=="Placebo_0",]
#' preds <- predict(emax, times=c(5:15),
#'   E0=10,
#'   ref.resp=paindata.ref)
#' summary(preds)
#'
#' # Repeat the above prediction but using a random effects meta-analysis of the
#' #network reference treatment response
#' preds <- predict(emax, times=c(5:15),
#'   E0=10,
#'   ref.resp=paindata.ref,
#'   synth="random")
#' summary(preds)
#' }
#'
#' @export
predict.mbnma <- function(object, times=c(0:max(object$model$data()$time, na.rm=TRUE)),
                          E0=0,
                          treats = NULL,
                          ref.resp=NULL, synth="fixed",
                          ...) {
  ######## CHECKS ########

  # Run checks
  argcheck <- checkmate::makeAssertCollection()
  checkmate::assertClass(object, "mbnma", add=argcheck)
  checkmate::assertNumeric(times, lower=0, finite=TRUE, any.missing=FALSE, unique=TRUE,
                sorted=TRUE, add=argcheck)
  checkmate::assertChoice(synth, choices=c("random", "fixed"), add=argcheck)
  #checkmate::assertClass(treats, classes=c("numeric", "character"), null.ok=TRUE, add=argcheck)
  checkmate::reportAssertions(argcheck)

  # Check whether class effects have been used and give error if so
  # if (any(object[["model.arg"]][["class.effect"]]=="random")) {
  #   stop("Random class effects have not yet been fully implemented in predict(`object`)")
  # }
  if (length(object[["model.arg"]][["class.effect"]]>0)) {
    stop("Class effects have not yet been fully implemented in predict(`object`)")
  }

  # Check whether UME has been used and stop if so
  if (object[["model.arg"]][["UME"]]!=FALSE) {
    stop("UME model cannot be used for prediction")
  }

  # Check ref.resp has been specified correctly if any mbnma parameters are "rel"
  if (check.betas(object)==TRUE) {
    if (is.null(ref.resp)) {

      # If ref.resp is not given then assign 0 to all rel time-course parameters
      ref.resp <- list()
      for (i in 1:4) {
        if (!is.null(object$model.arg[[paste0("beta.",i)]])) {
          if (object$model.arg[[paste0("beta.",i)]]$pool=="rel") {
            if (!is.null(object$model.arg$arg.params)) {
              elname <- object$model.arg$arg.params$wrap.params[object$model.arg$arg.params$run.params==
                                                                 paste0("beta.",i)]
            } else {
              elname <- paste0("beta.",i)
            }
            ref.resp[[elname]] <- 0
          }
        }
      }
    } else {

      # If ref.resp is given ensure it is of the correct class
      if (!(any(class(ref.resp) %in% c("data.frame", "tibble", "list")))) {
        stop("`object` includes time-course parameters modelled using relative effects (pool=`rel`).
      The reference treatment response for them must be provided to `ref.resp` as a list,
      or estimated from a dataset of reference treatment studies by providing a data frame.")
      }
    }
  } else if (check.betas(object)==FALSE) {
    ref.resp <- NULL
  }


  # If treats have not been specified then select all of them
  if (is.null(treats)) {
    #treats <- c(1:object[["model"]][["data"]]()[["NT"]])
    treats <- object$treatments
  } else if (!is.null(treats)) {
    if (is.numeric(treats)) {
      if (any(treats > object[["model"]][["data"]]()[["NT"]] | any(treats<1))) {
        stop("If given as numeric treatment codes, `treats` must be numbered similarly to treatment codes in `object`")
      }
      treats <- object$treatments[treats]
    }
    if (is.character(treats)) {
      if (!all(treats %in% object$treatments)) {
        stop("`treats` includes treatments not included in `object`")
      }
    }
  }

  #### Check E0 ####
  if (is.null(E0)) {
    stop("E0 has not been defined")
  }

  # Check that distribution for E0 is of the correct format
  if (is.character(E0)) {
    if (grepl("r[A-z]+\\(n,.+\\)", E0)==FALSE) {
      stop("Stochastic distribution for E0 must be expressed as a string in the form of a supported R distribution (e.g. `rnorm(n, 5,2)`)")
    }
  } else if (is.numeric(E0)) {
    if (length(E0)!=1) {
      stop("`E0` can only take a single numeric value if not expressed as a stochastic distribution")
    }
  }


  ###### Extract info from mbnma #######

  n <- object$BUGSoutput$n.sims

  # Initial predict parameters
  timecourse <- list(init.predict(object)[["timecourse"]])
  beta.incl <- init.predict(object)[["beta.incl"]]


  # Extract parameter values from MBNMA result
  model.vals <- get.model.vals(mbnma=object, timecourse=timecourse,
                               beta.incl=beta.incl, E0=E0)
  timecourse <- model.vals[["timecourse"]]
  time.params <- model.vals[["time.params"]]


  ########## Get reference treatment effect ###########
  mu.prior <- model.vals[["mu.prior"]]
  mu.params <- time.params[grepl("^mu.", time.params)]


  if (!is.null(ref.resp)) {

    # If ref.resp specified as values for each time-course parameter (in a list)
    if (any(class(ref.resp)=="list")) {
      msg <- paste0("Priors required for: ", paste(mu.prior, collapse=", "))
      message(msg)

      if (identical(sort(mu.prior),sort(names(ref.resp)))==FALSE) {
        msg <- "Named elements of `ref.resp` do not correspond to consistency time-course parameters monitored within the model."
        stop(msg)
      } else {
        message("Success: Elements in prior match consistency time-course treatment effect parameters")
      }

      # Assign ref.resp to mu values in model
      for (i in seq_along(ref.resp)) {

        if (is.character(ref.resp[[i]])) {
          if (grepl("r[A-z]+\\(n,.+\\)", ref.resp[[i]])==FALSE) {
            stop("Stochastic distribution for ref.resp must be expressed as a string in the form of a supported R distribution (e.g. `rnorm(n, 5,2)`)")
          }
        }
        assign(mu.params[which(names(ref.resp)[i]==mu.prior)],
               eval(parse(text=ref.resp[[i]])))
      }
    } else if (any(class(ref.resp) %in% c("data.frame", "tibble"))) {

      ### PLACEBO SYNTHESIS MODEL ###
      synth.result <- ref.synth(data.ab=ref.resp, mbnma=object, synth=synth)
      #synth.result <- ref.synth(data.ab=ref.resp, mbnma=object, synth=synth, ...)

      # Assign synth.result to mu values in model
      for (i in seq_along(mu.params)) {
        if (synth=="random") {
          assign(mu.params[i],
                 stats::rnorm(n,
                       synth.result[[paste0("m.", mu.params[i])]],
                       synth.result[[paste0("sd.", mu.params[i])]])
          )
        } else if (synth=="fixed") {
          assign(mu.params[i], rep(synth.result[[paste0("m.", mu.params[i])]],
                                   n))
        } else (stop("synth must be either `fixed` or `random`"))

      }
    }
  }


  ########## Predict responses ###########

  # Assign E0 to alpha in model
  #alpha <- eval(parse(text=E0)) # TO BE REMOVED
  alpha <- model.vals$alpha


  beta.params <- time.params[grepl("^beta.", time.params)]
  # Assign single beta results to beta values in model
  for (i in seq_along(beta.params)) {
    if (!is.matrix(model.vals[[beta.params[i]]])) {
      assign(beta.params[i], model.vals[[beta.params[i]]])
    }
  }

  d.params <- time.params[grepl("^d\\.", time.params)]

  predicts <- list()
  treatsnum <- which(object$treatments %in% treats)
  for (treat in seq_along(treatsnum)) {

    # Assign treatment beta results to beta values in model
    for (i in seq_along(beta.params)) {
      if (is.matrix(model.vals[[beta.params[i]]])) {
        assign(beta.params[i], model.vals[[beta.params[i]]][,treatsnum[treat]])
      }
    }

    # Assign d results to d values in model
    for (i in seq_along(d.params)) {
      assign(d.params[i], model.vals[[d.params[i]]][,treatsnum[treat]])
    }

    #treatpred <- vector(mode="numeric", length=n)
    treatpred <- data.frame("pred"=rep(NA,n))
    for (m in seq_along(times)) {
      time <- times[m]

      # Evaluate function
      pred <- eval(parse(text=timecourse))

      if (any(is.na(pred))) {
        pred[is.na(pred)] <- 0
      }

      #treatpred <- cbind(treatpred, pred)
      treatpred[[paste0("time", times[m])]] <- pred

    }

    predicts[[paste0(treats[treat])]] <- treatpred[,-1]
  }

  # Generate summary data frame
  sumpred <- list()
  for (i in seq_along(treats)) {
    summary <- data.frame("time"=times)

    summary[["mean"]] <- apply(predicts[[as.character(treats[i])]], MARGIN=2,
                               FUN=function(x) mean(x))
    summary[["sd"]] <- apply(predicts[[as.character(treats[i])]], MARGIN=2,
                             FUN=function(x) stats::sd(x))

    quantiles <- apply(predicts[[as.character(treats[i])]], MARGIN = 2,
                       function(x) stats::quantile(x,
                                            probs=c(0.025, 0.25, 0.5, 0.75, 0.975)))
    summary <- cbind(summary, t(quantiles))

    sumpred[[as.character(treats[i])]] <- summary
  }

  #predict.result <- list("summary"=sumpred, "pred.mat"=predicts, "treatments"=object$treatments, "mbnma"=object)
  predict.result <- list("summary"=sumpred, "pred.mat"=predicts, "mbnma"=object)
  class(predict.result) <- "mb.predict"

  return(predict.result)
}





#' Check if any relative effects are specified within time-course parameters in an mbnma model
#'
#' @inheritParams predict.mbnma
#'
#' @return A boolean object that takes `TRUE` if any time-course parameters specify relative effects and `FALSE` if not
#'
#' @noRd
check.betas <- function(mbnma) {
  mbnma.betas <- vector()
  for (i in 1:4) {
    mbnma.betas <- append(mbnma.betas, mbnma[["model.arg"]][[paste0("beta.", i)]]$pool)
    }

  if (any(mbnma.betas == "rel")) {
    return(TRUE)
  } else {return(FALSE)}
}



#' Get code parameter values from previous MBNMA result
#'
#' @inheritParams predict.mbnma
#'
#' @return A list with two elements:
#' * `timecourse` A character object that specifies the time-course used in `mbnma` (in terms of alpha, beta, and time)
#' * `beta.incl` A numeric vector that indicates the time-course parameters that were included in `mbnma`
#'
#' @noRd
init.predict <- function(mbnma) {

  # Check betas are specified correctly and prepare format for subsequent functions
  for (i in 1:4) {
    betaname <- paste0("beta.", i)
    if (!is.null(mbnma$model.arg[[betaname]])) {
      assign(paste0(betaname, ".str"), compound.beta(mbnma$model.arg[[betaname]]))
    } else if (is.null(mbnma$model.arg[[betaname]])) {
      assign(paste0(betaname, ".str"), NULL)
    }
  }

  timecourse <- time.fun(fun=mbnma$model.arg$fun, user.fun=mbnma$model.arg$user.fun,
                         alpha=mbnma$model.arg$alpha, beta.1=beta.1.str, beta.2=beta.2.str,
                         beta.3=beta.3.str, beta.4=beta.4.str)[["relationship"]]

  # Add piecewise function if fun==piecelinear
  # if (fun=="piecelinear") {
  #   timecourse <-
  #     "ifelse(time < beta.3, alpha, 0) + ifelse(time < beta.3, beta.1*time, 0) + ifelse(time >= beta.3, alpha + (beta.1*beta.3), 0) + ifelse(time >= beta.3, beta.2*time, 0)"
  # }

  # Add

  # Generate vector with indices of betas included in model
  beta.incl <- vector()
  for (i in 1:4) {
    if (grepl(paste0("beta.", i), timecourse)) {
      beta.incl <- append(beta.incl, i)
    }
  }

  return(list("timecourse"=timecourse, "beta.incl"=beta.incl))
}




#' Get MBNMA model values
#'
#' Extracts specific information required for prediction from a time-course
#' MBNMA model
#'
#' @inheritParams predict.mbnma
#' @inheritParams ref.synth
#' @param timecourse A character object that specifies the time-course used in
#'   `mbnma` (in terms of alpha, beta, and time), as generated by
#'   `init.predict()`
#' @param beta.incl A numeric vector that indicates the time-course parameters
#'   that were included in `mbnma`, as generated by `init.predict()`
#'
#' @return A list containing named elements that correspond to different
#'   time-course parameters in `mbnma`. These elements contain MCMC results
#'   either taken directly from `mbnma` or (in the case of random time-course
#'   parameters specified as `method="random"`) randomly
#'   generated using parameter values estimated in `mbnma`.
#'
#'   Additional elements contain the following values:
#'   * `timecourse` A character object that specifies the time-course used in `mbnma` in terms of
#'   alpha, beta, mu, d and time. Consistency relative time-course parameters
#'   are specified in terms of mu and d.
#'   * `mu.prior` A character vector that
#'   indicates for which time-course parameters a network reference treatment
#'   effect will be required for prediction.
#'   * `time.params` A character vector
#'   that indicates the different time-course parameters that are required for
#'   the prediction
#'
#'   @noRd
get.model.vals <- function(mbnma, timecourse, beta.incl, E0=0) {
  model.vals <- list()
  time.params <- "alpha"
  mu.prior <- vector()

  n <- mbnma$BUGSoutput$n.sims

  # Assign E0 to alpha in model.vals
  alpha <- eval(parse(text=E0))
  if (length(alpha)==1) {
    model.vals[["alpha"]] <- rep(alpha, n)
  } else if (length(alpha)>1) {
    model.vals[["alpha"]] <- alpha
  }


  # Get code parameter values from previous MBNMA result
  # model.arg <- names(mbnma[["model.arg"]])
  # for (i in seq_along(model.arg)) {
  #   assign(model.arg[i], mbnma[["model.arg"]][[model.arg[i]]])
  # }

  sims.matrix <- mbnma$BUGSoutput$sims.matrix

  arg.params <- mbnma[["parameters.to.save"]]
  for (i in seq_along(beta.incl)) {

    beta.name <- paste0("beta.", beta.incl[i])
    if (!is.null(mbnma$model.arg[[beta.name]])) {

      # Change named beta to named model parameter specific to time-course
      if (!is.null(mbnma$model.arg$arg.params)) {
        suffix <- mbnma$model.arg$arg.params[["wrap.params"]][
          which(mbnma$model.arg$arg.params[["run.params"]]==beta.name)
          ]
      } else {
        suffix <- beta.incl[i]
      }

      betatemp <- paste0("beta.", suffix)

      # Assign results from MBNMA to result variables for each beta
      if (betatemp %in% arg.params |
          paste0("d.", suffix) %in% arg.params) {
        if (mbnma$model.arg[[beta.name]]$method == "common" &  mbnma$model.arg[[beta.name]]$pool %in% c("const", "arm")) {

          # Store MCMC results for relevant parameters
          model.vals[[paste0("beta.", beta.incl[i])]] <-
            sims.matrix[,grepl(paste0("^beta\\.", suffix), colnames(sims.matrix))]

          # Add beta parameters to the vector of time-course parameters
          time.params <- append(time.params, paste0("beta.", beta.incl[i]))

        } else if (mbnma$model.arg[[beta.name]]$method=="random" & mbnma$model.arg[[beta.name]]$pool %in% c("const", "arm")) {
          # Store matrix of beta values generated from random distribution determined by model parameters
          # This section could be performed for each iteration (rather than on posterior medians)
          mat <- cbind(mbnma$BUGSoutput$median[[paste0("beta.", suffix)]],
                       mbnma$BUGSoutput$median[[paste0("sd.beta.", suffix)]])
          model.vals[[paste0("beta.", beta.incl[i])]] <-
            apply(mat, MARGIN = 1, FUN=function(x) {stats::rnorm(n, x[1], x[2])})

          # Add beta parameters to the vector of time-course parameters
          time.params <- append(time.params, paste0("beta.", beta.incl[i]))

        } else if (mbnma$model.arg[[beta.name]]$pool=="rel") {

          # Ammend time-course equation
          timecourse <- gsub(paste0("(beta\\.", beta.incl[i],")"),
                                  paste0("(mu.", i, " + d.", i, ")"), timecourse)

          # Add d parameters to the vector of time-course parameters
          time.params <- append(time.params, c(paste0("d.", beta.incl[i]), paste0("mu.", beta.incl[i])))

          if (!is.null(mbnma$model.arg$arg.params)) {
            mu.prior <- append(mu.prior, suffix)
          } else {
            mu.prior <- append(mu.prior, beta.name)
          }

          #if (get(beta.name)$pool=="rel") {
          if (mbnma$model.arg[[beta.name]]$method=="common") {
            # Store MCMC results for relevant parameters
            model.vals[[paste0("d.", beta.incl[i])]] <-
              sims.matrix[,grepl(paste0("^d\\.", suffix), colnames(sims.matrix))]
          } else if (mbnma$model.arg[[beta.name]]$method=="random") {
            # Store matrix of d values generated from random distribution determined by model parameters
            # This section could be performed for each iteration (rather than on posterior medians)
            mat <- cbind(mbnma$BUGSoutput$median[[paste0("d.", suffix)]],
                         mbnma$BUGSoutput$median[[paste0("sd.", suffix)]])
            model.vals[[paste0("d.", beta.incl[i])]] <-
              apply(mat, MARGIN = 1, FUN=function(x) {stats::rnorm(n, x[1], x[2])})
          }
          #}
        }
      } else if (is.numeric(mbnma$model.arg[[beta.name]]$method)) {

        # Store MCMC results for relevant parameters
        model.vals[[paste0("beta.", beta.incl[i])]] <-
          rep(mbnma$model.arg[[beta.name]]$method, n)

        # Add beta parameters to the vector of time-course parameters
        time.params <- append(time.params, paste0("beta.", beta.incl[i]))

      } else {
        stop("Parameter(s) in time-course function not included in those monitored in JAGS model")
      }
    }
  }
  model.vals[["mu.prior"]] <- mu.prior
  model.vals[["timecourse"]] <- timecourse
  model.vals[["time.params"]] <- time.params

  return(model.vals)
}






#' Synthesise single arm studies with repeated observations of the same
#' treatment over time
#'
#' Synthesises single arm studies with repeated measures by applying a
#' particular time-course function. Used in predicting mean responses from a
#' time-course MBNMA. The same parameterisation of the time course must be used
#' as in the MBNMA.
#'
#' @inheritParams predict.mbnma
#' @inheritParams R2jags::jags
#' @param mbnma An S3 object of class `"mbnma"` generated by running
#' a time-course MBNMA model
#' @param data.ab A data frame of arm-level data in "long" format containing the
#'   columns:
#'   * `studyID` Study identifiers
#'   * `time` Numeric data indicating follow-up times
#'   * `y` Numeric data indicating the mean response for a given observation
#'   * `se` Numeric data indicating the standard error for a given observation
#'
#' @details `data.ab` can be a collection of studies that closely resemble the
#'   population of interest intended for the prediction, which could be
#'   different to those used to estimate the MBNMA model, and could be include
#'   single arms of RCTs or observational studies. If other data is not
#'   available, the data used to estimate the MBNMA model can be used by
#'   selecting only the studies and arms that specify the network reference
#'   treatment responses.
#'
#' @return A list of named elements corresponding to each time-course parameter
#'   within an MBNMA model that contain the median posterior value for the
#'   network reference treatment response.
#'
#' @examples
#' \donttest{
#' # Create an mb.network object from a dataset
#' network <- mb.network(osteopain)
#'
#' # Run an MBNMA model with an Emax time-course
#' emax <- mb.emax(network,
#'   emax=list(pool="rel", method="common"),
#'   et50=list(pool="rel", method="random"),
#'   positive.scale=TRUE)
#'
#' # Generate a set of studies with which to estimate the network reference treatment response
#' paindata.ref <- osteopain[osteopain$treatname=="Placebo_0",]
#'
#' # Estimate the network reference treatment effect using fixed effects meta-analysis
#' ref.synth(data.ab=paindata.ref, mbnma=emax, synth="fixed")
#'
#' # Estimate the network reference treatment effect using random effects meta-analysis
#' ref.synth(data.ab=paindata.ref, mbnma=emax, synth="random")
#' }
#'
#' @export
ref.synth <- function(data.ab, mbnma, synth="random",
                      n.iter=mbnma$BUGSoutput$n.iter,
                      n.burnin=mbnma$BUGSoutput$n.burnin,
                      n.thin=mbnma$BUGSoutput$n.thin,
                      n.chains=mbnma$BUGSoutput$n.chains,
                      ...) {

  # First need to validate data.frame to check dataset is in correct format...maybe another function for this
  # Change it to correct format if it is not already
  #data.ab <- ref.validate(data.ab)[["data"]]
  data.ab <- ref.validate(data.ab)[["data.ab"]]

  # Run checks
  argcheck <- checkmate::makeAssertCollection()

  checkmate::assertClass(mbnma, "mbnma", add=argcheck)
  checkmate::assertChoice(synth, choices=c("random", "fixed"), add=argcheck)
  checkmate::assertInt(n.iter, lower=1, add=argcheck)
  checkmate::assertInt(n.burnin, lower=1, add=argcheck)
  checkmate::assertInt(n.thin, lower=1, add=argcheck)
  checkmate::assertInt(n.chains, lower=1, add=argcheck)

  checkmate::reportAssertions(argcheck)

  # To get model for meta-analysis of placebo must create v similar model
  #to study model
  # Do all the mb.write bits but without the consistency bits

  model.arg <- names(mbnma[["model.arg"]])
  for (i in seq_along(model.arg)) {
    assign(model.arg[i], mbnma[["model.arg"]][[model.arg[i]]])
  }

  # Check betas are specified correctly and prepare format for subsequent functions
  for (i in 1:4) {
    betaname <- paste0("beta.", i)
    #if (!is.null(get(betaname))) {
    if (!is.null(mbnma$model.arg[[betaname]])) {
      # SPECIFIC TO REFERENCE SYNTHESIS MODEL since "arm" is equivalent to "const" if there is only one treatment
      if (mbnma$model.arg[[betaname]]$pool == "arm") {
        assign(betaname, list(pool="const", method=mbnma$model.arg[[betaname]]$method))
      }

      assign(paste0(betaname, ".str"), compound.beta(mbnma$model.arg[[betaname]]))
    } else {
      assign(paste0(betaname, ".str"), NULL)
    }
  }

  jagsmodel <- write.ref.synth(fun=mbnma$model.arg$fun, user.fun=mbnma$model.arg$user.fun,
                               alpha=mbnma$model.arg$alpha,
                               beta.1=beta.1.str, beta.2=beta.2.str,
                               beta.3=beta.3.str, beta.4=beta.4.str,
                               positive.scale=mbnma$model.arg$positive.scale, intercept=mbnma$model.arg$intercept,
                               rho=mbnma$model.arg$rho, covar=mbnma$model.arg$covar,
                               mu.synth=synth,
                               class.effect=mbnma$model.arg$class.effect, UME=mbnma$model.arg$UME,
                               priors=mbnma$model.arg$priors
  )


  parameters.to.save <- vector()
  for (i in 1:4) {
    if (!is.null(mbnma$model.arg[[paste0("beta.", i)]])) {
      if (mbnma$model.arg[[paste0("beta.", i)]]$pool=="rel") {
        parameters.to.save <- append(parameters.to.save, paste0("m.mu.", i))
        if (synth=="random") {
          parameters.to.save <- append(parameters.to.save, paste0("sd.mu.", i))
        }
      }
    }
  }

  jags.result <- mb.jags(data.ab,
                            model=jagsmodel,
                            rho=mbnma$model.arg$rho, covar=mbnma$model.arg$covar,
                            parameters.to.save=parameters.to.save,
                            n.iter=n.iter, n.burnin=n.burnin,
                            n.thin=n.thin, n.chains=n.chains,
                            ...)[["jagsoutput"]]

  result <- jags.result$BUGSoutput$median
  result[["deviance"]] <- NULL

  if (any(jags.result$BUGSoutput$summary[,
                                         colnames(jags.result$BUGSoutput$summary)=="Rhat"
                                         ]>1.02)) {
    warning("Rhat values for parameter(s) in reference treatment synthesis model are >1.02. Suggest running for more iterations.")
  }

  return(result)

}



#' Checks the validity of ref.resp if given as data frame
#'
#' Ensures `ref.resp` takes the correct form to allow for synthesis of network
#' reference treatment response if data is provided for meta-analysis
#'
#' @inheritParams ref.synth
ref.validate <- function(data.ab) {

  argcheck <- checkmate::makeAssertCollection()
  checkmate::assertDataFrame(data.ab, any.missing=FALSE, add=argcheck)
  checkmate::assertNames(names(data.ab), must.include = c("studyID", "y", "se", "time"), add=argcheck)
  checkmate::reportAssertions(argcheck)

  # Sort data.ab
  data.ab <- dplyr::arrange(data.ab, studyID, time)

  # if (anyMissing(data.ab)) {
  #   stop("Data frame for synthesis of reference treatment contains NA values")
  # }

  message("Data frame must contain only data from reference treatment")

  #### Prepare data frame ####
  # Add arm index (=1 since only one arm in each study)
  data.ab[["arm"]] <- 1
  data.ab[["narm"]] <- 1
  data.ab[["treatment"]] <- 1

  # Ensuring studies are numbered sequentially
  if (!is.numeric(data.ab[["studyID"]])) {
    #message("Studies being recoded to allow sequential numbering")
    data.ab <- transform(data.ab,studyID=as.numeric(factor(studyID, levels=as.character(unique(data.ab$studyID)))))
    data.ab <- dplyr::arrange(data.ab, studyID, time)
  } else if (all(abs(diff(data.ab[["studyID"]])) != TRUE)) {
    #message("Studies being recoded to allow sequential numbering")
    data.ab <- transform(data.ab,studyID=as.numeric(factor(studyID, levels=as.character(unique(data.ab$studyID)))))
    data.ab <- dplyr::arrange(data.ab, studyID, time)
  }

  data.ab <- add_index(data.ab, reference=1)

  return(data.ab)

}


