% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/contagion.R
\name{contagion}
\alias{contagion}
\title{Contagion Simulations}
\usage{
contagion(exposures, 
          buffer, 
          shock = "all", 
          weights = NULL, 
          method = c("debtrank", "threshold"), 
          ...,
          exposure_type = c("assets", "liabilities", "impact", "vulnerability"),
          keep.history = FALSE, 
          abs.tol = .Machine$double.eps ^ 0.2, 
          max.it = min(1000, nrow(v)*10),
          verbose = TRUE)
}
\arguments{
\item{exposures}{an adjacency \code{\link{matrix}}, (sparse) \code{\link{Matrix}} or an \code{\link[igraph]{igraph}}
object with the network of bilateral exposures between vertices. By default, the function
expects the exposures in the form of an assets matrix
in which A -> B means that A has an asset with B. However, you can
change that with the parameter \code{exposure_type}. When using a matrix, preferably it should have
rows and columns names.}

\item{buffer}{a numeric vector with the capital buffer for each vertex.
Values should be in the same row/column order as the network of bilateral exposures. The
buffer is not needed if \code{exposure_type = "vulnerability"}.}

\item{shock}{a list with the shock vectors. If \code{"all"} (the default) the function will run a 
simulation for the default of each vertex in the network.}

\item{weights}{default is \code{NULL}. You can use a numeric
vector of weights to give some economic significance to the measures, like,
for instance, the total assets of the nodes.}

\item{method}{the contagion propagation method. Currently, you should use either "debtrank" for the DebtRank propagation method or "threshold" for the tradional default
cascades. The DebtRank version implemented is the one proposed in Bardoscia et al (2015). 
If you want to use the old "single-hit" DebtRank of Battiston et al (2012), simply provide the argument \code{single.hit = TRUE}.}

\item{...}{other arguments to be passed to the contagion propagation method.}

\item{exposure_type}{character vector indicating the type of the bilateral exposures. It can be
an \code{"assets"} network (where A -> B means that A has an asset with B),
a \code{"liabilities"} network (where A -> B means that A has a debt with B),
a (binary) \code{"impact"} matrix (where A -> B indicates the relative impact
 of A in B's capital buffer), or 
a (binary) \code{"vulnerability"} matrix 
(where A -> B indicates the relative impact A suffers from B's default). 
The default is \code{"assets"}.}

\item{keep.history}{keep all the history of stress levels? This can use a lot of memory, so
the default is \code{FALSE}.}

\item{abs.tol}{the desired accuracy.}

\item{max.it}{the maximum number of iterations.}

\item{verbose}{gives verbose output. Default is \code{TRUE}.}
}
\value{
The function returns an object of class \code{"contagion"} with the results of the simulation.
}
\description{
Given a matrix of exposures, a vector of buffers and weights (optional) 
the functions simulates contagion for all the shock vectors provided. You may
choose from the implemented propagation contagion method or create you own propagation method. Details on how to create
your own method will be provided in a future version.
}
\examples{
# Loads simulated banking data
data("sim_data")
head(sim_data)

# seed for reproducibility
set.seed(15)

# minimum density estimation
# verbose = F to prevent printing
md_mat <- matrix_estimation(sim_data$assets, sim_data$liabilities, method = "md", verbose = FALSE)

# rownames and colnames for the matrix
rownames(md_mat) <- colnames(md_mat) <- sim_data$bank

# DebtRank simulation
contdr <- contagion(exposures = md_mat, buffer = sim_data$buffer, weights = sim_data$weights, 
                    shock = "all", method = "debtrank", verbose = FALSE)
summary(contdr)

plot(contdr)

# Traditional default cascades simulation
contthr <-  contagion(exposures = md_mat, buffer = sim_data$buffer, weights = sim_data$weights, 
                      shock = "all", method = "threshold", verbose = FALSE)
summary(contthr)

# simulating shock scenarios 1\% to 25\% shock in all vertices
s <- seq(0.01, 0.25, by = 0.01)
shocks <- lapply(s, function(x) rep(x, nrow(md_mat)))
names(shocks) <- paste(s*100, "pct shock")

cont <- contagion(exposures = md_mat, buffer = sim_data$buffer, shock = shocks, 
                 weights = sim_data$weights, method = "debtrank", verbose = FALSE)
summary(cont)
plot(cont)
}
\references{
Bardoscia M, Battiston S, Caccioli F, Caldarelli G (2015) DebtRank: A Microscopic Foundation for Shock Propagation. PLoS ONE 10(6): e0130406. doi: 10.1371/journal.pone.0130406

Battiston, S., Puliga, M., Kaushik, R., Tasca, P., and Caldarelli, G. (2012). DebtRank: Too central to fail? Financial networks, the FED and systemic risk. Scientific reports, 2:541.
}

