/*
 * KAKASI (Kanji Kana Simple inversion program)
 * $Id: kk2.c,v 1.11 2006-05-15 07:23:30 knok Exp $
 * Copyright (C) 1992
 * Hironobu Takahashi (takahasi@tiny.or.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either versions 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with KAKASI, see the file COPYING.  If not, write to the Free
 * Software Foundation Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <stdio.h>
#ifdef HAVE_STRING_H
# include <string.h>
#else
# include <strings.h>
#endif
#include "kakasi.h"

struct K2rom_tbl {
    unsigned char kana[10];
    char romaji[7];
}
K2rom_h_table_old[] = { 

    {"\245\241", "a"}, {"\245\242", "a"},
    {"\245\243", "i"}, {"\245\244", "i"},
    {"\245\245", "u"}, {"\245\246", "u"},
    {"\245\247", "e"}, {"\245\250", "e"},
    {"\245\251", "o"}, {"\245\252", "o"},

    {"\245\253", "ka"}, {"\245\254", "ga"},
    {"\245\255", "ki"}, {"\245\255\245\343", "kya"},
    {"\245\255\245\345", "kyu"}, {"\245\255\245\347", "kyo"},
    {"\245\256", "gi"}, {"\245\256\245\343", "gya"},
    {"\245\256\245\345", "gyu"}, {"\245\256\245\347", "gyo"},
    {"\245\257", "ku"}, {"\245\260", "gu"},
    {"\245\261", "ke"}, {"\245\262", "ge"},
    {"\245\263", "ko"}, {"\245\264", "go"},

    {"\245\265", "sa"}, {"\245\266", "za"},
    {"\245\267", "shi"}, {"\245\267\245\343", "sha"},
    {"\245\267\245\345", "shu"}, {"\245\267\245\347", "sho"},
    {"\245\270", "ji"}, {"\245\270\245\343", "ja"},
    {"\245\270\245\345", "ju"}, {"\245\270\245\347", "jo"},
    {"\245\271", "su"}, {"\245\272", "zu"},
    {"\245\273", "se"}, {"\245\274", "ze"},
    {"\245\275", "so"}, {"\245\276", "zo"},

    {"\245\277", "ta"}, {"\245\300", "da"},
    {"\245\301", "chi"}, {"\245\301\245\343", "cha"},
    {"\245\301\245\345", "chu"}, {"\245\301\245\347", "cho"},
    {"\245\302", "di"}, {"\245\302\245\343", "dya"},
    {"\245\302\245\345", "dyu"}, {"\245\302\245\347", "dyo"},

    {"\245\303", "tsu"},
    {"\245\303\245\253", "kka"}, {"\245\303\245\254", "gga"},
    {"\245\303\245\255", "kki"}, {"\245\303\245\255\245\343", "kkya"},
    {"\245\303\245\255\245\345", "kkyu"}, {"\245\303\245\255\245\347", "kkyo"},
    {"\245\303\245\256", "ggi"}, {"\245\303\245\256\245\343", "ggya"},
    {"\245\303\245\256\245\345", "ggyu"}, {"\245\303\245\256\245\347", "ggyo"},
    {"\245\303\245\257", "kku"}, {"\245\303\245\260", "ggu"},
    {"\245\303\245\261", "kke"}, {"\245\303\245\262", "gge"},
    {"\245\303\245\263", "kko"}, {"\245\303\245\264", "ggo"},
    {"\245\303\245\265", "ssa"}, {"\245\303\245\266", "zza"},
    {"\245\303\245\267", "sshi"}, {"\245\303\245\267\245\343", "ssha"},
    {"\245\303\245\267\245\345", "sshu"}, {"\245\303\245\267\245\347", "ssho"},
    {"\245\303\245\270", "jji"}, {"\245\303\245\270\245\343", "jja"},
    {"\245\303\245\270\245\345", "jju"}, {"\245\303\245\270\245\347", "jjo"},
    {"\245\303\245\271", "ssu"}, {"\245\303\245\272", "zzu"},
    {"\245\303\245\273", "sse"}, {"\245\303\245\274", "zze"},
    {"\245\303\245\275", "sso"}, {"\245\303\245\276", "zzo"},
    {"\245\303\245\277", "tta"}, {"\245\303\245\300", "dda"},
    {"\245\303\245\301", "cchi"}, {"\245\303\245\301\245\343", "ccha"},
    {"\245\303\245\301\245\345", "cchu"}, {"\245\303\245\301\245\347", "ccho"},
    {"\245\303\245\302", "ddi"}, {"\245\303\245\302\245\343", "ddya"},
    {"\245\303\245\302\245\345", "ddyu"}, {"\245\303\245\302\245\347", "ddyo"},
    {"\245\303\245\304", "ttsu"}, {"\245\303\245\305", "ddu"},
    {"\245\303\245\306", "tte"}, {"\245\303\245\307", "dde"},
    {"\245\303\245\310", "tto"}, {"\245\303\245\311", "ddo"},
    {"\245\303\245\317", "hha"}, {"\245\303\245\320", "bba"},
    {"\245\303\245\321", "ppa"},
    {"\245\303\245\322", "hhi"}, {"\245\303\245\322\245\343", "hhya"},
    {"\245\303\245\322\245\345", "hhyu"}, {"\245\303\245\322\245\347", "hhyo"},
    {"\245\303\245\323", "bbi"}, {"\245\303\245\323\245\343", "bbya"},
    {"\245\303\245\323\245\345", "bbyu"}, {"\245\303\245\323\245\347", "bbyo"},
    {"\245\303\245\324", "ppi"}, {"\245\303\245\324\245\343", "ppya"},
    {"\245\303\245\324\245\345", "ppyu"}, {"\245\303\245\324\245\347", "ppyo"},
    {"\245\303\245\325", "ffu"}, {"\245\303\245\325\245\241", "ffa"},
    {"\245\303\245\325\245\243", "ffi"}, {"\245\303\245\325\245\247", "ffe"},
    {"\245\303\245\325\245\251", "ffo"},
    {"\245\303\245\326", "bbu"}, {"\245\303\245\327", "ppu"},
    {"\245\303\245\330", "hhe"}, {"\245\303\245\331", "bbe"},
    {"\245\303\245\332", "ppe"},
    {"\245\303\245\333", "hho"}, {"\245\303\245\334", "bbo"},
    {"\245\303\245\335", "ppo"},
    {"\245\303\245\344", "yya"}, {"\245\303\245\346", "yyu"},
    {"\245\303\245\350", "yyo"},
    {"\245\303\245\351", "rra"}, {"\245\303\245\352", "rri"},
    {"\245\303\245\352\245\343", "rrya"}, {"\245\303\245\352\245\345", "rryu"},
    {"\245\303\245\352\245\347", "rryo"},
    {"\245\303\245\353", "rru"}, {"\245\303\245\354", "rre"},
    {"\245\303\245\355", "rro"},
    {"\245\303\245\364", "vvu"}, {"\245\303\245\364\245\241", "vva"},
    {"\245\303\245\364\245\243", "vvi"}, {"\245\303\245\364\245\247", "vve"},
    {"\245\303\245\364\245\251", "vvo"},

    {"\245\304", "tsu"}, {"\245\305", "du"},
    {"\245\306", "te"}, {"\245\307", "de"},
    {"\245\310", "to"}, {"\245\311", "do"},

    {"\245\312", "na"},
    {"\245\313", "ni"}, {"\245\313\245\343", "nya"},
    {"\245\313\245\345", "nyu"}, {"\245\313\245\347", "nyo"},
    {"\245\314", "nu"}, {"\245\315", "ne"}, {"\245\316", "no"},

    {"\245\317", "ha"}, {"\245\320", "ba"}, {"\245\321", "pa"},
    {"\245\322", "hi"}, {"\245\322\245\343", "hya"},
    {"\245\322\245\345", "hyu"}, {"\245\322\245\347", "hyo"},
    {"\245\323", "bi"}, {"\245\323\245\343", "bya"},
    {"\245\323\245\345", "byu"}, {"\245\323\245\347", "byo"},
    {"\245\324", "pi"}, {"\245\324\245\343", "pya"},
    {"\245\324\245\345", "pyu"}, {"\245\324\245\347", "pyo"},
    {"\245\325", "fu"}, {"\245\325\245\241", "fa"},
    {"\245\325\245\243", "fi"}, {"\245\325\245\247", "fe"},
    {"\245\325\245\251", "fo"},
    {"\245\326", "bu"}, {"\245\327", "pu"},
    {"\245\330", "he"}, {"\245\331", "be"}, {"\245\332", "pe"},
    {"\245\333", "ho"}, {"\245\334", "bo"}, {"\245\335", "po"},

    {"\245\336", "ma"},
    {"\245\337", "mi"}, {"\245\337\245\343", "mya"},
    {"\245\337\245\345", "myu"}, {"\245\337\245\347", "myo"},
    {"\245\340", "mu"}, {"\245\341", "me"}, {"\245\342", "mo"},

    {"\245\343", "ya"}, {"\245\344", "ya"},
    {"\245\345", "yu"}, {"\245\346", "yu"},
    {"\245\347", "yo"}, {"\245\350", "yo"},

    {"\245\351", "ra"},
    {"\245\352", "ri"}, {"\245\352\245\343", "rya"},
    {"\245\352\245\345", "ryu"}, {"\245\352\245\347", "ryo"},
    {"\245\353", "ru"}, {"\245\354", "re"}, {"\245\355", "ro"},

    {"\245\356", "wa"}, {"\245\357", "wa"},
    {"\245\360", "i"}, {"\245\361", "e"}, 
    {"\245\362", "wo"}, {"\245\363", "n"},

    {"\245\363\245\242", "n'a"}, {"\245\363\245\244", "n'i"},
    {"\245\363\245\246", "n'u"}, {"\245\363\245\250", "n'e"},
    {"\245\363\245\252", "n'o"},
    {"\245\364", "vu"}, {"\245\364\245\241", "va"},
    {"\245\364\245\243", "vi"}, {"\245\364\245\247", "ve"},
    {"\245\364\245\251", "vo"},
    {"\245\365", "ka"}, {"\245\366", "ke"},

    {"", ""},

}, K2rom_h_table[] = { 

    {"\245\241", "a"}, {"\245\242", "a"},
    {"\245\243", "i"}, {"\245\244", "i"},
    {"\245\245", "u"}, {"\245\246", "u"},
    {"\245\247", "e"}, {"\245\250", "e"},
    {"\245\251", "o"}, {"\245\252", "o"},

    {"\245\253", "ka"}, {"\245\254", "ga"},
    {"\245\255", "ki"}, {"\245\255\245\343", "kya"},
    {"\245\255\245\345", "kyu"}, {"\245\255\245\347", "kyo"},
    {"\245\256", "gi"}, {"\245\256\245\343", "gya"},
    {"\245\256\245\345", "gyu"}, {"\245\256\245\347", "gyo"},
    {"\245\257", "ku"}, {"\245\260", "gu"},
    {"\245\261", "ke"}, {"\245\262", "ge"},
    {"\245\263", "ko"}, {"\245\264", "go"},

    {"\245\265", "sa"}, {"\245\266", "za"},
    {"\245\267", "shi"}, {"\245\267\245\343", "sha"},
    {"\245\267\245\345", "shu"}, {"\245\267\245\347", "sho"},
    {"\245\270", "ji"}, {"\245\270\245\343", "ja"},
    {"\245\270\245\345", "ju"}, {"\245\270\245\347", "jo"},
    {"\245\271", "su"}, {"\245\272", "zu"},
    {"\245\273", "se"}, {"\245\274", "ze"},
    {"\245\275", "so"}, {"\245\276", "zo"},

    {"\245\277", "ta"}, {"\245\300", "da"},
    {"\245\301", "chi"}, {"\245\301\245\247", "che"}, {"\245\301\245\343", "cha"},
    {"\245\301\245\345", "chu"}, {"\245\301\245\347", "cho"},
    {"\245\302", "ji"}, {"\245\302\245\343", "ja"},
    {"\245\302\245\345", "ju"}, {"\245\302\245\347", "jo"},

    {"\245\303", "tsu"},
    {"\245\303\245\253", "kka"}, {"\245\303\245\254", "gga"},
    {"\245\303\245\255", "kki"}, {"\245\303\245\255\245\343", "kkya"},
    {"\245\303\245\255\245\345", "kkyu"}, {"\245\303\245\255\245\347", "kkyo"},
    {"\245\303\245\256", "ggi"}, {"\245\303\245\256\245\343", "ggya"},
    {"\245\303\245\256\245\345", "ggyu"}, {"\245\303\245\256\245\347", "ggyo"},
    {"\245\303\245\257", "kku"}, {"\245\303\245\260", "ggu"},
    {"\245\303\245\261", "kke"}, {"\245\303\245\262", "gge"},
    {"\245\303\245\263", "kko"}, {"\245\303\245\264", "ggo"},
    {"\245\303\245\265", "ssa"}, {"\245\303\245\266", "zza"},
    {"\245\303\245\267", "sshi"}, {"\245\303\245\267\245\343", "ssha"},
    {"\245\303\245\267\245\345", "sshu"}, {"\245\303\245\267\245\347", "ssho"},
    {"\245\303\245\270", "jji"}, {"\245\303\245\270\245\343", "jja"},
    {"\245\303\245\270\245\345", "jju"}, {"\245\303\245\270\245\347", "jjo"},
    {"\245\303\245\271", "ssu"}, {"\245\303\245\272", "zzu"},
    {"\245\303\245\273", "sse"}, {"\245\303\245\274", "zze"},
    {"\245\303\245\275", "sso"}, {"\245\303\245\276", "zzo"},
    {"\245\303\245\277", "tta"}, {"\245\303\245\300", "dda"},
    {"\245\303\245\301", "tchi"}, {"\245\303\245\301\245\343", "tcha"},
    {"\245\303\245\301\245\345", "tchu"}, {"\245\303\245\301\245\347", "tcho"},
    {"\245\303\245\302", "jji"}, {"\245\303\245\302\245\343", "jjya"},
    {"\245\303\245\302\245\345", "jjyu"}, {"\245\303\245\302\245\347", "jjyo"},
    {"\245\303\245\304", "ttsu"}, {"\245\303\245\305", "zzu"},
    {"\245\303\245\306", "tte"}, {"\245\303\245\307", "dde"},
    {"\245\303\245\310", "tto"}, {"\245\303\245\311", "ddo"},
    {"\245\303\245\317", "hha"}, {"\245\303\245\320", "bba"},
    {"\245\303\245\321", "ppa"},
    {"\245\303\245\322", "hhi"}, {"\245\303\245\322\245\343", "hhya"},
    {"\245\303\245\322\245\345", "hhyu"}, {"\245\303\245\322\245\347", "hhyo"},
    {"\245\303\245\323", "bbi"}, {"\245\303\245\323\245\343", "bbya"},
    {"\245\303\245\323\245\345", "bbyu"}, {"\245\303\245\323\245\347", "bbyo"},
    {"\245\303\245\324", "ppi"}, {"\245\303\245\324\245\343", "ppya"},
    {"\245\303\245\324\245\345", "ppyu"}, {"\245\303\245\324\245\347", "ppyo"},
    {"\245\303\245\325", "ffu"}, {"\245\303\245\325\245\241", "ffa"},
    {"\245\303\245\325\245\243", "ffi"}, {"\245\303\245\325\245\247", "ffe"},
    {"\245\303\245\325\245\251", "ffo"},
    {"\245\303\245\326", "bbu"}, {"\245\303\245\327", "ppu"},
    {"\245\303\245\330", "hhe"}, {"\245\303\245\331", "bbe"},
    {"\245\303\245\332", "ppe"},
    {"\245\303\245\333", "hho"}, {"\245\303\245\334", "bbo"},
    {"\245\303\245\335", "ppo"},
    {"\245\303\245\344", "yya"}, {"\245\303\245\346", "yyu"},
    {"\245\303\245\350", "yyo"},
    {"\245\303\245\351", "rra"}, {"\245\303\245\352", "rri"},
    {"\245\303\245\352\245\343", "rrya"}, {"\245\303\245\352\245\345", "rryu"},
    {"\245\303\245\352\245\347", "rryo"},
    {"\245\303\245\353", "rru"}, {"\245\303\245\354", "rre"},
    {"\245\303\245\355", "rro"},
    {"\245\303\245\364", "vvu"}, {"\245\303\245\364\245\241", "vva"},
    {"\245\303\245\364\245\243", "vvi"}, {"\245\303\245\364\245\247", "vve"},
    {"\245\303\245\364\245\251", "vvo"},

    {"\245\304", "tsu"}, {"\245\305", "zu"},
    {"\245\306", "te"}, {"\245\307", "de"}, {"\245\307\245\243", "di"},
    {"\245\310", "to"}, {"\245\311", "do"},

    {"\245\312", "na"},
    {"\245\313", "ni"}, {"\245\313\245\343", "nya"},
    {"\245\313\245\345", "nyu"}, {"\245\313\245\347", "nyo"},
    {"\245\314", "nu"}, {"\245\315", "ne"}, {"\245\316", "no"},

    {"\245\317", "ha"}, {"\245\320", "ba"}, {"\245\321", "pa"},
    {"\245\322", "hi"}, {"\245\322\245\343", "hya"},
    {"\245\322\245\345", "hyu"}, {"\245\322\245\347", "hyo"},
    {"\245\323", "bi"}, {"\245\323\245\343", "bya"},
    {"\245\323\245\345", "byu"}, {"\245\323\245\347", "byo"},
    {"\245\324", "pi"}, {"\245\324\245\343", "pya"},
    {"\245\324\245\345", "pyu"}, {"\245\324\245\347", "pyo"},
    {"\245\325", "fu"}, {"\245\325\245\241", "fa"},
    {"\245\325\245\243", "fi"}, {"\245\325\245\247", "fe"},
    {"\245\325\245\251", "fo"},
    {"\245\326", "bu"}, {"\245\327", "pu"},
    {"\245\330", "he"}, {"\245\331", "be"}, {"\245\332", "pe"},
    {"\245\333", "ho"}, {"\245\334", "bo"}, {"\245\335", "po"},

    {"\245\336", "ma"},
    {"\245\337", "mi"}, {"\245\337\245\343", "mya"},
    {"\245\337\245\345", "myu"}, {"\245\337\245\347", "myo"},
    {"\245\340", "mu"}, {"\245\341", "me"}, {"\245\342", "mo"},

    {"\245\343", "ya"}, {"\245\344", "ya"},
    {"\245\345", "yu"}, {"\245\346", "yu"},
    {"\245\347", "yo"}, {"\245\350", "yo"},

    {"\245\351", "ra"},
    {"\245\352", "ri"}, {"\245\352\245\343", "rya"},
    {"\245\352\245\345", "ryu"}, {"\245\352\245\347", "ryo"},
    {"\245\353", "ru"}, {"\245\354", "re"}, {"\245\355", "ro"},

    {"\245\356", "wa"}, {"\245\357", "wa"},
    {"\245\360", "i"}, {"\245\361", "e"}, 
    {"\245\362", "wo"}, {"\245\363", "n"},
    {"\245\363\245\242", "n'a"}, {"\245\363\245\244", "n'i"},
    {"\245\363\245\246", "n'u"}, {"\245\363\245\250", "n'e"},
    {"\245\363\245\252", "n'o"},
    {"\245\364", "vu"}, {"\245\364\245\241", "va"},
    {"\245\364\245\243", "vi"}, {"\245\364\245\247", "ve"},
    {"\245\364\245\251", "vo"},
    {"\245\365", "ka"}, {"\245\366", "ke"},

    {"", ""},

}, K2rom_k_table_old[] = { 

    {"\245\241", "a"}, {"\245\242", "a"},
    {"\245\243", "i"}, {"\245\244", "i"},
    {"\245\245", "u"}, {"\245\246", "u"},
    {"\245\247", "e"}, {"\245\250", "e"},
    {"\245\251", "o"}, {"\245\252", "o"},

    {"\245\253", "ka"}, {"\245\254", "ga"},
    {"\245\255", "ki"}, {"\245\255\245\343", "kya"},
    {"\245\255\245\345", "kyu"}, {"\245\255\245\347", "kyo"},
    {"\245\256", "gi"}, {"\245\256\245\343", "gya"},
    {"\245\256\245\345", "gyu"}, {"\245\256\245\347", "gyo"},
    {"\245\257", "ku"}, {"\245\260", "gu"},
    {"\245\261", "ke"}, {"\245\262", "ge"},
    {"\245\263", "ko"}, {"\245\264", "go"},

    {"\245\265", "sa"}, {"\245\266", "za"},
    {"\245\267", "si"}, {"\245\267\245\343", "sya"},
    {"\245\267\245\345", "syu"}, {"\245\267\245\347", "syo"},
    {"\245\270", "zi"}, {"\245\270\245\343", "zya"},
    {"\245\270\245\345", "zyu"}, {"\245\270\245\347", "zyo"},
    {"\245\271", "su"}, {"\245\272", "zu"},
    {"\245\273", "se"}, {"\245\274", "ze"},
    {"\245\275", "so"}, {"\245\276", "zo"},

    {"\245\277", "ta"}, {"\245\300", "da"},
    {"\245\301", "ti"}, {"\245\301\245\343", "tya"},
    {"\245\301\245\345", "tyu"}, {"\245\301\245\347", "tyo"},
    {"\245\302", "di"}, {"\245\302\245\343", "dya"},
    {"\245\302\245\345", "dyu"}, {"\245\302\245\347", "dyo"},

    {"\245\303", "tu"},
    {"\245\303\245\253", "kka"}, {"\245\303\245\254", "gga"},
    {"\245\303\245\255", "kki"}, {"\245\303\245\255\245\343", "kkya"},
    {"\245\303\245\255\245\345", "kkyu"}, {"\245\303\245\255\245\347", "kkyo"},
    {"\245\303\245\256", "ggi"}, {"\245\303\245\256\245\343", "ggya"},
    {"\245\303\245\256\245\345", "ggyu"}, {"\245\303\245\256\245\347", "ggyo"},
    {"\245\303\245\257", "kku"}, {"\245\303\245\260", "ggu"},
    {"\245\303\245\261", "kke"}, {"\245\303\245\262", "gge"},
    {"\245\303\245\263", "kko"}, {"\245\303\245\264", "ggo"},
    {"\245\303\245\265", "ssa"}, {"\245\303\245\266", "zza"},
    {"\245\303\245\267", "ssi"}, {"\245\303\245\267\245\343", "ssya"},
    {"\245\303\245\267\245\345", "ssyu"}, {"\245\303\245\267\245\347", "ssho"},
    {"\245\303\245\270", "zzi"}, {"\245\303\245\270\245\343", "zzya"},
    {"\245\303\245\270\245\345", "zzyu"}, {"\245\303\245\270\245\347", "zzyo"},
    {"\245\303\245\271", "ssu"}, {"\245\303\245\272", "zzu"},
    {"\245\303\245\273", "sse"}, {"\245\303\245\274", "zze"},
    {"\245\303\245\275", "sso"}, {"\245\303\245\276", "zzo"},
    {"\245\303\245\277", "tta"}, {"\245\303\245\300", "dda"},
    {"\245\303\245\301", "tti"}, {"\245\303\245\301\245\343", "ttya"},
    {"\245\303\245\301\245\345", "ttyu"}, {"\245\303\245\301\245\347", "ttyo"},
    {"\245\303\245\302", "ddi"}, {"\245\303\245\302\245\343", "ddya"},
    {"\245\303\245\302\245\345", "ddyu"}, {"\245\303\245\302\245\347", "ddyo"},
    {"\245\303\245\304", "ttu"}, {"\245\303\245\305", "ddu"},
    {"\245\303\245\306", "tte"}, {"\245\303\245\307", "dde"},
    {"\245\303\245\310", "tto"}, {"\245\303\245\311", "ddo"},
    {"\245\303\245\317", "hha"}, {"\245\303\245\320", "bba"},
    {"\245\303\245\321", "ppa"},
    {"\245\303\245\322", "hhi"}, {"\245\303\245\322\245\343", "hhya"},
    {"\245\303\245\322\245\345", "hhyu"}, {"\245\303\245\322\245\347", "hhyo"},
    {"\245\303\245\323", "bbi"}, {"\245\303\245\323\245\343", "bbya"},
    {"\245\303\245\323\245\345", "bbyu"}, {"\245\303\245\323\245\347", "bbyo"},
    {"\245\303\245\324", "ppi"}, {"\245\303\245\324\245\343", "ppya"},
    {"\245\303\245\324\245\345", "ppyu"}, {"\245\303\245\324\245\347", "ppyo"},
    {"\245\303\245\325", "hhu"}, {"\245\303\245\325\245\241", "ffa"},
    {"\245\303\245\325\245\243", "ffi"}, {"\245\303\245\325\245\247", "ffe"},
    {"\245\303\245\325\245\251", "ffo"},
    {"\245\303\245\326", "bbu"}, {"\245\303\245\327", "ppu"},
    {"\245\303\245\330", "hhe"}, {"\245\303\245\331", "bbe"},
    {"\245\303\245\332", "ppe"},
    {"\245\303\245\333", "hho"}, {"\245\303\245\334", "bbo"},
    {"\245\303\245\335", "ppo"},
    {"\245\303\245\344", "yya"},
    {"\245\303\245\346", "yyu"},
    {"\245\303\245\350", "yyo"},
    {"\245\303\245\351", "rra"},
    {"\245\303\245\352", "rri"}, {"\245\303\245\352\245\343", "rrya"},
    {"\245\303\245\352\245\345", "rryu"}, {"\245\303\245\352\245\347", "rryo"},
    {"\245\303\245\353", "rru"},
    {"\245\303\245\354", "rre"},
    {"\245\303\245\355", "rro"},
    {"\245\303\245\364", "vvu"}, {"\245\303\245\364\245\241", "vva"},
    {"\245\303\245\364\245\243", "vvi"}, {"\245\303\245\364\245\247", "vve"},
    {"\245\303\245\364\245\251", "vvo"},

    {"\245\304", "tu"}, {"\245\305", "du"},
    {"\245\306", "te"}, {"\245\307", "de"},
    {"\245\310", "to"}, {"\245\311", "do"},

    {"\245\312", "na"},
    {"\245\313", "ni"}, {"\245\313\245\343", "nya"},
    {"\245\313\245\345", "nyu"}, {"\245\313\245\347", "nyo"},
    {"\245\314", "nu"}, {"\245\315", "ne"}, {"\245\316", "no"},

    {"\245\317", "ha"}, {"\245\320", "ba"}, {"\245\321", "pa"},
    {"\245\322", "hi"}, {"\245\322\245\343", "hya"},
    {"\245\322\245\345", "hyu"}, {"\245\322\245\347", "hyo"},
    {"\245\323", "bi"}, {"\245\323\245\343", "bya"},
    {"\245\323\245\345", "byu"}, {"\245\323\245\347", "byo"},
    {"\245\324", "pi"}, {"\245\324\245\343", "pya"},
    {"\245\324\245\345", "pyu"}, {"\245\324\245\347", "pyo"},
    {"\245\325", "hu"}, {"\245\325\245\241", "fa"},
    {"\245\325\245\243", "fi"}, {"\245\325\245\247", "fe"},
    {"\245\325\245\251", "fo"},
    {"\245\326", "bu"}, {"\245\327", "pu"},
    {"\245\330", "he"}, {"\245\331", "be"}, {"\245\332", "pe"},
    {"\245\333", "ho"}, {"\245\334", "bo"}, {"\245\335", "po"},

    {"\245\336", "ma"},
    {"\245\337", "mi"}, {"\245\337\245\343", "mya"},
    {"\245\337\245\345", "myu"}, {"\245\337\245\347", "myo"},
    {"\245\340", "mu"}, {"\245\341", "me"}, {"\245\342", "mo"},

    {"\245\343", "ya"}, {"\245\344", "ya"},
    {"\245\345", "yu"}, {"\245\346", "yu"},
    {"\245\347", "yo"}, {"\245\350", "yo"},

    {"\245\351", "ra"},
    {"\245\352", "ri"}, {"\245\352\245\343", "rya"},
    {"\245\352\245\345", "ryu"}, {"\245\352\245\347", "ryo"},
    {"\245\353", "ru"}, {"\245\354", "re"}, {"\245\355", "ro"},

    {"\245\356", "wa"}, {"\245\357", "wa"},
    {"\245\360", "i"}, {"\245\361", "e"}, 
    {"\245\362", "wo"}, {"\245\363", "n"},
    {"\245\363\245\242", "n'a"}, {"\245\363\245\244", "n'i"},
    {"\245\363\245\246", "n'u"}, {"\245\363\245\250", "n'e"},
    {"\245\363\245\252", "n'o"},
    {"\245\364", "vu"}, {"\245\364\245\241", "va"},
    {"\245\364\245\243", "vi"}, {"\245\364\245\247", "ve"},
    {"\245\364\245\251", "vo"},
    {"\245\365", "ka"}, {"\245\366", "ke"},
    {"", ""},

}, K2rom_k_table[] = { 

    {"\245\241", "a"}, {"\245\242", "a"},
    {"\245\243", "i"}, {"\245\244", "i"},
    {"\245\245", "u"}, {"\245\246", "u"},
    {"\245\247", "e"}, {"\245\250", "e"},
    {"\245\251", "o"}, {"\245\252", "o"},

    {"\245\253", "ka"}, {"\245\254", "ga"},
    {"\245\255", "ki"}, {"\245\255\245\343", "kya"},
    {"\245\255\245\345", "kyu"}, {"\245\255\245\347", "kyo"},
    {"\245\256", "gi"}, {"\245\256\245\343", "gya"},
    {"\245\256\245\345", "gyu"}, {"\245\256\245\347", "gyo"},
    {"\245\257", "ku"}, {"\245\260", "gu"},
    {"\245\261", "ke"}, {"\245\262", "ge"},
    {"\245\263", "ko"}, {"\245\264", "go"},

    {"\245\265", "sa"}, {"\245\266", "za"},
    {"\245\267", "si"}, {"\245\267\245\343", "sya"},
    {"\245\267\245\345", "syu"}, {"\245\267\245\347", "syo"},
    {"\245\270", "zi"}, {"\245\270\245\343", "zya"},
    {"\245\270\245\345", "zyu"}, {"\245\270\245\347", "zyo"},
    {"\245\271", "su"}, {"\245\272", "zu"},
    {"\245\273", "se"}, {"\245\274", "ze"},
    {"\245\275", "so"}, {"\245\276", "zo"},

    {"\245\277", "ta"}, {"\245\300", "da"},
    {"\245\301", "ti"}, {"\245\301\245\343", "tya"},
    {"\245\301\245\345", "tyu"}, {"\245\301\245\347", "tyo"},
    {"\245\302", "di"}, {"\245\302\245\343", "dya"},
    {"\245\302\245\345", "dyu"}, {"\245\302\245\347", "dyo"},

    {"\245\303", "tu"},
    {"\245\303\245\253", "kka"}, {"\245\303\245\254", "gga"},
    {"\245\303\245\255", "kki"}, {"\245\303\245\255\245\343", "kkya"},
    {"\245\303\245\255\245\345", "kkyu"}, {"\245\303\245\255\245\347", "kkyo"},
    {"\245\303\245\256", "ggi"}, {"\245\303\245\256\245\343", "ggya"},
    {"\245\303\245\256\245\345", "ggyu"}, {"\245\303\245\256\245\347", "ggyo"},
    {"\245\303\245\257", "kku"}, {"\245\303\245\260", "ggu"},
    {"\245\303\245\261", "kke"}, {"\245\303\245\262", "gge"},
    {"\245\303\245\263", "kko"}, {"\245\303\245\264", "ggo"},
    {"\245\303\245\265", "ssa"}, {"\245\303\245\266", "zza"},
    {"\245\303\245\267", "ssi"}, {"\245\303\245\267\245\343", "ssya"},
    {"\245\303\245\267\245\345", "ssyu"}, {"\245\303\245\267\245\347", "ssyo"},
    {"\245\303\245\270", "zzi"}, {"\245\303\245\270\245\343", "zzya"},
    {"\245\303\245\270\245\345", "zzyu"}, {"\245\303\245\270\245\347", "zzyo"},
    {"\245\303\245\271", "ssu"}, {"\245\303\245\272", "zzu"},
    {"\245\303\245\273", "sse"}, {"\245\303\245\274", "zze"},
    {"\245\303\245\275", "sso"}, {"\245\303\245\276", "zzo"},
    {"\245\303\245\277", "tta"}, {"\245\303\245\300", "dda"},
    {"\245\303\245\301", "tti"}, {"\245\303\245\301\245\343", "ttya"},
    {"\245\303\245\301\245\345", "ttyu"}, {"\245\303\245\301\245\347", "ttyo"},
    {"\245\303\245\302", "ddi"}, {"\245\303\245\302\245\343", "ddya"},
    {"\245\303\245\302\245\345", "ddyu"}, {"\245\303\245\302\245\347", "ddyo"},
    {"\245\303\245\304", "ttu"}, {"\245\303\245\305", "ddu"},
    {"\245\303\245\306", "tte"}, {"\245\303\245\307", "dde"},
    {"\245\303\245\310", "tto"}, {"\245\303\245\311", "ddo"},
    {"\245\303\245\317", "hha"}, {"\245\303\245\320", "bba"},
    {"\245\303\245\321", "ppa"},
    {"\245\303\245\322", "hhi"}, {"\245\303\245\322\245\343", "hhya"},
    {"\245\303\245\322\245\345", "hhyu"}, {"\245\303\245\322\245\347", "hhyo"},
    {"\245\303\245\323", "bbi"}, {"\245\303\245\323\245\343", "bbya"},
    {"\245\303\245\323\245\345", "bbyu"}, {"\245\303\245\323\245\347", "bbyo"},
    {"\245\303\245\324", "ppi"}, {"\245\303\245\324\245\343", "ppya"},
    {"\245\303\245\324\245\345", "ppyu"}, {"\245\303\245\324\245\347", "ppyo"},
    {"\245\303\245\325", "hhu"}, {"\245\303\245\325\245\241", "ffa"},
    {"\245\303\245\325\245\243", "ffi"}, {"\245\303\245\325\245\247", "ffe"},
    {"\245\303\245\325\245\251", "ffo"},
    {"\245\303\245\326", "bbu"}, {"\245\303\245\327", "ppu"},
    {"\245\303\245\330", "hhe"}, {"\245\303\245\331", "bbe"},
    {"\245\303\245\332", "ppe"},
    {"\245\303\245\333", "hho"}, {"\245\303\245\334", "bbo"},
    {"\245\303\245\335", "ppo"},
    {"\245\303\245\344", "yya"},
    {"\245\303\245\346", "yyu"},
    {"\245\303\245\350", "yyo"},
    {"\245\303\245\351", "rra"},
    {"\245\303\245\352", "rri"}, {"\245\303\245\352\245\343", "rrya"},
    {"\245\303\245\352\245\345", "rryu"}, {"\245\303\245\352\245\347", "rryo"},
    {"\245\303\245\353", "rru"},
    {"\245\303\245\354", "rre"},
    {"\245\303\245\355", "rro"},
    {"\245\303\245\364", "vvu"}, {"\245\303\245\364\245\241", "vva"},
    {"\245\303\245\364\245\243", "vvi"}, {"\245\303\245\364\245\247", "vve"},
    {"\245\303\245\364\245\251", "vvo"},

    {"\245\304", "tu"}, {"\245\305", "du"},
    {"\245\306", "te"}, {"\245\307", "de"},
    {"\245\310", "to"}, {"\245\311", "do"},

    {"\245\312", "na"},
    {"\245\313", "ni"}, {"\245\313\245\343", "nya"},
    {"\245\313\245\345", "nyu"}, {"\245\313\245\347", "nyo"},
    {"\245\314", "nu"}, {"\245\315", "ne"}, {"\245\316", "no"},

    {"\245\317", "ha"}, {"\245\320", "ba"}, {"\245\321", "pa"},
    {"\245\322", "hi"}, {"\245\322\245\343", "hya"},
    {"\245\322\245\345", "hyu"}, {"\245\322\245\347", "hyo"},
    {"\245\323", "bi"}, {"\245\323\245\343", "bya"},
    {"\245\323\245\345", "byu"}, {"\245\323\245\347", "byo"},
    {"\245\324", "pi"}, {"\245\324\245\343", "pya"},
    {"\245\324\245\345", "pyu"}, {"\245\324\245\347", "pyo"},
    {"\245\325", "hu"}, {"\245\325\245\241", "fa"},
    {"\245\325\245\243", "fi"}, {"\245\325\245\247", "fe"},
    {"\245\325\245\251", "fo"},
    {"\245\326", "bu"}, {"\245\327", "pu"},
    {"\245\330", "he"}, {"\245\331", "be"}, {"\245\332", "pe"},
    {"\245\333", "ho"}, {"\245\334", "bo"}, {"\245\335", "po"},

    {"\245\336", "ma"},
    {"\245\337", "mi"}, {"\245\337\245\343", "mya"},
    {"\245\337\245\345", "myu"}, {"\245\337\245\347", "myo"},
    {"\245\340", "mu"}, {"\245\341", "me"}, {"\245\342", "mo"},

    {"\245\343", "ya"}, {"\245\344", "ya"},
    {"\245\345", "yu"}, {"\245\346", "yu"},
    {"\245\347", "yo"}, {"\245\350", "yo"},

    {"\245\351", "ra"},
    {"\245\352", "ri"}, {"\245\352\245\343", "rya"},
    {"\245\352\245\345", "ryu"}, {"\245\352\245\347", "ryo"},
    {"\245\353", "ru"}, {"\245\354", "re"}, {"\245\355", "ro"},

    {"\245\356", "wa"}, {"\245\357", "wa"},
    {"\245\360", "i"}, {"\245\361", "e"}, 
    {"\245\362", "wo"}, {"\245\363", "n"},
    {"\245\363\245\242", "n'a"}, {"\245\363\245\244", "n'i"},
    {"\245\363\245\246", "n'u"}, {"\245\363\245\250", "n'e"},
    {"\245\363\245\252", "n'o"},
    {"\245\364", "vu"}, {"\245\364\245\241", "va"},
    {"\245\364\245\243", "vi"}, {"\245\364\245\247", "ve"},
    {"\245\364\245\251", "vo"},
    {"\245\365", "ka"}, {"\245\366", "ke"},
    {"", ""},
};

#define K2rom_buflen 11

int
K2rom(c, n, type)
     Character *c;
     Character *n;
     int type;
{
    static int index_table[0x81];
    static int index_made = UNINITIALIZED;
    static struct K2rom_tbl *K2rom_ptr;
    struct K2rom_tbl *p;
    int i, clen, ylen;
    unsigned char buffer[K2rom_buflen];
    int max_match, match_more;
    char *max_romaji;

    if (index_made != romaji_type) {
	int last;

	for (i = 0; i < 0x81; ++ i) {
	    index_table[i] = -1;
	}
	index_table[0x21] = 0;
	K2rom_ptr = (romaji_type == HEPBURN) ? K2rom_h_table : K2rom_k_table;
	if (use_old_romaji_table && romaji_type == HEPBURN)
	    K2rom_ptr = K2rom_h_table_old;
	if (use_old_romaji_table && romaji_type == KUNREI)
	    K2rom_ptr = K2rom_k_table_old;
	for (p = K2rom_ptr, i = 0; *(p->kana) != '\0'; ++ p, ++ i) {
	    index_table[(((p->kana)[1])&0x7f)+1] = i+1;
	}
	last = i;
	for (i = 0x80; i >= 0; -- i) {
	    if (index_table[i] == -1)
		index_table[i] = last;
	    else
		last = index_table[i];
	}
	index_made = romaji_type;
    }

    if ((c[0].c1 == 0xa1) && (c[0].c2 == 0xbc)) {
	n[0].type = type;
	n[0].c1 = '^';
	n[1].type = OTHER;
	n[1].c1 = 0;
	n[1].c2 = 0;
	return 1;
    }

    buffer[K2rom_buflen-1] = '\0'; clen = K2rom_buflen-1;
    for (i = 0; i < (K2rom_buflen-1)/2; i ++) {
	buffer[i*2  ] = c[i].c1;
	buffer[i*2+1] = c[i].c2;

	if (c[i].c1 == '\0') {
	    clen = i*2;
	    break;
	}
    }

    if (clen == 0) {
	n[0].type=OTHER;
	n[0].c1 = '\0';
	return 0;
    }

    max_match = 0;
    max_romaji = NULL;
    match_more = 0;
    for (i = index_table[buffer[1]&0x7f];
	 i < index_table[(buffer[1]&0x7f)+1];
	 ++ i) {
	ylen = strlen((const char *)(K2rom_ptr[i].kana));
	if (clen >= ylen) {
	    if (max_match < ylen) {
		if (strncmp((const char *)buffer, (const char *)K2rom_ptr[i].kana, ylen) == 0) {
		    max_match = ylen/2;
		    max_romaji = K2rom_ptr[i].romaji;
		}
	    }
	} else {
	    if (match_more == 0)
		if (strncmp((const char *)buffer, (const char *)K2rom_ptr[i].kana, clen) == 0)
		    match_more = 1;
	}
    }

    if (max_romaji == NULL) {
	i = 0;
	max_match = 1;
    } else {
	for (i = 0; max_romaji[i] != '\0'; ++ i) {
	    n[i].type=type;
	    n[i].c1 = max_romaji[i];
	}
    }
    n[i].type=OTHER;
    n[i].c1 = '\0';

    return (match_more == 0) ? max_match : -max_match;
}

int
K2a(c, n)
     Character *c;
     Character *n;
{
    return K2rom(c, n, ASCII);
}

int
K2j(c, n)
     Character *c;
     Character *n;
{
    return K2rom(c, n, JISROMAN);
}

int
K2H(c, n)
     Character *c;
     Character *n;
{
    if (c[0].c1 == 0xa5) {
	if (c[0].c2 < 0xf4) { /* != charcter code(\245\364),charcter code(\245\365),charcter code(\245\366) */
	    n[0].type = JIS83;
	    n[0].c1 = 0xa4;
	    n[0].c2 = c[0].c2;
	    n[1].type = OTHER;
	    n[1].c1 = 0;
	    n[1].c2 = 0;
	    return 1;
	} else if (c[0].c2 == 0xf4) { /* == charcter code(\245\364) */
	    n[0].type = JIS83;
	    n[0].c1 = 0xa4;
	    n[0].c2 = 0xa6;
	    n[1].type = JIS83;
	    n[1].c1 = 0xa1;
	    n[1].c2 = 0xab;
	    n[2].type = OTHER;
	    n[2].c1 = 0;
	    n[2].c2 = 0;
	    return 1;
	} else if (c[0].c2 == 0xf5) { /* == charcter code(\245\365) */
	    n[0].type = JIS83;
	    n[0].c1 = 0xa4;
	    n[0].c2 = 0xab;
	    n[1].type = OTHER;
	    n[1].c1 = 0;
	    n[1].c2 = 0;
	    return 1;
	} else if (c[0].c2 == 0xf6) { /* == charcter code(\245\366) */
	    n[0].type = JIS83;
	    n[0].c1 = 0xa4;
	    n[0].c2 = 0xb1;
	    n[1].type = OTHER;
	    n[1].c1 = 0;
	    n[1].c2 = 0;
	    return 1;
	}
    } else if (c[0].c1 == 0xa1) {
	if ((c[0].c2 == 0xbc) || (c[0].c2 == 0xab) || (c[0].c2 == 0xac)) {
	    n[0].type = JIS83;
	    n[0].c1 = c[0].c1;
	    n[0].c2 = c[0].c2;
	    n[1].type = OTHER;
	    n[1].c1 = 0;
	    n[1].c2 = 0;
	    return 1;
	} else if (c[0].c2 == 0xb3 || c[0].c2 == 0xb4) {
	    n[0].type = JIS83;
	    n[0].c1 = c[0].c1;
	    n[0].c2 = c[0].c2 + 2;
	    n[1].type = OTHER;
	    n[1].c1 = 0;
	    n[1].c2 = 0;
	    return 1;
	}
    }
    n[0].type = OTHER;
    n[0].c1 = 0;
    n[0].c2 = 0;
    return 1;
}

int
K2k(c, n)
     Character *c;
     Character *n;
{
    unsigned char *b;
    int i;

    static unsigned char K2k_table[0x60][3] = {
	"",   "'",  "1",  "(",  "2",  ")",  "3",  "*",
	"4",  "+",  "5",  "6",  "6^", "7",  "7^", "8",
	"8^", "9",  "9^", ":",  ":^", ";",  ";^", "<",
	"<^", "=",  "=^", ">",  ">^", "?",  "?^", "@",
	"@^", "A",  "A^", "/",  "B",  "B^", "C",  "C^",
	"D",  "D^", "E",  "F",  "G",  "H",  "I",  "J",
	"J^", "J_", "K",  "K^", "K_", "L",  "L^", "L_",
	"M",  "M^", "M_", "N",  "N^", "N_", "O",  "P",
	"Q",  "R",  "S",  ",",  "T",  "-",  "U",  ".",
	"V",  "W",  "X",  "Y",  "Z",  "[",  "\\",  "\\",
	"2",  "4",  "&",  "]",  "3^", "6",  "9",   "",
	"",   "",   "",   "",   "",   "",   "",   ""};

    if (c[0].c1 == 0xa5) {
	for (b = K2k_table[(c[0].c2 & 0x7f) - 0x20], i = 0;
	     *b != '\0'; ++ b, ++ i) {
	    n[i].type = KATAKANA;
	    n[i].c1 = *b;
	}
	n[i].type = OTHER;
	n[i].c1 = 0;
    } else if ((c[0].c1 == 0xa1) &&
	       ((c[0].c2 == 0xbc) || (c[0].c2 == 0xab) || (c[0].c2 == 0xac))) {
	n[0].type = KATAKANA;
	switch(c[0].c2) {
	  case 0xbc:
	    n[0].c1 = '0'; break;
	  case 0xab:
	    n[0].c1 = '^'; break;
	  case 0xac:
	    n[0].c1 = '_'; break;
	}
	n[1].type = OTHER;
	n[1].c1 = 0;
    } else {
	n[0].type = OTHER;
	n[0].c1 = 0;
    }
    return 1;
}

#ifdef WAKATIGAKI
int
K2K(c, n)
     Character *c;
     Character *n;
{
    if (c[0].c1 == 0xa5 || (c[0].c1 == 0xa1 && c[0].c2 == 0xbc)) {
	n[0].type = JIS83;
	n[0].c1 = c[0].c1;
	n[0].c2 = c[0].c2;
	n[1].type = OTHER;
	n[1].c1 = 0;
	n[1].c2 = 0;
    }
    return 1;
}
#endif /* WAKATIGAKI */
