% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/meshes.R
\name{Mesh}
\alias{Mesh}
\title{Make a 3D mesh}
\usage{
Mesh(vertices, faces, mesh = NULL, triangulate = FALSE, normals = FALSE)
}
\arguments{
\item{vertices}{a numeric matrix with three columns, or a \code{bigq}
matrix with three columns}

\item{faces}{either an integer matrix (each row provides the vertex indices
of the corresponding face) or a list of integer vectors, each one
providing the vertex indices of the corresponding face}

\item{mesh}{if not \code{NULL}, this argument takes precedence over \code{vertices}
and \code{faces}, and must be either a list containing the fields \code{vertices}
and \code{faces} (objects as described above), otherwise a \strong{rgl} mesh
(i.e. a \code{mesh3d} object)}

\item{triangulate}{Boolean, whether to triangulate the faces}

\item{normals}{Boolean, whether to compute the normals}
}
\value{
A list giving the vertices, the edges, the faces of the mesh, the
  exterior edges, the exterior vertices and optionally the normals. 
  If \code{triangulate=TRUE}, this list has two additional components 
  \code{edges0} and \code{normals0} giving the edges and the normals 
  before the triangulation, unless the mesh is already triangulated, 
  in which case the \code{triangulate} option is ignored.
}
\description{
Make a 3D mesh from given vertices and faces; the returned
  faces are coherently oriented, normals are computed if desired, and
  triangulation is performed if desired. The mesh is also cleaned: 
  duplicated vertices or faces are merged, and isolated vertices are removed.
}
\examples{
library(PolygonSoup)
library(rgl)

# a tetrahedron with ill-oriented faces ####
vertices <- rbind(
  c(-1, -1, -1),
  c(1, 1, -1),
  c(1, -1, 1),
  c(-1, 1, 1)
)
faces <- rbind(
  c(1, 2, 3),
  c(3, 4, 2),
  c(4, 2, 1),
  c(4, 3, 1)
)

# plot the tetrahedron, hiding the back of the faces
# then some faces do not appear, as their orientation is not correct
tmesh1 <- tmesh3d(
  vertices = t(vertices),
  indices = t(faces),
  homogeneous = FALSE
)
open3d(windowRect = c(50, 50, 562, 562))
shade3d(tmesh1, color = "green", back = "cull")

# now run the `Mesh` function
mesh2 <- Mesh(vertices, faces, normals = FALSE)
# plot the tetrahedron, hiding the back of the faces
# then all faces appear now
tmesh2 <- toRGL(mesh2)
open3d(windowRect = c(50, 50, 562, 562))
shade3d(tmesh2, color = "blue", back = "cull")

# illustration of the cleaning feature ####
# we construct a mesh with a lot of duplicated vertices
library(misc3d) # to compute a mesh of an isosurface
a <- 0.94; mu <- 0.56; c <- 0.34 # cyclide parameters
f <- function(x, y, z, a, c, mu){ # implicit equation of the cyclide
  b <- sqrt(a^2 - c^2)
  (x^2 + y^2 + z^2 - mu^2 + b^2)^2 - 4*(a*x - c*mu)^2 - 4*b^2*y^2
}
x <- seq(-c - mu - a, abs(mu - c) + a, length.out = 45)
y <- seq(-mu - a, mu + a, length.out = 45)
z <- seq(-mu - c, mu + c, length.out = 30)
g <- expand.grid(x = x, y = y, z = z)
voxel <- array(with(g, f(x, y, z, a, c, mu)), c(45, 45, 30))
cont <- computeContour3d(voxel, level = 0, x = x, y = y, z = z)
ids <- matrix(1:nrow(cont), ncol = 3, byrow = TRUE)
# run the `Mesh` function 
mesh <- Mesh(cont, ids, normals = TRUE)
# plot the cyclide
tmesh <- toRGL(mesh)
open3d(windowRect = c(50, 50, 562, 562), zoom = 0.9)
shade3d(tmesh, color = "green")

# illustration of the `triangulate` option ####
# the faces of the truncated icosahedron are hexagonal or pentagonal:
truncatedIcosahedron[["faces"]]
# so we triangulate them:
mesh <- Mesh(
  mesh = truncatedIcosahedron,
  triangulate = TRUE, normals = FALSE
)
# now we can plot the truncated icosahedron
tmesh <- toRGL(mesh)
open3d(windowRect = c(50, 50, 562, 562), zoom = 0.9)
shade3d(tmesh, color = "orange")
}
\seealso{
See \code{\link{plotEdges}} for more details about the edges 
  returned by this function.
}
