

# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~#
#                             FNs FOR MODEL 1 (FIXED EFFECTS)                         #
# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~#


################################ FN: CORRECTED ESTIMATE ################################

#' Estimate publication bias-corrected meta-analysis
#'
#' For a known selection ratio, eta, estimates a publication bias-corrected pooled point
#' estimate and confidence interval. Model options include fixed-effects, robust independent, and robust
#' clustered specifications.
#' @param yi A vector of point estimates to be meta-analyzed
#' @param vi A vector of estimated variances for the point estimates
#' @param eta The number of times more likely an affirmative study is to be published than a nonaffirmative study. See Details.
#' @param clustervar A character, factor, or numeric vector with the same length as yi. Unique values should indicate
#' unique clusters of point estimates. By default, assumes all point estimates are independent by default.
#' @param model "fixed" for fixed-effects or "robust" for robust random-effects
#' @param selection.tails 1 (for one-tailed selection, recommended for its conservatism) or 2 (for two-tailed selection)
#' @param CI.level Confidence interval level (as proportion) for the corrected point estimate
#' @param small Should inference allow for a small meta-analysis? We recommend always using TRUE.
#' @import
#' metafor
#' stats
#' robumeta
#' ggplot2
#' graphics
#' MetaUtility
#' @importFrom
#' dplyr %>% group_by summarise
#' @export
#' @details
#' The selection ratio, eta, represents the number of times more likely affirmative studies (i.e., those with a "statistically significant" and positive estimate)
#' are to be published than nonaffirmative studies (i.e., those with a "nonsignificant" or negative estimate).
#' @references
#' 1. Mathur MB & VanderWeele TJ (2019). Sensitivity analysis for publication bias in meta-analyses. Preprint available at XXX.
#' @examples
#' # calculate effect sizes from example dataset in metafor
#' require(metafor)
#' dat = metafor::escalc(measure="RR", ai=tpos, bi=tneg, ci=cpos, di=cneg, data=dat.bcg)
#'
#' # first fit fixed-effects model without any bias correction
#' rma( yi, vi, data = dat, method = "FE" )
#'
#' # notice this is the same as passing eta = 1 to corrected_meta, which makes sense
#' corrected_meta( yi = dat$yi,
#'                 vi = dat$vi,
#'                 eta = 1,
#'                 model = "fixed",
#'                 selection.tails = 1,
#'                 CI.level = 0.95,
#'                 small = TRUE )
#'
#' # assume a known selection ratio of 5
#' # i.e., affirmative results are 5x more likely to be published
#' #  than nonaffirmative
#' corrected_meta( yi = dat$yi,
#'                 vi = dat$vi,
#'                 eta = 5,
#'                 model = "fixed",
#'                 selection.tails = 1,
#'                 CI.level = 0.95,
#'                 small = TRUE )
#'
#' # same selection ratio, but now account for heterogeneity
#' # and clustering via robust specification
#' corrected_meta( yi = dat$yi,
#'                 vi = dat$vi,
#'                 eta = 5,
#'                 clustervar = dat$author,
#'                 model = "robust",
#'                 selection.tails = 1,
#'                 CI.level = 0.95,
#'                 small = TRUE )
#'
#' ##### Make sensitivity plot as in Mathur & VanderWeele (2019) #####
#' # range of parameters to try (more dense at the very small ones)
#' eta.list = as.list( c( 200, 150, 100, 50, 40, 30, 20, rev( seq(1,15,1) ) ) )
#' res.list = lapply( eta.list, function(x) {
#'                     cat("\n Working on eta = ", x)
#'                     return( corrected_meta( yi = dat$yi,
#'                                             vi = dat$vi,
#'                                             eta = x,
#'                                             model = "robust",
#'                                             clustervar = dat$author ) )
#'                                         }
#'                       )
#'
#' # put results for each eta in a dataframe
#' res.df = as.data.frame( do.call( "rbind", res.list ) )
#'
#' require(ggplot2)
#' ggplot( data = res.df, aes( x = eta, y = est ) ) +
#'
#'   geom_ribbon( data = res.df, aes( x = eta, ymin = lo, ymax = hi ), fill = "gray" ) +
#'
#'   geom_line( lwd = 1.2 ) +
#'   xlab( bquote( eta ) ) +
#'   ylab( bquote( hat(mu)[eta] ) ) +
#'
#'   theme_classic()


corrected_meta = function( yi,
                           vi,
                           eta,
                           clustervar = 1:length(yi),
                           model,
                           selection.tails = 1,
                           CI.level = 0.95,
                           small = TRUE ) {

  # stop if eta doesn't make sense
  if ( eta < 1 ) stop( "Eta must be at least 1.")

  # number of point estimates
  k = length(yi)

  alpha = 1 - CI.level

  # warn if clusters but user said fixed
  nclusters = length( unique( clustervar ) )
  if ( nclusters < k & model == "fixed" ) {
    warning( "Clusters exist, but will be ignored due to fixed-effects specification. To accommodate clusters, instead choose model = robust.")
  }

  # check and flip if naive point estimate is negative
  # do standard meta
  m0 = rma.uni(yi, vi)

  # reverse signs if needed to have pooled point estimate > 0
  if ( m0$b < 0 ) {
    # keep track so that we can flip back at the end
    flipped = TRUE
    yif = -yi
  } else {
    flipped = FALSE
    yif = yi
  }

  # 2-sided p-values for each study even if 1-tailed selection
  pvals = 2 * ( 1 - pnorm( abs(yif) / sqrt(vi) ) )

  # affirmative indicator based on selection tails
  if ( selection.tails == 1 ) A = (pvals < 0.05) & (yif > 0)
  if ( selection.tails == 2 ) A = (pvals < 0.05)

  k.affirmative = sum(A)
  k.nonaffirmative = k - sum(A)

  if ( k.affirmative == 0 | k.nonaffirmative == 0 ) {
    stop( "There are zero affirmative studies or zero nonaffirmative studies. Model estimation cannot proceed.")
  }

  dat = data.frame( yi, yif, vi, A, clustervar )


  ##### Fixed-Effects Model #####
  if ( model == "fixed" ) {

    # FE mean and sum of weights stratified by affirmative vs. nonaffirmative
    strat = dat %>% group_by(A) %>%
      summarise( nu = sum( 1 / vi ),
                 ybar = sum( yi / vi ) )

    # components of bias-corrected estimate by affirmative status
    ybarN = strat$ybar[ strat$A == 0 ]
    ybarS = strat$ybar[ strat$A == 1 ]
    nuN = strat$nu[ strat$A == 0 ]
    nuS = strat$nu[ strat$A == 1 ]

    # corrected pooled point estimate
    est = ( eta * ybarN + ybarS ) / ( eta * nuN + nuS )

    # inference
    var = ( eta^2 * nuN + nuS ) / ( eta * nuN + nuS )^2
    se = sqrt(var)

    # z-based inference
    if ( small == FALSE ) {
      lo = est - qnorm( 1 - (alpha/2) ) * sqrt(var)
      hi = est + qnorm( 1 - (alpha/2) ) * sqrt(var)
      z =  abs( est / sqrt(var) )
      pval.est = 2 * ( 1 - pnorm( z ) )
    }

    # t-based inference
    if ( small == TRUE ) {
      df = k - 1
      lo = est - qt( 1 - (alpha/2), df = df ) * sqrt(var)
      hi = est + qt( 1 - (alpha/2), df = df ) * sqrt(var)
      t =  abs( est / sqrt(var) )
      pval.est = 2 * ( 1 - pt( t, df = df ) )
    }
} # end fixed = TRUE

  ##### Robust Independent and Robust Clustered #####
  if ( model == "robust" ) {

    # weight for model
    weights = rep( 1, length(pvals) )
    weights[ A == FALSE ] = eta

    # initialize a dumb (unclustered and uncorrected) version of tau^2
    # which is only used for constructing weights
    meta.re = rma.uni( yi = yi,
                       vi = vi)
    t2hat.naive = meta.re$tau2

    # fit weighted robust model
    meta.robu = robu( yi ~ 1,
                      studynum = clustervar,
                      data = dat,
                      userweights = weights / (vi + t2hat.naive),
                      var.eff.size = vi,
                      small = small )

    est = as.numeric(meta.robu$b.r)
    se = meta.robu$reg_table$SE
    lo = meta.robu$reg_table$CI.L
    hi = meta.robu$reg_table$CI.U
    pval.est = meta.robu$reg_table$prob
    eta = eta
  } # end robust = TRUE

    return( data.frame( est,
                        se,
                        lo,
                        hi,
                        pval = pval.est,
                        eta = eta,
                        k.affirmative,
                        k.nonaffirmative ) )
}




###############################

#' Estimate publication bias needed to "explain away" results
#'
#' Estimates the S-value, defined as the severity of publication bias (i.e., the ratio
#' by which affirmative studies are more likely to be published than nonaffirmative studies)
#' that would be required to shift the pooled point estimate or its confidence interval limit
#' to the value \code{q}.
#' @param yi A vector of point estimates to be meta-analyzed
#' @param vi A vector of estimated variances for the point estimates
#' @param q The attenuated value to which to shift the point estimate or CI. Should be specified on the same scale as \code{yi}
#' (e.g., if \code{yi} is on the log-RR scale, then \code{q} should be as well).
#' @param clustervar A character, factor, or numeric vector with the same length as \code{yi}. Unique values should indicate
#' unique clusters of point estimates. By default, assumes all point estimates are independent.
#' @param model \code{"fixed"} for fixed-effects or \code{"robust"} for robust random-effects
#' @param eta.grid A vector of values of eta to try for the grid search. The vector should start at 1 and end at a large number, such
#' as 200. For a more precise estimate of eta, use more closely-spaced values. This argument is only needed when \code{model = "robust"}.
#' @param CI.level Confidence interval level (as proportion) for the corrected point estimate
#' @param small Should inference allow for a small meta-analysis? We recommend using \code{TRUE} even for large meta-analyses.
#' @import
#' metafor
#' stats
#' robumeta
#' ggplot2
#' @importFrom
#' dplyr %>% group_by summarise
#' @export
#' @details
#' For example, if the S-value for the point estimate is 30 with \code{q=0}, this indicates that affirmative studies
#' (i.e., those with a "statistically significant" and positive estimate) would need to be 30-fold more likely to be published
#' than nonaffirmative studies (i.e., those with a "nonsignificant" or negative estimate) to attenuate the pooled point estimate to
#' \code{q}.
#' @references
#' 1. Mathur MB & VanderWeele TJ (2019). Sensitivity analysis for publication bias in meta-analyses. Preprint available at XXX.
#' @examples
#' # calculate effect sizes from example dataset in metafor
#' require(metafor)
#' dat = metafor::escalc(measure="RR", ai=tpos, bi=tneg, ci=cpos, di=cneg, data=dat.bcg)
#'
#' ##### Fixed-Effects Specification #####
#' # S-values and worst-case meta-analysis under fixed-effects specification
#' svals.FE.0 = svalue( yi = dat$yi,
#'                    vi = dat$vi,
#'                    q = 0,
#'                    model = "fixed" )
#'
#' # publication bias required to shift point estimate to 0
#' svals.FE.0$svals$sval.est
#'
#' # and to shift CI to include 0
#' svals.FE.0$svals$sval.ci
#'
#' # now try shifting to a nonzero value (RR = 0.90)
#' svals.FE.q = svalue( yi = dat$yi,
#'                      vi = dat$vi,
#'                      q = log(.9),
#'                      model = "fixed" )
#'
#' # publication bias required to shift point estimate to RR = 0.90
#' svals.FE.q$svals$sval.est
#'
#' # and to shift CI to RR = 0.90
#' svals.FE.q$svals$sval.ci
#'
#' ##### Robust Clustered Specification #####
#' # takes ~1 min to run due to grid search
#' \dontrun{
#'   svalue( yi = dat$yi,
#'           vi = dat$vi,
#'           q = 0,
#'           model = "robust" )
#' }


svalue = function( yi,
                    vi,
                    q,
                    clustervar = 1:length(yi),
                    model,
                    eta.grid = seq(1, 200, 0.25),
                    CI.level = 0.95,
                    small = TRUE ) {

  # stop if eta doesn't make sense
  if ( any(eta.grid < 1 ) ) stop( "Eta must be at least 1.")

  # number of point estimates
  k.studies = length(yi)

  alpha = 1 - CI.level

  # warn if clusters but user said fixed
  nclusters = length( unique( clustervar ) )
  if ( nclusters < k.studies & model == "fixed" ) {
    warning( "You indicated there are clusters, but these will be ignored due to fixed-effects specification. To accommodate clusters, instead choose model = robust.")
  }

  # check and flip if naive point estimate is negative
  # fit uncorrected model
  m0 = corrected_meta( yi = yi,
                  vi = vi,
                  eta = 1,
                  model = model,
                  clustervar = clustervar,
                  selection.tails = 1,
                  CI.level = CI.level,
                  small = small )

  # stop if q is on wrong side of null
  if ( m0$est > 0 & q > m0$est ) stop( paste( "The uncorrected pooled point estimate is ", round2(m0$est),
                                              ". q must be less than this value (i.e., closer to zero).",
                                              sep = "" ) )
  if ( m0$est < 0 & q < m0$est ) stop( paste( "The uncorrected pooled point estimate is ", round2(m0$est),
                                              ". q must be greater than this value (i.e., closer to zero).",
                                              sep = "" ) )

  # reverse signs if needed to have pooled point estimate > 0
  if ( m0$est < 0 ) {
    # keep track so that we can flip back at the end
    flipped = TRUE
    yi = -yi
    q = -q
  } else {
    flipped = FALSE
  }

  # 2-sided p-values for each study even if 1-tailed selection
  pvals = 2 * ( 1 - pnorm( abs(yi) / sqrt(vi) ) )

  # affirmative indicator under 1-tailed selection
  A = (pvals < 0.05) & (yi > 0)

  k.affirmative = sum(A)
  k.nonaffirmative = k.studies - sum(A)

  if ( k.affirmative == 0 | k.nonaffirmative == 0 ) {
    stop( "There are zero affirmative studies or zero nonaffirmative studies. Model estimation cannot proceed.")
  }

  dat = data.frame( yi, vi, A, clustervar )


  ##### Fixed-Effects Model #####
  if ( model == "fixed" ) {

    # FE mean and sum of weights stratified by affirmative vs. nonaffirmative
    strat = dat %>% group_by(A) %>%
      summarise( nu = sum( 1 / vi ),
                 ybar = sum( yi / vi ) )

    # components of bias-corrected estimate by affirmative status
    ybarN = strat$ybar[ strat$A == 0 ]
    ybarA = strat$ybar[ strat$A == 1 ]
    nuN = strat$nu[ strat$A == 0 ]
    nuA = strat$nu[ strat$A == 1 ]

    # S-value for point estimate
    sval.est = ( nuA * q - ybarA ) / ( ybarN - nuN * q )

    # S-value for CI (to shift it to q)
    # match term names used in Wolfram Alpha
    a = ybarN
    b = ybarA
    c = nuN
    d = nuA

    if ( small == FALSE ) k = qnorm( 1 - (alpha/2) )
    if ( small == TRUE ) {
      df = k.studies - 1
      k = qt( 1 - (alpha/2), df = df )
    }

    # # version directly from Wolfram
    # termA = a^2 * d * k^2 - (2 * a * c * d * k^2 * q) +
    #           b^2 * c * k^2 -
    #           (2 * b * c * d * k^2 * q) +
    #           c^2 * d * k^2 * q^2 +
    #           c * d^2 * k^2 * q^2 -
    #           c * d * k^4

    # manually simplied version
    termA = k^2 * ( a^2 * d -
                      (2 * c * d * q) * (a + b) +
                      b^2 * c +
                      q^2 * (c^2 * d + d^2 * c) -
                      c * d * k^2 )

    termB = -a*b + a*d*q + b*c*q - c*d*q^2

    termC = a^2 - 2*a*c*q + c^2*q^2 - c*k^2

    sval.ci = ( -sqrt(termA) + termB ) / termC
    if ( sval.ci < 0 ) sval.ci = ( sqrt(termA) + termB ) / termC

    # # sanity check by inversion
    # # corrected CI limit
    # eta = sval.ci
    # termD = (eta * a + b) / (eta * c + d)
    # termE = k * sqrt( (eta^2 * c + d) / (eta * c + d)^2 )
    # expect_equal( termD - termE,
    #               q )
    # # WORKS!!!

  } # end fixed = TRUE


  ##### Robust Independent and Robust Clustered #####
  if ( model == "robust" ) {

    # grid search
    res.list = lapply( as.list(eta.grid),
                       function(x) {

                         cat("\n Doing grid search with eta =", x)

                         return( corrected_meta( yi = yi,
                                                 vi = vi,
                                                 eta = x,
                                                 model = model,
                                                 clustervar = clustervar,
                                                 selection.tails = 1,
                                                 CI.level = CI.level,
                                                 small = small ) )
                       } )

    grid.ests = as.data.frame( do.call( "rbind", res.list ) )

    # each column is secretly a list
    grid.ests = as.data.frame( apply(grid.ests, 2, unlist) )

    # find s-value for estimate
    ind1 = which.min( abs(grid.ests$est - q) )
    sval.est = grid.ests$eta[ ind1 ]

    # find s-value for CI
    ind2 = which.min( abs(grid.ests$lo - q) )
    sval.ci = grid.ests$eta[ ind2 ]

    # warn if eta grid doesn't go high enough
    if ( ind1 == length(eta.grid) ) {
      sval.est = paste( ">", sval.est )
      warning("Eta grid does not go high enough to find the S-value (or it is possible that no amount of publication bias would be sufficient). Try again with a grid that includes larger values.")
    }

    if ( ind2 == length(eta.grid) ) {
      sval.ci = paste( ">", sval.ci )
    }

  } # end robust = TRUE


  # ##### Worst-case bound #####
  # # as eta -> infinity
  # # not supposed to use KNHA for FE model
  #
  # # flip signs back to original direction if needed
  # if ( flipped == TRUE ) {
  #   yi = -yi
  # }
  #
  # meta.bd = corrected_meta( yi = yi[ A == 0 ],
  #                 vi = vi[ A == 0 ],
  #                 eta = 1,
  #                 model = model,
  #                 CI.level = CI.level,
  #                 small = small )

  # s-values less than 1 indicate complete robustness
  # is.numeric is in case we have a "< XXX" string instead of a number
  if ( is.numeric(sval.est) & !is.na(sval.est) & sval.est < 1) sval.est = "Not possible"
  if ( is.numeric(sval.ci) & !is.na(sval.ci) & sval.ci < 1) sval.ci = "Not possible"

  return( list( svals = data.frame( sval.est,
                                    sval.ci = sval.ci,
                                    k.affirmative,
                                    k.nonaffirmative ) ) )

}



############################# FN: SIGNIFICANCE FUNNEL PLOT #############################

#' Make significance funnel plot
#'
#' Creates a modified funnel plot that distinguishes between affirmative and nonaffirmative studies, helping detect the extent to which the nonaffirmative studies' point estimates are systematically smaller than
#' the entire set of point estimates. By default (\code{plot.pooled = TRUE}), also plots the fixed-effects pooled point
#' estimate within all studies (black diamond) and within only the nonaffirmative studies (blue diamond). The latter
#' represents a corrected fixed-effects estimate under worst-case publication bias.  When the diamonds are distant or if the
#' blue diamond represents a negligible effect size, then formal sensitivity analyses (via \code{PublicationBias::svalue})
#' may indicate that the meta-analysis is not robust.
#' @param yi A vector of point estimates to be meta-analyzed. The signs of the estimates should be chosen
#' such that publication bias is assumed to operate in favor of positive estimates.
#' @param vi A vector of estimated variances for the point estimates
#' @param xmin x-axis (point estimate) lower limit for plot
#' @param ymin y-axis (standard error) lower limit for plot
#' @param xmax x-axis (point estimate) upper limit for plot
#' @param ymax y-axis (standard error) upper limit for plot
#' @param plot.pooled Should the fixed-effects pooled estimates within all studies and within only the nonaffirmative
#' studies be plotted as well?
#' @import
#' metafor
#' stats
#' ggplot2
#' graphics
#' @export
#' @references
#' 1. Mathur MB & VanderWeele TJ (2019). Sensitivity analysis for publication bias in meta-analyses. Preprint available at XXX.
#' @examples
#'
#' # compute meta-analytic effect sizes
#' require(metafor)
#' dat = metafor::escalc(measure="RR", ai=tpos, bi=tneg, ci=cpos, di=cneg, data=dat.bcg)
#'
#' # flip signs since we think publication bias operates in favor of negative effects
#' dat$yi = -dat$yi
#'
#' significance_funnel( yi = dat$yi,
#'                       vi = dat$vi,
#'                       plot.pooled = TRUE )


significance_funnel = function( yi,
                                vi,
                                xmin = min(yi),
                                ymin = min( sqrt(vi) ),
                                xmax = max(yi),
                                ymax = max( sqrt(vi) ),
                                plot.pooled = TRUE ) {

  d = data.frame(yi, vi)
  d$sei = sqrt(vi)

  # calculate p-values
  d$pval = 2 * ( 1 - pnorm( abs(yi) / sqrt(vi) ) )

  # variable for positive vs. nonpositive studies
  d$positive = rep(NA, nrow(d))
  d$positive[ (d$yi > 0) & (d$pval < 0.05) ] = "Affirmative"
  d$positive[ (d$yi < 0) | (d$pval >= 0.05) ] = "Non-affirmative"

  # reorder levels for plotting joy
  d$positive = factor( d$positive, c("Non-affirmative", "Affirmative") )

  # stop if no studies in either group
  if ( sum( d$positive == "Non-affirmative" ) == 0 ) {
    stop("There are no non-affirmative studies. The plot would look silly.")
  }

  if ( sum( d$positive == "Affirmative" ) == 0 ) {
    stop("There are no affirmative studies. The plot would look silly.")
  }

  # pooled fixed-effects estimates
  est.N = rma.uni(yi = d$yi[ d$positive == "Non-affirmative" ],
                  vi = d$vi[ d$positive == "Non-affirmative" ],
                  method="FE")$b

  est.all = rma.uni(yi = d$yi,
                    vi = d$vi,
                    method="FE")$b

  # negative sei positions them below the horizontal divider line
  pooled.pts = data.frame( yi = c(est.N, est.all),
                           sei = c(0,0) )

  # for a given SE (y-value), return the "just significant" point estimate value (x-value)
  just_signif_est = function( .sei ) .sei * qnorm(.975)

  # polygon coordinates for the blue and orange shading
  # remember ymin is the min SE, etc.
  if ( !any( is.na( c(ymin, xmin, ymax, xmax) ) ) ) {
    poly.blue=data.frame(yi=c(xmin, just_signif_est(ymin), just_signif_est(ymax), xmin ),
                         sei=c(ymin, ymin, ymax, ymax),
                         positive = rep("Non-affirmative", 4),
                         alpha = 0.3)

    poly.orange=data.frame(yi=c(just_signif_est(ymin), xmax, xmax, just_signif_est(ymax)),
                           sei=c(ymin, ymin, ymax, ymax ),
                           positive = rep("Affirmative", 4),
                           alpha = 0.3)
  } else {
    # remove objects if they happen to exist
    # because will check for existence during plotting
    suppressWarnings( rm(poly.blue) )
    suppressWarnings( rm(poly.orange) )
  }

  colors = c("blue", "orange")

  p.funnel = ggplot( data = d, aes( x = d$yi,
                                    y = d$sei,
                                    color = d$positive ) )

  if ( exists("poly.orange") ) {
    p.funnel = p.funnel + geom_polygon(data=poly.blue, mapping=aes(x=poly.blue$yi, y=poly.blue$sei),
                                       fill = colors[1],
                                       alpha = 0.2,
                                       color = NA)

    p.funnel = p.funnel +  geom_polygon(data=poly.orange, mapping=aes(x=poly.orange$yi, y=poly.orange$sei),
                                        fill = colors[2],
                                        alpha = 0.2,
                                        color = NA)
  }

  if ( plot.pooled == TRUE ) {

    p.funnel = p.funnel + geom_point(
      data = pooled.pts,
      aes( x = pooled.pts$yi, y = pooled.pts$sei ),
      size = 4,
      shape = 5,
      fill = NA,
      color = c(colors[1], "black")
    ) +

      geom_point(
        data = pooled.pts,
        aes( x = pooled.pts$yi, y = pooled.pts$sei ),
        size = 4,
        shape = 18,
        color = c(colors[1], "black"),
        alpha = .3
      ) +

      # just for visual separation of pooled ests
      geom_hline( yintercept = 0 )
  }

  p.funnel = p.funnel +

    # semi-transparent points with solid circles around them
    geom_point( size = 3, alpha=.3) +
    geom_point( size = 3, shape = 1) +

    scale_color_manual(values = colors) +

    xlab( bquote( hat(theta) ) ) +
    ylab( bquote( hat(SE) ) ) +

    theme_classic() +
    theme(legend.title=element_blank())

  plot(p.funnel)
  return(p.funnel)
}



############################# FN: P-VALUE PLOT #############################

#' Plot one-tailed p-values
#'
#' Plots the one-tailed p-values. The leftmost red line indicates the cutoff for one-tailed p-values less than 0.025
#' (corresponding to "affirmative" studies; i.e., those with a positive point estimate and a two-tailed p-value
#' less than 0.05). The rightmost red line indicates one-tailed p-values greater than 0.975 (i.e., studies with a
#' negative point estimate and a two-tailed p-value less than 0.05). If there is a substantial point mass of p-values
#' to the right of the rightmost red line, this suggests that selection may be two-tailed rather than one-tailed.
#' @param yi A vector of point estimates to be meta-analyzed. The signs of the estimates should be chosen
#' such that publication bias is assumed to operate in favor of positive estimates.
#' @param vi A vector of estimated variances for the point estimates
#' @import
#' stats
#' ggplot2
#' @export
#' @references
#' 1. Mathur MB & VanderWeele TJ (2019). Sensitivity analysis for publication bias in meta-analyses. Preprint available at XXX.
#' @examples
#'
#' # compute meta-analytic effect sizes
#' require(metafor)
#' dat = metafor::escalc(measure="RR", ai=tpos, bi=tneg, ci=cpos, di=cneg, data=dat.bcg)
#'
#' # flip signs since we think publication bias operates in favor of negative effects
#' dat$yi = -dat$yi
#'
#' pval_plot( yi = dat$yi,
#'            vi = dat$vi )


pval_plot = function( yi,
                      vi ) {

  # calculate 1-tailed p-values
  pval = 1 - pnorm( yi / sqrt(vi) )

  ggplot( data = data.frame(pval = pval),
          aes( x = pval ) ) +
    geom_vline(xintercept = 0.025, color = "red", lwd = 1.2) +
    geom_vline(xintercept = 0.975, color = "red", lwd = 1.2) +
    geom_histogram( binwidth = 0.025 ) +
    xlab("One-tailed p-value") +
    theme_classic() +
    theme( panel.grid = element_blank(),
           axis.title.y = element_blank(),
           axis.text.y = element_blank(),
           axis.ticks.y = element_blank(),
           axis.text=element_text(size=16),
           axis.title=element_text(size=16, face = "bold") )
}

