# S-Plus script developed by Professor Alexander McNeil, mcneil@math.ethz.ch
# R-version adapted by Scott Ulman (scottulman@hotmail.com)
# This free script using QRMLib is distributed in the hope that it will be useful, 
# but WITHOUT ANY WARRANTY; without even the implied warranty of 
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
# GNU General Public License for more details. 

######Load the QRMlib##################
#QRMlib.pdf is a help file for the functions used by QRMlib.  It is available at
#...\Program Files\R\R-2.2.1\library\QRMlib\Docs
#If you have created the QRMBook workspace and .Rprofile  as described in QRMlib.pdf
#topics 'QRMBook-workspace' and 'profileLoadLibrary', then you may comment out the
#following line:
library(QRMlib);
#################################################


# COPULA FITTING AND VISUALIZATION
# A)Obtain 'pseudo-copula' (cumulative probability) data in two-step process: 
#    1) simulate random data and 
#    2) get the corresponding cumulative probabilities for the copula.
#Generate 5 multivariate normal and t variables with linear (equi)correlation 0.7
sim.data.normal <- rmnorm(4000, d=5, rho=0.7)
sim.data.t <- rmt(4000, df=4, d=5, rho=0.7)
#You may actually import simulated data from S-Plus to make sure you get the same answers in R
#Generate pseudo-copula data:
#Apply the pnorm() function normal distribution function (df) to columns (since parameter 2 is 2) 
#of the data since we simulated from normal distribution!  In S3-Fitting_Ch5, we use apply() 
#but use the empirical df (edf()) as the function (rather than pt() or pnorm()) when using REAL 
#data from an unknown distribution.  Here we are using simulated data from a KNOWN distribution.
#Get the corresponding cumulative probabilities for the multivariate normal an t simulated data:
data.gausscopula <- apply(sim.data.normal,2,pnorm)
#Here we are NOT applying the edf()but the t-distribution function:
data.tcopula <- apply(sim.data.t,2,pt,df=4)
#The previous line has the same result as:
#data2.tcopula <- pt(sim.data.t, df=4)

#produce a matrix of scatterplots paired together:
pairs(data.gausscopula)
#look at histogram for first column 
hist(data.gausscopula[,1])
pairs(data.tcopula)

# Bivariate Visualization
ll <- c(0.01,0.99)
BiDensPlot(func=dcopula.gauss,xpts=ll,ypts=ll,P=equicorr(2,0.5))
BiDensPlot(func=dcopula.t,xpts=ll,ypts=ll,nu=4,P=equicorr(2,0.5))

#Method B
# As an alternative to (A) above where we simulate random data and then generate the
#corresponding cumulative probability for the copula, we can reverse the procedure by
#   1) simulating cumulative probabilities from the copula functions and then
#   2) generating corresponding random outcomes (data) via the quantile function. 
# Note that d=2 in (B) vs. d=5 in (A).
# Figure 5.3: 
sim.data.gausscopula <- rcopula.gauss(2000,d=2, rho=0.7)
sim.data.gumbelcopula <- rcopula.gumbel(2000,theta=2, d=2)
sim.data.claytoncopula <- rcopula.clayton(2000, theta=2.2, d=2)
sim.data.tcopula <- rcopula.t(2000, df=4, d=2, rho=0.71) #rho=.71 rather than .7
par(mfrow=c(2,2))
plot(sim.data.gausscopula)
plot(sim.data.gumbelcopula)
plot(sim.data.claytoncopula)
plot(sim.data.tcopula)
par(mfrow=c(1,1))


# Figure 5.4
data.metagauss <- apply(sim.data.gausscopula,2,qnorm) #use gauss quantile with gauss copula data
# Meta Distributions. Marginals extracted from DIFFERENT quantile functions than those which 
# generated the copula probabilities.
# If we use a diffent quantile distribution than that associated with the copula which
#generated the data, we are creating a 'Meta distribution'. In the following we use quantiles for
#'normal marginals' (via the normal quantile function) with probabilities generated by
#the gumbel, clayton, and t copulas.  Resulting data values are all known as 'meta-Gaussian' distributions.
data.metagumbel <- apply(sim.data.gumbelcopula,2,qnorm)
data.metaclayton <- apply(sim.data.claytoncopula,2,qnorm)
data.metat <- apply(sim.data.tcopula,2,qnorm)
par(mfrow=c(2,2))
plot(data.metagauss)
plot(data.metagumbel)
plot(data.metaclayton)
plot(data.metat)
par(mfrow=c(1,1))

# the 3D view
#Define limits
ll <- c(-3,3)
#Define three functions:
normal.metagumbel <- function(x,theta){
  exp(dcopula.gumbel(apply(x,2,pnorm),theta,logvalue=T) + apply(log(apply(x,2,dnorm)),1,sum))
}
normal.metaclayton <- function(x,theta){
  exp(dcopula.clayton(apply(x,2,pnorm),theta,logvalue=T) + apply(log(apply(x,2,dnorm)),1,sum))
}
normal.metat <- function(x,nu,P){
  exp(dcopula.t(apply(x,2,pnorm),nu,P,logvalue=T) + apply(log(apply(x,2,dnorm)),1,sum))
}
par(mfrow=c(2,2))
#BiVariate Density Plot() 1st agrument is function that evaluates on an n x 2 matrix
#Perspective Plots:
BiDensPlot(dmnorm,xpts=ll,ypts=ll,mu=c(0,0),Sigma=equicorr(2,0.7))
BiDensPlot(normal.metagumbel,xpts=ll,ypts=ll,npts=80,theta=2)
BiDensPlot(normal.metaclayton,xpts=ll,ypts=ll,npts=80,theta=2.2)
BiDensPlot(normal.metat,xpts=ll,ypts=ll,npts=80,nu=4,P=equicorr(2,0.71))
#Contour Plots:
BiDensPlot(dmnorm,type="contour",xpts=ll,ypts=ll,mu=c(0,0),Sigma=equicorr(2,0.7))
BiDensPlot(normal.metagumbel,type="contour",xpts=ll,ypts=ll,npts=80,theta=2)
BiDensPlot(normal.metaclayton,type="contour",xpts=ll,ypts=ll,npts=80,theta=2.2)
BiDensPlot(normal.metat,type="contour",xpts=ll,ypts=ll,npts=80,nu=4,P=equicorr(2,0.71))
par(mfrow=c(1,1))

