#! /usr/bin/perl

# ---------------------------------------------------------------------------
# Reads in src/init.c
# Extracts all exported C level functions (by means of macro 'RREGDEF')
# Create wrapper functions
# Write to stdout 
# ---------------------------------------------------------------------------
# Run this script in the top-level Runuran directory.
# The script updates the wrapper functions in 
#    inst/include/Runuran_API.h
# ---------------------------------------------------------------------------

use strict;

# --- Constants -------------------------------------------------------------

my $unuran_h_file = "./inst/include/unuran.h";
my $init_c_file = "./src/init.c";

# --- Usage -----------------------------------------------------------------

sub usage {
    my $progname = $0;
    $progname =~ s#^.*/##g;
        
    print STDERR <<EOM;

usage: $progname

EOM

exit -1;
}

# --- Read file 'DESCRIPTION' -----------------------------------------------

open DESC, "DESCRIPTION" 
    or die "You must run this script in the top-level Runuran directory";
my $description;
while (<DESC>) {
    $description .= $_;
}
close DESC; 

# check name of package
die "You must run this script in the top-level Runuran directory"
    unless $description =~ /^\s*Package:\s+Runuran\s*\n/;

# --- Read file 'unuran.h' --------------------------------------------------

# Get list of all UNU.RAN functions
my %unuran_funcs;

open UNURAN_H, $unuran_h_file
    or die "Cannot find '$unuran_h_file'";

while (<UNURAN_H>) {
    my $line = $_;
    while ($line =~ /\\\s*$/) {
	$line =~ s/\\\s*$//;
	$line .= <UNURAN_H>
    }
    next if $line =~ /^\s*$/;       ## empty line
    next if $line =~ /^\s*\/\*/;    ## C comment
    next if $line =~ /^\s*\#/;      ## macro
    next if $line =~ /^\s*__/;      ## internal 

    until ($line =~ /;\s*$/) {
	$line =~ s/\s*\n/ /;
	$line .= <UNURAN_H>
    }

    next if $line =~ /^\s*typedef/;
    next unless $line =~ /^([\w\s]*)\s+(\*?)\s*(\w+)\s*\((.*)\)(.*)$/;

    my $valtype = $1;
    my $pointer = $2;
    my $name = $3;
    my $args = $4;

    $valtype =~ s/(^\s+|\s+$)//g;
    $valtype =~ s/\s+/ /g;
    $name =~ s/(^\s+|\s+$)//g;
    $args =~ s/(^\s+|\s+$)//g;
    $args =~ s/\s+/ /g;
    
    next if $name =~ /^\s*_/;       ## internal function

    next if $args =~ /\.\.\./;      ## cannot handle '...' arguments

    ## cannot handle the following functions correctly
    next if $name eq "unur_urng_set_sample_array"; 
    next if $name eq "unur_urng_set_seed";
    next if $name eq "unur_urng_set_anti";

    my @arglist = split /\,\s*/, $args;

    my @argtypes;
    my @argnames;
    foreach my $a (@arglist) {
	if ($a eq "void") {
	    push @argtypes, "void";
	    push @argnames, "";
	}
	elsif ($a =~ /\(/) {
	    $a =~ /(.+?)\(\s*\*?(.+?)\s*\)\s*\(\s*(.+?)\w+\s*\)/;
	    push @argtypes, "$1(*)($3)";
	    push @argnames, "$2";
	}
	else {
	    $a =~ /(.*[\s|\*])(\w+)/;
	    push @argtypes, $1;
	    push @argnames, $2;
	}
    }
    my $argtypelist = join ",", @argtypes;
    my $argnamelist = join ",", @argnames;
    
    my $func = 
	"$valtype attribute_hidden $pointer $name($args) {\n" .
	"\tstatic $valtype$pointer(*func)($argtypelist) = NULL;\n" .
	"\tif (func == NULL)\n" .
	"\t\tfunc = ($valtype$pointer(*)($argtypelist))\n" .
	"\t\t\tR_GetCCallable(\"Runuran\", \"$name\");\n" .
	(($valtype eq "void") ? "\t" : "\treturn ") .
	"func($argnamelist);\n" .
	"}\n";

    $unuran_funcs{$name} = $func;
}

close UNURAN_H; 

#foreach my $n (keys %unuran_funcs) {
#    print "$n:\n";
#    print $unuran_funcs{$n};
#}

# --- Read file 'src/init.c' ------------------------------------------------

open INIT_C, $init_c_file
    or die "Cannot find '$init_c_file'";

my $wrapper;
while (<INIT_C>) {
    next unless /^\s*RREGDEF\s*\(\s*(\w+)\s*\)\s*;/;
    my $func = $1;
    $wrapper .= "$unuran_funcs{$func}\n";
}

close INIT_C;

# ---------------------------------------------------------------------------

print <<EOH;
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Wrapper functions for calling routines exported from package 'Runuran'    */
/*                                                                           */
/*---------------------------------------------------------------------------*/
/* File generated by Runuran/src/scripts/update-API.pl: do not edit by hand  */
/*---------------------------------------------------------------------------*/

#ifndef _RUNURAN_API_H
#define _RUNURAN_API_H

#include "Runuran.h"
/* also includes 
 *   <R.h>
 *   <Rinternals.h>
 *   <Rdefines.h>
 *   <R_ext/Rdynload.h> 
 *   <unuran.h>
 */

#ifdef HAVE_VISIBILITY_ATTRIBUTE
  # define attribute_hidden __attribute__ ((visibility ("hidden")))
#else
  # define attribute_hidden
#endif

__BEGIN_DECLS   /* for C++  (defined in <unuran.h> */

/*---------------------------------------------------------------------------*/

EOH

print $wrapper;

print <<EOF;
/*---------------------------------------------------------------------------*/

__END_DECLS

/*---------------------------------------------------------------------------*/
#endif /* end of _RUNURAN_API_H */
/*---------------------------------------------------------------------------*/
EOF

# ---------------------------------------------------------------------------

exit 0;

# --- end -------------------------------------------------------------------
