#' Mock function used to document all main function.
#'
#' @param sample \strong{\code{\link[SeuratObject]{Seurat}}} | A Seurat object, generated by \link[Seurat]{CreateSeuratObject}.
#' @param font.size \strong{\code{\link[base]{numeric}}} | Overall font size of the plot. All plot elements will have a size relationship with this font size.
#' @param font.type \strong{\code{\link[base]{character}}} | Base font family for the plot. One of:
#' \itemize{
#'   \item \emph{\code{mono}}: Mono spaced font.
#'   \item \emph{\code{serif}}: Serif font family.
#'   \item \emph{\code{sans}}: Default font family.
#' }
#' @param legend.type \strong{\code{\link[base]{character}}} | Type of legend to display. One of:
#' \itemize{
#'   \item \emph{\code{normal}}: Default legend displayed by \pkg{ggplot2}.
#'   \item \emph{\code{colorbar}}: Redefined colorbar legend, using \link[ggplot2]{guide_colorbar}.
#'   \item \emph{\code{colorsteps}}: Redefined legend with colors going by range, in steps, using \link[ggplot2]{guide_colorsteps}.
#' }
#' @param legend.position \strong{\code{\link[base]{character}}} | Position of the legend in the plot. One of:
#' \itemize{
#'   \item \emph{\code{top}}: Top of the figure.
#'   \item \emph{\code{bottom}}: Bottom of the figure.
#'   \item \emph{\code{left}}: Left of the figure.
#'   \item \emph{\code{right}}: Right of the figure.
#'   \item \emph{\code{none}}: No legend is displayed.
#' }
#' @param legend.title \strong{\code{\link[base]{character}}} | Title for the legend.
#' @param legend.title.position \strong{\code{\link[base]{character}}} | Position for the title of the legend. One of:
#' \itemize{
#'   \item \emph{\code{top}}: Top of the legend.
#'   \item \emph{\code{bottom}}: Bottom of the legend.
#'   \item \emph{\code{left}}: Left of the legend.
#'   \item \emph{\code{right}}: Right of the legend.
#' }
#' @param legend.framewidth,legend.tickwidth \strong{\code{\link[base]{numeric}}} | Width of the lines of the box in the legend.
#' @param legend.framecolor \strong{\code{\link[base]{character}}} | Color of the lines of the box in the legend.
#' @param legend.tickcolor \strong{\code{\link[base]{character}}} | Color of the ticks of the box in the legend.
#' @param legend.length,legend.width \strong{\code{\link[base]{numeric}}} | Length and width of the legend. Will adjust automatically depending on legend side.
#' @param legend.icon.size \strong{\code{\link[base]{numeric}}} | Size of the icons in legend.
#' @param legend.ncol,legend.nrow \strong{\code{\link[base]{numeric}}} | Number of columns/rows in the legend.
#' @param legend.byrow \strong{\code{\link[base]{logical}}} | Whether the legend is filled by row or not.
#' @param plot.title,plot.subtitle,plot.caption \strong{\code{\link[base]{character}}} | Title, subtitle or caption to use in the plot.
#' @param individual.titles,individual.subtitles,individual.captions \strong{\code{\link[base]{character}}} | Vector. Title, subtitle or caption to use in the plot when multiple features are passed on. Use NA to keep the original title.
#' @param reduction \strong{\code{\link[base]{character}}} | Reduction to use. Can be the canonical ones such as "umap", "pca", or any custom ones, such as "diffusion". If you are unsure about which reductions you have, use `Seurat::Reductions(sample)`. Defaults to "umap" if present or to the last computed reduction if the argument is not provided.
#' @param assay \strong{\code{\link[base]{character}}} | Assay to use. Defaults to the current assay.
#' @param slot \strong{\code{\link[base]{character}}} | Data slot to use. Only one of: counts, data, scale.data. Defaults to "data".
#' @param viridis_color_map \strong{\code{\link[base]{character}}} | A capital letter from A to H or the scale name as in \link[viridis]{scale_fill_viridis}.
#' @param raster \strong{\code{\link[base]{logical}}} | Whether to raster the resulting plot. This is recommendable if plotting a lot of cells.
#' @param raster.dpi \strong{\code{\link[base]{numeric}}} | Pixel resolution for rasterized plots. Defaults to 1024. Only activates on Seurat versions higher or equal than 4.1.0.
#' @param plot_cell_borders \strong{\code{\link[base]{logical}}} | Whether to plot border around cells.
#' @param border.size \strong{\code{\link[base]{numeric}}} | Width of the border of the cells.
#' @param border.color \strong{\code{\link[base]{character}}} | Color to use for the border of the cells.
#' @param na.value \strong{\code{\link[base]{character}}} | Color value for NA.
#' @param rotate_x_axis_labels \strong{\code{\link[base]{numeric}}} | Degree to rotate the X labels. One of: 0, 45, 90.
#' @param xlab,ylab \strong{\code{\link[base]{character}}} | Titles for the X and Y axis.
#' @param pt.size \strong{\code{\link[base]{numeric}}} | Size of the dots.
#' @param flip \strong{\code{\link[base]{logical}}} | Whether to invert the axis of the displayed plot.
#' @param verbose \strong{\code{\link[base]{logical}}} | Whether to show extra comments, warnings,etc.
#' @param group.by \strong{\code{\link[base]{character}}} | Metadata variable to group the output by. Has to be a character of factor column.
#' @param split.by \strong{\code{\link[base]{character}}} | Secondary metadata variable to further group (split) the output by. Has to be a character of factor column.
#' @param colors.use \strong{\code{\link[SCpubr]{named_vector}}} | Named vector of valid color representations (either name of HEX codes) with as many named colors as unique values of group.by. If group.by is not provided, defaults to the unique values of \link[Seurat]{Idents}. If not provided, a color scale will be set by default.
#' @param plot_marginal_distributions \strong{\code{\link[base]{logical}}} |  Whether to plot marginal distributions on the figure or not.
#' @param marginal.type \strong{\code{\link[base]{character}}} | One of:
#' \itemize{
#'   \item \emph{\code{density}}: Compute density plots on the margins.
#'   \item \emph{\code{histogram}}: Compute histograms on the margins.
#'   \item \emph{\code{boxplot}}: Compute boxplot on the margins.
#'   \item \emph{\code{violin}}: Compute violin plots on the margins.
#'   \item \emph{\code{densigram}}: Compute densigram plots on the margins.
#' }
#' @param marginal.size \strong{\code{\link[base]{numeric}}} | Size ratio between the main and marginal plots. A value of 5 means that the main plot is 5 times bigger than the marginal plots.
#' @param marginal.group \strong{\code{\link[base]{logical}}} | Whether to group the marginal distribution by group.by or current identities.
#' @param enforce_symmetry \strong{\code{\link[base]{logical}}} | Return a symmetrical plot axes-wise or continuous color scale-wise, when applicable.
#' @param column_title \strong{\code{\link[base]{character}}} | Title for the columns of the heatmaps. Only works with single heatmaps.
#' @param row_title \strong{\code{\link[base]{character}}} | Title for the rows of the heatmaps. Only works with single heatmaps.
#' @param cluster_cols,cluster_rows \strong{\code{\link[base]{logical}}} | Cluster the columns or rows of the heatmaps.
#' @param column_names_rot \strong{\code{\link[base]{numeric}}} | Degree in which to rotate the column labels.
#' @param row_names_rot \strong{\code{\link[base]{numeric}}} | Degree in which to rotate the row labels.
#' @param cell_size \strong{\code{\link[base]{numeric}}} | Size of each cell in the heatmap.
#' @param input_gene_list \strong{\code{\link[SCpubr]{named_list}}} | Named list of lists of genes to be used as input.
#' @param column_title_rot \strong{\code{\link[base]{numeric}}} | Degree in which to rotate the column titles.
#' @param row_title_rot \strong{\code{\link[base]{numeric}}} | Degree in which to rotate the row titles.
#' @param column_names_side \strong{\code{\link[base]{character}}} | Side to put the column names. Either left or right.
#' @param row_names_side \strong{\code{\link[base]{character}}} | Side to put the row names. Either left or right.
#' @param column_title_side \strong{\code{\link[base]{character}}} | Side to put the column titles Either left or right.
#' @param row_title_side \strong{\code{\link[base]{character}}} | Side to put the row titles Either left or right.
#' @param heatmap.legend.length,heatmap.legend.width \strong{\code{\link[base]{numeric}}} | Width and length of the legend in the heatmap.
#' @param heatmap.legend.framecolor \strong{\code{\link[base]{character}}} | Color of the edges and ticks of the legend in the heatmap.
#' @param scale_direction \strong{\code{\link[base]{numeric}}} | Direction of the viridis scales. Either -1 or 1.
#' @param heatmap_gap \strong{\code{\link[base]{numeric}}} | Gap in cm between heatmaps.
#' @param legend_gap \strong{\code{\link[base]{numeric}}} | Gap in cm between legends.
#' @param cells.highlight,idents.highlight \strong{\code{\link[base]{character}}} | Vector of cells/identities to focus into. The identities have to much those in \code{Seurat::Idents(sample)} The rest of the cells will be grayed out. Both parameters can be used at the same time.
#' @param dims \strong{\code{\link[base]{numeric}}} | Vector of 2 numerics indicating the dimensions to plot out of the selected reduction. Defaults to c(1, 2) if not specified.
#' @param ncol \strong{\code{\link[base]{numeric}}} | Number of columns used in the arrangement of the output plot using "split.by" parameter.
#' @param features \strong{\code{\link[base]{character}}} | Features to represent.
#' @param feature \strong{\code{\link[base]{character}}} | Feature to represent.
#' @param use_viridis \strong{\code{\link[base]{logical}}} | Whether to use viridis color scales.
#' @param viridis_direction \strong{\code{\link[base]{numeric}}} | Either 1 or -1. Controls how the gradient of viridis scale is formed.
#' @param plot.grid \strong{\code{\link[base]{logical}}} | Whether to plot grid lines.
#' @param grid.color \strong{\code{\link[base]{character}}} | Color of the grid in the panels.
#' @param grid.type \strong{\code{\link[base]{character}}} | One of the possible linetype options:
#' \itemize{
#'   \item \emph{\code{blank}}.
#'   \item \emph{\code{solid}}.
#'   \item \emph{\code{dashed}}.
#'   \item \emph{\code{dotted}}.
#'   \item \emph{\code{dotdash}}.
#'   \item \emph{\code{longdash}}.
#'   \item \emph{\code{twodash}}.
#' }
#' @param plot.axes \strong{\code{\link[base]{logical}}} | Whether to plot axes or not.
#' @param nbin \strong{\code{\link[base]{numeric}}} | Number of bins to use in \link[Seurat]{AddModuleScore}.
#' @param ctrl \strong{\code{\link[base]{numeric}}} | Number of genes in the control set to use in \link[Seurat]{AddModuleScore}.
#' @param repel \strong{\code{\link[base]{logical}}} | Whether to repel the text labels.
#' @param plot_density_contour \strong{\code{\link[base]{logical}}} | Whether to plot density contours in the UMAP.
#' @param contour.position \strong{\code{\link[base]{character}}} | Whether to plot density contours on top or at the bottom of the visualization layers, thus overlapping the clusters/cells or not.
#' @param contour.color \strong{\code{\link[base]{character}}} | Color of the density lines.
#' @param contour.lineend \strong{\code{\link[base]{character}}} | Line end style (round, butt, square).
#' @param contour.linejoin \strong{\code{\link[base]{character}}} | Line join style (round, mitre, bevel).
#' @param contour_expand_axes \strong{\code{\link[base]{numeric}}} | To make the contours fit the plot, the limits of the X and Y axis are expanding a given percentage from the min and max values for each axis. This controls such percentage.
#' @param min.cutoff,max.cutoff \strong{\code{\link[base]{numeric}}} | Set the min/max ends of the color scale. Any cell/group with a value lower than min.cutoff will turn into min.cutoff and any cell with a value higher than max.cutoff will turn into max.cutoff. In FeaturePlots, provide as many values as features. Use NAs to skip a feature.
#' @usage NULL
#' @return Nothing. This is a mock function.
#' @keywords internal
#' @examples
#'
#' # This a mock function that stores the documentation for many other functions.
#' # It is not intended for user usage.
doc_function <- function(sample,
                         font.size,
                         font.type,
                         legend.type,
                         legend.position,
                         legend.framewidth,
                         legend.tickwidth,
                         legend.framecolor,
                         legend.tickcolor,
                         legend.length,
                         legend.width,
                         plot.title,
                         plot.subtitle,
                         plot.caption,
                         assay,
                         slot,
                         reduction,
                         viridis_color_map,
                         raster,
                         raster.dpi,
                         plot_cell_borders,
                         border.size,
                         border.color,
                         na.value,
                         rotate_x_axis_labels,
                         xlab,
                         ylab,
                         pt.size,
                         verbose,
                         flip,
                         group.by,
                         split.by,
                         colors.use,
                         legend.title,
                         legend.icon.size,
                         legend.byrow,
                         legend.ncol,
                         legend.nrow,
                         plot_marginal_distributions,
                         marginal.type,
                         marginal.size,
                         marginal.group,
                         enforce_symmetry,
                         column_title,
                         row_title,
                         cluster_cols,
                         cluster_rows,
                         column_names_rot,
                         row_names_rot,
                         cell_size,
                         input_gene_list,
                         column_title_rot,
                         row_title_rot,
                         column_names_side,
                         row_names_side,
                         column_title_side,
                         row_title_side,
                         heatmap.legend.length,
                         heatmap.legend.width,
                         heatmap.legend.framecolor,
                         scale_direction,
                         heatmap_gap,
                         legend_gap,
                         cells.highlight,
                         idents.highlight,
                         ncol,
                         dims,
                         feature,
                         features,
                         use_viridis,
                         viridis_direction,
                         plot.grid,
                         grid.color,
                         grid.type,
                         plot.axes,
                         individual.titles,
                         individual.subtitles,
                         individual.captions,
                         legend.title.position,
                         repel,
                         plot_density_contour,
                         contour.position,
                         contour.color,
                         contour.lineend,
                         contour.linejoin,
                         contour_expand_axes){}

#' Named vector.
#'
#' @return Nothing. This is a mock function.
#' @keywords internal
#' @usage NULL
#' @examples
#' # This is a named vector.
#' x <- c("first_element" = 3,
#'        "second_element" = TRUE)
#' print(x)
#'
named_vector <- function(){}

#' Named list.
#'
#' @return Nothing. This is a mock function.
#' @keywords internal
#' @usage NULL
#' @examples
#' # This is a named vector.
#' x <- list("first_element" = c("GENE A", "GENE B"),
#'           "second_element" = c("GENE C", "GENE D"))
#' print(x)
#'
named_list <- function(){}

# Operators.

# Not in operator.
`%!in%` <- function(x, y) {return(!(x %in% y))}


#' Checks for Suggests.
#'
#' @noRd
#' @return None
#' @examples
#' \donttest{
#' TBD
#' }
check_suggests <- function(function_name, passive = FALSE){
  pkg_list <- list("core" = c("Seurat",
                              "rlang",
                              "dplyr",
                              "magrittr",
                              "dplyr",
                              "tidyr",
                              "tibble",
                              "stringr",
                              "patchwork",
                              "plyr",
                              "grDevices",
                              "stats",
                              "viridis",
                              "scales",
                              "grid",
                              "assertthat"),
                   "do_AlluvialPlot" = c("ggalluvial", "ggrepel"),
                   "do_BarPlot" = c("colorspace", "ggrepel"),
                   "do_BeeSwarmPlot" = c("colorspace", "ggbeeswarm", "ggrastr"),
                   "do_BoxPlot" = c("ggsignif"),
                   "do_CellularStatesPlot" = c("pbapply", "ggExtra", "ggplotify", "scattermore"),
                   "do_ChordDiagramPlot" = c("circlize"),
                   "do_ColorPalette" = c(),
                   "do_CopyNumberVariantPlot" = c("ggdist"),
                   "do_CorrelationPlot" = c("ComplexHeatmap", "ComplexHeatmap", "circlize"),
                   "do_DimPlot" = c("colorspace", "ggplotify", "scattermore"),
                   "do_DotPlot" = c(),
                   "do_EnrichmentHeatmap" = c("ComplexHeatmap", "circlize"),
                   "do_FeaturePlot" = c("scattermore"),
                   "do_GeyserPlot" = c("ggdist"),
                   "do_GroupwiseDEPlot" = c("ComplexHeatmap"),
                   "do_LigandReceptorPlot" = c("liana"),
                   "do_NebulosaPlot" = c("Nebulosa"),
                   "do_PathwayActivityPlot" = c("ComplexHeatmap"),
                   "do_PseudotimePlot" = c("monocle3", "ggdist"),
                   "do_RidgePlot" = c("ggridges"),
                   "do_SankeyPlot" = c("ggsankey"),
                   "do_TermEnrichmentPlot" = c(),
                   "do_TFActivityPlot" = c("ComplexHeatmap"),
                   "do_ViolinPlot" = c(),
                   "do_VolcanoPlot" = c("ggrepel"),
                   "save_Plot" = c("ComplexHeatmap", "svglite"),
                   "testing" = c("Does_not_exist"))

  # The function is not in the current list of possibilities.
  if (function_name %!in% names(pkg_list)){
    stop(paste0(function_name, " is not an accepted function name."), call. = FALSE)
  }
  pkgs <- c(pkg_list[[function_name]], pkg_list[["core"]])

  non_seurat_functions <- c("save_Plot",
                            "do_VolcanoPlot",
                            "do_LigandReceptorPlot",
                            "do_ColorPalette")

  if (function_name %in% non_seurat_functions){
    pkgs <- pkgs[pkgs != "Seurat"]
  }

  value <- TRUE
  for (pkg in pkgs){
    if (!requireNamespace(pkg, quietly = TRUE)) {
      if (isFALSE(passive)){
        stop(paste0("Package ", pkg, " must be installed to use ", function_name, "."), call. = FALSE)
      } else{
        value <- FALSE
      }
    }
  }
  if (isTRUE(passive)) {return(value)}
}
#' State SCpubr current function dependencies.
#'
#' @param function_name \strong{\code{\link[base]{character}}} | Name of an exported function from SCpubr. If NULL, return all functions.
#' @param return_dependencies \strong{\code{\link[base]{logical}}} | Whether to have the dependencies as an output object instead of a printed message.
#' @return None
#' @export
#'
#' @examples
#'
#' # See all dependencies.
#' SCpubr::state_dependencies()
#'
#' # See the dependencies for a single package.
#' SCpubr::state_dependencies(function_name = "do_DimPlot")
state_dependencies <- function(function_name = NULL, return_dependencies = FALSE){
  pkg_list <- list("core" = c("Seurat",
                              "rlang",
                              "dplyr",
                              "magrittr",
                              "dplyr",
                              "tidyr",
                              "tibble",
                              "stringr",
                              "patchwork",
                              "plyr",
                              "grDevices",
                              "stats",
                              "viridis",
                              "forcats",
                              "scales",
                              "grid",
                              "assertthat"),
                   "do_BarPlot" = c("colorspace", "ggrepel"),
                   "do_BeeSwarmPlot" = c("colorspace", "ggbeeswarm", "ggrastr"),
                   "do_BoxPlot" = c("ggsignif"),
                   "do_CellularStatesPlot" = c("pbapply", "ggExtra", "ggplotify", "scattermore"),
                   "do_ChordDiagramPlot" = c("circlize"),
                   "do_ColorPalette" = c(),
                   "do_CopyNumberVariantPlot" = c("ggdist"),
                   "do_CorrelationPlot" = c("ComplexHeatmap", "ComplexHeatmap", "circlize"),
                   "do_DimPlot" = c("colorspace", "ggplotify", "scattermore"),
                   "do_DotPlot" = c(),
                   "do_EnrichmentHeatmap" = c("ComplexHeatmap", "circlize"),
                   "do_FeaturePlot" = c("scattermore"),
                   "do_GeyserPlot" = c("ggdist"),
                   "do_GroupwiseDEPlot" = c("ComplexHeatmap"),
                   "do_LigandReceptorPlot" = c("liana"),
                   "do_NebulosaPlot" = c("Nebulosa"),
                   "do_PathwayActivityPlot" = c("ComplexHeatmap"),
                   "do_PseudotimePlot" = c("monocle3", "ggdist"),
                   "do_RidgePlot" = c("ggridges"),
                   "do_SankeyPlot" = c("ggsankey"),
                   "do_TermEnrichmentPlot" = c(),
                   "do_TFActivityPlot" = c("ComplexHeatmap"),
                   "do_ViolinPlot" = c(),
                   "do_VolcanoPlot" = c("ggrepel"),
                   "save_Plot" = c("ComplexHeatmap", "svglite"))
  # The function is not in the current list of possibilities.
  if (!(is.null(function_name))){
    for (func in function_name){
      if (func %!in% names(pkg_list)){
        stop(paste0(function_name, " is not an accepted function name."), call. = FALSE)
      }
    }
  }
  cran_packages <- c("assertthat",
                     "circlize",
                     "colorspace",
                     "dplyr",
                     "ggbeeswarm",
                     "ggdist",
                     "ggExtra",
                     "ggplot2",
                     "ggplotify",
                     "ggrastr",
                     "ggrepel",
                     "ggridges",
                     "ggsignif",
                     "graphics",
                     "magrittr",
                     "patchwork",
                     "pheatmap",
                     "plyr",
                     "rlang",
                     "scales",
                     "scattermore",
                     "Seurat",
                     "tibble",
                     "tidyr",
                     "forcats",
                     "Matrix",
                     "purrr",
                     "stringr",
                     "svglite",
                     "viridis")

  bioconductor_packages <- c("ComplexHeatmap",
                             "infercnv",
                             "Nebulosa")
  github_packages <- c("ggsankey",
                       "liana",
                       "monocle3")

  func_list <- sort(names(pkg_list))
  if (!(is.null(function_name))){
    func_list <- function_name
  } else {
    func_list <- names(pkg_list)
  }

  if (isFALSE(return_dependencies)){
    message("\n---LIST OF PACKAGE DEPENDENCIES---\n")
    for (func in func_list){
      packages <- c(pkg_list[[func]], pkg_list[["core"]])
      cran_packages_individual <- sort(packages[packages %in% cran_packages])
      bioconductor_packages_individual <- sort(packages[packages %in% bioconductor_packages])
      github_packages_individual <- sort(packages[packages %in% github_packages])
      message("Dependencies for ", func, ":")
      if (length(cran_packages_individual >= 1)){message("  CRAN packages: ", paste(cran_packages_individual, collapse = ", "))}
      if (length(bioconductor_packages_individual >= 1)){message("  Bioconductor packages: ", paste(bioconductor_packages_individual, collapse = ", "))}
      if (length(github_packages_individual >= 1)){message("  Github packages: ", paste(github_packages_individual, collapse = ", "))}
      message("")
    }
  } else {
    list_output <- list()
    for (func in func_list){
      packages <- c(pkg_list[[func]], pkg_list[["core"]])
      list_output[[func]] <- packages
    }
    return(list_output)
  }

}





#' Check for Seurat class.
#'
#' @param sample Seurat object.
#'
#' @noRd
#' @return None
#'
#' @examples
#' \donttest{
#' TBD
#' }
check_Seurat <- function(sample){
  assertthat::assert_that("Seurat" %in% class(sample),
                          msg = "Object provided is not a Seurat object.")
}

#' Internal check for colors.
#'
#' Adapted from: https://stackoverflow.com/a/13290832.
#
#' @param colors Vector of colors.
#' @param parameter_name The name of the parameter for which we are testing the colors.
#' @return None
#' @noRd
#' @examples
#' \donttest{
#' TBD
#' }
check_colors <- function(colors, parameter_name = "") {
  check <- sapply(colors, function(color) {
    tryCatch(is.matrix(grDevices::col2rgb(colors)),
             error = function(e) FALSE)
  })
  # Check for cols.highlight.
  assertthat::assert_that(sum(check) == length(colors),
                          msg = paste0("The value/s for ", parameter_name, " is/are not a valid color representation. Please check whether it is an accepted R name or a HEX code."))
}

#' Internal check for named colors and unique values of the grouping variable.
#'
#' @param sample Seurat object.
#' @param colors Named vector of colors.
#' @param grouping_variable Metadata variable in sample to obtain its unique values.
#' @return None
#' @noRd
#' @examples
#' \donttest{
#' TBD
#' }
check_consistency_colors_and_names <- function(sample, colors, grouping_variable = NULL){
  if (is.null(grouping_variable)){
    check_values <- levels(sample)
  } else {
    check_values <- unique(sample@meta.data[, grouping_variable])
  }
  # Remove NAs.
  check_values <- check_values[!(is.na(check_values))]

  # Remove values that are not in the vector.
  if (sum(names(colors) %in% check_values) == length(check_values) & length(names(colors)) > length(check_values)){
    colors <- colors[names(colors) %in% check_values]
  }

  assertthat::assert_that(length(colors) == length(check_values),
                          msg = "The number of provided colors is lower than the unique values in the selected grouping variable (levels(object), group.by or split.by).")

  assertthat::assert_that(sum(names(colors) %in% check_values) == length(check_values),
                          msg = "The names of provided colors does not match the number of unique values in the selected grouping variable (levels(object), group.by or split.by).")

  return(colors)
}

#' Generate custom color scale.
#'
#' @param names_use Vector of the names that will go alongside the color scale.
#' @return None
#' @noRd
#' @examples
#' \donttest{
#' TBD
#' }
generate_color_scale <- function(names_use){
  # Generate a vector of colors equal to the number of identities in the sample.
  colors <- colorspace::qualitative_hcl(length(names_use), palette = "Dark 3")
  colors <- grDevices::col2rgb(colors)
  colors <- grDevices::rgb2hsv(colors)
  colors["v", ] <- colors["v", ] - 0.1
  colors["s", ] <- colors["s", ] + 0.2
  colors["s", ][colors["s", ] > 1] <- 1
  colors <- grDevices::hsv(h = colors["h", ],
                           s = colors["s", ],
                           v = colors["v", ],
                           alpha = 1)
  names(colors) <- names_use
  return(colors)
}


#' Compute the max and min value of a variable provided to FeaturePlot.
#'
#' @param sample Seurat object.
#' @param feature Feature to plot.
#' @param assay Assay used.
#' @param reduction Reduction used.
#'
#' @return None
#' @noRd
#' @examples
#' \donttest{
#' TBD
#' }
compute_scale_limits <- function(sample, feature, assay = NULL, reduction = NULL){
  if (is.null(assay)){
    assay <- Seurat::DefaultAssay(sample)
  }
  if (is.null(reduction)){
    dim_colnames <- c()
    for(red in Seurat::Reductions(object = sample)){
      if (feature %in% colnames(sample@reductions[[red]][[]])){
        reduction <- red
      }
    }
  }

  if (feature %in% rownames(sample)){
    scale.begin <- min(sample@assays[[assay]]@data[feature, ])
    scale.end <- max(sample@assays[[assay]]@data[feature, ])
  } else if (feature %in% colnames(sample@meta.data)){
    if (is.factor(sample@meta.data[, feature])){
      sample@meta.data[, feature] <- as.character(sample@meta.data[, feature])
    }
    scale.begin <- min(sample@meta.data[, feature])
    scale.end <- max(sample@meta.data[, feature])
  } else if (feature %in% colnames(sample@reductions[[reduction]][[]])){
    scale.begin <- min(sample@reductions[[reduction]][[]][, feature])
    scale.end <- max(sample@reductions[[reduction]][[]][, feature])
  }
  return(list("scale.begin" = scale.begin,
              "scale.end" = scale.end))
}

#' Check if a value is in the range of the values.
#'
#' @param sample Seurat object.
#' @param feature Feature to plot.
#' @param assay Assay used.
#' @param reduction Reduction used.
#' @param value Value to check.
#' @param value_name Name of the value.
#'
#' @return None
#' @noRd
#' @examples
#' \donttest{
#' TBD
#' }
check_limits <- function(sample, feature, value_name, value, assay = NULL, reduction = NULL){
  limits <- compute_scale_limits(sample = sample, feature = feature, assay = assay, reduction = reduction)


  assertthat::assert_that(limits[["scale.begin"]] <= value & limits[["scale.end"]] >= value,
                          msg = paste0("The value provided for ", value_name, " (", value, ") is not in the range of the feature (", feature, "), which is: Min: ", limits[["scale.begin"]], ", Max: ", limits[["scale.end"]], "."))

}

#' Check if the feature to plot is in the Seurat object.
#'
#' @param sample Seurat object.
#' @param features Feature to plot.
#' @param dump_reduction_names Whether to return the reduction colnames.
#' @param permissive Throw a warning or directly stops if the feature is not found.
#'
#' @return None
#' @noRd
#' @examples
#' \donttest{
#' TBD
#' }
check_feature <- function(sample, features, permissive = FALSE, dump_reduction_names = FALSE, enforce_check = NULL, enforce_parameter = NULL){
  if (is.list(features)){
    features_check <- unlist(features)
  } else {
    features_check <- features
  }
  check_enforcers <- list() # Store the results of the checks.
  not_found_features <- c() # Store the features not found.
  # Check each of the features.
  for (feature in features_check){
    check <- 0
    if (!(feature %in% rownames(sample))){
      check <- check + 1
      check_enforcers[["gene"]] <- FALSE
    } else {
      check_enforcers[["gene"]] <- TRUE
    }

    if (!(feature %in% colnames(sample@meta.data))){
      check <- check + 1
      check_enforcers[["metadata"]] <- FALSE
    } else {
      check_enforcers[["metadata"]] <- TRUE
    }

    dim_colnames <- c()
    for(red in Seurat::Reductions(object = sample)){
      dim_colnames <- c(dim_colnames, colnames(sample@reductions[[red]][[]]))
    }
    if (!(feature %in% dim_colnames)){
      check <- check + 1
      check_enforcers[["reductions"]] <- FALSE
    } else {
      check_enforcers[["reductions"]] <- TRUE
    }

    if (check == 3) {
      not_found_features <- c(not_found_features, feature)
    }
  }
  # Return the error logs if there were features not found.
  if (length(not_found_features) > 0){
    if (isTRUE(permissive)){
      # Stop if neither of the features are found.
      assertthat::assert_that(length(unlist(not_found_features)) != length(unlist(features)),
                              msg = "Neither of the provided features are found.")
      warning("The requested features (",
              not_found_features,
              ") could not be found:\n",
              "    - Not matching any gene name (rownames of the provided object).\n",
              "    - Not matching any metadata column (in sample@meta.data).\n",
              "    - Not part of the dimension names in any of the following reductions: ",
              paste(Seurat::Reductions(object = sample), collapse = ", "),
              ".\n\n", call. = FALSE)
      features_out <- remove_not_found_features(features = features, not_found_features = not_found_features)

    } else if (isFALSE(permissive)){
      assertthat::assert_that(length(not_found_features) == 0,
                              msg = paste0("The requested features (",
                                           not_found_features,
                                           ") could not be found:\n",
                                           "    - Not matching any gene name (rownames of the provided object).\n",
                                           "    - Not matching any metadata column (in sample@meta.data).\n",
                                           "    - Not part of the dimension names in any of the following reductions: ",
                                           paste(Seurat::Reductions(object = sample), collapse = ", "),
                                           ".\n\n"))
    }
  } else {
    features_out <- features
  }
  # If we are enforcing a given check (i.e: the feature being in the metadata).
  if (!(is.null(enforce_check))){
    assertthat::assert_that(enforce_check %in% names(check_enforcers),
                            msg = "The variable enforcer is not in the current list of checked variable types.")

    assertthat::assert_that(isTRUE(check_enforcers[[enforce_check]]),
                            msg = paste0("The provided feature (", enforce_parameter, " = ", feature, ") not found in ", enforce_check, "."))
  }

  # Return options.
  if (isTRUE(dump_reduction_names) & isFALSE(permissive)){return(dim_colnames)}
  if (isTRUE(permissive) & isFALSE(dump_reduction_names)){return(features_out)}
  if (isTRUE(dump_reduction_names) & isTRUE(permissive)){return(list("features" = features_out, "reduction_names" = dim_colnames))}
}

#' Remove not found features
#'
#' @param features Features to check.
#' @param not_found_features Features to exclude.
#'
#' @return None
#' @noRd
#' @examples
#' \donttest{
#' TBD
#' }
remove_not_found_features <- function(features, not_found_features){
  if (is.character(features)){
    features_out <- features[!(features %in% not_found_features)]
  } else if (is.list(features)){
    features_out <- list()
    for (list_name in names(features)){
      genes <- features[[list_name]]
      genes_out <- genes[!(genes %in% not_found_features)]
      features_out[[list_name]] <- genes_out
    }
  }
  return(features_out)
}

#' Remove duplicated features.
#'
#' @param features Features to check.
#'
#' @return None
#' @noRd
#' @examples
#' \donttest{
#' TBD
#' }
remove_duplicated_features <- function(features){
  if (is.character(features)){
    check <- sum(duplicated(features))
    if (check > 0){
      warning("Found duplicated features (", paste(features[duplicated(features)], collapse = ", "), "). Excluding them from the analysis.", call. = FALSE)
      features <- features[!(duplicated(features))]
    }
  } else if (is.list(features)){
    features_out <- list()
    all_genes <- c() # Will update with the genes as they iterate to check duplicates.
    for (list_name in names(features)){
      genes <- features[[list_name]]
      # Remove genes duplicated within the list.
      if (sum(duplicated(genes)) > 0){
        warning("Found duplicated features (", paste(genes[duplicated(genes)], collapse = ", "), ") in the list '", list_name, "'. Excluding them from the analysis.", call. = FALSE)
      }
      genes <- genes[!(duplicated(genes))]
      # Remove genes duplicated in the vector of all genes.
      duplicated_features <- genes[genes %in% all_genes]
      all_genes <- c(all_genes, genes[!(genes %in% all_genes)])
      genes <- genes[!(genes %in% duplicated_features)]
      if (length(duplicated_features) > 0){
        warning("Found duplicated features (", paste(duplicated_features, collapse = ", "), ") in list '", list_name, "' with regard to lists. Excluding them from the analysis.", call. = FALSE)
      }
      features_out[[list_name]] <- genes
    }
    features <- features_out
  }
  return(features)
}

#' Check if the identity provided is in the current Seurat identities.
#'
#' @param sample Seurat object.
#' @param identities Identities to test.
#'
#' @return None
#' @noRd
#' @examples
#' \donttest{
#' TBD
#' }
check_identity <- function(sample, identities){
  for (identity in identities){
    assertthat::assert_that(identity %in% levels(sample),
                            msg = paste0("Could not find provided identity (", identity, ") in the current active identities of the object.\n Try running 'levels(your_seurat_object)' and see whether any typos were introduced."))
  }
}

#' Check the reduction provided and set it up.
#'
#' @param sample Seurat sample.
#' @param reduction Reduction.
#'
#' @return None
#' @noRd
#' @examples
#' \donttest{
#' TBD
#' }
check_and_set_reduction <- function(sample, reduction = NULL){
  # Check if the object has a reduction computed.
  assertthat::assert_that(length(Seurat::Reductions(sample)) != 0,
                          msg = "This object has no reductions computed!")
  # If no reduction was provided by the user.
  if (is.null(reduction)){
    # Select umap if computed.
    if ("umap" %in% Seurat::Reductions(sample)){
      reduction <- "umap"
    } else {
      # Select the last computed one.
      reduction <- Seurat::Reductions(sample)[length(Seurat::Reductions(sample))]
    }
  # If the user provided a value for reduction.
  } else if (!(is.null(reduction))){
    # Check if the provided reduction is in the list.
    assertthat::assert_that(reduction %in% Seurat::Reductions(sample),
                            msg = paste0("The provided reduction could not be found in the object: ", reduction))
  }
  return(reduction)
}

#' Check the provided dimensions and set them up.
#'
#' @param sample Seurat object.
#' @param reduction Provided reduction.
#' @param dims Provided dimensions.
#'
#' @return None
#' @noRd
#' @examples
#' \donttest{
#' TBD
#' }
check_and_set_dimensions <- function(sample, reduction = NULL, dims = NULL){
  # If reduction is null, select the last computed one.
  if (is.null(reduction)){
    reduction <- Seurat::Reductions(sample)[length(Seurat::Reductions(sample))]
  }

  # Check that the dimensions is a 2 item vector.
  if (!is.null(dims)){
    assertthat::assert_that(length(dims) == 2,
                            msg = "Provided dimensions need to be a 2-item vector.")

    # Check that at least 2 dimensions are present.
    aval_dims <- length(colnames(Seurat::Embeddings(sample[[reduction]])))

    assertthat::assert_that(aval_dims >= 2,
                            msg = "Available dimensions need to be at least a 2-item vector.")

    # Check that the dimensions are integers.
    null_check <- is.null(dims[1]) & is.null(dims[2])
    integer_check <- is.numeric(dims[1]) & is.numeric(dims[1])

    assertthat::assert_that(isFALSE(null_check) &  isTRUE(integer_check),
                            msg = "Provided dimensions need to be numerics.")

    # Check that the dimensions are in the requested embedding.
    assertthat::assert_that(dims[1] %in% seq_len(aval_dims) & dims[2] %in% seq_len(aval_dims),
                            msg = paste0("Dimension could not be found in the following reduction: ", reduction))
  } else {
    # If no dimensions were provided, fall back to first and second.
    dims <- c(1, 2)
  }
  return(dims)
}

#' Check and set the provided assay.
#'
#' @param sample Seurat object.
#' @param assay Provided assay.
#'
#' @return None
#' @noRd
#' @examples
#' \donttest{
#' TBD
#' }
check_and_set_assay <- function(sample, assay = NULL){
  # Check that at least one assay is computed.
  assertthat::assert_that(length(Seurat::Assays(sample)) != 0,
                          msg = "There must be at least one computed assay in the object.")
  # If assay is null, set it to the active one.
  if (is.null(assay)){
    assay <- Seurat::DefaultAssay(sample)
  } else {
    # Check if the assay is a character.
    assertthat::assert_that(is.character(assay),
                            msg = "The value for assay has to be a character.")
    # Check that the assay is in the available assays.
    aval_assays <- Seurat::Assays(sample)
    assertthat::assert_that(assay %in% aval_assays,
                            msg = paste0("The following assay could not be found: ", assay))
  }
  # Set up the assay the user has defined.
  if (assay != Seurat::DefaultAssay(sample)){
    Seurat::DefaultAssay(sample) <- assay
  }
  return(list("sample" = sample,
              "assay" = assay))
}


#' Check a parameter for a given class.
#'
#' @param parameters List of named parameters to test.
#' @param required_type Name of the required class.
#' @param test_function Testing function.
#'
#' @return None
#' @noRd
#' @examples
#' \donttest{
#' TBD
#' }
check_type <- function(parameters, required_type, test_function){
  for(parameter_name in names(parameters)){
    # Get each individual parameter from the list.
    parameter <- parameters[[parameter_name]]
    # Cases in which the user has to provide a vector.
    # Check if the parameter is not NULL already.
    if (!(is.null(parameter))){
      # For each parameter in the vector.
      for (item in parameter){
        # If not null.
        if (!(is.null(item))){
          # If not NA, if the testing function fails, report it.
          if (sum(!(is.na(item))) > 0){
            assertthat::assert_that(sum(test_function(item)) > 0,
                                    msg = paste0("Parameter ", parameter_name, " needs to be a ", required_type, "."))
          }
        }
      }
    }
  }
}

#' Check the slots.
#'
#' @param slot Slot provided.
#'
#' @return None
#' @noRd
#' @examples
#' \donttest{
#' TBD
#' }
check_and_set_slot <- function(slot){
  if (is.null(slot)){
    slot <- "data"
  } else {
    assertthat::assert_that(slot %in% c("counts", "data", "scale.data"),
                            msg = "Only one of these 3 options can be passed to slot parameter: counts, data, scale.data.")
  }

  return(slot)
}


#' Compute the order of the plotted bars for do_BarPlot.
#'
#' @param sample Seurat object.
#' @param feature Feature to plot.
#' @param group.by Feature to group the output by.
#' @param order Whether to arrange the values.
#' @param order.by Unique value in group.by to reorder labels in descending order.
#'
#' @return None
#' @noRd
#' @examples
#' \donttest{
#' TBD
#' }
compute_factor_levels <- function(sample, feature, position, group.by = NULL, order = FALSE, order.by = FALSE, assay = "SCT", slot = "data"){
  `%>%` <- magrittr::`%>%`

  assertthat::assert_that(position %in% c("stack", "fill"),
                          msg = "Position needs to be either stack or fill.")

  if (is.null(group.by)){
    sample@meta.data[, "group.by"] <- sample@active.ident
  } else {
    sample@meta.data[, "group.by"] <- sample@meta.data[, group.by]
  }
  group.by <- "group.by"

  if (isFALSE(order)){
    factor_levels <- as.character(rev(sort(unique(sample@meta.data[, group.by]))))
  } else if (isTRUE(order)){
    factor_levels <- get_data_column_in_context(sample = sample,
                                                feature = feature,
                                                group.by = group.by,
                                                assay = assay,
                                                slot = slot) %>%
                     dplyr::group_by(.data$group.by) %>%
                     dplyr::summarise("value" = if(is.double(.data$feature)){dplyr::across(.cols = dplyr::all_of("feature"), mean)} else {"feature" <- dplyr::n()}) %>%
                     dplyr::mutate("feature" = if (position == "fill") {.data$value / sum(.data$value)} else {.data$value}) %>%
                     dplyr::arrange(dplyr::desc(.data$feature)) %>%
                     dplyr::pull(.data$group.by) %>%
                     as.character()

  }


  return(factor_levels)
}

#' Check viridis color map.
#'
#' @param viridis_color_map Viridis color map provided.
#' @param verbose Verbosity choice.
#'
#' @return None
#' @noRd
#' @examples
#' \donttest{
#' TBD
#' }
check_viridis_color_map <- function(viridis_color_map, verbose = FALSE){
  check_parameters(viridis_color_map, parameter_name = "viridis_color_map")
}




#' Check length of parameters compared to features.
#'
#' @param vector_of_parameters Vector of parameters to test.
#' @param vector_of_features  Vector of features to test against.
#' @param parameters_name Name of the parameters variable.
#' @param features_name Name of the features variable.

#' @return None
#' @noRd
#' @examples
#' \donttest{
#' TBD
#' }
check_length <- function(vector_of_parameters,
                         vector_of_features,
                         parameters_name,
                         features_name){
  assertthat::assert_that(length(vector_of_parameters) == length(vector_of_features),
                          msg = paste0("Length of ", parameters_name, " not equal to ", features_name, "."))
}


#' Return a SC count matrix
#'
#' @return None
#' @noRd
#' @examples
#' \donttest{
#' TBD
#' }
use_dataset <- function(n_cells = 180){
  # We want this function to be completely silent.
  suppressWarnings({
    genes <- readRDS(system.file("extdata/genes_example.rds", package = "SCpubr"))
    values <- seq(0, 15, 0.1)
    counts <- matrix(ncol = n_cells, nrow = length(genes))
    cols <- c()
    for (i in seq(1, n_cells)){
      cts <- sample(values, size = length(genes), replace = TRUE, prob = c(0.66, rep((0.34 / 150), length(values) - 1)))
      counts[, i] <- cts
      cols <- c(cols, paste0("Cell_", i))
    }
    rownames(counts) <- genes
    colnames(counts) <- cols
    sample <- Seurat::CreateSeuratObject(counts)
    sample <- Seurat::PercentageFeatureSet(sample, pattern = "^MT-", col.name = "percent.mt")
    # Compute QC.
    mask1 <- sample$nCount_RNA >= 1000
    mask2 <- sample$nFeature_RNA >= 500
    mask3 <- sample$percent.mt <= 20
    mask <- mask1 & mask2 & mask3
    sample <- sample[, mask]
    # Normalize.
    sample <- suppressWarnings({Seurat::SCTransform(sample, verbose = FALSE)})

    # Dimensional reduction.
    sample <- Seurat::RunPCA(sample, verbose = FALSE)
    sample <- Seurat::RunUMAP(sample, dims = 1:30, verbose = FALSE)
    # Find clusters.
    sample <- Seurat::FindNeighbors(sample, dims = 1:30, verbose = FALSE)
    sample <- Seurat::FindClusters(sample, resolution = 0.5, verbose = FALSE)
    sample$seurat_clusters <- as.character(sample$seurat_clusters)
    sample$seurat_clusters[1:20] <- "0"
    sample$seurat_clusters[21:40] <- "1"
    sample$seurat_clusters[41:60] <- "2"
    sample$seurat_clusters[61:80] <- "3"
    sample$seurat_clusters[81:100] <- "4"
    sample$seurat_clusters[101:120] <- "5"
    sample$seurat_clusters[121:140] <- "6"
    sample$seurat_clusters[141:160] <- "7"
    sample$seurat_clusters[161:180] <- "8"
    Seurat::Idents(sample) <- sample$seurat_clusters
  })

  return(sample)
}

#' Add viridis color scale while suppressing the warning that comes with adding a second scale.
#'
#' @param p GGplot2 plot.
#' @param num_plots Number of plots.
#' @param function_use Coloring function to use.
#' @param scale Name of the scale. Either fill or color.
#' @param limits Whether to put limits.
#'
#' @return None
#' @noRd
#' @examples
#' \donttest{
#' TBD
#' }
add_scale <- function(p, scale, function_use, num_plots = 1, limits = NULL){
  if (scale == "color"){scale <- "colour"}
  # Compute the number of plots in this object (maybe a more efficient solution exists).
  if (num_plots == 1){
    # Find the index in which the scale is stored.
    # Adapted from: https://stackoverflow.com/a/46003178
    x <- which(sapply(p$scales$scales, function(x) scale %in% x$aesthetics))
    # Remove it.
    p$scales$scales[[x]] <- NULL
  } else {
    for (i in seq(1, num_plots)){
      # Find the index in which the scale is stored.
      # Adapted from: https://stackoverflow.com/a/46003178
      x <- which(sapply(p[[i]]$scales$scales, function(x) scale %in% x$aesthetics))
      # Remove it.
      p[[i]]$scales$scales[[x]] <- NULL
    }
  }
  # Add the scale and now it will now show up a warning since we removed the previous scale.
  p <- p & function_use
  return(p)
}




#' Compute the data frame of the annotation for barplot annotation in heatmaps.
#'
#' @param sample Seurat object.
#' @param group.by Variable to group by.
#' @param annotation Annotation variable to use.
#'
#' @return None
#' @noRd
#' @examples
#' \donttest{
#' TBD
#' }
compute_barplot_annotation <- function(sample,
                                       group.by,
                                       annotation){
  `%>%`<- magrittr::`%>%`
  # Compute column/row annotation. Obtain the percentage of a group per variable.
  annotation <- sample@meta.data %>%
                dplyr::select(dplyr::all_of(c(group.by, annotation))) %>%
                dplyr::mutate(cluster = !!rlang::sym(group.by)) %>%
                dplyr::mutate(subgroup = !!rlang::sym(annotation)) %>%
                dplyr::select(dplyr::all_of(c("cluster", "subgroup"))) %>%
                dplyr::group_by(.data$cluster, .data$subgroup) %>%
                dplyr::summarise(n = dplyr::n()) %>%
                dplyr::mutate(freq = .data$n / sum(.data$n)) %>%
                dplyr::select(dplyr::all_of(c("cluster", "subgroup", "freq"))) %>%
                tidyr::pivot_wider(values_from = "freq", names_from = "subgroup")
  return(annotation)
}




#' Inner helper for heatmaps
#'
#' @param data Matrix ready to be plotted. Use it alongside from_matrix.
#' @param legend.title Name of the general legend.
#' @param data_range One of:
#' - "both": Will compute a color scale equally balanced to both sides. Use when the values to plot are positive and negative.
#' - "only_pos": Will compute a color scale based only on the positive values. Will take the positive end of colors.use as well. Use when the values to plot are only positive.
#' - "only_neg": Will compute a color scale based only on the negative values. Will take the negative end of colors.use as well. Use when the values to plot are only negative
#' @param colors.use Vector of 2 colors defining a gradient. White color will be inserted in the middle.
#' @param grid_color Color for the grid.
#' @param range.data Numeric. Min or max value (data_range = "only_pos" or "only_neg") or vector of min and max (data_range = "both") that will determine the span of the color scale.
#' @param outlier.data Logical. Whether there is outlier data to take into account.
#' @param fontsize General fontsize of the plot.
#' @param cell_size Size of each of the cells in the heatmap.
#' @param row_names_side,column_names_side Where to place the column or row names. "top", "bottom", "left", "right".
#' @param cluster_columns,cluster_rows Logical. Whether to cluster the rows or the columns of the heatmap.
#' @param border Logical. Whether to draw the border of the heatmap.
#' @param row_dendogram,column_dendogram Logical. Whether to plot row and column dendograms.
#' @param row_annotation,column_annotation Annotation objects.
#' @param row_annotation_side,column_annotation_side Where to place the annotation. "top", "bottom", "left", "right".
#' @param row_title,column_title Titles for the axes.
#' @param column_title_side,row_title_side Side for the titles.
#' @param column_title_rotation,row_title_rotation Angle of rotation of the titles.
#' @param row_names_rot,column_names_rot Angle of rotation of the text.
#' @param legend.framecolor Color of the lines of the box in the legend.
#' @param legend.length,legend.width Length and width of the legend. Will adjust automatically depending on legend side.
#' @param na.value Color for NAs
#' @param use_viridis Logical. Whether to use viridis color palettes.
#' @param viridis_color_map Character. Palette to use.
#' @param viridis_direction Numeric. Direction of the scale.
#' @param zeros_are_white Logical.
#' @return None
#' @noRd
#' @examples
#' \donttest{
#' TBD
#' }
heatmap_inner <- function(data,
                          legend.title = "Values",
                          data_range = "both",
                          colors.use = NULL,
                          grid_color = "grey50",
                          fontsize = 12,
                          cell_size = 5,
                          range.data = NULL,
                          outlier.data = FALSE,
                          outlier.up.color = "#4b010b",
                          outlier.down.color = "#02294b",
                          outlier.up.label = NULL,
                          outlier.down.label = NULL,
                          round_value_outlier = 2,
                          column_title = NULL,
                          row_title = NULL,
                          row_names_side = "left",
                          column_names_side = "bottom",
                          cluster_columns = TRUE,
                          cluster_rows = TRUE,
                          border = TRUE,
                          legend.position = "bottom",
                          legend.length = 20,
                          legend.width = 1,
                          legend.framecolor = "grey50",
                          row_dendogram = FALSE,
                          column_dendogram = FALSE,
                          column_title_side = "top",
                          row_title_rotation = 90,
                          column_title_rotation = 0,
                          row_names_rot = 0,
                          column_names_rot = 90,
                          row_title_side = "left",
                          row_annotation = NULL,
                          row_annotation_side = "right",
                          column_annotation = NULL,
                          column_annotation_side = "top",
                          na.value = "grey75",
                          use_viridis = FALSE,
                          viridis_color_map = "D",
                          viridis_direction = 1,
                          zeros_are_white = FALSE,
                          symmetrical_scale = FALSE){
  `%>%`<- magrittr::`%>%`

  assertthat::assert_that((nrow(data) >= 1 & ncol(data) > 1) | (nrow(data) > 1 & ncol(data) >= 1),
                          msg = "Please provide a matrix that is not 1x1.")

  if (!(is.null(range.data)) & data_range == "both"){
    assertthat::assert_that(length(range.data) == 2,
                            msg = "When providing data_range = both and range data, you need to specify the two ends of the scale in range.data. Please provide two numbers to range.data.")
  }

  assertthat::assert_that(sum(dim(unique(data))) > 2,
                          msg = "Please provide a matrix with at least 2 different values.")

  if (legend.position %in% c("top", "bottom")){
    legend_width <- grid::unit(legend.length, "mm")
    legend_height <- NULL
    grid_height <- grid::unit(legend.width, "mm")
    grid_width <- grid::unit(4, "mm")
    direction <- "horizontal"
    title_position <- "topcenter"
  } else if (legend.position %in% c("left", "right")){
    grid_width <- grid::unit(legend.width, "mm")
    legend_height <- grid::unit(legend.length, "mm")
    legend_width <- NULL
    grid_height <- grid::unit(4, "mm")
    direction <- "vertical"
    title_position <- "topleft"
  }

  if (!is.null(range.data)){

    if (data_range == "both"){
      if (isTRUE(symmetrical_scale)){
        abs_value <- max(abs(range.data), na.rm = TRUE)
        q100 <- abs(abs_value)
        q0 <- -abs(abs_value)
      } else {
        q0 <- range.data[1]
        q100 <- range.data[2]
        abs_value <- q100
      }
    } else if (data_range == "only_pos"){
      abs_value <- abs(range.data)
      q0 <- 0
      q100 <- abs_value
    } else if (data_range == "only_neg"){
      abs_value <- abs(range.data)
      q100 <- 0
      q0 <- range.data
    }
  } else {
    q0 <- min(data, na.rm = TRUE)
    q100 <- max(data, na.rm = TRUE)
    abs_value <- max(c(abs(q0), abs(q100)), na.rm = TRUE)
  }

  q50 <- mean(c(q0, q100))
  q25 <- mean(c(q0, q50))
  q75 <- mean(c(q50, q100))

  # Checks.
  if (data_range == "only_neg"){
    assertthat::assert_that(q0 < 0,
                            msg = "There are no negative values in the matrix.")
  } else if (data_range == "only_pos"){
    assertthat::assert_that(q100 > 0,
                            msg = "There are no positive values in the matrix.")
  }

  if (is.null(colors.use)){
    colors.use <- c("#023f73", "white", "#7a0213")
  } else {
    colors.use <- c(colors.use[1], "white", colors.use[2])
  }
  if (data_range == "both"){
    if (isTRUE(symmetrical_scale)){
      breaks <-  round(c(-abs_value, (-abs_value / 2), 0, (abs_value / 2), abs_value), 1)
      counter <- 0
      while (sum(duplicated(breaks)) > 0){
        counter <- counter + 1
        breaks <-  round(c(-abs_value, (-abs_value / 2), 0, (abs_value / 2), abs_value), 1 + counter)
      }
    } else if (isFALSE(symmetrical_scale)){
      breaks <-  round(c(q0, q25, q50, q75, q100), 1)
      counter <- 0
      while (sum(duplicated(breaks)) > 0){
        counter <- counter + 1
        breaks <-  round(c(q0, q25, q50, q75, q100), 1 + counter)
      }
    }
    labels <- as.character(breaks)
    colors.use <- grDevices::colorRampPalette(colors.use)(length(breaks))
    if (isTRUE(outlier.data) & !is.null(range.data)){
      breaks <- c(-abs_value - 0.00001, breaks, abs_value + 0.00001)
      colors.use <- c(outlier.down.color, colors.use, outlier.up.color)
      labels <- c(if(is.null(outlier.down.label)){paste0("< ", -round(abs_value, round_value_outlier))} else {outlier.down.label},
                  labels,
                  if(is.null(outlier.up.label)){paste0("> ", -round(abs_value, round_value_outlier))} else {outlier.up.label})
    }

    names(colors.use) <- labels
  } else if (data_range == "only_neg"){
    if (isTRUE(zeros_are_white)){
      breaks <-  round(c(-abs_value, (-abs_value * 0.75), (-abs_value * 0.5), (-abs_value * 0.25), (-abs_value * 0.01), 0), 1)
      counter <- 0
      while (sum(duplicated(breaks)) > 0){
        counter <- counter + 1
        breaks <-  round(c(-abs_value, (-abs_value * 0.75), (-abs_value * 0.5), (-abs_value * 0.25), (-abs_value * 0.01), 0), 1 + counter)
      }
    } else {
      breaks <-  round(c(-abs_value, (-abs_value * 0.75), (-abs_value * 0.5), (-abs_value * 0.25), 0), 1)
      counter <- 0
      while (sum(duplicated(breaks)) > 0){
        counter <- counter + 1
        breaks <-  round(c(-abs_value, (-abs_value * 0.75), (-abs_value * 0.5), (-abs_value * 0.25), 0), 1 + counter)
      }
    }
    labels <- as.character(breaks)
    colors.use <- grDevices::colorRampPalette(colors.use[c(1, 2)])(length(breaks))
    if (isTRUE(outlier.data) & !is.null(range.data)){
      breaks <- c(-abs_value - 0.00001, breaks)
      colors.use <- c(outlier.down.color, colors.use)
      labels <- c(if(is.null(outlier.down.label)){paste0("< ", -round(abs_value, round_value_outlier))} else {outlier.down.label},
                  labels)
    }
    names(colors.use) <- labels
  } else if (data_range == "only_pos"){
    if (isTRUE(zeros_are_white)){
      breaks <-  round(c(0, (abs_value * 0.01), (abs_value * 0.25), (abs_value * 0.5), (abs_value * 0.75), abs_value), 1)
      counter <- 0
      while (sum(duplicated(breaks)) > 0){
        counter <- counter + 1
        breaks <-  round(c(0, (abs_value * 0.01), (abs_value * 0.25), (abs_value * 0.5), (abs_value * 0.75), abs_value), 1 + counter)
      }
    } else {
      breaks <-  round(c(0, (abs_value * 0.25), (abs_value * 0.5), (abs_value * 0.75), abs_value), 1)
      counter <- 0
      while (sum(duplicated(breaks)) > 0){
        counter <- counter + 1
        breaks <-  round(c(0, (abs_value * 0.25), (abs_value * 0.5), (abs_value * 0.75), abs_value), 1 + counter)
      }
    }
    labels <- as.character(breaks)
    colors.use <- grDevices::colorRampPalette(colors.use[c(2, 3)])(length(breaks))
    if (isTRUE(outlier.data) & !is.null(range.data)){
      breaks <- c(breaks, abs_value + 0.00001)
      colors.use <- c(colors.use, outlier.up.color)
      labels <- c(labels,
                  if(is.null(outlier.up.label)){paste0("> ", -round(abs_value, round_value_outlier))} else {outlier.up.label})
    }
    names(colors.use) <- labels
  }

  if (isTRUE(use_viridis)){
    if (isTRUE(zeros_are_white) & data_range %in% c("only_pos", "only_neg")){
      col_fun <- circlize::colorRamp2(breaks = breaks, colors = c("white", viridis::viridis(n = length(breaks) - 1,
                                                                                            option = viridis_color_map,
                                                                                            direction = viridis_direction)))
    } else {
      col_fun <- circlize::colorRamp2(breaks = breaks, colors = viridis::viridis(n = length(breaks),
                                                                                 option = viridis_color_map,
                                                                                 direction = viridis_direction))
    }
  } else {
    col_fun <- circlize::colorRamp2(breaks = breaks, colors = colors.use)
  }



  lgd <- ComplexHeatmap::Legend(at = breaks,
                                labels = labels,
                                col_fun = col_fun,
                                title = legend.title,
                                direction = direction,
                                legend_height = legend_height,
                                legend_width = legend_width,
                                grid_width = grid_width,
                                grid_height = grid_height,
                                border = legend.framecolor,
                                title_position = title_position,
                                break_dist = rep(1, length(breaks) - 1),
                                labels_gp = grid::gpar(fontsize = fontsize,
                                                       fontface = "bold"),
                                title_gp = grid::gpar(fontsize = fontsize,
                                                      fontface = "bold"))


  if (!(is.null(row_annotation))){
    if (row_annotation_side == "right"){
      right_annotation <- row_annotation
      left_annotation <- NULL
    } else {
      right_annotation <- NULL
      left_annotation <- row_annotation
    }
  } else {
    right_annotation <- NULL
    left_annotation <- NULL
  }

  if (!(is.null(column_annotation))){
    if (column_annotation_side == "top"){
      top_annotation <- column_annotation
      bottom_annotation <- NULL
    } else {
      top_annotation <- NULL
      bottom_annotation <- column_annotation
    }
  } else {
    top_annotation <- NULL
    bottom_annotation <- NULL
  }


  h <- ComplexHeatmap::Heatmap(matrix = data,
                               name = legend.title,
                               col = col_fun,
                               na_col = na.value,
                               show_heatmap_legend = FALSE,
                               cluster_rows = cluster_rows,
                               cluster_columns = cluster_columns,
                               show_row_dend = row_dendogram,
                               show_column_dend = column_dendogram,
                               top_annotation = top_annotation,
                               bottom_annotation = bottom_annotation,
                               right_annotation = right_annotation,
                               left_annotation = left_annotation,
                               width = ncol(data)*grid::unit(cell_size, "mm"),
                               height = nrow(data)*grid::unit(cell_size, "mm"),
                               column_names_gp = grid::gpar(fontsize = fontsize,
                                                            fontface = "bold"),
                               row_names_gp = grid::gpar(fontsize = fontsize,
                                                         fontface = "bold"),
                               row_names_side = row_names_side,
                               column_names_side = column_names_side,
                               column_title = column_title,
                               column_title_side = column_title_side,
                               row_title_side = row_title_side,
                               row_title = row_title,
                               column_title_rot = column_title_rotation,
                               row_title_rot = row_title_rotation,
                               column_names_rot = column_names_rot,
                               row_names_rot = row_names_rot,
                               column_title_gp = grid::gpar(fontsize = fontsize,
                                                            fontface = "bold"),
                               row_title_gp = grid::gpar(fontsize = fontsize,
                                                         fontface = "bold"),
                               border = border,
                               rect_gp = grid::gpar(col= grid_color),
                               cell_fun = function(j, i, x, y, w, h, fill) {
                                 grid::grid.rect(x, y, w, h, gp = grid::gpar(alpha = 0))
                               },
                               column_names_centered = FALSE,
                               row_names_centered = FALSE)

  return_list <- list("heatmap" = h,
                      "legend" = lgd)

  return(return_list)
}


#' Modify a string to wrap it around the middle point.
#'
#' @param string_to_modify
#'
#' @return None
#' @noRd
#' @examples
#' \donttest{
#' TBD
#' }
modify_string <- function(string_to_modify){
  words <- stringr::str_split(string_to_modify, " ")[[1]]
  num_words <- length(words)
  middle_point <- round(num_words / 2, 0)
  string_to_modify <- paste(paste(words[1:middle_point], collapse = " "), "\n",
                            paste(words[(middle_point + 1):num_words], collapse = " "))
  return(string_to_modify)
}


#' Compute Enrichment scores using Seurat::AddModuleScore()
#'
#' @param sample  Seurat object.
#' @param input_gene_list  Named list of genes to compute enrichment for.
#' @param verbose  Verbose output.
#' @param nbin Number of bins.
#' @param ctrl Number of control genes.
#'
#' @return None
#' @noRd
#' @examples
#' \donttest{
#' TBD
#' }
compute_enrichment_scores <- function(sample,
                                      input_gene_list,
                                      verbose = FALSE,
                                      nbin = 24,
                                      ctrl = 100,
                                      assay = NULL,
                                      slot = NULL,
                                      flavor = "Seurat",
                                      ncores = 1,
                                      storeRanks = TRUE){
  # Checks for UCell.
  if (flavor == "UCell"){
    R_version <- paste0(R.version$major, ".", R.version$minor)
    assertthat::assert_that(R_version >= "4.2.0",
                            msg = "To run UCell scoring, R version 4.2.0 is required. Please select flavor = 'Seurat' if you are running a version inferior to this.")
    if (!requireNamespace("UCell", quietly = TRUE)) {
      stop(paste0("Package UCell must be installed to run UCell scoring."), call. = FALSE)
    }
  }

  if (!is.list(input_gene_list) & is.character(input_gene_list)){
    input_gene_list <- list("Input" = input_gene_list)
  }
  for (celltype in names(input_gene_list)){
    list_markers <- list(input_gene_list[[celltype]])

    if (flavor == "Seurat"){
      # Compute Seurat AddModuleScore as well.
      if (verbose){
        sample <- Seurat::AddModuleScore(sample,
                                         list_markers,
                                         name = celltype,
                                         search = TRUE,
                                         verbose = TRUE,
                                         nbin = nbin,
                                         ctrl = ctrl,
                                         assay = assay)
      } else {
        sample <- suppressMessages(suppressWarnings(Seurat::AddModuleScore(sample,
                                                                           list_markers,
                                                                           name = celltype,
                                                                           search = TRUE,
                                                                           verbose = FALSE,
                                                                           nbin = nbin,
                                                                           ctrl = ctrl,
                                                                           assay = assay)))
      }


      # Retrieve the scores.
      col_name <- stringr::str_replace_all(paste0(celltype, "1"), " ", ".")
      col_name <- stringr::str_replace_all(col_name, "-", ".")
      col_name <- stringr::str_replace_all(col_name, "\\+", ".")

      # Modify the name that Seurat::AddModuleScore gives by default.
      sample@meta.data[, celltype] <- sample@meta.data[, col_name]
      # Remove old metadata.
      sample@meta.data[, col_name] <- NULL
    }
  }
  if (flavor == "UCell"){
    list.names <- c()
    for (celltype in names(input_gene_list)){
      col_name <- celltype
      col_name <- stringr::str_replace_all(col_name, "-", ">")
      col_name <- stringr::str_replace_all(col_name, " ", "_")
      col_name <- stringr::str_replace_all(col_name, "\\+", ".")
      list.names <- append(list.names, col_name)
    }
    list.originals <- names(input_gene_list)
    names(input_gene_list) <- list.names

    sample <- UCell::AddModuleScore_UCell(obj = sample,
                                          features = input_gene_list,
                                          assay = assay,
                                          slot = if (is.null(slot)){"data"} else {slot},
                                          name = "",
                                          ncores = ncores,
                                          storeRanks = storeRanks)

    for (i in seq_len(length(list.names))){
      old.name <- list.originals[i]
      mod.name <- list.names[i]
      # Modify the name that Seurat::AddModuleScore gives by default.
      sample@meta.data[, old.name] <- sample@meta.data[, mod.name]
      # Remove old metadata.
      if (old.name != mod.name){
        sample@meta.data[, mod.name] <- NULL
      }
    }
  }
  return(sample)
}




#' Modify the aspect of the legend.
#'
#' @param p Plot.
#' @param legend.aes Character. Either color or fill.
#' @param legend.type Character. Type of legend to display. One of: normal, colorbar, colorsteps.
#' @param legend.position Position of the legend in the plot. Will only work if legend is set to TRUE.
#' @param legend.framewidth,legend.tickwidth Width of the lines of the box in the legend.
#' @param legend.framecolor,legend.tickcolor Color of the lines of the box in the legend.
#' @param legend.length,legend.width Length and width of the legend. Will adjust automatically depending on legend side.
#' @param legend.title Character. Title for the legend.
#'
#' @return None
#' @noRd
#' @examples
#' \donttest{
#' TBD
#' }
modify_continuous_legend <- function(p,
                                     legend.aes,
                                     legend.type,
                                     legend.position,
                                     legend.length,
                                     legend.width,
                                     legend.framecolor,
                                     legend.tickcolor,
                                     legend.tickwidth,
                                     legend.framewidth,
                                     legend.title = NULL){
  # Define legend parameters. Width and height values will change depending on the legend orientation.
  if (legend.position %in% c("top", "bottom")){
    legend.barwidth <- legend.length
    legend.barheight <- legend.width
  } else if (legend.position %in% c("left", "right")){
    legend.barwidth <- legend.width
    legend.barheight <- legend.length
  }

  legend.title <- if (is.null(legend.title)){ggplot2::waiver()} else {legend.title}

  if (legend.aes == "color" | legend.aes == "colour"){
    if (legend.type == "normal"){
      p <- p +
        ggplot2::guides(color = ggplot2::guide_colorbar(title = legend.title,
                                                        title.position = "top",
                                                        title.hjust = 0.5))
    } else if (legend.type == "colorbar"){
      p <- p +
        ggplot2::guides(color = ggplot2::guide_colorbar(title = legend.title,
                                                        title.position = "top",
                                                        barwidth = legend.barwidth,
                                                        barheight = legend.barheight,
                                                        title.hjust = 0.5,
                                                        ticks.linewidth = legend.tickwidth,
                                                        frame.linewidth = legend.framewidth,
                                                        frame.colour = legend.framecolor,
                                                        ticks.colour = legend.tickcolor))
    } else if (legend.type == "colorsteps"){
      p <- p +
        ggplot2::guides(color = ggplot2::guide_colorsteps(title = legend.title,
                                                          title.position = "top",
                                                          barwidth = legend.barwidth,
                                                          barheight = legend.barheight,
                                                          title.hjust = 0.5,
                                                          ticks.linewidth = legend.tickwidth,
                                                          frame.linewidth = legend.framewidth,
                                                          frame.colour = legend.framecolor,
                                                          ticks.colour = legend.tickcolor))
    }
  } else if (legend.aes == "fill"){
    if (legend.type == "normal"){
      p <- p +
        ggplot2::guides(fill = ggplot2::guide_colorbar(title = legend.title,
                                                       title.position = "top",
                                                        title.hjust = 0.5))
    } else if (legend.type == "colorbar"){
      p <- p +
        ggplot2::guides(fill = ggplot2::guide_colorbar(title = legend.title,
                                                       title.position = "top",
                                                        barwidth = legend.barwidth,
                                                        barheight = legend.barheight,
                                                        title.hjust = 0.5,
                                                        ticks.linewidth = legend.tickwidth,
                                                        frame.linewidth = legend.framewidth,
                                                        frame.colour = legend.framecolor,
                                                        ticks.colour = legend.tickcolor))
    } else if (legend.type == "colorsteps"){
      p <- p +
        ggplot2::guides(fill = ggplot2::guide_colorsteps(title = legend.title,
                                                         title.position = "top",
                                                          barwidth = legend.barwidth,
                                                          barheight = legend.barheight,
                                                          title.hjust = 0.5,
                                                          ticks.linewidth = legend.tickwidth,
                                                          frame.linewidth = legend.framewidth,
                                                          frame.colour = legend.framecolor,
                                                          ticks.colour = legend.tickcolor))
    }
  }

  return(p)
}

#' Return a column with the desired values of a feature per cell.
#'
#' @param sample Seurat object.
#' @param feature Feature to retrieve data.
#' @param assay Assay to pull data from.
#' @param slot Slot from the assay to pull data from.
#'
#' @noRd
#' @examples
#' \donttest{
#' TBD
#' }
get_data_column <- function(sample,
                            feature,
                            assay,
                            slot){
  `%>%` <- magrittr::`%>%`
  dim_colnames <- c()
  for(red in Seurat::Reductions(object = sample)){
    col.names <- colnames(sample@reductions[[red]][[]])
    dim_colnames <- c(dim_colnames, col.names)
    if (feature %in% col.names){
      # Get the reduction in which the feature is, if this is the case.
      reduction <- red
    }
  }

  if (isTRUE(feature %in% colnames(sample@meta.data))){
    feature_column <- sample@meta.data %>%
                      dplyr::select(dplyr::all_of(c(feature))) %>%
                      tibble::rownames_to_column(var = "cell") %>%
                      dplyr::rename("feature" = dplyr::all_of(c(feature)))
  } else if (isTRUE(feature %in% rownames(sample))){
    feature_column <- Seurat::GetAssayData(object = sample,
                                           assay = assay,
                                           slot = slot)[feature, , drop = FALSE] %>%
      as.matrix() %>%
      t() %>%
      as.data.frame() %>%
      tibble::rownames_to_column(var = "cell") %>%
      dplyr::rename("feature" = dplyr::all_of(c(feature)))
  } else if (isTRUE(feature %in% dim_colnames)){
    feature_column <- sample@reductions[[reduction]][[]][, feature, drop = FALSE] %>%
      as.data.frame() %>%
      tibble::rownames_to_column(var = "cell") %>%
      dplyr::rename("feature" = dplyr::all_of(c(feature)))
  }
  return(feature_column)
}

#' Return a column with the desired values of a feature per cell.
#'
#' @param sample Seurat object.
#' @param feature Feature to retrieve data.
#' @param assay Assay to pull data from.
#' @param group.by Parameter used later on for grouping.
#' @param split.by Parameter used later on for splitting.
#' @param slot Slot from the assay to pull data from.
#'
#' @noRd
#' @examples
#' \donttest{
#' TBD
#' }
get_data_column_in_context <- function(sample,
                                       feature,
                                       assay,
                                       slot,
                                       group.by = NULL,
                                       split.by = NULL){
  `%>%` <- magrittr::`%>%`
  if (is.null(group.by)){
    sample@meta.data[, "group.by"] <- sample@active.ident
  } else {
    sample@meta.data[, "group.by"] <- sample@meta.data[, group.by]
  }
  group.by <- "group.by"

  vars <- c("cell", "group.by")
  if (!is.null(split.by)){
    sample@meta.data[, "split.by"] <- sample@meta.data[, split.by]
    vars <- c(vars, "split.by")
  }

  data <- sample@meta.data %>%
          tibble::rownames_to_column(var = "cell") %>%
          dplyr::select(dplyr::all_of(vars)) %>%
          dplyr::left_join(y = get_data_column(sample = sample,
                                               feature = feature,
                                               assay = assay,
                                               slot = slot),
                           by = "cell") %>%
          tibble::as_tibble()

  return(data)
}

#' Check parameters.
#'
#' @param parameter Parameter to check
#' @param parameter_name Name of the parameter.
#'
#' @noRd
#' @examples
#' \donttest{
#' TBD
#' }
check_parameters <- function(parameter,
                             parameter_name){
  if (parameter_name == "font.type"){
    # Check font.type.
    assertthat::assert_that(parameter %in% c("sans", "serif", "mono"),
                           msg = "Please select one of the following for font.type: sans, serif, mono.")
  } else if (parameter_name == "legend.type"){
    # Check the legend.type.
    assertthat::assert_that(parameter %in% c("normal", "colorbar", "colorsteps"),
                            msg = "Please select one of the following for legend.type: normal, colorbar, colorsteps.")
  } else if (parameter_name == "legend.position"){
    # Check the legend.position.
    assertthat::assert_that(parameter %in% c("top", "bottom", "left", "right", "none"),
                            msg = "Please select one of the following for legend.position: top, bottom, left, right, none.")
  } else if (parameter_name == "marginal.type"){
    # Check marginal.type.
    assertthat::assert_that(parameter %in% c("density", "histogram", "boxplot", "violin", "densigram"),
                            msg = "Please select one of the following for marginal.type: density, histogram, boxplot, violin, densigram.")
  } else if (parameter_name == "viridis_direction"){
    assertthat::assert_that(parameter %in% c(1, -1),
                            msg = "Please provide a value for viridis_direction of -1 or 1.")
  } else if (parameter_name == "viridis_color_map"){
    viridis_options <- c("A", "B", "C", "D", "E", "F", "G", "H", "magma", "inferno", "plasma", "viridis", "cividis", "rocket", "mako", "turbo")
    assertthat::assert_that(parameter %in% viridis_options,
                            msg = "The option provided to viridis_color_map is not an accepted option.")
  } else if (parameter_name == "grid.type"){
    assertthat::assert_that(parameter %in% c("blank", "solid", "dashed", "dotted", "dotdash", "longdash", "twodash"),
                            msg = "Please select one of the following for grid.type: blank, solid, dashed, dotted, dotdash, longdash, twodash.")
  } else if (parameter_name == "direction.type"){
    for (item in parameter){
      assertthat::assert_that(item %in% c("diffHeight", "arrows"),
                              msg = "Please set direction.type as either diffHeight, arrows or both.")
    }
  } else if (parameter_name == "self.link"){
    assertthat::assert_that(parameter %in% c(1, 2),
                            msg = "Please set self.link as either 1 or 2.")
  } else if (parameter_name == "directional"){
    assertthat::assert_that(parameter %in% c(0, 1, 2, -1),
                            msg = "Please set directional as either 0, 1, 2 or -1.")
  } else if (parameter_name == "link.arr.type"){
    assertthat::assert_that(parameter %in% c("big.arrow", "triangle"),
                            msg = "Please set link.arr.type as either big.arrow or triangle.")
  } else if (parameter_name == "alignment"){
    assertthat::assert_that(parameter %in% c("default", "vertical", "horizontal"),
                            msg = "Please set alignment as either default or vertical or horizontal.")
  } else if (parameter_name == "alpha.highlight"){
    assertthat::assert_that(parameter %in% c(seq(1, 99), "FF"),
                            msg = "Please provide either FF or a number between 1 and 99 to alpha.highlight.")
  } else if (parameter_name == "scale_type"){
    assertthat::assert_that(parameter %in% c("categorical", "continuous"),
                            msg = "Please provide one of the following to scale_type: continuous, categorical.")
  } else if (parameter_name == "rotate_x_axis_labels"){
    assertthat::assert_that(parameter %in% c(0, 45, 90),
                            msg = "Please provide one of the following to rotate_x_axis_labels: 0, 45, 90.")
  } else if (parameter_name == "contour.lineend"){
    assertthat::assert_that(parameter %in% c("round", "butt", "square"),
                            msg = "Please provide one of the following to contour_lineend: round, butt, square.")
  } else if (parameter_name == "contour.linejoin"){
    assertthat::assert_that(parameter %in% c("round", "mitre", "bevel"),
                            msg = "Please provide one of the following to contour_linejoin: round, mitre, bevel.")
  } else if (parameter_name == "contour.position"){
    assertthat::assert_that(parameter %in% c("bottom", "top"),
                            msg = "Please provide one of the following to contour_position: top, bottom.")
  } else if (parameter_name == "flavor"){
    assertthat::assert_that(parameter %in% c("Seurat", "UCell"),
                            msg = "Please provide one of the following to contour_position: Seurat, UCell")
  }
}

#' Helper for do_AlluvialPlot.
#'
#' @param data  Data to plot.
#' @param vars.use  Names of the variables.
#'
#' @noRd
#' @examples
#' \donttest{
#' TBD
#' }
prepare_ggplot_alluvial_plot <- function(data,
                                         vars.use){
  items <- length(vars.use)
  `%>%` <- magrittr::`%>%`
  assertthat::assert_that(items <= 10,
                          msg = "Please provide between first_group, middle_groups, and last_group only up to 10 different elements.")
  if (items == 2){
    p <- data %>%
         ggplot2::ggplot(mapping = ggplot2::aes(y = data$n,
                                                axis1 = data[[vars.use[1]]],
                                                axis2 = data[[vars.use[2]]]))
  } else if (items == 3){
    p <- data %>%
         ggplot2::ggplot(mapping = ggplot2::aes(y = data$n,
                                                axis1 = data[[vars.use[1]]],
                                                axis2 = data[[vars.use[2]]],
                                                axis3 = data[[vars.use[3]]]))
  } else if (items == 4){
    p <- data %>%
         ggplot2::ggplot(mapping = ggplot2::aes(y = data$n,
                                                axis1 = data[[vars.use[1]]],
                                                axis2 = data[[vars.use[2]]],
                                                axis3 = data[[vars.use[3]]],
                                                axis4 = data[[vars.use[4]]]))
  } else if (items == 5){
    p <- data %>%
         ggplot2::ggplot(mapping = ggplot2::aes(y = data$n,
                                                axis1 = data[[vars.use[1]]],
                                                axis2 = data[[vars.use[2]]],
                                                axis3 = data[[vars.use[3]]],
                                                axis4 = data[[vars.use[4]]],
                                                axis5 = data[[vars.use[5]]]))
  } else if (items == 6){
    p <- data %>%
         ggplot2::ggplot(mapping = ggplot2::aes(y = data$n,
                                                axis1 = data[[vars.use[1]]],
                                                axis2 = data[[vars.use[2]]],
                                                axis3 = data[[vars.use[3]]],
                                                axis4 = data[[vars.use[4]]],
                                                axis5 = data[[vars.use[5]]],
                                                axis6 = data[[vars.use[6]]]))
  } else if (items == 7){
    p <- data %>%
         ggplot2::ggplot(mapping = ggplot2::aes(y = data$n,
                                                axis1 = data[[vars.use[1]]],
                                                axis2 = data[[vars.use[2]]],
                                                axis3 = data[[vars.use[3]]],
                                                axis4 = data[[vars.use[4]]],
                                                axis5 = data[[vars.use[5]]],
                                                axis6 = data[[vars.use[6]]],
                                                axis7 = data[[vars.use[7]]]))
  } else if (items == 8) {
    p <- data %>%
         ggplot2::ggplot(mapping = ggplot2::aes(y = data$n,
                                                axis1 = data[[vars.use[1]]],
                                                axis2 = data[[vars.use[2]]],
                                                axis3 = data[[vars.use[3]]],
                                                axis4 = data[[vars.use[4]]],
                                                axis5 = data[[vars.use[5]]],
                                                axis6 = data[[vars.use[6]]],
                                                axis7 = data[[vars.use[7]]],
                                                axis8 = data[[vars.use[8]]]))
  } else if (items == 9){
    p <- data %>%
         ggplot2::ggplot(mapping = ggplot2::aes(y = data$n,
                                                axis1 = data[[vars.use[1]]],
                                                axis2 = data[[vars.use[2]]],
                                                axis3 = data[[vars.use[3]]],
                                                axis4 = data[[vars.use[4]]],
                                                axis5 = data[[vars.use[5]]],
                                                axis6 = data[[vars.use[6]]],
                                                axis7 = data[[vars.use[7]]],
                                                axis8 = data[[vars.use[8]]],
                                                axis9 = data[[vars.use[9]]]))
  } else if (items == 10){
    p <- data %>%
         ggplot2::ggplot(mapping = ggplot2::aes(y = data$n,
                                                axis1 = data[[vars.use[1]]],
                                                axis2 = data[[vars.use[2]]],
                                                axis3 = data[[vars.use[3]]],
                                                axis4 = data[[vars.use[4]]],
                                                axis5 = data[[vars.use[5]]],
                                                axis6 = data[[vars.use[6]]],
                                                axis7 = data[[vars.use[7]]],
                                                axis8 = data[[vars.use[8]]],
                                                axis9 = data[[vars.use[9]]],
                                                axis10 = data[[vars.use[10]]]))
  }
  return(p)
}

#' Helper for rotate_x_axis_labels.
#'
#' @param angle Angle of rotation.
#' @param flip Whether the plot if flipped or not.
#'
#' @noRd
#' @examples
#' \donttest{
#' TBD
#' }
get_axis_parameters <- function(angle,
                               flip){
  if (isTRUE(flip)){
    if (angle == 0){
      out <- list("angle" = angle,
                  "hjust" = 0.5,
                  "vjust" = 0.5)
    } else if (angle == 45){
      out <- list("angle" = angle,
                  "hjust" = 1,
                  "vjust" = 1)
    } else if (angle == 90){
      out <- list("angle" = angle,
                  "hjust" = 0.5,
                  "vjust" = 0.5)
    }
  } else if (isFALSE(flip)){
    if (angle == 0){
      out <- list("angle" = angle,
                  "hjust" = 0.5,
                  "vjust" = 0)
    } else if (angle == 45){
      out <- list("angle" = angle,
                  "hjust" = 1,
                  "vjust" = 1)
    } else if (angle == 90){
      out <- list("angle" = angle,
                  "hjust" = 0.5,
                  "vjust" = 0.5)
    }
  }
  return(out)
}




