% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/xEnrichCompare.r
\name{xEnrichCompare}
\alias{xEnrichCompare}
\title{Function to compare enrichment results using side-by-side barplots}
\usage{
xEnrichCompare(list_eTerm, displayBy = c("fc", "adjp", "fdr", "zscore",
"pvalue"), FDR.cutoff = 0.05, bar.label = TRUE, bar.label.size = 3,
wrap.width = NULL, sharings = NULL, signature = TRUE)
}
\arguments{
\item{list_eTerm}{a list of "eTerm" objects}

\item{displayBy}{which statistics will be used for comparison. It can
be "fc" for enrichment fold change (by default), "adjp" or "fdr" for
adjusted p value (or FDR), "pvalue" for p value, "zscore" for
enrichment z-score}

\item{FDR.cutoff}{FDR cutoff used to declare the significant terms. By
default, it is set to 0.05}

\item{bar.label}{logical to indicate whether to label each bar with
FDR. By default, it sets to true for bar labelling}

\item{bar.label.size}{an integer specifying the bar labelling text
size. By default, it sets to 3}

\item{wrap.width}{a positive integer specifying wrap width of name}

\item{sharings}{a numeric vector specifying whether only shared terms
will be displayed. For example, when comparing three groups of
enrichment results, it can be set into c(2,3) to display only shared
terms by any two or all three. By default, it is NULL meaning no such
restriction}

\item{signature}{logical to indicate whether the signature is assigned
to the plot caption. By default, it sets TRUE showing which function is
used to draw this graph}
}
\value{
an object of class "ggplot", but appended with a 'g' (an igraph object
to represent DAG after being unionised)
}
\description{
\code{xEnrichCompare} is supposed to compare enrichment results using
side-by-side barplots. It returns an object of class "ggplot".
}
\note{
none
}
\examples{
\dontrun{
# Load the XGR package and specify the location of built-in data
library(XGR)
RData.location <- "http://galahad.well.ox.ac.uk/bigdata_dev/"

# 1) load eQTL mapping results: cis-eQTLs significantly induced by IFN
cis <- xRDataLoader(RData.customised='JKscience_TS2A',
RData.location=RData.location)
ind <- which(cis$IFN_t > 0 & cis$IFN_fdr < 0.05)
df_cis <- cis[ind, c('variant','Symbol','IFN_t','IFN_fdr')]
data <- df_cis$variant

# 2) Enrichment analysis using Experimental Factor Ontology (EFO)
# 2a) Without considering LD SNPs and without respecting ontology tree
eTerm_noLD_noTree <- xEnricherSNPs(data, ontology="EF_disease",
include.LD=NA, ontology.algorithm="none",
RData.location=RData.location)
# 2b) Without considering LD SNPs but respecting ontology tree
eTerm_noLD_Tree <- xEnricherSNPs(data, ontology="EF_disease",
include.LD=NA, ontology.algorithm="lea", RData.location=RData.location)
# 2c) Considering LD SNPs but without respecting ontology tree
eTerm_LD_noTree <- xEnricherSNPs(data, ontology="EF_disease",
include.LD="EUR", LD.r2=0.8, ontology.algorithm="none",
RData.location=RData.location)
# 2d) Considering LD SNPs and respecting ontology tree
eTerm_LD_Tree <- xEnricherSNPs(data, ontology="EF_disease",
include.LD="EUR", LD.r2=0.8, ontology.algorithm="lea",
RData.location=RData.location)

# 3) Compare enrichment results
list_eTerm <- list(eTerm_noLD_noTree, eTerm_noLD_Tree, eTerm_LD_noTree,
eTerm_LD_Tree)
names(list_eTerm) <- c('LD(-) & Tree(-)','LD(-) & Tree(+)','LD(+) &
Tree(-)','LD(+) & Tree(+)')
## side-by-side comparisons 
bp <- xEnrichCompare(list_eTerm, displayBy="fc")
#pdf(file="enrichment_compared.pdf", height=6, width=12, compress=TRUE)
print(bp)
#dev.off()
## modify y axis text
bp + theme(axis.text.y=element_text(size=10,color="black"))
## modify x axis title
bp + theme(axis.title.x=element_text(color="black"))
## modify fill colors
bp + scale_fill_manual(values=c("black","#888888"))
## show legend title but hide strip
bp + theme(legend.position="right", strip.text=element_blank())

# 4) DAGplot of comparative enrichment results in the context of ontology tree
xEnrichDAGplotAdv(bp, graph.node.attrs=list(fontsize=100))
}
}
\seealso{
\code{\link{xEnricherGenes}}, \code{\link{xEnricherSNPs}},
\code{\link{xEnrichDAGplotAdv}}
}
