% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/activity_code.r
\name{fitact}
\alias{fitact}
\title{Fit activity model to time-of-day data}
\usage{
fitact(dat, wt = NULL, reps = 999, bw = NULL, adj = 1,
  sample = c("none", "data", "model"), bounds = NULL, show = TRUE)
}
\arguments{
\item{dat}{A numeric vector of radian time-of-day data.}

\item{wt}{A numeric vector of weights for each \code{dat} value.}

\item{reps}{Number of boostrap iterations to perform. Ignored if sample=="none".}

\item{bw}{Numeric value for kernel bandwidth. If NULL, calculated internally.}

\item{adj}{Numeric bandwidth adjustment multiplier.}

\item{sample}{Character string defining sampling method for bootstrapping errors (see details).}

\item{bounds}{A two-element vector defining radian bounds at which to truncate.}

\item{show}{Logical whether or not to show a progress bar while bootstrapping.}
}
\value{
An object of type \code{actmod}
}
\description{
Fits kernel density to radian time-of-day data and estimates activity level from this distribution.
Optionally: 1. bootstraps the distribution, in which case SEs and confidence limits are also
stored for activity level and PDF; 2. weights the distribution; 3. truncates the distribution at given times.
}
\details{
When no \code{bounds} are given (default), a circular kernel distribution is fitted using \code{dvmkern}.
Otherwise, a normal kernel distribution is used, truncated at the values of \code{bounds}, using \code{density2}.

The bandwidth adjustment multiplier \code{adj} is provided to allow
exploration of the effect of adjusting the internally calculated bandwidth on
accuracy of activity level estimates.

The alternative bootstrapping methods defined by \code{sample} are:
 \code{none}: no bootstrapping.
 \code{data}: sample from the data;
 \code{model}: sample from the fitted probability density distribution;
Confidence interval coverage seems to be better at large sample size
(greater than around 100) using \code{"model"}, but better at small sample size
when using \code{"data"}.
}
\examples{
#Fit without confidence limits
data(BCItime)
tm <- 2*pi*subset(BCItime, species=="brocket")$time
mod1 <- fitact(tm)
plot(mod1)

#Fit with confidence limits (limited reps to speed up)
mod2 <- fitact(tm, sample="data", reps=10)
plot(mod2)

#Fit weighted function to correct for detection radius 1.2 times higher
#by day than by night, assuming day between pi/2 (6 am) and pi*2/3 (6 pm)
weight <- 1/ifelse(tm>pi/2 & tm<pi*3/2, 1.2, 1)
mod3 <- fitact(tm, wt=weight)
plot(mod3)
#Overplot unweighted version for comparison
plot(mod1, add=TRUE, tline=list(col=2))

#Fit truncated function to consider only night time records,
#assuming night between pi*3/2 (6 pm) and pi/3 (6 am)
mod4 <- fitact(tm, bounds=c(pi*3/2, pi/2))
plot(mod4, centre="night")
}
