#' Prepares input for \code{apollo_estimate}
#'
#' Searches the user work space (.GlobalEnv) for all necessary input to run \code{apollo_estimate}, and packs it in a single list.
#'
#' All arguments to this function are optional. If the function is called without arguments, then it it will look in
#' the user workspace (i.e. the global environment) for variables with the same name as its ommited arguments.
#' We strongly recommend users to visit www.ApolloChoiceModelling.com for examples on how to use Apollo.
#' In the website, users will also find a detailed manual and a user-group for help and further reference.
#' @param apollo_beta Named numeric vector. Names and values for parameters.
#' @param apollo_fixed Character vector. Names (as defined in \code{apollo_beta}) of parameters whose value should not change during estimation.
#' @param database data.frame. Data used by model.
#' @param apollo_control List. Options controlling the running of the code.
#'                       \itemize{
#'                         \item \code{modelName}: Character. Name of the model. Used when saving the output to files. Avoid characters not allowed in file names, such as \code{\\}, \code{*}, \code{:}, etc.
#'                         \item \code{modelDescr}: Character. Description of the model. Used in output files.
#'                         \item \code{indivID}: Character. Name of column in the database with each decision maker's ID.
#'                         \item \code{mixing}: Boolean. TRUE for models that include random parameters.
#'                         \item \code{nCores}: Numeric>0. Number of threads (processors) to use in estimation of the model.
#'                         \item \code{workInLogs}: Boolean. TRUE for higher numeric stability at the expense of computational time.
#'                                                  Useful for panel models only. Default is FALSE.
#'                         \item \code{seed}: Numeric. Seed for random number generation.
#'                         \item \code{HB}: Boolean. TRUE if using RSGHB for Bayesian estimation of model.
#'                         \item \code{noValidation}: Boolean. TRUE if user does not wish model input to be validated before estimation - FALSE by default.
#'                         \item \code{noDiagnostics}: Boolean. TRUE if user does not wish model diagnostics to be printed - FALSE by default.
#'                         \item \code{panelData}: Boolean. TRUE if using panelData data (created automatically by \code{apollo_validateControl}).
#'                         \item \code{weights}: Character. Name of column in database containing weights for estimation.
#'                       }
#' @param apollo_HB List. Contains options for bayesian estimation. See \code{?RSGHB::doHB} for details.
#'                   Parameters \code{modelname}, \code{gVarNamesFixed}, \code{gVarNamesNormal},
#'                   \code{gDIST}, \code{svN} and \code{FC} are automatically set based on the
#'                   other arguments of this function.
#'                   It should also include a named character vector called \code{hbDist} identifying 
#'                   the distribution of each parameter to be estimated. Possible values are as follows.
#'                   \itemize{
#'                     \item \code{"DNE"}: Parameter kept at its starting value (not estimated).
#'                     \item \code{"F"}: Fixed (as in non-random) parameter.
#'                     \item \code{"N"}: Normal.
#'                     \item \code{"LN+"}: Positive log-normal.
#'                     \item \code{"LN-"}: Negative log-normal.
#'                     \item \code{"CN+"}: Positive censored normal.
#'                     \item \code{"CN-"}: Negative censored normal.
#'                     \item \code{"JSB"}: Johnson SB.
#'                   }
#' @param apollo_draws List of arguments describing the inter and intra individual draws. Required only if \code{apollo_control$mixing = TRUE}. Unused elements can be ommited.
#'                  \itemize{
#'                    \item \code{interDrawsType}: Character. Type of inter-individual draws ('halton','mlhs','pmc','sobol','sobolOwen',
#'                                                 'sobolFaureTezuka', 'sobolOwenFaureTezuka' or the name of an object loaded in memory,
#'                                                 see manual in www.ApolloChoiceModelling.com for details).
#'                    \item \code{interNDraws}: Numeric scalar (>=0). Number of inter-individual draws per individual. Should be set to 0 if not using them.
#'                    \item \code{interUnifDraws}: Character vector. Names of uniform-distributed inter-individual draws.
#'                    \item \code{interNormDraws}: Character vector. Names of normaly distributed inter-individual draws.
#'                    \item \code{intraDrawsType}: Character. Type of intra-individual draws ('halton','mlhs','pmc','sobol','sobolOwen','sobolFaureTezuka', 'sobolOwenFaureTezuka' or the name of an object loaded in memory).
#'                    \item \code{intraNDraws}: Numeric scalar (>=0). Number of intra-individual draws per individual. Should be set to 0 if not using them.
#'                    \item \code{intraUnifDraws}: Character vector. Names of uniform-distributed intra-individual draws.
#'                    \item \code{intraNormDraws}: Character vector. Names of normaly distributed intra-individual draws.
#'                  }
#' @param apollo_randCoeff Function. Used with mixing models. Constructs the random parameters of a mixing model. Receives two arguments:
#'                      \itemize{
#'                        \item \code{apollo_beta}: Named numeric vector. Names and values of model parameters. 
#'                        \item \code{apollo_inputs}: The output of this function (\code{apollo_validateInputs}).
#'                      }
#' @param apollo_lcPars Function. Used with latent class models. Constructs a list of parameters for each latent class. Receives two arguments:
#'                      \itemize{
#'                        \item \code{apollo_beta}: Named numeric vector. Names and values of model parameters. 
#'                        \item \code{apollo_inputs}: The output of this function (\code{apollo_validateInputs}).
#'                      }
#' @param silent Boolean. TRUE to keep the function from printing to the console. Default is FALSE.
#' @return List grouping several required input for model estimation.
#' @export
apollo_validateInputs <- function(apollo_beta=NA, apollo_fixed=NA, database=NA,
                                  apollo_control=NA, 
                                  apollo_HB=NA, apollo_draws=NA,
                                  apollo_randCoeff=NA, apollo_lcPars=NA,
                                  silent=FALSE){
  
  ### Try to recover mandatory variables from global environment if not provided
  tmp <- c("database", paste0("apollo_", c("beta", "fixed", "control"))) 
  for(i in tmp){
    x <- get(i, envir=environment(), inherits=FALSE)
    if(length(x)==1 && is.na(x)) x <- tryCatch( get(i, envir=globalenv()), error=function(e) NA )
    if(length(x)==1 && is.na(x)) stop("No variable called ", i, " found in user workspace (i.e. global environment).") else assign(i, x, envir=environment())
  }; rm(tmp, x, i)
  
  ### Validate apollo_beta & apollo_fixed
  if(!is.numeric(apollo_beta) | !is.vector(apollo_beta) | is.null(names(apollo_beta))) stop("The \"apollo_beta\" argument needs to be a named vector")
  if(length(apollo_fixed)>0 && !is.character(apollo_fixed)) stop("'apollo_fixed' is not an empty vector nor a vector of names.")
  if(length(unique(names(apollo_beta)))<length(apollo_beta)) stop("The \"apollo_beta\" argument contains duplicate elements")
  if(length(unique(apollo_fixed))<length(apollo_fixed)) stop("The \"apollo_fixed\" argument contains duplicate elements")
  if(!all(apollo_fixed %in% names(apollo_beta))) stop("Some parameters included in 'apollo_fixed' are not included in 'apollo_beta'.")
  
  ### Validate apollo_control, database
  apollo_control <- apollo_validateControl(database, apollo_control, silent=silent)
  database       <- apollo_validateData(database, apollo_control, silent=silent)
  
  ### Check that workInLogs is only used with panelData
  if(apollo_control$workInLogs & !apollo_control$panelData){
    apollo_control$workInLogs <- FALSE
    cat("\nworkInLogs set to FALSE. workInLogs can only be used with panel data")
  }
  
  ### Try to recover apollo_HB if appropiate, and sets the default value for the missing parts
  if(!apollo_control$HB) apollo_HB <- NA else{
    if(length(apollo_HB)==1 && is.na(apollo_HB)) apollo_HB <- tryCatch( get("apollo_HB", envir=globalenv()), error=function(e) NA )
    if(length(apollo_HB)==1 && is.na(apollo_HB)) stop("No variable called 'apollo_HB' found in user workspace (i.e. global environment).")
    apollo_HB <- apollo_validateHBControl(apollo_HB, apollo_beta, apollo_fixed, apollo_control)
  }
  if(apollo_control$HB && anyNA(apollo_HB)) stop("Argument 'apollo_HB' must be provided when using Bayesian estimation.")
  
  ### Try to recover apollo_draws and apollo_randCoeff if appropiate, and sets the default value for the missing parts
  if(!apollo_control$mixing){
    if(!is.function(apollo_randCoeff)) apollo_randCoeff <- tryCatch( get("apollo_randCoeff", envir=globalenv()), error=function(e) NA )
    if(is.function(apollo_randCoeff)) cat("\nFunction called 'apollo_randCoeff' found in user workspace will be ignored as model not using mixing.")
    if(length(apollo_draws)==1 && is.na(apollo_draws)) apollo_draws <- tryCatch( get("apollo_draws", envir=globalenv()), error=function(e) NA )
    if(length(apollo_draws)==1 && !is.na(apollo_draws)) cat("\nVariable called 'apollo_draws' found in user workspace will be ignored as model not using mixing.")
    apollo_draws <- NA
    draws <- NA
    apollo_randCoeff <- NA
  } else{
    if(length(apollo_draws)==1 && is.na(apollo_draws)) apollo_draws <- tryCatch( get("apollo_draws", envir=globalenv()), error=function(e) NA )
    if(length(apollo_draws)==1 && is.na(apollo_draws)) stop("Mixing set to TRUE in 'apollo_control', but no variable called 'apollo_draws' found in user workspace (i.e. global environment).")
    default <- list(interDrawsType="halton", interNDraws=0, interUnifDraws=c(), interNormDraws=c(), 
                    intraDrawsType='halton', intraNDraws=0, intraUnifDraws=c(), intraNormDraws=c())
    for(i in names(default)) if(!(i %in% names(apollo_draws))) apollo_draws[[i]] <- default[[i]]
    
    if(!is.function(apollo_randCoeff)) apollo_randCoeff <- tryCatch( get("apollo_randCoeff", envir=globalenv()), error=function(e) NA )
    if(!is.function(apollo_randCoeff)) stop("Mixing set to TRUE in 'apollo_control', but no function called 'apollo_randCoeff' found in user workspace (i.e. global environment).")
  }
  
  ### Try to recover apollo_lcPars if not provided
  if(length(apollo_lcPars)==1 && is.na(apollo_lcPars)) apollo_lcPars <- tryCatch( get("apollo_lcPars", envir=globalenv()), error=function(e) NA )
  
  ### Create apolloLog
  apolloLog <- new.env(parent=emptyenv())
  
  ### Pack everything into a single list
  apollo_inputs <- list(database=database, apollo_control=apollo_control, 
                        apollo_HB=apollo_HB, apollo_draws=apollo_draws, apollo_randCoeff=apollo_randCoeff,
                        apollo_lcPars=apollo_lcPars, draws=NA, apolloLog=apolloLog)
  
  ### Make draws
  if(apollo_control$mixing) apollo_inputs$draws <- apollo_makeDraws(apollo_inputs, silent=silent)
  
  ### Check that if mixing, everything necessary was provided
  if(apollo_control$mixing){
    if(anyNA(apollo_inputs$draws)) stop("Argument 'draws' must be provided when estimating mixture models. Use apollo_makeDraws.")
    if(!is.function(apollo_inputs$apollo_randCoeff)) stop("Argument 'apollo_randCoeff' must be provided when estimating mixture models.")
    if(!apollo_inputs$apollo_control$panel & dim(apollo_inputs$draws[[1]])[2]>1) cat("\nInter-person draws are used without a panel structure. This is unusual.")
  }
  
  ### Pre-calculate exponentials if required
  if(apollo_control$fastExp){
    apollo_inputs$exp_min   = ifelse(!is.null(apollo_control$exp_min), apollo_control$exp_min, -10)
    apollo_inputs$exp_max   = ifelse(!is.null(apollo_control$exp_max), apollo_control$exp_max,  10)
    apollo_inputs$exp_steps = ifelse(!is.null(apollo_control$exp_steps), apollo_control$exp_steps,  10^6)
    apollo_inputs$exp_delta = (apollo_inputs$exp_max - apollo_inputs$exp_min)/(apollo_inputs$exp_steps - 1)
    x=seq(apollo_inputs$exp_min, apollo_inputs$exp_max, length=apollo_inputs$exp_steps)
    apollo_inputs$expVecY =exp(x)
    apollo_inputs$expVecM = (c(apollo_inputs$expVec[-1], exp(x[length(x)] + apollo_inputs$exp_delta))- apollo_inputs$expVec)/apollo_inputs$exp_delta
    apollo_inputs$expVecX = x
  }
  
  return(apollo_inputs)
}
