#' Lists variables names and definitions used in V
#' 
#' Returns a list containing the names and definitions of variables used in V
#' 
#' It looks for variables in apollo_beta, apollo_randCoeff, draws, and 
#' apollo_probabilities. It returns them in a list ordered by origin.
#' 
#' @param apollo_probabilities Likelihood function of the whole model.
#' @param apollo_beta Named numeric vector of parameters to be estimated.
#' @param apollo_inputs List of arguments and settings generated by \link{apollo_validateInputs}.
#' @param V Named list of functions.
#' @param cpp Scalar logical. If TRUE, expressions are modified to match C++ syntax (e.g. x^y -> pow(x,y)). FALSE by default.
#' 
#' @return A list containing the following elements (all of type character):
#'         \itemize{
#'           \item \code{b}: Vector with the names of variables contained in \code{apollo_beta}.
#'           \item \code{x}: Vector with the names of variables contained in \code{database}.
#'           \item \code{d}: Vector with the names of variables contained in \code{draws}.
#'           \item \code{r}: Matrix with the names (first column) and definitions (second column) of variables contained in \code{apollo_randCoeff}.
#'           \item \code{p}: Matrix with the names (first column) and definitions (second column) of variables contained in \code{apollo_probabilities}.
#'           \item \code{v}: Matrix with the names (first column) and definitions (second column) of utilities contained in \code{V}.
#'         }
#' @importFrom utils capture.output
#' @export
apollo_varList <- function(apollo_probabilities, apollo_beta, apollo_inputs, V, cpp=FALSE){
  # Useful function
  is.val <- function(e) if(is.symbol(e) || is.numeric(e) || is.character(e) || is.logical(e) ) return(TRUE) else return(FALSE)
  
  # Check that V only containts functions
  test <- sapply(V, is.function)
  if(!all(test)){
    if(!is.null(names(test))) test <- paste0(" (", (names(test)[test])[1], ")") else test <- ""
    apollo_print(paste0("At least one element", test, " is not defined as a function."))
    return(NULL)
  }
  
  # Check there are no assignments inside the V functions
  for(v in V){
    bF <- body(v)
    if(is.call(bF)) for(i in 1:length(bF)){
      test <- length(bF[[i]])==3 && (bF[[i]][[1]]=="=" || bF[[i]][[1]]=="<-")
      if(test) stop("Utilities should not contain assignments")
    }
  }
  
  # Check there are no assignments inside the randCoeff functions
  if(is.function(apollo_inputs$apollo_randCoeff)){
    rndCoeff <- apollo_inputs$apollo_randCoeff
    environment(rndCoeff) <- list2env(c(as.list(apollo_beta), 
                                        apollo_inputs$database, 
                                        apollo_inputs$draws), hash=TRUE)
    rndCoeff <- rndCoeff(apollo_beta, apollo_inputs)
    test <- which(!sapply(rndCoeff, is.function))
    if(length(test)>0 && !is.null(names(rndCoeff))) test <- paste0('(', paste0(names(rndCoeff)[test], collapse=", "), ')')
    if(length(test)>0){
      apollo_print(paste0("At least one random component inside 'apollo_randCoeff'", 
                          ifelse(is.character(test), test, ""), " is not defined as a function."))
      return(NULL)
    }
    if(!anyNA(rndCoeff)) for(r in rndCoeff){
      bF <- body(r)
      if(!is.val(bF)) for(i in 1:length(bF)){
        test <- length(bF[[i]])==3 && (bF[[i]][[1]]=="=" || bF[[i]][[1]]=="<-")
        if(test) stop("Random coefficients should not contain assignments")
      }
    }
  } else rndCoeff <- NA
  
  # Utility function to transform language object to string
  lang2str <- function(e){
    if(is.numeric(e) && length(e)==1) return(as.character(e))
    if(is.function(e)) e <- body(e)
    if(is.numeric(e)) return(as.character(e))
    e <- paste0(capture.output(print(e)),collapse="")
    e <- gsub("\\s+", " ", e)
    return(e)
  }
  
  # Read names and definitions of V
  v <- matrix("", nrow=length(V), ncol=2)
  for(i in 1:length(V)){
    v[i,1] <- names(V)[i]
    v[i,2] <- lang2str(V[[i]])
  }
  
  # Read random parameters names and definitions
  r <- NULL
  if(is.list(rndCoeff)){
    r <- matrix("", nrow=length(rndCoeff), ncol=2)
    r[,1] <- names(rndCoeff)
    for(i in 1:length(rndCoeff)) r[i,2] <- lang2str(rndCoeff[[i]])
  }
  
  # Read names and definitions of variables inside apollo_probabilities
  extractDef <- function(e){
    if(is.function(e)) e <- body(e)
    # If it's a value
    if(is.symbol(e) || is.numeric(e) || is.character(e) || is.logical(e)) return(NULL)
    # If it's an assignment
    test <- length(e)==3 && (e[[1]]=="=" || e[[1]]=="<-")
    test <- test && length(e[[2]])==1 && is.symbol(e[[2]])
    test2<- test && (is.expression(e[[3]]) || is.call(e[[3]]))
    test <- test && !(test2 && e[[3]][[1]]=="list")
    test <- test && !(test2 && e[[3]][[1]]=="apollo_avgIntraDraws")
    test <- test && !(test2 && e[[3]][[1]]=="apollo_panelProd")
    test <- test && !(test2 && e[[3]][[1]]=="apollo_avgInterDraws")
    test <- test && !(test2 && e[[3]][[1]]=="apollo_prepareProb")
    test <- test && !(test2 && e[[3]][[1]]=="apollo_combineModels")
    if(test) return(c(as.character(e[[2]]), lang2str(e[[3]])))
    # If it's an expression or call but not an assignment
    if(!test && (is.expression(e) || is.call(e))){
      ans <- c()
      for(i in 1:length(e)) if(!is.null(e[[i]])) ans <- c(ans, extractDef(e[[i]]))
      return(ans)
    }
    # In any other case
    return(NULL)
  }
  p <- extractDef(apollo_probabilities)
  if(length(p)>0) p <- matrix(p, ncol=2, byrow=TRUE)
  
  ## Read names and definitions of variables defined in apollo_probabilities
  #bL     <- body(apollo_probabilities)
  #vars.p <- c(); def.p <- c()
  #for(i in 1:length(bL)){
  #  # An assignment has three elements: 
  #  # [1] "=" or "<-", [2] names of new var, [3] value assigned
  #  bLi  <- bL[[i]]
  #  test <- length(bLi)==3 && (bLi[[1]]=="=" || bLi[[1]]=="<-")
  #  test <- test && length(bLi[[2]])==1
  #  test2<- test && (is.expression(bLi[[3]]) || is.call(bLi[[3]]))
  #  test <- test && !(test2 && bLi[[3]][[1]]=="list")
  #  test <- test && !(test2 && bLi[[3]][[1]]=="apollo_avgIntraDraws")
  #  test <- test && !(test2 && bLi[[3]][[1]]=="apollo_panelProd")
  #  test <- test && !(test2 && bLi[[3]][[1]]=="apollo_avgInterDraws")
  #  test <- test && !(test2 && bLi[[3]][[1]]=="apollo_prepareProb")
  #  test <- test && !(test2 && bLi[[3]][[1]]=="apollo_combineModels")
  #  if(test){
  #    vars.p <- c(vars.p, as.character(bLi[[2]]))
  #    def.p  <- c(def.p, lang2str(bLi[[3]]))
  #  }
  #  if(length(vars.p)>0) p <- cbind(vars.p, def.p) else p <- NULL
  #  if(is.vector(p)) p <- matrix(p, nrow=1)
  #}; rm(vars.p, def.p)
  
  # Read explanatory variables, parameters and draws names
  vars.x <- names(apollo_inputs$database)
  vars.b <- names(apollo_beta)
  if(is.list(apollo_inputs$draws)) vars.d <- names(apollo_inputs$draws) else vars.d <- c()
  
  # Remove unused variables
  def <- v[,2]
  if(!is.null(r)) def <- c(def, r[,2])
  if(!is.null(p)) def <- c(def, p[,2])
  f <- function(x){
    if(length(x)==0) return(x)
    if(is.matrix(x)) nam <- x[,1] else nam <- x
    #drop <- c()
    #for(i in 1:length(nam)) if(all(regexpr(nam[i], def, fixed=TRUE)<0)) drop <- c(i, drop)
    drop <- which(sapply(as.list(nam), function(n) all(regexpr(n, text=def, fixed=TRUE)<0)))
    if(length(drop)==0) return(x)
    if(is.matrix(x)) x <- x[-drop,,drop=FALSE] else x <- x[-drop]
    if(length(x)==0) return(NULL)
    return(x)
  }
  r <- f(r)
  p <- f(p)
  vars.x <- f(vars.x)
  vars.b <- f(vars.b)
  vars.d <- f(vars.d)
  
  # timing
  #system.time(f(vars.x)) # sapply 78.9 sec
  #system.time(f(vars.x)) # for    80.2 sec
  
  # If C++ transformations are not necesary, return
  if(!cpp){
    vars <- list(b=vars.b, x=vars.x, d=vars.d, r=r, p=p, v=v)
    return(vars)
  }
  
  # Check there are no boolean comparisons in the definitions
  test <- FALSE
  if(!is.null(v)) if(any(grepl("==|!=|>=|<=|<|>", v[,2]))) test <- TRUE
  if(!is.null(r)) if(any(grepl("==|!=|>=|<=|<|>", r[,2]))) test <- TRUE
  if(!is.null(p)) if(any(grepl("==|!=|>=|<=|<|>", p[,2]))) test <- TRUE
  if(test){
    apollo_print("Logical comparisons (e.g. == or <) cannot be converted to C++.\n Create new variables in the database instead.\n")
    return(NULL)
  }
  
  # Substitute "^" by "pow", if necessary. 
  # The function below receives an expression, function or
  # character as input, and returns a language object (call 
  # or symbol) as output.
  insertPow <- function(e){
    if(is.function(e)) e <- body(e)
    if(is.character(e)) e <- str2lang(e)
    if(is.symbol(e)) return(e)
    if(length(e)>0) if(e[[1]]=="^") e[[1]] <- quote(pow)
    if(length(e)>1) for(i in 2:length(e)) e[[i]] <- insertPow(e[[i]])
    return(e)
  }
  if(!is.null(v)) for(i in 1:nrow(v))  v[i,2] <- lang2str( insertPow(v[i,2]) )
  if(!is.null(r)) for(i in 1:nrow(r))  r[i,2] <- lang2str( insertPow(r[i,2]) )
  if(!is.null(p)) for(i in 1:nrow(p))  p[i,2] <- lang2str( insertPow(p[i,2]) )
  
  # Stitch together
  vars <- list(b=vars.b, x=vars.x, d=vars.d, r=r, p=p, v=v)
  return(vars)
}