% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bama.R
\name{bama}
\alias{bama}
\title{Bayesian Mediation Analysis}
\usage{
bama(
  Y,
  A,
  M,
  C1,
  C2,
  beta.m,
  alpha.a,
  burnin,
  ndraws,
  weights = NULL,
  k = 2,
  lm0 = 1e-04,
  lm1 = 1,
  l = 1
)
}
\arguments{
\item{Y}{Length \code{n} numeric outcome vector}

\item{A}{Length \code{n} numeric exposure vector}

\item{M}{\code{n x p} numeric matrix of mediators of Y and A}

\item{C1}{\code{n x nc1} numeric matrix of extra covariates to include in the
outcome model}

\item{C2}{\code{n x nc2} numeric matrix of extra covariates to include in the
mediator model}

\item{beta.m}{Length \code{p} numeric vector of initial \code{beta.m} in the
outcome model}

\item{alpha.a}{Length \code{p} numeric vector of initial \code{alpha.a} in
the mediator model}

\item{burnin}{number of iterations to run the MCMC before sampling}

\item{ndraws}{number of draws to take from MCMC after the burnin period}

\item{weights}{Length \code{n} numeric vector of weights}

\item{k}{Shape parameter prior for inverse gamma}

\item{lm0}{Scale parameter prior for inverse gamma for the small normal
components}

\item{lm1}{Scale parameter prior for inverse gamma for the large normal
components}

\item{l}{Scale parameter prior for the other inverse gamma distributions}
}
\value{
\code{bama} returns a object of type "bama" with 12 elements:
\describe{
\item{beta.m}{\code{ndraws x p} matrix containing outcome model mediator
coefficients.
}
\item{r1}{\code{ndraws x p} matrix indicating whether or not each beta.m
belongs to the larger normal component (1) or smaller normal
component (0).
}
\item{alpha.a}{\code{ndraws x p} matrix containing the mediator model
exposure coefficients.
}
\item{r3}{\code{ndraws x p} matrix indicating whether or not each alpha.a
belongs to the larger normal component (1) or smaller normal component (0).
}
\item{beta.a}{Vector of length \code{ndraws} containing the beta.a coefficient.}
\item{pi.m}{Vector of length \code{ndraws} containing the proportion of
non zero beta.m coefficients.
}
\item{pi.a}{Vector of length \code{ndraws} containing the proportion of
non zero alpha.a coefficients.
}
\item{sigma.m0}{Vector of length \code{ndraws} containing the standard
deviation of the smaller normal component for mediator-outcome
coefficients (beta.m).
}
\item{sigma.m1}{Vector of length \code{ndraws} containing standard deviation
of the larger normal component for mediator-outcome coefficients (beta.m).
}
\item{sigma.ma0}{Vector of length \code{ndraws} containing standard
deviation of the smaller normal component for exposure-mediator
coefficients (alpha.a).
}
\item{sigma.ma1}{Vector of length \code{ndraws} containing standard deviation
of the larger normal component for exposure-mediator coefficients
(alpha.a).
}
\item{call}{The R call that generated the output.}
}
}
\description{
\code{bama} is a Bayesian inference method that uses continuous shrinkage priors
for high-dimensional Bayesian mediation analysis, developed by Song et al
(2019). \code{bama} provides estimates for the regression coefficients as
well as the posterior inclusion probability for ranking mediators.
}
\details{
\code{bama} uses two regression models for the two conditional relationships,
\eqn{Y | A, M, C} and \eqn{M | A, C}. For the outcome model, \code{bama}
uses
\deqn{Y = M \beta_M  + A * \beta_A + C* \beta_C + \epsilon_Y}
For the mediator model, \code{bama} uses the model
\deqn{M = A * \alpha_A + C * \alpha_C + \epsilon_M}

For high dimensional tractability, \code{bama} employs continuous Bayesian
shrinkage priors to select mediators and makes the two following assumptions:
First, it assumes that all the potential mediators contribute small effects
in mediating the exposure-outcome relationship. Second, it assumes
that only a small proportion of mediators exhibit large effects
("active" mediators). \code{bama} uses a Metropolis-Hastings within Gibbs
MCMC to generate posterior samples from the model.
}
\examples{
library(bama)

Y <- bama.data$y
A <- bama.data$a

# grab the mediators from the example data.frame
M <- as.matrix(bama.data[, paste0("m", 1:100)], nrow(bama.data))

# We just include the intercept term in this example as we have no covariates
C1 <- matrix(1, 1000, 1)
C2 <- matrix(1, 1000, 1)
beta.m  <- rep(0, 100)
alpha.a <- rep(0, 100)

set.seed(12345)
out <- bama(Y, A, M, C1, C2, beta.m, alpha.a, burnin = 1000, ndraws = 100)

# The package includes a function to summarise output from 'bama'
summary <- summary(out)
head(summary)
}
\references{
Song, Y, Zhou, X, Zhang, M, et al. Bayesian shrinkage estimation of high
dimensional causal mediation effects in omics studies. Biometrics. 2019;
1-11. \href{http://doi.org/10.1111/biom.13189}{doi:10.1111/biom.13189}
}
\author{
Alexander Rix
}
