# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# The bigMap Package for R.

# Copyright (c) 2018, Joan Garriga <jgarriga@ceab.csic.es>, Frederic Bartumeus <fbartu@ceab.csic.es> (Blanes Centre for Advanced Studies, CEAB-CSIC).

# bigMap is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.

# bigMap is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.

# You should have received a copy of the GNU General Public License along with this program. If not, see http://www.gnu.org/licenses.
# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

#' ptSNE quantile-maps
#'
#' @param bdm A \var{bdm} instance as generated by \code{bdm.init()}.
#'
#' @param layer The number of a layer (1 by default).
#'
#' @param data A \code{matrix/data.frame} to be mapped. By default, the input data \var{bdm$data} is mapped.
#'
#' @param labels If available, labels can be included as a separate vector of length equal to \code{nrow(dSet.data)}. Label values are factorized as \code{as.numeric(as.factor(labels))}.
#'
#' @param subset A numeric vector with the indexes of a subset of data. Data-points in the subset are heat-mapped and the rest are shown in light grey. By default all data-points are heat-mapped.
#'
#' @param levels The number of levels of the heat-map (16 by default).
#'
#' @param cex The size of the data-points (as in \code{par()}).
#'
#' @return None.
#'
#' @examples
#'
#' bdm.example()
#' bdm.qMap(exMap)
#' # --- show only components (1, 2, 4, 8) of the GMM
#' bdm.qMap(exMap, subset = which(exMap$lbls %in% c(1, 4, 8, 16)))

bdm.qMap <- function(bdm, layer = 1, data = NULL, labels = NULL, subset = NULL, levels = 16, cex = 0.3)
{
	# get labels
	if (!is.null(labels)) labels <- as.numeric(as.factor(labels))
	else labels <- bdm$lbls
	# get data
	if (is.null(data)) data <- as.matrix(bdm$data)

	# get var names
	if (is.null(colnames(data))) {
		colnames(data) <- paste('V', formatC(seq(ncol(data)), width = 2, flag = '0'), sep = '.')
	}

	# join labels & data
	data <- cbind(labels, data)

	# check number of vars
	if (ncol(data) > 16) {
		data <- data[, 1:16]
		cat('+++ WARNING: plotting first ', ncol(data) - !is.null(labels), ' columns !, \n', sep='')
	}

	# get mapping
	l <- c(1, 2) + (layer -1) *2
	Y <- bdm$ptsne$Y[ , l]

	# set graphic environment
	parbdm.set(oma = c(1, 1, 3, 1), mar = c(2, 2, 0.5, 0.5))
	l.mtx <- layout.get(ncol(data)) +1
	l.mtx <- cbind(l.mtx, rep(1, nrow(l.mtx)))
	layout(l.mtx, widths = c(rep(1, ncol(l.mtx) -1), 0.3))

	# plot palette legend
	hmap.pltt <- c(pltt.heat(levels), '#DDDDDDFF')
	plot(1, 1, xlab = '', ylab = '', xlim = c(0, 1), ylim = c(1, levels+1), xaxt = "n", yaxt = "n", bty = "n", type = "n")
	rect(0, seq(1, levels), 1, seq(2, levels+1), col = hmap.pltt, border = NA)

	# plot heatmaps
	nulL <- lapply(seq(ncol(data)), function(j){
		if (j == 1 & !is.null(labels)) {
			x <- data[, j]
			pltt <- c(pltt.get(s = length(unique(x)) +1), '#DDDDDDFF')
		}
		else {
			# factor data
			x <- get.lvls(data[, j], levels)
			pltt <- hmap.pltt
		}
		if (!is.null(subset)) {
			# plot shadow
			plot(Y[-subset, ], xlab = '', ylab = '', xlim = range(Y[,1]), ylim = range(Y[,2]), col = pltt[length(pltt)], cex = cex, pch = 20, asp = 1)
			# plot subset heatmap
			points(Y[subset, ], col = pltt[x[subset]], cex = cex, pch = 20, asp = 1)
		}
		else {
			# plot heat-map
			plot(Y, xlab = '', ylab = '', xlim = range(Y[, 1]), ylim = range(Y[, 2]), col = pltt[x], cex = cex, pch = 20, asp = 1)
		}
		# plot title
		text(0.8 *min(Y[, 1]), 0.9 *max(Y[, 2]), labels = colnames(data)[j], cex = 1.0)
	})

	# fill layout
	if (length(ncol(data)) < max(layout.get(length(ncol(data))))) plot.null()
	# layout title
	title(bdm$dSet, outer = T, cex.main = 1.2)
	# reset graphic environment
	parbdm.def()
}
