% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/boundary_layer_conductance.r
\name{Gb.Su}
\alias{Gb.Su}
\title{Boundary Layer Conductance according to Su et al. 2001}
\usage{
Gb.Su(data, Tair = "Tair", pressure = "pressure", ustar = "ustar",
  wind = "wind", H = "H", zh, zr, d, Dl, fc = NULL, LAI = NULL, N = 2,
  Cd = 0.2, hs = 0.01, stab_formulation = c("Dyer_1970", "Businger_1971"),
  constants = bigleaf.constants())
}
\arguments{
\item{data}{Data.frame or matrix containing all required variables}

\item{Tair}{Air temperature (degC)}

\item{pressure}{Atmospheric pressure (kPa)}

\item{ustar}{Friction velocity (m s-1)}

\item{wind}{Wind speed (m s-1)}

\item{H}{Sensible heat flux (W m-2)}

\item{zh}{Canopy height (m)}

\item{zr}{Reference height (m)}

\item{d}{Zero-plane displacement height (-), can be calculated using \code{roughness.parameters}}

\item{Dl}{Leaf characteristic dimension (m)}

\item{fc}{Fractional vegetation cover [0-1] (if not provided, calculated from LAI)}

\item{LAI}{One-sided leaf area index (-)}

\item{N}{Number of leaf sides participating in heat exchange (defaults to 2)}

\item{Cd}{Foliage drag coefficient (-)}

\item{hs}{Roughness height of the soil (m)}

\item{stab_formulation}{Stability correction function used (If \code{stab_correction = TRUE}).
Either \code{"Dyer_1970"} or \code{"Businger_1971"}.}

\item{constants}{Kelvin - conversion degree Celsius to Kelvin \cr
pressure0 - reference atmospheric pressure at sea level (Pa) \cr
Tair0 - reference air temperature (K) \cr
Rbwc - Ratio of the transfer efficiency through the boundary layer for water vapor and CO2 (-)}
}
\value{
A data.frame with the following columns:
    \item{Rb}{Boundary layer resistance for heat and water (s m-1)}
    \item{Rb_CO2}{Boundary layer resistance for CO2 (s m-1)}
    \item{Gb}{Boundary layer conductance (m s-1)}
    \item{kB}{kB-1 parameter (-)}
}
\description{
A physically based formulation for the canopy boundary layer conductance
             to heat/water vapor according to Su et al. 2001.
}
\details{
The formulation is based on the kB-1 model developed by Massman 1999. 
         Su et al. 2001 derived the following approximation:
          
           \deqn{kB-1 = (k Cd fc^2) / (4Ct ustar/u(zh)) + kBs-1(1 - fc)^2}
         
         If fc (fractional vegetation cover) is missing, it is estimated from LAI:

           \deqn{fc = 1 - exp(-LAI/2)}
         
         The wind speed at the top of the canopy is calculated using function
         \code{\link{wind.profile}}.
         
         Ct is the heat transfer coefficient of the leaf (Massman 1999):
         
           \deqn{Ct = Pr^-2/3 Reh^-1/2 N}
         
         where Pr is the Prandtl number (set to 0.71), and Reh is the Reynolds number for leaves:
         
           \deqn{Dl wind(zh) / v}
          
         kBs-1, the kB-1 value for bare soil surface, is calculated according 
         to Su et al. 2001:
         
           \deqn{kBs^-1 = 2.46(Re)^0.25 - ln(7.4)}
         
         Rb for water vapor and heat is assumed to be equal. Rb for CO2 (Rb_CO2) is given as:
 
           \deqn{Rb_CO2 = 1.37 * Rb}
 
         The factor 1.37 arises due the lower molecular diffusivity of CO2 compared to water.
         It is lower than the ratio of the molecular diffusivities (Dw/DCO2 = 1.6), as movement
         across the boundary layer is assumed to be partly by diffusion and partly by turbulent
         mixing (Nobel 2005).
}
\examples{
# Canopy boundary layer resistance (and kB-1 parameter) for a set of meteorological conditions,
# a leaf characteristic dimension of 1cm, and an LAI of 5
df <- data.frame(Tair=25,pressure=100,wind=c(3,4,5),ustar=c(0.5,0.6,0.65),H=c(200,230,250)) 
Gb.Su(data=df,zh=25,zr=40,d=17.5,Dl=0.01,LAI=5)

# the same meteorological conditions, but larger leaves
Gb.Su(data=df,zh=25,zr=40,d=17.5,Dl=0.1,LAI=5)

# same conditions, large leaves, and sparse canopy cover (LAI = 1.5)
Gb.Su(data=df,zh=25,zr=40,d=17.5,Dl=0.1,LAI=1.5)

}
\references{
Su, Z., Schmugge, T., Kustas, W. & Massman, W., 2001: An evaluation of
            two models for estimation of the roughness height for heat transfer between
            the land surface and the atmosphere. Journal of Applied Meteorology 40, 1933-1951.

            Massman, W., 1999: A model study of kB H- 1 for vegetated surfaces using
           'localized near-field' Lagrangian theory. Journal of Hydrology 223, 27-43.
           
            Nobel, P. S., 2005: Physicochemical and Environmental Plant Physiology. Third 
            Edition. Elsevier Academic Press, Burlington, USA.
}
\seealso{
\code{\link{Gb.Thom}}, \code{\link{Gb.Choudhury}}, \code{\link{aerodynamic.conductance}}
}
