% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/surface_roughness.r
\name{roughness.parameters}
\alias{roughness.parameters}
\title{Roughness Parameters}
\usage{
roughness.parameters(
  method = c("canopy_height", "canopy_height&LAI", "wind_profile"),
  zh,
  frac_d = 0.7,
  frac_z0m = 0.1,
  LAI,
  zr,
  cd = 0.2,
  hs = 0.01,
  data,
  Tair = "Tair",
  pressure = "pressure",
  wind = "wind",
  ustar = "ustar",
  H = "H",
  d = NULL,
  z0m = NULL,
  stab_roughness = TRUE,
  stab_formulation = c("Dyer_1970", "Businger_1971"),
  constants = bigleaf.constants()
)
}
\arguments{
\item{method}{Method to use, one of \code{"canopy_height","canopy_height&LAI","wind_profile"} \cr
NOTE: if \code{method = "canopy_height"}, only the following three arguments
are used. If \code{method = "canopy_height&LAI"}, only \code{zh, LAI, cd}, 
and \code{hs} are required.}

\item{zh}{Vegetation height (m)}

\item{frac_d}{Fraction of displacement height on canopy height (-)}

\item{frac_z0m}{Fraction of roughness length on canopy height (-)}

\item{LAI}{Leaf area index (-)}

\item{zr}{Instrument (reference) height (m)}

\item{cd}{Mean drag coefficient for individual leaves. Defaults to 0.2. 
Only needed if \code{method = "canopy_height&LAI"}.}

\item{hs}{roughness length of the soil surface (m). Only needed if \code{method = "canopy_height&LAI"}
The following arguments are only needed if \code{method = "wind_profile"}!}

\item{data}{Data.frame or matrix containing all required variables}

\item{Tair}{Air temperature (deg C)}

\item{pressure}{Atmospheric pressure (kPa)}

\item{wind}{Wind speed at height zr (m s-1)}

\item{ustar}{Friction velocity (m s-1)}

\item{H}{Sensible heat flux (W m-2)}

\item{d}{Zero-plane displacement height (m); optional}

\item{z0m}{Roughness length for momentum (m); optional}

\item{stab_roughness}{Should stability correction be considered? Default is \code{TRUE}.}

\item{stab_formulation}{Stability correction function used (If \code{stab_correction = TRUE}).
Either \code{"Dyer_1970"} or \code{"Businger_1971"}.}

\item{constants}{k - von-Karman constant (-) \cr
Kelvin - conversion degree Celsius to Kelvin \cr
cp - specific heat of air for constant pressure (J K-1 kg-1) \cr
g - gravitational acceleration (m s-2) \cr
se_median - conversion standard error (SE) of the mean to SE of the median}
}
\value{
a data.frame with the following columns:
        \item{d}{Zero-plane displacement height (m)}
        \item{z0m}{Roughness length for momentum (m)}
        \item{z0m_se}{Only if \code{method = wind_profile}: Standard Error of the median for z0m (m)}
}
\description{
A simple approximation of the two roughness parameters displacement height (d)
             and roughness length for momentum (z0m).
}
\details{
The two main roughness parameters, the displacement height (d)
         and the roughness length for momentum (z0m) can be estimated from simple
         empirical relationships with canopy height (zh). If \code{method = "canopy_height"},
         the following formulas are used:  
         
           \deqn{d = frac_d * zh}
         
           \deqn{z0m = frac_z0m * zh}
         
         where frac_d defaults to 0.7 and frac_z0m to 0.1.
         
         Alternatively, d and z0m can be estimated from both canopy height and LAI
         (If \code{method = "canopy_height&LAI"}).
         Based on data from Shaw & Pereira 1982, Choudhury & Monteith 1988 proposed 
         the following semi-empirical relations:
         
           \deqn{X = cd * LAI}
         
           \deqn{d = 1.1 * zh * ln(1 + X^(1/4))} 
         
           \deqn{z0m = hs + 0.3 * zh * X^(1/2)   for 0 <= X <= 0.2}
         
           \deqn{z0m = hs * zh * (1 - d/zh)   for 0.2 < X} 
         
         If \code{method = "wind_profile"}, z0m is estimated by solving
         the wind speed profile for z0m:
         
           \deqn{z0m = median((zr - d) * exp(-k*wind / ustar - psi_m)}
                 
         By default, d in this equation is fixed to 0.7*zh, but can be set to any
         other value. psi_m is 0 if \code{stab_roughness = FALSE}.
}
\examples{
# estimate d and z0m from canopy height for a dense (LAI=5) and open (LAI=2) canopy
roughness.parameters(method="canopy_height&LAI",zh=25,LAI=5)
roughness.parameters(method="canopy_height&LAI",zh=25,LAI=2)   
   
# fix d to 0.7*zh and estimate z0m from the wind profile
df <- data.frame(Tair=c(25,25,25),pressure=100,wind=c(3,4,5),ustar=c(0.5,0.6,0.65),H=200)
roughness.parameters(method="wind_profile",zh=25,zr=40,frac_d=0.7,data=df)

# assume d = 0.8*zh
roughness.parameters(method="wind_profile",zh=25,zr=40,frac_d=0.8,data=df) 

}
\references{
Choudhury, B. J., Monteith J.L., 1988: A four-layer model for the heat
            budget of homogeneous land surfaces. Q. J. R. Meteorol. Soc. 114, 373-398.
            
            Shaw, R. H., Pereira, A., 1982: Aerodynamic roughness of a plant canopy: 
            a numerical experiment. Agricultural Meteorology, 26, 51-65.
}
\seealso{
\code{\link{wind.profile}}
}
