% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/brmsfit-methods.R, R/generics.R,
%   R/misc-methods.R
\name{hypothesis.brmsfit}
\alias{hypothesis.brmsfit}
\alias{hypothesis}
\alias{hypothesis.default}
\title{Non-Linear Hypothesis Testing}
\usage{
\method{hypothesis}{brmsfit}(x, hypothesis, class = "b", group = "",
  alpha = 0.05, seed = 1234, ...)

hypothesis(x, ...)

\method{hypothesis}{default}(x, hypothesis, alpha = 0.05, ...)
}
\arguments{
\item{x}{An \code{R} object. If it is no \code{brmsfit} object,
it must be coercible to a \code{data.frame}.}

\item{hypothesis}{A character vector specifying one or more 
non-linear hypothesis concerning parameters of the model.}

\item{class}{A string specifying the class of parameters being tested. 
Default is "b" for population-level effects. 
Other typical options are "sd" or "cor". 
If \code{class = NULL}, all parameters can be tested
against each other, but have to be specified with their full name 
(see also \code{\link[brms:parnames]{parnames}})}

\item{group}{Name of a grouping factor to evaluate only 
group-level effects parameters related to this grouping factor.
Ignored if \code{class} is not \code{"sd"} or \code{"cor"}.}

\item{alpha}{The alpha-level of the tests (default is 0.05;
see 'Details' for more information).}

\item{seed}{A single numeric value passed to \code{set.seed} 
to make results reproducible.}

\item{...}{Currently ignored.}
}
\value{
A \code{\link[brms:brmshypothesis]{brmshypothesis}} object.
}
\description{
Perform non-linear hypothesis testing for all model parameters.
}
\details{
Among others, \code{hypothesis} computes an 
 evidence ratio (\code{Evid.Ratio}) for each hypothesis. 
 For a directed hypothesis, this is just the posterior probability 
 under the hypothesis against its alternative.
 That is, when the hypothesis if of the form \code{a > b}, 
 the evidence ratio is the ratio of the posterior probability 
 of \code{a > b} and the posterior probability of \code{a < b}.
 In this example, values greater than one indicate that the evidence in
 favour of \code{a > b} is larger than evidence in favour of \code{a < b}.
 For an undirected (point) hypothesis, the evidence ratio 
 is a Bayes factor between the hypothesis and its alternative
 computed via the Savage-Dickey density ratio method.
 That is the posterior density at the point of interest divided
 by the prior density at that point.
 Values greater than one indicate that evidence in favour of the point
 hypothesis has increased after seeing the data.
 In order to calculate this Bayes factor, all parameters related 
 to the hypothesis must have proper priors
 and argument \code{sample_prior} of function \code{brm} 
 must be set to \code{TRUE}. 
 When interpreting Bayes factors, make sure 
 that your priors are reasonable and carefully chosen,
 as the result will depend heavily on the priors. 
 In particular, avoid using default priors.
 
 The argument \code{alpha} specifies the size of the credible interval
 (i.e., Bayesian confidence interval).
 For instance, if \code{alpha = 0.05} (5\%), the credible interval
 will contain \code{1 - alpha = 0.95} (95\%) of the posterior values.
 Hence, \code{alpha * 100}\% of the posterior values will lie
 outside of the credible interval. Although this allows testing of
 hypotheses in a similar manner as in the frequentist null-hypothesis
 testing framework, we strongly argue against using arbitrary cutoffs 
 (e.g., \code{p < .05}) to determine the 'existence' of an effect.
}
\examples{
\dontrun{
## define priors
prior <- c(set_prior("normal(0,2)", class = "b"),
           set_prior("student_t(10,0,1)", class = "sigma"),
           set_prior("student_t(10,0,1)", class = "sd"))

## fit a linear mixed effects models
fit <- brm(time ~ age + sex + disease + (1 + age|patient),
           data = kidney, family = lognormal(),
           prior = prior, sample_prior = TRUE, 
           control = list(adapt_delta = 0.95))

## perform two-sided hypothesis testing
(hyp1 <- hypothesis(fit, "sexfemale = age + diseasePKD"))
plot(hyp1)
hypothesis(fit, "exp(age) - 3 = 0", alpha = 0.01)

## perform one-sided hypothesis testing
hypothesis(fit, "diseasePKD + diseaseGN - 3 < 0")

hypothesis(fit, "age < Intercept", 
           class = "sd", group  = "patient")

## test the amount of random intercept variance on all variance
h <- paste("sd_patient_Intercept^2 / (sd_patient_Intercept^2 +",
           "sd_patient_age^2 + sigma^2) = 0")
(hyp2 <- hypothesis(fit, h, class = NULL))
plot(hyp2)

## test more than one hypothesis at once
(hyp3 <- hypothesis(fit, c("diseaseGN = diseaseAN", 
                           "2 * diseaseGN - diseasePKD = 0")))
plot(hyp3, ignore_prior = TRUE)

## use the default method
dat <- as.data.frame(fit)
hypothesis(dat, "b_age > 0")
}

}
\seealso{
\code{\link[brms:brmshypothesis]{brmshypothesis}}
}
\author{
Paul-Christian Buerkner \email{paul.buerkner@gmail.com}
}
