#' Generate the sample select panel of the shiny app
#' @description These are the UI and server components of the sample selection
#' panel of the shiny app. It is generated by including 'SampleSelect' in the 
#' panels.default argument of \code{\link{generateShinyApp}}.
#' @inheritParams DEpanel
#' @return The UI and Server components of the shiny module, that can be used
#' within the UI and Server definitions of a shiny app.
#' @name sampleSelectPanel
NULL

#' @rdname sampleSelectPanel
#' @export
sampleSelectPanelUI <- function(id){
  ns <- NS(id)
  
  tabPanel(
    'Sample select',
    actionButton(ns('goSamples'), label = 'Use the selected samples!', 
                 width = "100%", class = "btn-primary btn-lg"),
    DT::dataTableOutput(ns('tbl'))
  )
}

#' @rdname sampleSelectPanel
#' @export
sampleSelectPanelServer <- function(id, expression.matrix, metadata){
  ns <- NS(id)
  # check whether inputs (other than id) are reactive or not
  stopifnot({
    !is.reactive(expression.matrix)
    !is.reactive(metadata)
  })
  
  moduleServer(id, function(input, output, session){
    # create a character vector of shiny inputs
    shinyInput = function(FUN, len, id, value, ...) {
      if (length(value) == 1) value <- rep(value, len)
      inputs = character(len)
      for (i in seq_len(len)) {
        inputs[i] = as.character(FUN(ns(paste0(id, i)), label = NULL, value = value[i]))
      }
      inputs
    }
    
    # obtain the values of inputs
    shinyValue = function(id, len) {
      unlist(lapply(seq_len(len), function(i) {
        value = input[[paste0(id, i)]]
        if (is.null(value)) TRUE else value
      }))
    }
    
    n <- nrow(metadata)
    df = cbind(
      data.frame(selected = shinyInput(checkboxInput, n, 'cb_', value = TRUE, width='1px')),
      metadata
    )
    
    loopData = reactive({
      df$selected <<- shinyInput(checkboxInput, n, 'cb_', value = shinyValue('cb_', n), width='1px')
      df
    })
    
    tbl <- DT::renderDataTable(
      isolate(loopData()),
      escape = FALSE, 
      selection = 'none',
      options = list(
        dom = 't', paging = FALSE, ordering = FALSE,
        preDrawCallback = DT::JS('function() { Shiny.unbindAll(this.api().table().node()); }'),
        drawCallback = DT::JS('function() { Shiny.bindAll(this.api().table().node()); } ')
      ))
    
    output[["tbl"]] = tbl
    
    proxy = DT::dataTableProxy('tbl')
    
    observe({
      DT::replaceData(proxy, loopData(), resetPaging = FALSE)
    })
    
    filteredInputs <- reactive({
      list("expression.matrix" = expression.matrix[, shinyValue('cb_', n)],
           "metadata" = metadata[shinyValue('cb_', n), ])
    }) %>%
      bindEvent(input[["goSamples"]], ignoreNULL = FALSE)
    
  })
}

# sampleSelectPanelApp <- function(){
#   shinyApp(
#     ui = fluidPage(sampleSelectPanelUI('SampleSelect')),
#     server = function(input, output, session){
#       sampleSelectPanelServer('SampleSelect', expression.matrix, metadata)
#     }
#   )
# }