% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/postestimate_assess.R
\name{assess}
\alias{assess}
\title{Assess model}
\usage{
assess(
  .object              = NULL, 
  .only_common_factors = TRUE, 
  .quality_criterion   = c("all", "ave", "rho_C", "rho_C_mm", "rho_C_weighted", 
                           "rho_C_weighted_mm", "cronbachs_alpha", 
                          "cronbachs_alpha_weighted", "dg", "dl", "dml", "df",
                          "esize", "cfi", "gfi", "ifi", "nfi", "nnfi", 
                          "reliability", 
                          "rmsea", "rms_theta", "srmr",
                          "gof", "htmt", "r2", "r2_adj",
                          "rho_T", "rho_T_weighted", "vif", 
                          "vifmodeB",  "fl_criterion"),
  ...
)
}
\arguments{
\item{.object}{An R object of class \link{cSEMResults} resulting from a call to \code{\link[=csem]{csem()}}.}

\item{.only_common_factors}{Logical. Should only concepts modeled as common
factors be included when calculating one of the following quality critera:
AVE, the Fornell-Larcker criterion, HTMT, and all reliability estimates.
Defaults to \code{TRUE}.}

\item{.quality_criterion}{Character string. A single character string or a
vector of character strings naming the quality criterion to compute. See
the Details section for a list of possible candidates.
Defaults to "\emph{all}" in which case all possible quality criteria are computed.}

\item{...}{Further arguments passed to functions called by \code{\link[=assess]{assess()}}.
See \link{args_assess_dotdotdot} for a complete list of available arguments.}
}
\value{
A named list of quality criteria. Note that if only a single quality
criteria is computed the return value is still a list!
}
\description{
Assess a model using common quality criteria.
See the \href{https://m-e-rademaker.github.io/cSEM/articles/Using-assess.html}{Postestimation: Assessing a model}
article on the
\href{https://m-e-rademaker.github.io/cSEM/index.html}{cSEM website} for details.
}
\details{
The function is essentially a wrapper around a number of internal functions
that perform an "assessment task" (called a \strong{quality criterion} in \pkg{cSEM}
parlance) like computing reliability estimates,
the effect size, the heterotrait-monotrait ratio of correlations (HTMT) etc.

By default every possible quality criterion is calculated (\code{.quality_criterion = "all"}).
If only a subset of quality criteria are needed a single character string
or a vector of character strings naming the criteria to be computed may be
supplied to \code{\link[=assess]{assess()}} via the \code{.quality_criterion} argument. Currently, the
following quality criteria are implemented (in alphabetical order):

\describe{
\item{Average variance extracted (AVE); "ave"}{An estimate of the
amount of variation in the indicators that is due to the underlying latent variable.
Practically, it is calculated as the ratio of the (indicator) true score variances
(i.e., the sum of the squared loadings)
relative to the sum of the total indicator variances. Calculation is done
by \code{\link[=calculateAVE]{calculateAVE()}}.}
\item{Congeneric reliability; "rho_C", "rho_C_mm,", "rho_C_weighted", "rho_C_weighted_mm"}{
An estimate of the reliability assuming a congeneric measurement model (i.e., loadings are
allowed to differ) and a test score (proxy) based on unit weights.
There are four different versions implemented. See the
\href{https://m-e-rademaker.github.io/cSEM/articles/Using-assess.html#methods}{Methods and Formulae} section
of the \href{https://m-e-rademaker.github.io/cSEM/articles/Using-assess.html}{Postestimation: Assessing a model}
article on the on the
\href{https://m-e-rademaker.github.io/cSEM/index.html}{cSEM website} for details.
Alternative but synonemmous names for \code{"rho_C_mm"} are:
composite reliability, construct reliablity, reliability coefficient,
Joereskog's rho, coefficient omega, or Dillon-Goldstein's rho.
For \code{"rho_C_weighted_mm"}: (Dijkstra-Henselers) rhoA, or rhoB.
Calculation is done by \code{\link[=calculateRhoC]{calculateRhoC()}}.}
\item{Cronbach's alpha; "cronbachs_alpha"}{An estimate of the
reliability assuming a tau-equivalent measurement model (i.e., a measurement
model with equal loadings) and a test score (proxy) based on unit weights.
To compute Cronbach's alpha based on a score that uses the weights of the
weight approach used to obtain \code{.object}, use \code{"cronbachs_alpha_weighted"} instead.
Cronbach's alpha is an alias for \code{"rho_T"} the tau-equivalent
reliability which is
the prefered name for this kind of reliability in \pkg{cSEM}, as it clearly states what
it actually estimates (the tau-equivalent reliability as opposed to
the congeneric reliability). "rho_T" and "cronbachs_alpha" are therefore
always identical. Calculation is done by \code{\link[=calculateRhoT]{calculateRhoT()}}}
\item{Distance measures; "dg", "dl", "dml"}{Measures of the distance
between the model-implied and the empirical indicator correlation matrix.
Currently, the geodesic distance (\code{"dg"}), the squared Euclidian distance
(\code{"dl"}) and the the maximum likelihood-based distance function are implemented
(\code{"dml"}). Calculation is done by \code{\link[=calculateDL]{calculateDL()}}, \code{\link[=calculateDG]{calculateDG()}},
and \code{\link[=calculateDML]{calculateDML()}}.}
\item{Degrees of freedom, "df"}{
Returns the degrees of freedom. Calculation is done by \code{\link[=calculateDf]{calculateDf()}}.
}
\item{Effect size; "esize"}{An index of the effect size of an independent
variable in a structural regression equation. The effect size of the k'th
independent variable in this case
is definied as the ratio (R2_included - R2_excluded)/(1 - R2_included), where
R2_included and R2_excluded are the R squares of the
original structural model regression equation (R2_included) and the
alternative specification with the k'th variable dropped (R2_excluded).
This measure is commonly known as Cohen's f^2.
Calculation is done by \code{\link[=calculateEffectSize]{calculateEffectSize()}}.}
\item{Fit indices; "cfi", "gfi", "ifi", "nfi", "nnfi",  "rmsea", "rms_theta"
"srmr"}{
Several absolute and incremental fit indices. Note that their suitability
for models containing constructs modeled as common factors is still an
open research question. Also note that fit indices are not tests in a
hypothesis testing sense and
decisions based on common cut-offs proposed in the literature should be
considered with caution!. Calculation is done by \code{\link[=calculateCFI]{calculateCFI()}},
\code{\link[=calculateGFI]{calculateGFI()}}, \code{\link[=calculateIFI]{calculateIFI()}}, \code{\link[=calculateNFI]{calculateNFI()}}, \code{\link[=calculateNNFI]{calculateNNFI()}},
\code{\link[=calculateRMSEA]{calculateRMSEA()}}, \code{\link[=calculateRMSTheta]{calculateRMSTheta()}} and \code{\link[=calculateSRMR]{calculateSRMR()}}.}
\item{Fornell-Larcker criterion; "fl_criterion"}{A rule suggested by \insertCite{Fornell1981;textual}{cSEM}
to assess discriminant validity. The Fornell-Larcker
criterion is a decision rule based on a comparison between the squared
construct correlations and the average variance extracted. FL returns
a matrix with the squared construct correlations on the off-diagonal and
the AVE's on the main diagonal. Calculation is done by \code{assess()}.}
\item{Goodness of Fit (GoF); "gof"}{The GoF is defined as the square root
of the mean of the R squares of the structural model times the mean
of the variances in the indicators that are explained by their
related constructs (i.e., the average over all lambda^2_k).
For the latter, only constructs modeled as common factors are considered
as they explain their indicator variance in contrast to a composite where
indicators actually build the construct.
Note that, contrary to what the name suggests, the GoF is \strong{not} a
measure of model fit in a Chi-square fit test sense. Calculation is done
by \code{\link[=calculateGoF]{calculateGoF()}}.}
\item{Heterotrait-monotrait ratio of correlations (HTMT); "htmt"}{
An estimate of the latent variable correlation used to assess
convergent and/or discriminant validity of a construct. Calculation is done
by \code{\link[=calculateHTMT]{calculateHTMT()}}.}
\item{Reliability: "reliability"}{
As described in the \href{https://m-e-rademaker.github.io/cSEM/articles/Using-assess.html#methods}{Methods and Formulae}
section of the \href{https://m-e-rademaker.github.io/cSEM/articles/Using-assess.html}{Postestimation: Assessing a model}
article on the on the \href{https://m-e-rademaker.github.io/cSEM/index.html}{cSEM website}
there are many different estimators for the (internal consistency) reliability.
Choosing \code{.quality_criterion = "reliability"} computes the three most common
measures, namely: "Cronbachs alpha" (identical to "rho_T"), "Jöreskogs rho" (identical to "rho_C_mm"),
and "Dijkstra-Henselers rho A" (identical to "rho_C_weighted_mm").}
\item{R square and R square adjusted; "r2", "r2_adj"}{The R square and the adjusted
R square for each structural regression equation.
Calculated when running \code{\link[=csem]{csem()}}.}
\item{Tau-equivalent reliability; "rho_T"}{An estimate of the
reliability assuming a tau-equivalent measurement model (i.e. a measurement
model with equal loadings) and a test score (proxy) based on unit weights.
Tau-equivalent reliability is the preferred name for reliability estimates
that assume a tau-equivalent measurment model such as Cronbach's alpha.
Calculation is done by \code{\link[=calculateRhoT]{calculateRhoT()}}.}
\item{Variance inflation factors (VIF); "vif"}{An index for the amount of (multi-)
collinearity between independent variables of a regression equation. Computed
for each structural equation. Practically, VIF_k is defined
as the ratio of 1 over (1 - R2_k) where R2_k is the R squared from a regression
of the k'th independent variable on all remaining independent variables.
Calculated when running \code{\link[=csem]{csem()}}.}
\item{Variance inflation factors for PLS-PM mode B (VIF-ModeB); "vifmodeB"}{An index for
the amount of (multi-) collinearity between independent variables (indicators) in
mode B regression equations. Computed only if \code{.object} was obtained using
\code{.weight_approach = "PLS-PM"} and at least one mode was mode B.
Practically, VIF-ModeB_k is defined as the ratio of 1 over (1 - R2_k) where
R2_k is the R squared from a regression of the k'th indicator of block j on
all remaining indicators of the same block.
Calculation is done by \code{\link[=calculateVIFModeB]{calculateVIFModeB()}}.}
}

For details on the most important quality criteria see the \href{https://m-e-rademaker.github.io/cSEM/articles/Using-assess.html#methods}{Methods and Formulae} section
of the \href{https://m-e-rademaker.github.io/cSEM/articles/Using-assess.html}{Postestimation: Assessing a model}
article on the on the
\href{https://m-e-rademaker.github.io/cSEM/index.html}{cSEM website}.

Some of the quality criteria are inherently tied to the classical common
factor model and therefore only meaningfully interpreted within a common
factor model (see the
\href{https://m-e-rademaker.github.io/cSEM/articles/Using-assess.html}{Postestimation: Assessing a model}
article for details).
It is possible to force computation of all quality criteria for constructs
modeled as composites by setting \code{.only_common_factors = FALSE}, however,
we explicitly warn to interpret quality criteria in this case with caution,
as they may not even have a conceptual meaning.

\subsection{Resampling}{
To resample a given quality criterion supply the name of the function
that calculates the desired quality criterion to \code{\link[=csem]{csem()}}'s \code{.user_funs} argument.
See \code{\link[=resamplecSEMResults]{resamplecSEMResults()}} for details.
}
}
\examples{
\donttest{# ===========================================================================
# Using the threecommonfactors dataset
# ===========================================================================
model <- "
# Structural model
eta2 ~ eta1
eta3 ~ eta1 + eta2

# Each concept is measured by 3 indicators, i.e., modeled as latent variable
eta1 =~ y11 + y12 + y13
eta2 =~ y21 + y22 + y23
eta3 =~ y31 + y32 + y33
"

res <- csem(threecommonfactors, model)
a   <- assess(res) # computes all quality criteria (.quality_criterion = "all")
a

## The return value is a named list
a$HTMT

# You may also just compute a subset of the quality criteria
assess(res, .quality_criterion = c("ave", "rho_C", "htmt"))

## Resampling ---------------------------------------------------------------
# To resample a given quality criterion use csem()'s .user_funs argument
# Note: The output of the quality criterion needs to be a vector or a matrix.
#       Matrices will be vectorized columnwise.
res <- csem(threecommonfactors, model, 
            .resample_method = "bootstrap", 
            .R               = 40,
            .user_funs       = cSEM:::calculateHTMT
)

## Look at the resamples
res$Estimates$Estimates_resample$Estimates1$User_fun$Resampled[1:4, ]

## Use infer() to compute e.g. the 95\% percentile confidence interval
res_infer <- infer(res, .quantity = "CI_percentile")
res_infer$User_fun 
}
}
\seealso{
\code{\link[=csem]{csem()}}, \code{\link[=resamplecSEMResults]{resamplecSEMResults()}}
}
