#' List announcements
#' @param limit integer optional. Number of results to return. Defaults to 10. Maximum allowed is 50.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to released_at. Must be one of: released_at.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of this announcement}
#' \item{subject}{string, The subject of this announcement.}
#' \item{body}{string, The body of this announcement.}
#' \item{releasedAt}{string, The date and time this announcement was released.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' @export
announcements_list <- function(limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/announcements/"
  path_params  <- list()
  query_params <- list(limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' List apps
#' 
#' @return  An array containing the following fields:
#' \item{slug}{string, The slug for the application.}
#' \item{id}{integer, The unique id of the application.}
#' \item{instanceName}{string, A word that describes an instance of this app.}
#' \item{name}{string, The name of the application.}
#' @export
apps_list <- function() {

  args <- as.list(match.call())[-1]
  path <- "/apps/"
  path_params  <- list()
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' List details of a Decision Application
#' @param slug string required. The slug for the application.
#' 
#' @return  A list containing the following elements:
#' \item{slug}{string, The slug for the application.}
#' \item{id}{integer, The unique id of the application.}
#' \item{instanceName}{string, A word that describes an instance of this app.}
#' \item{name}{string, The name of the application.}
#' \item{currentRelease}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The unique id of the release.
#' \item appId integer, The id of the app the release belongs to.
#' \item reportTemplateId integer, ID of the report template for this release.
#' \item resources object, A hash of resources associated with this release.
#' }}
#' \item{features}{object, App features.}
#' @export
apps_get <- function(slug) {

  args <- as.list(match.call())[-1]
  path <- "/apps/{slug}"
  path_params  <- list(slug = slug)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' List the instances of a Decision Application
#' @param slug string required. The slug for the application.
#' @param archived string optional. The archival status of the requested object(s).
#' @param app_release_id integer optional. If supplied, return only instances matching this release.
#' @param limit integer optional. Number of results to return. Defaults to its maximum of 50.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to id. Must be one of: id, created_at.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to asc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The unique id of the instance.}
#' \item{name}{string, The name of the instance.}
#' \item{appReleaseId}{integer, The id of the app release the instance belongs to.}
#' \item{reportId}{integer, The id of the report the instance belongs to.}
#' \item{createdAt}{string, The time the instance was created at.}
#' \item{user}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{projectId}{integer, The id of the project collecting all the objects that belong to this app instance.}
#' @export
apps_list_instances <- function(slug, archived = NULL, app_release_id = NULL, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/apps/{slug}/instances"
  path_params  <- list(slug = slug)
  query_params <- list(archived = archived, app_release_id = app_release_id, limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Create a new instance of an application of the given slug
#' @param slug string required. The slug for the application.
#' @param name string optional. The name of the instance.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The unique id of the instance.}
#' \item{name}{string, The name of the instance.}
#' \item{appReleaseId}{integer, The id of the app release the instance belongs to.}
#' \item{reportId}{integer, The id of the report the instance belongs to.}
#' \item{createdAt}{string, The time the instance was created at.}
#' \item{user}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{projectId}{integer, The id of the project collecting all the objects that belong to this app instance.}
#' \item{authCodeUrl}{string, }
#' \item{apiKey}{string, A Civis API key that can be used by this app instance.}
#' @export
apps_post_instances <- function(slug, name = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/apps/{slug}/instances"
  path_params  <- list(slug = slug)
  query_params <- list()
  body_params  <- list(name = name)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' List the releases of a particular Decision Application
#' @param slug string required. The slug for the application.
#' @param archived string optional. The archival status of the requested object(s).
#' @param limit integer optional. Number of results to return. Defaults to its maximum of 50.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to id. Must be one of: id.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to asc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The unique id of the release.}
#' \item{appId}{integer, The id of the app the release belongs to.}
#' \item{reportTemplateId}{integer, ID of the report template for this release.}
#' \item{resources}{object, A hash of resources associated with this release.}
#' @export
apps_list_releases <- function(slug, archived = NULL, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/apps/{slug}/releases"
  path_params  <- list(slug = slug)
  query_params <- list(archived = archived, limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Return a given app release
#' @param id integer required. The unique id of the release.
#' @param slug string required. The slug for the application.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The unique id of the release.}
#' \item{appId}{integer, The id of the app the release belongs to.}
#' \item{reportTemplateId}{integer, ID of the report template for this release.}
#' \item{resources}{object, A hash of resources associated with this release.}
#' @export
apps_get_releases <- function(id, slug) {

  args <- as.list(match.call())[-1]
  path <- "/apps/{slug}/releases/{id}"
  path_params  <- list(id = id, slug = slug)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Return a given app instance
#' @param id integer required. The unique id of the instance.
#' @param slug string required. The slug for the application.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The unique id of the instance.}
#' \item{name}{string, The name of the instance.}
#' \item{appReleaseId}{integer, The id of the app release the instance belongs to.}
#' \item{reportId}{integer, The id of the report the instance belongs to.}
#' \item{createdAt}{string, The time the instance was created at.}
#' \item{user}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{projectId}{integer, The id of the project collecting all the objects that belong to this app instance.}
#' \item{authCodeUrl}{string, }
#' \item{apiKey}{string, A Civis API key that can be used by this app instance.}
#' @export
apps_get_instances <- function(id, slug) {

  args <- as.list(match.call())[-1]
  path <- "/apps/{slug}/instances/{id}"
  path_params  <- list(id = id, slug = slug)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update a given app instance
#' @param id integer required. The unique id of the instance.
#' @param slug string required. The slug for the application.
#' @param name string optional. The name of the instance.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The unique id of the instance.}
#' \item{name}{string, The name of the instance.}
#' \item{appReleaseId}{integer, The id of the app release the instance belongs to.}
#' \item{reportId}{integer, The id of the report the instance belongs to.}
#' \item{createdAt}{string, The time the instance was created at.}
#' \item{user}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{projectId}{integer, The id of the project collecting all the objects that belong to this app instance.}
#' \item{authCodeUrl}{string, }
#' \item{apiKey}{string, A Civis API key that can be used by this app instance.}
#' @export
apps_patch_instances <- function(id, slug, name = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/apps/{slug}/instances/{id}"
  path_params  <- list(id = id, slug = slug)
  query_params <- list()
  body_params  <- list(name = name)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("patch", path, path_params, query_params, body_params)

  return(resp)

 }


#' List users and groups permissioned on this object
#' @param id integer required. The ID of the object.
#' @param slug string required. The slug for the application.
#' 
#' @return  An array containing the following fields:
#' \item{readers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
apps_list_instances_shares <- function(id, slug) {

  args <- as.list(match.call())[-1]
  path <- "/apps/{slug}/instances/{id}/shares"
  path_params  <- list(id = id, slug = slug)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions users have on this object
#' @param id integer required. ID of the resource to be shared
#' @param slug string required. The slug for the application.
#' @param user_ids array required. An array of one or more user IDs
#' @param permission_level string required. Options are: "read", "write", or "manage"
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
apps_put_instances_shares_users <- function(id, slug, user_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/apps/{slug}/instances/{id}/shares/users"
  path_params  <- list(id = id, slug = slug)
  query_params <- list()
  body_params  <- list(userIds = user_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a user has on this object
#' @param id integer required. ID of the resource to be revoked
#' @param user_id integer required. ID of the user
#' @param slug string required. The slug for the application.
#' 
#' @return  An empty HTTP response
#' @export
apps_delete_instances_shares_users <- function(id, user_id, slug) {

  args <- as.list(match.call())[-1]
  path <- "/apps/{slug}/instances/{id}/shares/users/{user_id}"
  path_params  <- list(id = id, user_id = user_id, slug = slug)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions groups has on this object
#' @param id integer required. ID of the resource to be shared
#' @param slug string required. The slug for the application.
#' @param group_ids array required. An array of one or more group IDs
#' @param permission_level string required. Options are: "read", "write", or "manage"
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
apps_put_instances_shares_groups <- function(id, slug, group_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/apps/{slug}/instances/{id}/shares/groups"
  path_params  <- list(id = id, slug = slug)
  query_params <- list()
  body_params  <- list(groupIds = group_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a group has on this object
#' @param id integer required. ID of the resource to be revoked
#' @param group_id integer required. ID of the group
#' @param slug string required. The slug for the application.
#' 
#' @return  An empty HTTP response
#' @export
apps_delete_instances_shares_groups <- function(id, group_id, slug) {

  args <- as.list(match.call())[-1]
  path <- "/apps/{slug}/instances/{id}/shares/groups/{group_id}"
  path_params  <- list(id = id, group_id = group_id, slug = slug)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update the archive status of this object
#' @param id integer required. The ID of the object.
#' @param slug string required. The slug for the application.
#' @param status boolean required. The desired archived status of the object.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The unique id of the instance.}
#' \item{name}{string, The name of the instance.}
#' \item{appReleaseId}{integer, The id of the app release the instance belongs to.}
#' \item{reportId}{integer, The id of the report the instance belongs to.}
#' \item{createdAt}{string, The time the instance was created at.}
#' \item{user}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{projectId}{integer, The id of the project collecting all the objects that belong to this app instance.}
#' \item{authCodeUrl}{string, }
#' \item{apiKey}{string, A Civis API key that can be used by this app instance.}
#' @export
apps_put_instances_archive <- function(id, slug, status) {

  args <- as.list(match.call())[-1]
  path <- "/apps/{slug}/instances/{id}/archive"
  path_params  <- list(id = id, slug = slug)
  query_params <- list()
  body_params  <- list(status = status)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' List the projects a AppInstance belongs to
#' @param id integer required. The ID of the resource.
#' @param slug string required. The slug for the application.
#' @param hidden boolean optional. If specified to be true, returns hidden objects. Defaults to false, returning non-hidden objects.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID for this project.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{name}{string, The name of this project.}
#' \item{description}{string, A description of the project}
#' \item{users}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{autoShare}{boolean, }
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{archived}{string, The archival status of the requested object(s).}
#' @export
apps_list_instances_projects <- function(id, slug, hidden = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/apps/{slug}/instances/{id}/projects"
  path_params  <- list(id = id, slug = slug)
  query_params <- list(hidden = hidden)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Add a AppInstance to a project
#' @param id integer required. ID of the resource
#' @param project_id integer required. The ID of the project
#' @param slug string required. The slug for the application.
#' 
#' @return  An empty HTTP response
#' @export
apps_put_instances_projects <- function(id, project_id, slug) {

  args <- as.list(match.call())[-1]
  path <- "/apps/{slug}/instances/{id}/projects/{project_id}"
  path_params  <- list(id = id, project_id = project_id, slug = slug)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Remove a AppInstance from a project
#' @param id integer required. ID of the resource
#' @param project_id integer required. The ID of the project
#' @param slug string required. The slug for the application.
#' 
#' @return  An empty HTTP response
#' @export
apps_delete_instances_projects <- function(id, project_id, slug) {

  args <- as.list(match.call())[-1]
  path <- "/apps/{slug}/instances/{id}/projects/{project_id}"
  path_params  <- list(id = id, project_id = project_id, slug = slug)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' List Worker Clusters
#' @param limit integer optional. Number of results to return. Defaults to its maximum of 50.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to id. Must be one of: id, created_at.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to asc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of this cluster.}
#' \item{instanceType}{string, The EC2 instance types in this cluster.}
#' \item{minInstances}{integer, The minimum number of instances in this cluster.}
#' \item{maxInstances}{integer, The maximum number of instances in this cluster.}
#' \item{region}{string, The AWS region that this cluster is in.}
#' \item{activeJobsCount}{integer, The number of jobs currently being run in the cluster.}
#' \item{queuedJobsCount}{integer, The number of jobs currently waiting to be run on the cluster.}
#' @export
clusters_list_workers <- function(limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/clusters/workers"
  path_params  <- list()
  query_params <- list(limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Describe a Worker Cluster
#' @param id integer required. The ID of this cluster.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this cluster.}
#' \item{instanceType}{string, The EC2 instance types in this cluster.}
#' \item{minInstances}{integer, The minimum number of instances in this cluster.}
#' \item{maxInstances}{integer, The maximum number of instances in this cluster.}
#' \item{instances}{integer, The number of instances currently in this cluster.}
#' \item{instanceMaxMemory}{integer, The amount of memory available to a single instance.}
#' \item{instanceMaxCpu}{integer, The number of processor shares available to a single instance.}
#' \item{instanceMaxDiskSpace}{number, The amount of memory available to a single instance.}
#' \item{region}{string, The AWS region that this cluster is in.}
#' \item{activeJobsCount}{integer, The number of jobs currently being run in the cluster.}
#' \item{queuedJobsCount}{integer, The number of jobs currently waiting to be run on the cluster.}
#' @export
clusters_get_workers <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/clusters/workers/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' List Active Jobs for a Worker Cluster
#' @param id integer required. The ID of this cluster.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, }
#' \item{name}{string, }
#' \item{type}{string, }
#' \item{fromTemplateId}{integer, }
#' \item{state}{string, Whether the job is idle, queued, running, cancelled, or failed.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{runs}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{hidden}{boolean, The hidden status of the object.}
#' \item{requiredCpu}{integer, The CPU shares required by the script.}
#' \item{requiredDiskSpace}{integer, The disk space in GB required by the script.}
#' \item{requiredMemory}{integer, The memory in MB required by the script.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{runningAs}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' @export
clusters_list_workers_active_jobs <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/clusters/workers/{id}/active_jobs"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' List Queued Jobs for a Worker Cluster
#' @param id integer required. The ID of this cluster.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, }
#' \item{name}{string, }
#' \item{type}{string, }
#' \item{fromTemplateId}{integer, }
#' \item{state}{string, Whether the job is idle, queued, running, cancelled, or failed.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{runs}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{hidden}{boolean, The hidden status of the object.}
#' \item{requiredCpu}{integer, The CPU shares required by the script.}
#' \item{requiredDiskSpace}{integer, The disk space in GB required by the script.}
#' \item{requiredMemory}{integer, The memory in MB required by the script.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{runningAs}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' @export
clusters_list_workers_queued_jobs <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/clusters/workers/{id}/queued_jobs"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' List Kubernetes Clusters
#' @param organization_slug string optional. The slug of this cluster's organization.
#' @param limit integer optional. Number of results to return. Defaults to its maximum of 50.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to organization_id. Must be one of: organization_id, created_at.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to asc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of this cluster.}
#' \item{instanceTypes}{array, An array containing the following fields: 
#' \itemize{
#' \item type string, An EC2 instance type.
#' \item minInstances integer, The minimum number of instances of that type in this cluster.
#' \item maxInstances integer, The maximum number of instances of that type in this cluster.
#' \item instanceMaxMemory integer, The amount of memory available to a single instance of that type.
#' \item instanceMaxCpu integer, The number of processor shares available to a single instance of that type.
#' }}
#' \item{defaultInstanceType}{string, The default EC2 instance type in this cluster.}
#' \item{organizationId}{string, The id of this cluster's organization.}
#' \item{organizationSlug}{string, The slug of this cluster's organization.}
#' \item{securityGroupId}{string, The security group to be added to the nodes of this cluster}
#' @export
clusters_list_kubernetes <- function(organization_slug = NULL, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/clusters/kubernetes"
  path_params  <- list()
  query_params <- list(organization_slug = organization_slug, limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Describe a Kubernetes Cluster
#' @param id integer required. The ID of this cluster.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this cluster.}
#' \item{instanceTypes}{array, An array containing the following fields: 
#' \itemize{
#' \item type string, An EC2 instance type.
#' \item minInstances integer, The minimum number of instances of that type in this cluster.
#' \item maxInstances integer, The maximum number of instances of that type in this cluster.
#' \item instanceMaxMemory integer, The amount of memory available to a single instance of that type.
#' \item instanceMaxCpu integer, The number of processor shares available to a single instance of that type.
#' }}
#' \item{defaultInstanceType}{string, The default EC2 instance type in this cluster.}
#' \item{organizationId}{string, The id of this cluster's organization.}
#' \item{organizationSlug}{string, The slug of this cluster's organization.}
#' \item{securityGroupId}{string, The security group to be added to the nodes of this cluster}
#' @export
clusters_get_kubernetes <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/clusters/kubernetes/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' List codes
#' @param limit integer optional. The maximum number of codes to return.
#' @param type string optional. The code's type; e.g., Code::FrontEnd.
#' @param featured boolean optional. If true, return featured codes.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID for this code.}
#' \item{name}{string, Name of code.}
#' \item{type}{string, The code's type; e.g., Code::FrontEnd.}
#' \item{createdAt}{string, The creation time for this code.}
#' \item{updatedAt}{string, The last modification time for this code.}
#' @export
codes_list <- function(limit = NULL, type = NULL, featured = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/codes/"
  path_params  <- list()
  query_params <- list(limit = limit, type = type, featured = featured)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Create a new code
#' @param name string required. Name of code.
#' @param type string required. The code's type; e.g., Code::FrontEnd.
#' @param body string required. Actual code contents; e.g., HTML, SQL, etc.
#' @param readme string required. User specified information about this code. Markdown format accepted.
#' @param score integer optional. Internal Civis Use Only.
#' @param auth_s3_url string optional. Authorized_s3_link to file.
#' 
#' @return  An empty HTTP response
#' @export
codes_post <- function(name, type, body, readme, score = NULL, auth_s3_url = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/codes/"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(name = name, type = type, body = body, readme = readme, score = score, authS3Url = auth_s3_url)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' Show basic code info
#' @param id integer required. The ID for this code.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for this code.}
#' \item{name}{string, Name of code.}
#' \item{type}{string, The code's type; e.g., Code::FrontEnd.}
#' \item{body}{string, Actual code contents; e.g., HTML, SQL, etc.}
#' \item{readme}{string, User specified information about this code. Markdown format accepted.}
#' \item{score}{integer, Internal Civis Use Only.}
#' \item{authS3Url}{string, Authorized_s3_link to file.}
#' \item{createdAt}{string, The creation time for this code.}
#' \item{updatedAt}{string, The last modification time for this code.}
#' @export
codes_get <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/codes/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update a code
#' @param id integer required. The ID for this code.
#' @param name string required. Name of code.
#' @param type string required. The code's type; e.g., Code::FrontEnd.
#' @param body string required. Actual code contents; e.g., HTML, SQL, etc.
#' @param readme string required. User specified information about this code. Markdown format accepted.
#' @param score integer optional. Internal Civis Use Only.
#' @param auth_s3_url string optional. Authorized_s3_link to file.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for this code.}
#' \item{name}{string, Name of code.}
#' \item{type}{string, The code's type; e.g., Code::FrontEnd.}
#' \item{body}{string, Actual code contents; e.g., HTML, SQL, etc.}
#' \item{readme}{string, User specified information about this code. Markdown format accepted.}
#' \item{score}{integer, Internal Civis Use Only.}
#' \item{authS3Url}{string, Authorized_s3_link to file.}
#' \item{createdAt}{string, The creation time for this code.}
#' \item{updatedAt}{string, The last modification time for this code.}
#' @export
codes_put <- function(id, name, type, body, readme, score = NULL, auth_s3_url = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/codes/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(name = name, type = type, body = body, readme = readme, score = score, authS3Url = auth_s3_url)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update a code
#' @param id integer required. The ID for this code.
#' @param name string optional. Name of code.
#' @param type string optional. The code's type; e.g., Code::FrontEnd.
#' @param body string optional. Actual code contents; e.g., HTML, SQL, etc.
#' @param readme string optional. User specified information about this code. Markdown format accepted.
#' @param score integer optional. Internal Civis Use Only.
#' @param auth_s3_url string optional. Authorized_s3_link to file.
#' 
#' @return  An empty HTTP response
#' @export
codes_patch <- function(id, name = NULL, type = NULL, body = NULL, readme = NULL, score = NULL, auth_s3_url = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/codes/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(name = name, type = type, body = body, readme = readme, score = score, authS3Url = auth_s3_url)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("patch", path, path_params, query_params, body_params)

  return(resp)

 }


#' Delete a code
#' @param id integer required. The ID for this code.
#' 
#' @return  An empty HTTP response
#' @export
codes_delete <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/codes/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' List credentials
#' @param type string optional. The type (or types) of credentials to return. One or more of: Amazon Web Services S3, Bitbucket, BSD::API, CASS/NCOA PAF, Catalist::API, Catalist::SFTP, Certificate, Civis Platform, Custom, Database, Google, Github, JobTraits::Ftp, Salesforce User, Salesforce Client, Silverpop Application, Silverpop Refresh Token, Silverpop User, TableauUser, VAN::MyVoterFile, VAN::MyCampaign, and VAN::BothModes. Specify multiple values as a comma-separated list (e.g., "A,B").
#' @param default boolean optional. If true, will return a list with a single credential which is the current user's default credential.
#' @param limit integer optional. Number of results to return. Defaults to its maximum of 1000.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to updated_at. Must be one of: updated_at, created_at, name.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of the credential.}
#' \item{name}{string, The name identifying the credential}
#' \item{type}{string, The credential's type.}
#' \item{username}{string, The username for the credential.}
#' \item{description}{string, A long description of the credential.}
#' \item{owner}{string, The name of the user who this credential belongs to.}
#' \item{remoteHostId}{integer, The ID of the remote host associated with this credential.}
#' \item{remoteHostName}{string, The name of the remote host associated with this credential.}
#' \item{createdAt}{string, The creation time for this credential.}
#' \item{updatedAt}{string, The last modification time for this credential.}
#' @export
credentials_list <- function(type = NULL, default = NULL, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/credentials/"
  path_params  <- list()
  query_params <- list(type = type, default = default, limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Create a credential
#' @param type string required. 
#' @param username string required. The username for the credential.
#' @param password string required. The password for the credential.
#' @param name string optional. The name identifying the credential.
#' @param description string optional. A long description of the credential.
#' @param remote_host_id integer optional. The ID of the remote host associated with the credential.
#' @param system_credential boolean optional. 
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the credential.}
#' \item{name}{string, The name identifying the credential}
#' \item{type}{string, The credential's type.}
#' \item{username}{string, The username for the credential.}
#' \item{description}{string, A long description of the credential.}
#' \item{owner}{string, The name of the user who this credential belongs to.}
#' \item{remoteHostId}{integer, The ID of the remote host associated with this credential.}
#' \item{remoteHostName}{string, The name of the remote host associated with this credential.}
#' \item{createdAt}{string, The creation time for this credential.}
#' \item{updatedAt}{string, The last modification time for this credential.}
#' @export
credentials_post <- function(type, username, password, name = NULL, description = NULL, remote_host_id = NULL, system_credential = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/credentials/"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(type = type, username = username, password = password, name = name, description = description, remoteHostId = remote_host_id, systemCredential = system_credential)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update an existing credential
#' @param id integer required. The ID of the credential.
#' @param type string required. 
#' @param username string required. The username for the credential.
#' @param password string required. The password for the credential.
#' @param name string optional. The name identifying the credential.
#' @param description string optional. A long description of the credential.
#' @param remote_host_id integer optional. The ID of the remote host associated with the credential.
#' @param system_credential boolean optional. 
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the credential.}
#' \item{name}{string, The name identifying the credential}
#' \item{type}{string, The credential's type.}
#' \item{username}{string, The username for the credential.}
#' \item{description}{string, A long description of the credential.}
#' \item{owner}{string, The name of the user who this credential belongs to.}
#' \item{remoteHostId}{integer, The ID of the remote host associated with this credential.}
#' \item{remoteHostName}{string, The name of the remote host associated with this credential.}
#' \item{createdAt}{string, The creation time for this credential.}
#' \item{updatedAt}{string, The last modification time for this credential.}
#' @export
credentials_put <- function(id, type, username, password, name = NULL, description = NULL, remote_host_id = NULL, system_credential = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/credentials/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(type = type, username = username, password = password, name = name, description = description, remoteHostId = remote_host_id, systemCredential = system_credential)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get a credential
#' @param id integer required. The ID of the credential.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the credential.}
#' \item{name}{string, The name identifying the credential}
#' \item{type}{string, The credential's type.}
#' \item{username}{string, The username for the credential.}
#' \item{description}{string, A long description of the credential.}
#' \item{owner}{string, The name of the user who this credential belongs to.}
#' \item{remoteHostId}{integer, The ID of the remote host associated with this credential.}
#' \item{remoteHostName}{string, The name of the remote host associated with this credential.}
#' \item{createdAt}{string, The creation time for this credential.}
#' \item{updatedAt}{string, The last modification time for this credential.}
#' @export
credentials_get <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/credentials/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Authenticate against a remote host
#' @param url string required. The URL to your host.
#' @param remote_host_type string required. The type of remote host. One of: RemoteHostTypes::Bitbucket, RemoteHostTypes::Ftp, RemoteHostTypes::Github, RemoteHostTypes::GoogleDoc, RemoteHostTypes::JDBC, RemoteHostTypes::Redshift, RemoteHostTypes::Salesforce, and RemoteHostTypes::Van
#' @param username string required. The username for the credential.
#' @param password string required. The password for the credential.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the credential.}
#' \item{name}{string, The name identifying the credential}
#' \item{type}{string, The credential's type.}
#' \item{username}{string, The username for the credential.}
#' \item{description}{string, A long description of the credential.}
#' \item{owner}{string, The name of the user who this credential belongs to.}
#' \item{remoteHostId}{integer, The ID of the remote host associated with this credential.}
#' \item{remoteHostName}{string, The name of the remote host associated with this credential.}
#' \item{createdAt}{string, The creation time for this credential.}
#' \item{updatedAt}{string, The last modification time for this credential.}
#' @export
credentials_post_authenticate <- function(url, remote_host_type, username, password) {

  args <- as.list(match.call())[-1]
  path <- "/credentials/authenticate"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(url = url, remoteHostType = remote_host_type, username = username, password = password)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' Generate a temporary credential for accessing S3
#' @param id integer required. The ID of the credential.
#' @param duration integer optional. The number of seconds the temporary credential should be valid. Defaults to 15 minutes. Must not be less than 15 minutes or greater than 36 hours.
#' 
#' @return  A list containing the following elements:
#' \item{accessKey}{string, The identifier of the credential.}
#' \item{secretAccessKey}{string, The secret part of the credential.}
#' \item{sessionToken}{string, The session token identifier.}
#' @export
credentials_post_temporary <- function(id, duration = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/credentials/{id}/temporary"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(duration = duration)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' List users and groups permissioned on this object
#' @param id integer required. The ID of the object.
#' 
#' @return  An array containing the following fields:
#' \item{readers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
credentials_list_shares <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/credentials/{id}/shares"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions users have on this object
#' @param id integer required. ID of the resource to be shared
#' @param user_ids array required. An array of one or more user IDs
#' @param permission_level string required. Options are: "read", "write", or "manage"
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
credentials_put_shares_users <- function(id, user_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/credentials/{id}/shares/users"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(userIds = user_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a user has on this object
#' @param id integer required. ID of the resource to be revoked
#' @param user_id integer required. ID of the user
#' 
#' @return  An empty HTTP response
#' @export
credentials_delete_shares_users <- function(id, user_id) {

  args <- as.list(match.call())[-1]
  path <- "/credentials/{id}/shares/users/{user_id}"
  path_params  <- list(id = id, user_id = user_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions groups has on this object
#' @param id integer required. ID of the resource to be shared
#' @param group_ids array required. An array of one or more group IDs
#' @param permission_level string required. Options are: "read", "write", or "manage"
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
credentials_put_shares_groups <- function(id, group_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/credentials/{id}/shares/groups"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(groupIds = group_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a group has on this object
#' @param id integer required. ID of the resource to be revoked
#' @param group_id integer required. ID of the group
#' 
#' @return  An empty HTTP response
#' @export
credentials_delete_shares_groups <- function(id, group_id) {

  args <- as.list(match.call())[-1]
  path <- "/credentials/{id}/shares/groups/{group_id}"
  path_params  <- list(id = id, group_id = group_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' List databases
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID for the database.}
#' \item{name}{string, The name of the database.}
#' @export
databases_list <- function() {

  args <- as.list(match.call())[-1]
  path <- "/databases/"
  path_params  <- list()
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' List schemas in this database
#' @param id integer required. The ID of the database.
#' 
#' @return  An array containing the following fields:
#' \item{schema}{string, The name of a schema.}
#' @export
databases_list_schemas <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/databases/{id}/schemas"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' List tables in the specified database, deprecated use "GET /tables" instead
#' @param id integer required. The ID of the database.
#' @param name string optional. If specified, will be used to filter the tables returned. Substring matching is supported (e.g., "name=table" will return both "table1" and "my table").
#' @param limit integer optional. Number of results to return. Defaults to 200. Maximum allowed is 1000.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to name. Must be one of: name.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to asc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of the table.}
#' \item{databaseId}{integer, The ID of the database.}
#' \item{schema}{string, The name of the schema containing the table.}
#' \item{name}{string, Name of the table.}
#' \item{description}{string, The description of the table, as specified by the table owner}
#' \item{isView}{boolean, True if this table represents a view. False if it represents a regular table.}
#' \item{rowCount}{integer, The number of rows in the table.}
#' \item{columnCount}{integer, The number of columns in the table.}
#' \item{sizeMb}{number, The size of the table in megabytes.}
#' \item{owner}{string, The database username of the table's owner.}
#' \item{distkey}{string, The column used as the Amazon Redshift distkey.}
#' \item{sortkeys}{string, The column used as the Amazon Redshift sortkey.}
#' \item{refreshStatus}{string, How up-to-date the table's statistics on row counts, null counts, distinct counts, and values distributions are. One of: refreshing, stale, or current.}
#' \item{lastRefresh}{string, The time of the last statistics refresh.}
#' \item{refreshId}{string, The ID of the most recent statistics refresh.}
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' @export
databases_list_tables <- function(id, name = NULL, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/databases/{id}/tables"
  path_params  <- list(id = id)
  query_params <- list(name = name, limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' List tables in the specified database, deprecated use "GET /tables" instead
#' @param id integer required. The ID of the database.
#' @param name string optional. If specified, will be used to filter the tables returned. Substring matching is supported (e.g., "name=table" will return both "table1" and "my table").
#' @param column_name string optional. Search for tables containing a column with the given name.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of the table.}
#' \item{databaseId}{integer, The ID of the database.}
#' \item{schema}{string, The name of the schema containing the table.}
#' \item{name}{string, Name of the table.}
#' \item{description}{string, The description of the table, as specified by the table owner}
#' \item{isView}{boolean, True if this table represents a view. False if it represents a regular table.}
#' \item{rowCount}{integer, The number of rows in the table.}
#' \item{columnCount}{integer, The number of columns in the table.}
#' \item{sizeMb}{number, The size of the table in megabytes.}
#' \item{owner}{string, The database username of the table's owner.}
#' \item{distkey}{string, The column used as the Amazon Redshift distkey.}
#' \item{sortkeys}{string, The column used as the Amazon Redshift sortkey.}
#' \item{refreshStatus}{string, How up-to-date the table's statistics on row counts, null counts, distinct counts, and values distributions are. One of: refreshing, stale, or current.}
#' \item{lastRefresh}{string, The time of the last statistics refresh.}
#' \item{refreshId}{string, The ID of the most recent statistics refresh.}
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{columnNames}{array, The names of each column in the table.}
#' @export
databases_list_tables_search <- function(id, name = NULL, column_name = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/databases/{id}/tables-search"
  path_params  <- list(id = id)
  query_params <- list(name = name, column_name = column_name)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' List whitelisted IPs for the specified database
#' @param id integer required. The ID for the database.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of this whitelisted IP address.}
#' \item{remoteHostId}{integer, The ID of the database this rule is applied to.}
#' \item{securityGroupId}{string, The ID of the security group this rule is applied to.}
#' \item{subnetMask}{string, The subnet mask that is allowed by this rule.}
#' \item{createdAt}{string, The time this rule was created.}
#' \item{updatedAt}{string, The time this rule was last updated.}
#' @export
databases_list_whitelist_ips <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/databases/{id}/whitelist-ips"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Whitelist an IP address
#' @param id integer required. The ID of the database this rule is applied to.
#' @param subnet_mask string required. The subnet mask that is allowed by this rule.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this whitelisted IP address.}
#' \item{remoteHostId}{integer, The ID of the database this rule is applied to.}
#' \item{securityGroupId}{string, The ID of the security group this rule is applied to.}
#' \item{subnetMask}{string, The subnet mask that is allowed by this rule.}
#' \item{authorizedBy}{string, The user who authorized this rule.}
#' \item{isActive}{boolean, True if the rule is applied, false if it has been revoked.}
#' \item{createdAt}{string, The time this rule was created.}
#' \item{updatedAt}{string, The time this rule was last updated.}
#' @export
databases_post_whitelist_ips <- function(id, subnet_mask) {

  args <- as.list(match.call())[-1]
  path <- "/databases/{id}/whitelist-ips"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(subnetMask = subnet_mask)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' View details about a whitelisted IP
#' @param id integer required. The ID of the database this rule is applied to.
#' @param whitelisted_ip_id integer required. The ID of this whitelisted IP address.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this whitelisted IP address.}
#' \item{remoteHostId}{integer, The ID of the database this rule is applied to.}
#' \item{securityGroupId}{string, The ID of the security group this rule is applied to.}
#' \item{subnetMask}{string, The subnet mask that is allowed by this rule.}
#' \item{authorizedBy}{string, The user who authorized this rule.}
#' \item{isActive}{boolean, True if the rule is applied, false if it has been revoked.}
#' \item{createdAt}{string, The time this rule was created.}
#' \item{updatedAt}{string, The time this rule was last updated.}
#' @export
databases_get_whitelist_ips <- function(id, whitelisted_ip_id) {

  args <- as.list(match.call())[-1]
  path <- "/databases/{id}/whitelist-ips/{whitelisted_ip_id}"
  path_params  <- list(id = id, whitelisted_ip_id = whitelisted_ip_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Remove a whitelisted IP address
#' @param id integer required. The ID of the database this rule is applied to.
#' @param whitelisted_ip_id integer required. The ID of this whitelisted IP address.
#' 
#' @return  An empty HTTP response
#' @export
databases_delete_whitelist_ips <- function(id, whitelisted_ip_id) {

  args <- as.list(match.call())[-1]
  path <- "/databases/{id}/whitelist-ips/{whitelisted_ip_id}"
  path_params  <- list(id = id, whitelisted_ip_id = whitelisted_ip_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' List API endpoints
#' 
#' @return  An empty HTTP response
#' @export
endpoints_list <- function() {

  args <- as.list(match.call())[-1]
  path <- "/endpoints/"
  path_params  <- list()
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' List available enhancement types
#' 
#' @return  An array containing the following fields:
#' \item{name}{string, The name of the type.}
#' @export
enhancements_list_types <- function() {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/types"
  path_params  <- list()
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' List Enhancements
#' @param type string optional. If specified, return objects of these types.
#' @param author string optional. If specified, return objects from this author. Must use user IDs. A comma separated list of IDs is also accepted to return objects from multiple authors.
#' @param status string optional. If specified, returns objects with one of these statuses. It accepts a comma-separated list, possible values are 'running', 'failed', 'succeeded', 'idle', 'scheduled'.
#' @param archived string optional. The archival status of the requested object(s).
#' @param limit integer optional. Number of results to return. Defaults to 20. Maximum allowed is 50.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to updated_at. Must be one of: updated_at, name, created_at, last_run.updated_at.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID for the enhancement.}
#' \item{name}{string, The name of the enhancement.}
#' \item{type}{string, The type of the enhancement (e.g CASS-NCOA)}
#' \item{createdAt}{string, The time this enhancement was created.}
#' \item{updatedAt}{string, The time the enhancement was last updated.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the enhancement's last run}
#' \item{archived}{string, The archival status of the requested object(s).}
#' @export
enhancements_list <- function(type = NULL, author = NULL, status = NULL, archived = NULL, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/"
  path_params  <- list()
  query_params <- list(type = type, author = author, status = status, archived = archived, limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Create a CASS/NCOA Enhancement
#' @param name string required. The name of the enhancement.
#' @param source object required. A list containing the following elements: 
#' \itemize{
#' \item databaseTable object, 
#' }
#' @param schedule object optional. A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }
#' @param parent_id integer optional. Parent ID that triggers this enhancement.
#' @param notifications object optional. A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }
#' @param destination object optional. A list containing the following elements: 
#' \itemize{
#' \item databaseTable object, 
#' }
#' @param column_mapping object optional. A list containing the following elements: 
#' \itemize{
#' \item address1 string, The first address line.
#' \item address2 string, The second address line.
#' \item city string, The city of an address.
#' \item state string, The state of an address.
#' \item zip string, The zip code of an address.
#' \item name string, The full name of the resident at this address. If needed, separate multiple columns with `+`, e.g. `first_name+last_name`
#' \item company string, The name of the company located at this address.
#' }
#' @param use_default_column_mapping boolean optional. Defaults to true, where the existing column mapping on the input table will be used. If false, a custom column mapping must be provided.
#' @param perform_ncoa boolean optional. Whether to update addresses for records matching the National Change of Address (NCOA) database.
#' @param ncoa_credential_id integer optional. Credential to use when performing NCOA updates. Required if 'performNcoa' is true.
#' @param output_level string optional. The set of fields persisted by a CASS or NCOA enhancement.For CASS enhancements, one of 'cass' or 'all.'For NCOA enhancements, one of 'cass', 'ncoa' , 'coalesced' or 'all'.By default, all fields will be returned.
#' @param limiting_sql string optional. The limiting sql.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the enhancement.}
#' \item{name}{string, The name of the enhancement.}
#' \item{type}{string, The type of the enhancement (e.g CASS-NCOA)}
#' \item{createdAt}{string, The time this enhancement was created.}
#' \item{updatedAt}{string, The time the enhancement was last updated.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the enhancement's last run}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{parentId}{integer, Parent ID that triggers this enhancement.}
#' \item{notifications}{object, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{runningAs}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{source}{object, A list containing the following elements: 
#' \itemize{
#' \item databaseTable object, 
#' }}
#' \item{destination}{object, A list containing the following elements: 
#' \itemize{
#' \item databaseTable object, 
#' }}
#' \item{columnMapping}{object, A list containing the following elements: 
#' \itemize{
#' \item address1 string, The first address line.
#' \item address2 string, The second address line.
#' \item city string, The city of an address.
#' \item state string, The state of an address.
#' \item zip string, The zip code of an address.
#' \item name string, The full name of the resident at this address. If needed, separate multiple columns with `+`, e.g. `first_name+last_name`
#' \item company string, The name of the company located at this address.
#' }}
#' \item{useDefaultColumnMapping}{boolean, Defaults to true, where the existing column mapping on the input table will be used. If false, a custom column mapping must be provided.}
#' \item{performNcoa}{boolean, Whether to update addresses for records matching the National Change of Address (NCOA) database.}
#' \item{ncoaCredentialId}{integer, Credential to use when performing NCOA updates. Required if 'performNcoa' is true.}
#' \item{outputLevel}{string, The set of fields persisted by a CASS or NCOA enhancement.For CASS enhancements, one of 'cass' or 'all.'For NCOA enhancements, one of 'cass', 'ncoa' , 'coalesced' or 'all'.By default, all fields will be returned.}
#' \item{limitingSQL}{string, The limiting sql.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' @export
enhancements_post_cass_ncoa <- function(name, source, schedule = NULL, parent_id = NULL, notifications = NULL, destination = NULL, column_mapping = NULL, use_default_column_mapping = NULL, perform_ncoa = NULL, ncoa_credential_id = NULL, output_level = NULL, limiting_sql = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/cass-ncoa"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(name = name, source = source, schedule = schedule, parentId = parent_id, notifications = notifications, destination = destination, columnMapping = column_mapping, useDefaultColumnMapping = use_default_column_mapping, performNcoa = perform_ncoa, ncoaCredentialId = ncoa_credential_id, outputLevel = output_level, limitingSQL = limiting_sql)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get a CASS/NCOA Enhancement
#' @param id integer required. 
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the enhancement.}
#' \item{name}{string, The name of the enhancement.}
#' \item{type}{string, The type of the enhancement (e.g CASS-NCOA)}
#' \item{createdAt}{string, The time this enhancement was created.}
#' \item{updatedAt}{string, The time the enhancement was last updated.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the enhancement's last run}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{parentId}{integer, Parent ID that triggers this enhancement.}
#' \item{notifications}{object, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{runningAs}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{source}{object, A list containing the following elements: 
#' \itemize{
#' \item databaseTable object, 
#' }}
#' \item{destination}{object, A list containing the following elements: 
#' \itemize{
#' \item databaseTable object, 
#' }}
#' \item{columnMapping}{object, A list containing the following elements: 
#' \itemize{
#' \item address1 string, The first address line.
#' \item address2 string, The second address line.
#' \item city string, The city of an address.
#' \item state string, The state of an address.
#' \item zip string, The zip code of an address.
#' \item name string, The full name of the resident at this address. If needed, separate multiple columns with `+`, e.g. `first_name+last_name`
#' \item company string, The name of the company located at this address.
#' }}
#' \item{useDefaultColumnMapping}{boolean, Defaults to true, where the existing column mapping on the input table will be used. If false, a custom column mapping must be provided.}
#' \item{performNcoa}{boolean, Whether to update addresses for records matching the National Change of Address (NCOA) database.}
#' \item{ncoaCredentialId}{integer, Credential to use when performing NCOA updates. Required if 'performNcoa' is true.}
#' \item{outputLevel}{string, The set of fields persisted by a CASS or NCOA enhancement.For CASS enhancements, one of 'cass' or 'all.'For NCOA enhancements, one of 'cass', 'ncoa' , 'coalesced' or 'all'.By default, all fields will be returned.}
#' \item{limitingSQL}{string, The limiting sql.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' @export
enhancements_get_cass_ncoa <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/cass-ncoa/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Replace all attributes of this CASS/NCOA Enhancement
#' @param id integer required. The ID for the enhancement.
#' @param name string required. The name of the enhancement.
#' @param source object required. A list containing the following elements: 
#' \itemize{
#' \item databaseTable object, 
#' }
#' @param schedule object optional. A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }
#' @param parent_id integer optional. Parent ID that triggers this enhancement.
#' @param notifications object optional. A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }
#' @param destination object optional. A list containing the following elements: 
#' \itemize{
#' \item databaseTable object, 
#' }
#' @param column_mapping object optional. A list containing the following elements: 
#' \itemize{
#' \item address1 string, The first address line.
#' \item address2 string, The second address line.
#' \item city string, The city of an address.
#' \item state string, The state of an address.
#' \item zip string, The zip code of an address.
#' \item name string, The full name of the resident at this address. If needed, separate multiple columns with `+`, e.g. `first_name+last_name`
#' \item company string, The name of the company located at this address.
#' }
#' @param use_default_column_mapping boolean optional. Defaults to true, where the existing column mapping on the input table will be used. If false, a custom column mapping must be provided.
#' @param perform_ncoa boolean optional. Whether to update addresses for records matching the National Change of Address (NCOA) database.
#' @param ncoa_credential_id integer optional. Credential to use when performing NCOA updates. Required if 'performNcoa' is true.
#' @param output_level string optional. The set of fields persisted by a CASS or NCOA enhancement.For CASS enhancements, one of 'cass' or 'all.'For NCOA enhancements, one of 'cass', 'ncoa' , 'coalesced' or 'all'.By default, all fields will be returned.
#' @param limiting_sql string optional. The limiting sql.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the enhancement.}
#' \item{name}{string, The name of the enhancement.}
#' \item{type}{string, The type of the enhancement (e.g CASS-NCOA)}
#' \item{createdAt}{string, The time this enhancement was created.}
#' \item{updatedAt}{string, The time the enhancement was last updated.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the enhancement's last run}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{parentId}{integer, Parent ID that triggers this enhancement.}
#' \item{notifications}{object, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{runningAs}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{source}{object, A list containing the following elements: 
#' \itemize{
#' \item databaseTable object, 
#' }}
#' \item{destination}{object, A list containing the following elements: 
#' \itemize{
#' \item databaseTable object, 
#' }}
#' \item{columnMapping}{object, A list containing the following elements: 
#' \itemize{
#' \item address1 string, The first address line.
#' \item address2 string, The second address line.
#' \item city string, The city of an address.
#' \item state string, The state of an address.
#' \item zip string, The zip code of an address.
#' \item name string, The full name of the resident at this address. If needed, separate multiple columns with `+`, e.g. `first_name+last_name`
#' \item company string, The name of the company located at this address.
#' }}
#' \item{useDefaultColumnMapping}{boolean, Defaults to true, where the existing column mapping on the input table will be used. If false, a custom column mapping must be provided.}
#' \item{performNcoa}{boolean, Whether to update addresses for records matching the National Change of Address (NCOA) database.}
#' \item{ncoaCredentialId}{integer, Credential to use when performing NCOA updates. Required if 'performNcoa' is true.}
#' \item{outputLevel}{string, The set of fields persisted by a CASS or NCOA enhancement.For CASS enhancements, one of 'cass' or 'all.'For NCOA enhancements, one of 'cass', 'ncoa' , 'coalesced' or 'all'.By default, all fields will be returned.}
#' \item{limitingSQL}{string, The limiting sql.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' @export
enhancements_put_cass_ncoa <- function(id, name, source, schedule = NULL, parent_id = NULL, notifications = NULL, destination = NULL, column_mapping = NULL, use_default_column_mapping = NULL, perform_ncoa = NULL, ncoa_credential_id = NULL, output_level = NULL, limiting_sql = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/cass-ncoa/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(name = name, source = source, schedule = schedule, parentId = parent_id, notifications = notifications, destination = destination, columnMapping = column_mapping, useDefaultColumnMapping = use_default_column_mapping, performNcoa = perform_ncoa, ncoaCredentialId = ncoa_credential_id, outputLevel = output_level, limitingSQL = limiting_sql)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update some attributes of this CASS/NCOA Enhancement
#' @param id integer required. The ID for the enhancement.
#' @param name string optional. The name of the enhancement.
#' @param schedule object optional. A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }
#' @param parent_id integer optional. Parent ID that triggers this enhancement.
#' @param notifications object optional. A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }
#' @param source object optional. A list containing the following elements: 
#' \itemize{
#' \item databaseTable object, 
#' }
#' @param destination object optional. A list containing the following elements: 
#' \itemize{
#' \item databaseTable object, 
#' }
#' @param column_mapping object optional. A list containing the following elements: 
#' \itemize{
#' \item address1 string, The first address line.
#' \item address2 string, The second address line.
#' \item city string, The city of an address.
#' \item state string, The state of an address.
#' \item zip string, The zip code of an address.
#' \item name string, The full name of the resident at this address. If needed, separate multiple columns with `+`, e.g. `first_name+last_name`
#' \item company string, The name of the company located at this address.
#' }
#' @param use_default_column_mapping boolean optional. Defaults to true, where the existing column mapping on the input table will be used. If false, a custom column mapping must be provided.
#' @param perform_ncoa boolean optional. Whether to update addresses for records matching the National Change of Address (NCOA) database.
#' @param ncoa_credential_id integer optional. Credential to use when performing NCOA updates. Required if 'performNcoa' is true.
#' @param output_level string optional. The set of fields persisted by a CASS or NCOA enhancement.For CASS enhancements, one of 'cass' or 'all.'For NCOA enhancements, one of 'cass', 'ncoa' , 'coalesced' or 'all'.By default, all fields will be returned.
#' @param limiting_sql string optional. The limiting sql.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the enhancement.}
#' \item{name}{string, The name of the enhancement.}
#' \item{type}{string, The type of the enhancement (e.g CASS-NCOA)}
#' \item{createdAt}{string, The time this enhancement was created.}
#' \item{updatedAt}{string, The time the enhancement was last updated.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the enhancement's last run}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{parentId}{integer, Parent ID that triggers this enhancement.}
#' \item{notifications}{object, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{runningAs}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{source}{object, A list containing the following elements: 
#' \itemize{
#' \item databaseTable object, 
#' }}
#' \item{destination}{object, A list containing the following elements: 
#' \itemize{
#' \item databaseTable object, 
#' }}
#' \item{columnMapping}{object, A list containing the following elements: 
#' \itemize{
#' \item address1 string, The first address line.
#' \item address2 string, The second address line.
#' \item city string, The city of an address.
#' \item state string, The state of an address.
#' \item zip string, The zip code of an address.
#' \item name string, The full name of the resident at this address. If needed, separate multiple columns with `+`, e.g. `first_name+last_name`
#' \item company string, The name of the company located at this address.
#' }}
#' \item{useDefaultColumnMapping}{boolean, Defaults to true, where the existing column mapping on the input table will be used. If false, a custom column mapping must be provided.}
#' \item{performNcoa}{boolean, Whether to update addresses for records matching the National Change of Address (NCOA) database.}
#' \item{ncoaCredentialId}{integer, Credential to use when performing NCOA updates. Required if 'performNcoa' is true.}
#' \item{outputLevel}{string, The set of fields persisted by a CASS or NCOA enhancement.For CASS enhancements, one of 'cass' or 'all.'For NCOA enhancements, one of 'cass', 'ncoa' , 'coalesced' or 'all'.By default, all fields will be returned.}
#' \item{limitingSQL}{string, The limiting sql.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' @export
enhancements_patch_cass_ncoa <- function(id, name = NULL, schedule = NULL, parent_id = NULL, notifications = NULL, source = NULL, destination = NULL, column_mapping = NULL, use_default_column_mapping = NULL, perform_ncoa = NULL, ncoa_credential_id = NULL, output_level = NULL, limiting_sql = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/cass-ncoa/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(name = name, schedule = schedule, parentId = parent_id, notifications = notifications, source = source, destination = destination, columnMapping = column_mapping, useDefaultColumnMapping = use_default_column_mapping, performNcoa = perform_ncoa, ncoaCredentialId = ncoa_credential_id, outputLevel = output_level, limitingSQL = limiting_sql)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("patch", path, path_params, query_params, body_params)

  return(resp)

 }


#' Start a run
#' @param id integer required. The ID of the cass_ncoa.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the run.}
#' \item{cassNcoaId}{integer, The ID of the cass_ncoa.}
#' \item{state}{string, The state of the run, one of 'queued' 'running' 'succeeded' 'failed' or 'cancelled'.}
#' \item{isCancelRequested}{boolean, True if run cancel requested, else false.}
#' \item{startedAt}{string, The time the last run started at.}
#' \item{finishedAt}{string, The time the last run completed.}
#' \item{error}{string, The error, if any, returned by the run.}
#' @export
enhancements_post_cass_ncoa_runs <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/cass-ncoa/{id}/runs"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' List runs for the given cass_ncoa
#' @param id integer required. The ID of the cass_ncoa.
#' @param limit integer optional. Number of results to return. Defaults to 20. Maximum allowed is 100.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to id. Must be one of: id.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of the run.}
#' \item{cassNcoaId}{integer, The ID of the cass_ncoa.}
#' \item{state}{string, The state of the run, one of 'queued' 'running' 'succeeded' 'failed' or 'cancelled'.}
#' \item{isCancelRequested}{boolean, True if run cancel requested, else false.}
#' \item{startedAt}{string, The time the last run started at.}
#' \item{finishedAt}{string, The time the last run completed.}
#' \item{error}{string, The error, if any, returned by the run.}
#' @export
enhancements_list_cass_ncoa_runs <- function(id, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/cass-ncoa/{id}/runs"
  path_params  <- list(id = id)
  query_params <- list(limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Check status of a run
#' @param id integer required. The ID of the cass_ncoa.
#' @param run_id integer required. The ID of the run.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the run.}
#' \item{cassNcoaId}{integer, The ID of the cass_ncoa.}
#' \item{state}{string, The state of the run, one of 'queued' 'running' 'succeeded' 'failed' or 'cancelled'.}
#' \item{isCancelRequested}{boolean, True if run cancel requested, else false.}
#' \item{startedAt}{string, The time the last run started at.}
#' \item{finishedAt}{string, The time the last run completed.}
#' \item{error}{string, The error, if any, returned by the run.}
#' @export
enhancements_get_cass_ncoa_runs <- function(id, run_id) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/cass-ncoa/{id}/runs/{run_id}"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Cancel a run
#' @param id integer required. The ID of the cass_ncoa.
#' @param run_id integer required. The ID of the run.
#' 
#' @return  An empty HTTP response
#' @export
enhancements_delete_cass_ncoa_runs <- function(id, run_id) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/cass-ncoa/{id}/runs/{run_id}"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get the logs for a run
#' @param id integer required. The ID of the cass_ncoa.
#' @param run_id integer required. The ID of the run.
#' @param last_id integer optional. The ID of the last log message received. Log entries with this ID value or lower will be omitted.Logs are sorted by ID if this value is provided, and are otherwise sorted by createdAt.
#' @param limit integer optional. The maximum number of log messages to return. Default of 10000.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of the log.}
#' \item{createdAt}{string, The time the log was created.}
#' \item{message}{string, The log message.}
#' \item{level}{string, The level of the log. One of unknown,fatal,error,warn,info,debug.}
#' @export
enhancements_list_cass_ncoa_runs_logs <- function(id, run_id, last_id = NULL, limit = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/cass-ncoa/{id}/runs/{run_id}/logs"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list(last_id = last_id, limit = limit)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Cancel a run
#' @param id integer required. The ID of the job.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the run.}
#' \item{state}{string, The state of the run, one of 'queued', 'running' or 'cancelled'.}
#' \item{isCancelRequested}{boolean, True if run cancel requested, else false.}
#' @export
enhancements_post_cass_ncoa_cancel <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/cass-ncoa/{id}/cancel"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' List the outputs for a run
#' @param id integer required. The ID of the job.
#' @param run_id integer required. The ID of the run.
#' @param limit integer optional. Number of results to return. Defaults to its maximum of 50.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to created_at. Must be one of: created_at, id.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{objectType}{string, The type of the output. Valid values are File, Table, Report, Project, Credential, or JSONValue}
#' \item{objectId}{integer, The ID of the output object.}
#' \item{name}{string, The name of the output object.}
#' \item{link}{string, The link to retrieve the output object.}
#' \item{value}{string, }
#' @export
enhancements_list_cass_ncoa_runs_outputs <- function(id, run_id, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/cass-ncoa/{id}/runs/{run_id}/outputs"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list(limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Create a Person Matching Enhancement
#' @param name string required. The name of the enhancement.
#' @param configuration object required. A list containing the following elements: 
#' \itemize{
#' \item task string, The type of person matching task. Options are: "table_to_table", "dedupe_table", or "table_to_civis_data".
#' \item source string, The input source of your data. Options are: "redshift" or "s3".
#' \item inputDatabaseName string, The Redshift database name for input data.
#' \item inputSchema string, The schema name for the input data.
#' \item inputTable string, The table name for the input data.
#' \item inputFileId string, The ID of the input S3 file.
#' \item inputFieldMapping string, The column mapping for the input in JSON or YAML.
#' \item inputFileHeaders string, Provide your headers in a list if the first row of your input does not have the headers, and make them JSON-decodable. For example: ["col1","col2","col3"].
#' \item targetDatabaseName string, The Redshift database for target data.
#' \item targetSchema string, The schema for target data.
#' \item targetTable string, The table for target data.
#' \item targetFieldMapping string, The column mapping for the target in JSON or YAML.
#' \item targetFileId string, The ID of the target S3 file.
#' \item matchTargetId integer, The ID of the match target.
#' \item matchDatabaseName string, The Redshift database for the match output table.
#' \item matchSchema string, The schema for the match output table.
#' \item matchTable string, The name of the match output table.
#' \item matchCsvFilename string, The name of the match output file.
#' \item matchSourceIdCol string, The name of the column in the output table that will hold the id from the source for each match.
#' \item matchTargetIdCol string, The name the column in the output table that will hold the id from the target for each match.
#' \item maxMatches integer, The maximum number of matches to return.
#' \item threshold number, The score threshold (between 0 and 1).
#' }
#' @param schedule object optional. A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }
#' @param parent_id integer optional. Parent ID that triggers this enhancement.
#' @param notifications object optional. A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the enhancement.}
#' \item{name}{string, The name of the enhancement.}
#' \item{type}{string, The type of the enhancement (e.g CASS-NCOA)}
#' \item{createdAt}{string, The time this enhancement was created.}
#' \item{updatedAt}{string, The time the enhancement was last updated.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the enhancement's last run}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{parentId}{integer, Parent ID that triggers this enhancement.}
#' \item{notifications}{object, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{runningAs}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{configuration}{object, A list containing the following elements: 
#' \itemize{
#' \item task string, The type of person matching task. Options are: "table_to_table", "dedupe_table", or "table_to_civis_data".
#' \item source string, The input source of your data. Options are: "redshift" or "s3".
#' \item inputDatabaseName string, The Redshift database name for input data.
#' \item inputSchema string, The schema name for the input data.
#' \item inputTable string, The table name for the input data.
#' \item inputFileId string, The ID of the input S3 file.
#' \item inputFieldMapping string, The column mapping for the input in JSON or YAML.
#' \item inputFileHeaders string, Provide your headers in a list if the first row of your input does not have the headers, and make them JSON-decodable. For example: ["col1","col2","col3"].
#' \item targetDatabaseName string, The Redshift database for target data.
#' \item targetSchema string, The schema for target data.
#' \item targetTable string, The table for target data.
#' \item targetFieldMapping string, The column mapping for the target in JSON or YAML.
#' \item targetFileId string, The ID of the target S3 file.
#' \item matchTargetId integer, The ID of the match target.
#' \item matchDatabaseName string, The Redshift database for the match output table.
#' \item matchSchema string, The schema for the match output table.
#' \item matchTable string, The name of the match output table.
#' \item matchCsvFilename string, The name of the match output file.
#' \item matchSourceIdCol string, The name of the column in the output table that will hold the id from the source for each match.
#' \item matchTargetIdCol string, The name the column in the output table that will hold the id from the target for each match.
#' \item maxMatches integer, The maximum number of matches to return.
#' \item threshold number, The score threshold (between 0 and 1).
#' }}
#' @export
enhancements_post_person_matching <- function(name, configuration, schedule = NULL, parent_id = NULL, notifications = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/person-matching"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(name = name, configuration = configuration, schedule = schedule, parentId = parent_id, notifications = notifications)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get a Person Matching Enhancement
#' @param id integer required. 
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the enhancement.}
#' \item{name}{string, The name of the enhancement.}
#' \item{type}{string, The type of the enhancement (e.g CASS-NCOA)}
#' \item{createdAt}{string, The time this enhancement was created.}
#' \item{updatedAt}{string, The time the enhancement was last updated.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the enhancement's last run}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{parentId}{integer, Parent ID that triggers this enhancement.}
#' \item{notifications}{object, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{runningAs}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{configuration}{object, A list containing the following elements: 
#' \itemize{
#' \item task string, The type of person matching task. Options are: "table_to_table", "dedupe_table", or "table_to_civis_data".
#' \item source string, The input source of your data. Options are: "redshift" or "s3".
#' \item inputDatabaseName string, The Redshift database name for input data.
#' \item inputSchema string, The schema name for the input data.
#' \item inputTable string, The table name for the input data.
#' \item inputFileId string, The ID of the input S3 file.
#' \item inputFieldMapping string, The column mapping for the input in JSON or YAML.
#' \item inputFileHeaders string, Provide your headers in a list if the first row of your input does not have the headers, and make them JSON-decodable. For example: ["col1","col2","col3"].
#' \item targetDatabaseName string, The Redshift database for target data.
#' \item targetSchema string, The schema for target data.
#' \item targetTable string, The table for target data.
#' \item targetFieldMapping string, The column mapping for the target in JSON or YAML.
#' \item targetFileId string, The ID of the target S3 file.
#' \item matchTargetId integer, The ID of the match target.
#' \item matchDatabaseName string, The Redshift database for the match output table.
#' \item matchSchema string, The schema for the match output table.
#' \item matchTable string, The name of the match output table.
#' \item matchCsvFilename string, The name of the match output file.
#' \item matchSourceIdCol string, The name of the column in the output table that will hold the id from the source for each match.
#' \item matchTargetIdCol string, The name the column in the output table that will hold the id from the target for each match.
#' \item maxMatches integer, The maximum number of matches to return.
#' \item threshold number, The score threshold (between 0 and 1).
#' }}
#' @export
enhancements_get_person_matching <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/person-matching/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Replace all attributes of this Person Matching Enhancement
#' @param id integer required. The ID for the enhancement.
#' @param name string required. The name of the enhancement.
#' @param configuration object required. A list containing the following elements: 
#' \itemize{
#' \item task string, The type of person matching task. Options are: "table_to_table", "dedupe_table", or "table_to_civis_data".
#' \item source string, The input source of your data. Options are: "redshift" or "s3".
#' \item inputDatabaseName string, The Redshift database name for input data.
#' \item inputSchema string, The schema name for the input data.
#' \item inputTable string, The table name for the input data.
#' \item inputFileId string, The ID of the input S3 file.
#' \item inputFieldMapping string, The column mapping for the input in JSON or YAML.
#' \item inputFileHeaders string, Provide your headers in a list if the first row of your input does not have the headers, and make them JSON-decodable. For example: ["col1","col2","col3"].
#' \item targetDatabaseName string, The Redshift database for target data.
#' \item targetSchema string, The schema for target data.
#' \item targetTable string, The table for target data.
#' \item targetFieldMapping string, The column mapping for the target in JSON or YAML.
#' \item targetFileId string, The ID of the target S3 file.
#' \item matchTargetId integer, The ID of the match target.
#' \item matchDatabaseName string, The Redshift database for the match output table.
#' \item matchSchema string, The schema for the match output table.
#' \item matchTable string, The name of the match output table.
#' \item matchCsvFilename string, The name of the match output file.
#' \item matchSourceIdCol string, The name of the column in the output table that will hold the id from the source for each match.
#' \item matchTargetIdCol string, The name the column in the output table that will hold the id from the target for each match.
#' \item maxMatches integer, The maximum number of matches to return.
#' \item threshold number, The score threshold (between 0 and 1).
#' }
#' @param schedule object optional. A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }
#' @param parent_id integer optional. Parent ID that triggers this enhancement.
#' @param notifications object optional. A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the enhancement.}
#' \item{name}{string, The name of the enhancement.}
#' \item{type}{string, The type of the enhancement (e.g CASS-NCOA)}
#' \item{createdAt}{string, The time this enhancement was created.}
#' \item{updatedAt}{string, The time the enhancement was last updated.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the enhancement's last run}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{parentId}{integer, Parent ID that triggers this enhancement.}
#' \item{notifications}{object, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{runningAs}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{configuration}{object, A list containing the following elements: 
#' \itemize{
#' \item task string, The type of person matching task. Options are: "table_to_table", "dedupe_table", or "table_to_civis_data".
#' \item source string, The input source of your data. Options are: "redshift" or "s3".
#' \item inputDatabaseName string, The Redshift database name for input data.
#' \item inputSchema string, The schema name for the input data.
#' \item inputTable string, The table name for the input data.
#' \item inputFileId string, The ID of the input S3 file.
#' \item inputFieldMapping string, The column mapping for the input in JSON or YAML.
#' \item inputFileHeaders string, Provide your headers in a list if the first row of your input does not have the headers, and make them JSON-decodable. For example: ["col1","col2","col3"].
#' \item targetDatabaseName string, The Redshift database for target data.
#' \item targetSchema string, The schema for target data.
#' \item targetTable string, The table for target data.
#' \item targetFieldMapping string, The column mapping for the target in JSON or YAML.
#' \item targetFileId string, The ID of the target S3 file.
#' \item matchTargetId integer, The ID of the match target.
#' \item matchDatabaseName string, The Redshift database for the match output table.
#' \item matchSchema string, The schema for the match output table.
#' \item matchTable string, The name of the match output table.
#' \item matchCsvFilename string, The name of the match output file.
#' \item matchSourceIdCol string, The name of the column in the output table that will hold the id from the source for each match.
#' \item matchTargetIdCol string, The name the column in the output table that will hold the id from the target for each match.
#' \item maxMatches integer, The maximum number of matches to return.
#' \item threshold number, The score threshold (between 0 and 1).
#' }}
#' @export
enhancements_put_person_matching <- function(id, name, configuration, schedule = NULL, parent_id = NULL, notifications = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/person-matching/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(name = name, configuration = configuration, schedule = schedule, parentId = parent_id, notifications = notifications)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update some attributes of this Person Matching Enhancement
#' @param id integer required. The ID for the enhancement.
#' @param name string optional. The name of the enhancement.
#' @param schedule object optional. A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }
#' @param parent_id integer optional. Parent ID that triggers this enhancement.
#' @param notifications object optional. A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }
#' @param configuration object optional. A list containing the following elements: 
#' \itemize{
#' \item task string, The type of person matching task. Options are: "table_to_table", "dedupe_table", or "table_to_civis_data".
#' \item source string, The input source of your data. Options are: "redshift" or "s3".
#' \item inputDatabaseName string, The Redshift database name for input data.
#' \item inputSchema string, The schema name for the input data.
#' \item inputTable string, The table name for the input data.
#' \item inputFileId string, The ID of the input S3 file.
#' \item inputFieldMapping string, The column mapping for the input in JSON or YAML.
#' \item inputFileHeaders string, Provide your headers in a list if the first row of your input does not have the headers, and make them JSON-decodable. For example: ["col1","col2","col3"].
#' \item targetDatabaseName string, The Redshift database for target data.
#' \item targetSchema string, The schema for target data.
#' \item targetTable string, The table for target data.
#' \item targetFieldMapping string, The column mapping for the target in JSON or YAML.
#' \item targetFileId string, The ID of the target S3 file.
#' \item matchTargetId integer, The ID of the match target.
#' \item matchDatabaseName string, The Redshift database for the match output table.
#' \item matchSchema string, The schema for the match output table.
#' \item matchTable string, The name of the match output table.
#' \item matchCsvFilename string, The name of the match output file.
#' \item matchSourceIdCol string, The name of the column in the output table that will hold the id from the source for each match.
#' \item matchTargetIdCol string, The name the column in the output table that will hold the id from the target for each match.
#' \item maxMatches integer, The maximum number of matches to return.
#' \item threshold number, The score threshold (between 0 and 1).
#' }
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the enhancement.}
#' \item{name}{string, The name of the enhancement.}
#' \item{type}{string, The type of the enhancement (e.g CASS-NCOA)}
#' \item{createdAt}{string, The time this enhancement was created.}
#' \item{updatedAt}{string, The time the enhancement was last updated.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the enhancement's last run}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{parentId}{integer, Parent ID that triggers this enhancement.}
#' \item{notifications}{object, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{runningAs}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{configuration}{object, A list containing the following elements: 
#' \itemize{
#' \item task string, The type of person matching task. Options are: "table_to_table", "dedupe_table", or "table_to_civis_data".
#' \item source string, The input source of your data. Options are: "redshift" or "s3".
#' \item inputDatabaseName string, The Redshift database name for input data.
#' \item inputSchema string, The schema name for the input data.
#' \item inputTable string, The table name for the input data.
#' \item inputFileId string, The ID of the input S3 file.
#' \item inputFieldMapping string, The column mapping for the input in JSON or YAML.
#' \item inputFileHeaders string, Provide your headers in a list if the first row of your input does not have the headers, and make them JSON-decodable. For example: ["col1","col2","col3"].
#' \item targetDatabaseName string, The Redshift database for target data.
#' \item targetSchema string, The schema for target data.
#' \item targetTable string, The table for target data.
#' \item targetFieldMapping string, The column mapping for the target in JSON or YAML.
#' \item targetFileId string, The ID of the target S3 file.
#' \item matchTargetId integer, The ID of the match target.
#' \item matchDatabaseName string, The Redshift database for the match output table.
#' \item matchSchema string, The schema for the match output table.
#' \item matchTable string, The name of the match output table.
#' \item matchCsvFilename string, The name of the match output file.
#' \item matchSourceIdCol string, The name of the column in the output table that will hold the id from the source for each match.
#' \item matchTargetIdCol string, The name the column in the output table that will hold the id from the target for each match.
#' \item maxMatches integer, The maximum number of matches to return.
#' \item threshold number, The score threshold (between 0 and 1).
#' }}
#' @export
enhancements_patch_person_matching <- function(id, name = NULL, schedule = NULL, parent_id = NULL, notifications = NULL, configuration = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/person-matching/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(name = name, schedule = schedule, parentId = parent_id, notifications = notifications, configuration = configuration)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("patch", path, path_params, query_params, body_params)

  return(resp)

 }


#' Start a run
#' @param id integer required. The ID of the person_matching.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the run.}
#' \item{personMatchingId}{integer, The ID of the person_matching.}
#' \item{state}{string, The state of the run, one of 'queued' 'running' 'succeeded' 'failed' or 'cancelled'.}
#' \item{isCancelRequested}{boolean, True if run cancel requested, else false.}
#' \item{startedAt}{string, The time the last run started at.}
#' \item{finishedAt}{string, The time the last run completed.}
#' \item{error}{string, The error, if any, returned by the run.}
#' @export
enhancements_post_person_matching_runs <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/person-matching/{id}/runs"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' List runs for the given person_matching
#' @param id integer required. The ID of the person_matching.
#' @param limit integer optional. Number of results to return. Defaults to 20. Maximum allowed is 100.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to id. Must be one of: id.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of the run.}
#' \item{personMatchingId}{integer, The ID of the person_matching.}
#' \item{state}{string, The state of the run, one of 'queued' 'running' 'succeeded' 'failed' or 'cancelled'.}
#' \item{isCancelRequested}{boolean, True if run cancel requested, else false.}
#' \item{startedAt}{string, The time the last run started at.}
#' \item{finishedAt}{string, The time the last run completed.}
#' \item{error}{string, The error, if any, returned by the run.}
#' @export
enhancements_list_person_matching_runs <- function(id, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/person-matching/{id}/runs"
  path_params  <- list(id = id)
  query_params <- list(limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Check status of a run
#' @param id integer required. The ID of the person_matching.
#' @param run_id integer required. The ID of the run.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the run.}
#' \item{personMatchingId}{integer, The ID of the person_matching.}
#' \item{state}{string, The state of the run, one of 'queued' 'running' 'succeeded' 'failed' or 'cancelled'.}
#' \item{isCancelRequested}{boolean, True if run cancel requested, else false.}
#' \item{startedAt}{string, The time the last run started at.}
#' \item{finishedAt}{string, The time the last run completed.}
#' \item{error}{string, The error, if any, returned by the run.}
#' @export
enhancements_get_person_matching_runs <- function(id, run_id) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/person-matching/{id}/runs/{run_id}"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Cancel a run
#' @param id integer required. The ID of the person_matching.
#' @param run_id integer required. The ID of the run.
#' 
#' @return  An empty HTTP response
#' @export
enhancements_delete_person_matching_runs <- function(id, run_id) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/person-matching/{id}/runs/{run_id}"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get the logs for a run
#' @param id integer required. The ID of the person_matching.
#' @param run_id integer required. The ID of the run.
#' @param last_id integer optional. The ID of the last log message received. Log entries with this ID value or lower will be omitted.Logs are sorted by ID if this value is provided, and are otherwise sorted by createdAt.
#' @param limit integer optional. The maximum number of log messages to return. Default of 10000.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of the log.}
#' \item{createdAt}{string, The time the log was created.}
#' \item{message}{string, The log message.}
#' \item{level}{string, The level of the log. One of unknown,fatal,error,warn,info,debug.}
#' @export
enhancements_list_person_matching_runs_logs <- function(id, run_id, last_id = NULL, limit = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/person-matching/{id}/runs/{run_id}/logs"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list(last_id = last_id, limit = limit)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Cancel a run
#' @param id integer required. The ID of the job.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the run.}
#' \item{state}{string, The state of the run, one of 'queued', 'running' or 'cancelled'.}
#' \item{isCancelRequested}{boolean, True if run cancel requested, else false.}
#' @export
enhancements_post_person_matching_cancel <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/person-matching/{id}/cancel"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' List the outputs for a run
#' @param id integer required. The ID of the job.
#' @param run_id integer required. The ID of the run.
#' @param limit integer optional. Number of results to return. Defaults to its maximum of 50.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to created_at. Must be one of: created_at, id.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{objectType}{string, The type of the output. Valid values are File, Table, Report, Project, Credential, or JSONValue}
#' \item{objectId}{integer, The ID of the output object.}
#' \item{name}{string, The name of the output object.}
#' \item{link}{string, The link to retrieve the output object.}
#' \item{value}{string, }
#' @export
enhancements_list_person_matching_runs_outputs <- function(id, run_id, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/person-matching/{id}/runs/{run_id}/outputs"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list(limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' List users and groups permissioned on this object
#' @param id integer required. The ID of the object.
#' 
#' @return  An array containing the following fields:
#' \item{readers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
enhancements_list_cass_ncoa_shares <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/cass-ncoa/{id}/shares"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions users have on this object
#' @param id integer required. ID of the resource to be shared
#' @param user_ids array required. An array of one or more user IDs
#' @param permission_level string required. Options are: "read", "write", or "manage"
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
enhancements_put_cass_ncoa_shares_users <- function(id, user_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/cass-ncoa/{id}/shares/users"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(userIds = user_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a user has on this object
#' @param id integer required. ID of the resource to be revoked
#' @param user_id integer required. ID of the user
#' 
#' @return  An empty HTTP response
#' @export
enhancements_delete_cass_ncoa_shares_users <- function(id, user_id) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/cass-ncoa/{id}/shares/users/{user_id}"
  path_params  <- list(id = id, user_id = user_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions groups has on this object
#' @param id integer required. ID of the resource to be shared
#' @param group_ids array required. An array of one or more group IDs
#' @param permission_level string required. Options are: "read", "write", or "manage"
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
enhancements_put_cass_ncoa_shares_groups <- function(id, group_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/cass-ncoa/{id}/shares/groups"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(groupIds = group_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a group has on this object
#' @param id integer required. ID of the resource to be revoked
#' @param group_id integer required. ID of the group
#' 
#' @return  An empty HTTP response
#' @export
enhancements_delete_cass_ncoa_shares_groups <- function(id, group_id) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/cass-ncoa/{id}/shares/groups/{group_id}"
  path_params  <- list(id = id, group_id = group_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' List the projects a JobTypes::CassNcoa belongs to
#' @param id integer required. The ID of the resource.
#' @param hidden boolean optional. If specified to be true, returns hidden objects. Defaults to false, returning non-hidden objects.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID for this project.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{name}{string, The name of this project.}
#' \item{description}{string, A description of the project}
#' \item{users}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{autoShare}{boolean, }
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{archived}{string, The archival status of the requested object(s).}
#' @export
enhancements_list_cass_ncoa_projects <- function(id, hidden = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/cass-ncoa/{id}/projects"
  path_params  <- list(id = id)
  query_params <- list(hidden = hidden)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Add a JobTypes::CassNcoa to a project
#' @param id integer required. ID of the resource
#' @param project_id integer required. The ID of the project
#' 
#' @return  An empty HTTP response
#' @export
enhancements_put_cass_ncoa_projects <- function(id, project_id) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/cass-ncoa/{id}/projects/{project_id}"
  path_params  <- list(id = id, project_id = project_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Remove a JobTypes::CassNcoa from a project
#' @param id integer required. ID of the resource
#' @param project_id integer required. The ID of the project
#' 
#' @return  An empty HTTP response
#' @export
enhancements_delete_cass_ncoa_projects <- function(id, project_id) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/cass-ncoa/{id}/projects/{project_id}"
  path_params  <- list(id = id, project_id = project_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update the archive status of this object
#' @param id integer required. The ID of the object.
#' @param status boolean required. The desired archived status of the object.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the enhancement.}
#' \item{name}{string, The name of the enhancement.}
#' \item{type}{string, The type of the enhancement (e.g CASS-NCOA)}
#' \item{createdAt}{string, The time this enhancement was created.}
#' \item{updatedAt}{string, The time the enhancement was last updated.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the enhancement's last run}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{parentId}{integer, Parent ID that triggers this enhancement.}
#' \item{notifications}{object, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{runningAs}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{source}{object, A list containing the following elements: 
#' \itemize{
#' \item databaseTable object, 
#' }}
#' \item{destination}{object, A list containing the following elements: 
#' \itemize{
#' \item databaseTable object, 
#' }}
#' \item{columnMapping}{object, A list containing the following elements: 
#' \itemize{
#' \item address1 string, The first address line.
#' \item address2 string, The second address line.
#' \item city string, The city of an address.
#' \item state string, The state of an address.
#' \item zip string, The zip code of an address.
#' \item name string, The full name of the resident at this address. If needed, separate multiple columns with `+`, e.g. `first_name+last_name`
#' \item company string, The name of the company located at this address.
#' }}
#' \item{useDefaultColumnMapping}{boolean, Defaults to true, where the existing column mapping on the input table will be used. If false, a custom column mapping must be provided.}
#' \item{performNcoa}{boolean, Whether to update addresses for records matching the National Change of Address (NCOA) database.}
#' \item{ncoaCredentialId}{integer, Credential to use when performing NCOA updates. Required if 'performNcoa' is true.}
#' \item{outputLevel}{string, The set of fields persisted by a CASS or NCOA enhancement.For CASS enhancements, one of 'cass' or 'all.'For NCOA enhancements, one of 'cass', 'ncoa' , 'coalesced' or 'all'.By default, all fields will be returned.}
#' \item{limitingSQL}{string, The limiting sql.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' @export
enhancements_put_cass_ncoa_archive <- function(id, status) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/cass-ncoa/{id}/archive"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(status = status)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' List users and groups permissioned on this object
#' @param id integer required. The ID of the object.
#' 
#' @return  An array containing the following fields:
#' \item{readers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
enhancements_list_person_matching_shares <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/person-matching/{id}/shares"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions users have on this object
#' @param id integer required. ID of the resource to be shared
#' @param user_ids array required. An array of one or more user IDs
#' @param permission_level string required. Options are: "read", "write", or "manage"
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
enhancements_put_person_matching_shares_users <- function(id, user_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/person-matching/{id}/shares/users"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(userIds = user_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a user has on this object
#' @param id integer required. ID of the resource to be revoked
#' @param user_id integer required. ID of the user
#' 
#' @return  An empty HTTP response
#' @export
enhancements_delete_person_matching_shares_users <- function(id, user_id) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/person-matching/{id}/shares/users/{user_id}"
  path_params  <- list(id = id, user_id = user_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions groups has on this object
#' @param id integer required. ID of the resource to be shared
#' @param group_ids array required. An array of one or more group IDs
#' @param permission_level string required. Options are: "read", "write", or "manage"
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
enhancements_put_person_matching_shares_groups <- function(id, group_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/person-matching/{id}/shares/groups"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(groupIds = group_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a group has on this object
#' @param id integer required. ID of the resource to be revoked
#' @param group_id integer required. ID of the group
#' 
#' @return  An empty HTTP response
#' @export
enhancements_delete_person_matching_shares_groups <- function(id, group_id) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/person-matching/{id}/shares/groups/{group_id}"
  path_params  <- list(id = id, group_id = group_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' List the projects a container docker belongs to
#' @param id integer required. The ID of the resource.
#' @param hidden boolean optional. If specified to be true, returns hidden objects. Defaults to false, returning non-hidden objects.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID for this project.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{name}{string, The name of this project.}
#' \item{description}{string, A description of the project}
#' \item{users}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{autoShare}{boolean, }
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{archived}{string, The archival status of the requested object(s).}
#' @export
enhancements_list_person_matching_projects <- function(id, hidden = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/person-matching/{id}/projects"
  path_params  <- list(id = id)
  query_params <- list(hidden = hidden)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Add a container docker to a project
#' @param id integer required. ID of the resource
#' @param project_id integer required. The ID of the project
#' 
#' @return  An empty HTTP response
#' @export
enhancements_put_person_matching_projects <- function(id, project_id) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/person-matching/{id}/projects/{project_id}"
  path_params  <- list(id = id, project_id = project_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Remove a container docker from a project
#' @param id integer required. ID of the resource
#' @param project_id integer required. The ID of the project
#' 
#' @return  An empty HTTP response
#' @export
enhancements_delete_person_matching_projects <- function(id, project_id) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/person-matching/{id}/projects/{project_id}"
  path_params  <- list(id = id, project_id = project_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update the archive status of this object
#' @param id integer required. The ID of the object.
#' @param status boolean required. The desired archived status of the object.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the enhancement.}
#' \item{name}{string, The name of the enhancement.}
#' \item{type}{string, The type of the enhancement (e.g CASS-NCOA)}
#' \item{createdAt}{string, The time this enhancement was created.}
#' \item{updatedAt}{string, The time the enhancement was last updated.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the enhancement's last run}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{parentId}{integer, Parent ID that triggers this enhancement.}
#' \item{notifications}{object, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{runningAs}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{configuration}{object, A list containing the following elements: 
#' \itemize{
#' \item task string, The type of person matching task. Options are: "table_to_table", "dedupe_table", or "table_to_civis_data".
#' \item source string, The input source of your data. Options are: "redshift" or "s3".
#' \item inputDatabaseName string, The Redshift database name for input data.
#' \item inputSchema string, The schema name for the input data.
#' \item inputTable string, The table name for the input data.
#' \item inputFileId string, The ID of the input S3 file.
#' \item inputFieldMapping string, The column mapping for the input in JSON or YAML.
#' \item inputFileHeaders string, Provide your headers in a list if the first row of your input does not have the headers, and make them JSON-decodable. For example: ["col1","col2","col3"].
#' \item targetDatabaseName string, The Redshift database for target data.
#' \item targetSchema string, The schema for target data.
#' \item targetTable string, The table for target data.
#' \item targetFieldMapping string, The column mapping for the target in JSON or YAML.
#' \item targetFileId string, The ID of the target S3 file.
#' \item matchTargetId integer, The ID of the match target.
#' \item matchDatabaseName string, The Redshift database for the match output table.
#' \item matchSchema string, The schema for the match output table.
#' \item matchTable string, The name of the match output table.
#' \item matchCsvFilename string, The name of the match output file.
#' \item matchSourceIdCol string, The name of the column in the output table that will hold the id from the source for each match.
#' \item matchTargetIdCol string, The name the column in the output table that will hold the id from the target for each match.
#' \item maxMatches integer, The maximum number of matches to return.
#' \item threshold number, The score threshold (between 0 and 1).
#' }}
#' @export
enhancements_put_person_matching_archive <- function(id, status) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/person-matching/{id}/archive"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(status = status)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' List 
#' @param type string optional. If specified, return exports of these types. It accepts a comma-separated list, possible values are 'database' and 'gdoc'.
#' @param author string optional. If specified, return exports from this author. It accepts a comma-separated list of author ids.
#' @param status string optional. If specified, returns export with one of these statuses. It accepts a comma-separated list, possible values are 'running', 'failed', 'succeeded', 'idle', 'scheduled'.
#' @param hidden boolean optional. If specified to be true, returns hidden objects. Defaults to false, returning non-hidden objects.
#' @param archived string optional. The archival status of the requested object(s).
#' @param limit integer optional. Number of results to return. Defaults to 20. Maximum allowed is 50.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to updated_at. Must be one of: updated_at, name, created_at, last_run.updated_at.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID for this export.}
#' \item{name}{string, The name of this export.}
#' \item{type}{string, The type of export.}
#' \item{createdAt}{string, The creation time for this export.}
#' \item{updatedAt}{string, The last modification time for this export.}
#' \item{state}{string, }
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' @export
exports_list <- function(type = NULL, author = NULL, status = NULL, hidden = NULL, archived = NULL, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/exports/"
  path_params  <- list()
  query_params <- list(type = type, author = author, status = status, hidden = hidden, archived = archived, limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' List the projects a Data::S3File belongs to
#' @param id integer required. The ID of the resource.
#' @param hidden boolean optional. If specified to be true, returns hidden objects. Defaults to false, returning non-hidden objects.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID for this project.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{name}{string, The name of this project.}
#' \item{description}{string, A description of the project}
#' \item{users}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{autoShare}{boolean, }
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{archived}{string, The archival status of the requested object(s).}
#' @export
files_list_projects <- function(id, hidden = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/files/{id}/projects"
  path_params  <- list(id = id)
  query_params <- list(hidden = hidden)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Add a Data::S3File to a project
#' @param id integer required. ID of the resource
#' @param project_id integer required. The ID of the project
#' 
#' @return  An empty HTTP response
#' @export
files_put_projects <- function(id, project_id) {

  args <- as.list(match.call())[-1]
  path <- "/files/{id}/projects/{project_id}"
  path_params  <- list(id = id, project_id = project_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Remove a Data::S3File from a project
#' @param id integer required. ID of the resource
#' @param project_id integer required. The ID of the project
#' 
#' @return  An empty HTTP response
#' @export
files_delete_projects <- function(id, project_id) {

  args <- as.list(match.call())[-1]
  path <- "/files/{id}/projects/{project_id}"
  path_params  <- list(id = id, project_id = project_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' Initiate an upload of a file into the platform
#' @param name string required. The file name.
#' @param expires_at string optional. The date and time the file will expire. If not specified, the file will expire in 30 days. To keep a file indefinitely, specify null.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the file object.}
#' \item{name}{string, The file name.}
#' \item{createdAt}{string, The date and time the file was created.}
#' \item{fileSize}{integer, The file size.}
#' \item{expiresAt}{string, The date and time the file will expire. If not specified, the file will expire in 30 days. To keep a file indefinitely, specify null.}
#' \item{uploadUrl}{string, The URL that may be used to upload a file. To use the upload URL, initiate a POST request to the given URL with the file you wish to import as the "file" form field.}
#' \item{uploadFields}{object, A hash containing the form fields to be included with the POST request.}
#' @export
files_post <- function(name, expires_at = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/files/"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(name = name, expiresAt = expires_at)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' Initiate a multipart upload
#' @param name string required. The file name.
#' @param num_parts integer required. The number of parts in which the file will be uploaded. This parameter determines the number of presigned URLs that are returned.
#' @param expires_at string optional. The date and time the file will expire. If not specified, the file will expire in 30 days. To keep a file indefinitely, specify null.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the file object.}
#' \item{name}{string, The file name.}
#' \item{createdAt}{string, The date and time the file was created.}
#' \item{fileSize}{integer, The file size.}
#' \item{expiresAt}{string, The date and time the file will expire. If not specified, the file will expire in 30 days. To keep a file indefinitely, specify null.}
#' \item{uploadUrls}{array, An array of URLs that may be used to upload file parts. Use separate PUT requests to complete the part uploads. Links expire after 12 hours.}
#' @export
files_post_multipart <- function(name, num_parts, expires_at = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/files/multipart"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(name = name, numParts = num_parts, expiresAt = expires_at)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' Complete a multipart upload
#' @param id integer required. The ID of the file object.
#' 
#' @return  An empty HTTP response
#' @export
files_post_multipart_complete <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/files/multipart/{id}/complete"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get details about a file
#' @param id integer required. The ID of the file object.
#' @param link_expires_at string optional. The date and time the download link will expire. Must be a time between now and 36 hours from now. Defaults to 30 minutes from now.
#' @param inline boolean optional. If true, will return a url that can be displayed inline in HTML
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the file object.}
#' \item{name}{string, The file name.}
#' \item{createdAt}{string, The date and time the file was created.}
#' \item{fileSize}{integer, The file size.}
#' \item{expiresAt}{string, The date and time the file will expire. If not specified, the file will expire in 30 days. To keep a file indefinitely, specify null.}
#' \item{downloadUrl}{string, A JSON string containing information about the URL of the file.}
#' \item{fileUrl}{string, The URL that may be used to download the file.}
#' @export
files_get <- function(id, link_expires_at = NULL, inline = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/files/{id}"
  path_params  <- list(id = id)
  query_params <- list(link_expires_at = link_expires_at, inline = inline)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' List users and groups permissioned on this object
#' @param id integer required. The ID of the object.
#' 
#' @return  An array containing the following fields:
#' \item{readers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
files_list_shares <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/files/{id}/shares"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions users have on this object
#' @param id integer required. ID of the resource to be shared
#' @param user_ids array required. An array of one or more user IDs
#' @param permission_level string required. Options are: "read", "write", or "manage"
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
files_put_shares_users <- function(id, user_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/files/{id}/shares/users"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(userIds = user_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a user has on this object
#' @param id integer required. ID of the resource to be revoked
#' @param user_id integer required. ID of the user
#' 
#' @return  An empty HTTP response
#' @export
files_delete_shares_users <- function(id, user_id) {

  args <- as.list(match.call())[-1]
  path <- "/files/{id}/shares/users/{user_id}"
  path_params  <- list(id = id, user_id = user_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions groups has on this object
#' @param id integer required. ID of the resource to be shared
#' @param group_ids array required. An array of one or more group IDs
#' @param permission_level string required. Options are: "read", "write", or "manage"
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
files_put_shares_groups <- function(id, group_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/files/{id}/shares/groups"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(groupIds = group_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a group has on this object
#' @param id integer required. ID of the resource to be revoked
#' @param group_id integer required. ID of the group
#' 
#' @return  An empty HTTP response
#' @export
files_delete_shares_groups <- function(id, group_id) {

  args <- as.list(match.call())[-1]
  path <- "/files/{id}/shares/groups/{group_id}"
  path_params  <- list(id = id, group_id = group_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' List bookmarked git repositories
#' @param limit integer optional. Number of results to return. Defaults to 20. Maximum allowed is 100.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to repo_url. Must be one of: repo_url, created_at.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to asc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID for this git repository.}
#' \item{repoUrl}{string, The URL for this git repository.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' @export
git_repos_list <- function(limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/git_repos/"
  path_params  <- list()
  query_params <- list(limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Bookmark a git repository
#' @param repo_url string required. The URL for this git repository.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for this git repository.}
#' \item{repoUrl}{string, The URL for this git repository.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' @export
git_repos_post <- function(repo_url) {

  args <- as.list(match.call())[-1]
  path <- "/git_repos/"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(repoUrl = repo_url)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get a bookmarked git repository
#' @param id integer required. The ID for this git repository.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for this git repository.}
#' \item{repoUrl}{string, The URL for this git repository.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' @export
git_repos_get <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/git_repos/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Remove the bookmark on a git repository
#' @param id integer required. The ID for this git repository.
#' 
#' @return  An empty HTTP response
#' @export
git_repos_delete <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/git_repos/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' List Groups
#' @param query string optional. If specified, it will filter the groups returned. Prefix matching is supported (e.g., "query=group" will return "group" and "group of people", but not "my group".
#' @param permission string optional. A permissions string, one of "read", "write", or "manage". Lists only groups for which the current user has that permission.
#' @param limit integer optional. Number of results to return. Defaults to 50. Maximum allowed is 1000.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to name. Must be one of: name, created_at.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to asc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of this group.}
#' \item{name}{string, This group's name.}
#' \item{createdAt}{string, The date and time when this group was created.}
#' \item{slug}{string, The slug for this group.}
#' \item{organizationId}{integer, The organization associated with this group.}
#' @export
groups_list <- function(query = NULL, permission = NULL, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/groups/"
  path_params  <- list()
  query_params <- list(query = query, permission = permission, limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' List users and groups permissioned on this object
#' @param id integer required. The ID of the object.
#' 
#' @return  An array containing the following fields:
#' \item{readers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
imports_list_shares <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/imports/{id}/shares"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions users have on this object
#' @param id integer required. ID of the resource to be shared
#' @param user_ids array required. An array of one or more user IDs
#' @param permission_level string required. Options are: "read", "write", or "manage"
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
imports_put_shares_users <- function(id, user_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/imports/{id}/shares/users"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(userIds = user_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a user has on this object
#' @param id integer required. ID of the resource to be revoked
#' @param user_id integer required. ID of the user
#' 
#' @return  An empty HTTP response
#' @export
imports_delete_shares_users <- function(id, user_id) {

  args <- as.list(match.call())[-1]
  path <- "/imports/{id}/shares/users/{user_id}"
  path_params  <- list(id = id, user_id = user_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions groups has on this object
#' @param id integer required. ID of the resource to be shared
#' @param group_ids array required. An array of one or more group IDs
#' @param permission_level string required. Options are: "read", "write", or "manage"
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
imports_put_shares_groups <- function(id, group_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/imports/{id}/shares/groups"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(groupIds = group_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a group has on this object
#' @param id integer required. ID of the resource to be revoked
#' @param group_id integer required. ID of the group
#' 
#' @return  An empty HTTP response
#' @export
imports_delete_shares_groups <- function(id, group_id) {

  args <- as.list(match.call())[-1]
  path <- "/imports/{id}/shares/groups/{group_id}"
  path_params  <- list(id = id, group_id = group_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' List the projects a JobTypes::Import belongs to
#' @param id integer required. The ID of the resource.
#' @param hidden boolean optional. If specified to be true, returns hidden objects. Defaults to false, returning non-hidden objects.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID for this project.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{name}{string, The name of this project.}
#' \item{description}{string, A description of the project}
#' \item{users}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{autoShare}{boolean, }
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{archived}{string, The archival status of the requested object(s).}
#' @export
imports_list_projects <- function(id, hidden = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/imports/{id}/projects"
  path_params  <- list(id = id)
  query_params <- list(hidden = hidden)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Add a JobTypes::Import to a project
#' @param id integer required. ID of the resource
#' @param project_id integer required. The ID of the project
#' 
#' @return  An empty HTTP response
#' @export
imports_put_projects <- function(id, project_id) {

  args <- as.list(match.call())[-1]
  path <- "/imports/{id}/projects/{project_id}"
  path_params  <- list(id = id, project_id = project_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Remove a JobTypes::Import from a project
#' @param id integer required. ID of the resource
#' @param project_id integer required. The ID of the project
#' 
#' @return  An empty HTTP response
#' @export
imports_delete_projects <- function(id, project_id) {

  args <- as.list(match.call())[-1]
  path <- "/imports/{id}/projects/{project_id}"
  path_params  <- list(id = id, project_id = project_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update the archive status of this object
#' @param id integer required. The ID of the object.
#' @param status boolean required. The desired archived status of the object.
#' 
#' @return  A list containing the following elements:
#' \item{name}{string, The name of the import.}
#' \item{syncType}{string, The type of sync to perform; one of Dbsync, AutoImport, SilverpopDataImport, SilverpopContactImport, GdocImport, GdocExport, and Salesforce.}
#' \item{source}{object, A list containing the following elements: 
#' \itemize{
#' \item remoteHostId integer, 
#' \item credentialId integer, 
#' \item additionalCredentials array, Array that holds additional credentials used for specific imports. For salesforce imports, the first and only element is the client credential id.
#' \item name string, 
#' }}
#' \item{destination}{object, A list containing the following elements: 
#' \itemize{
#' \item remoteHostId integer, 
#' \item credentialId integer, 
#' \item additionalCredentials array, Array that holds additional credentials used for specific imports. For salesforce imports, the first and only element is the client credential id.
#' \item name string, 
#' }}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{notifications}{object, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{parentId}{integer, Parent id to trigger this import from}
#' \item{id}{integer, The ID for the import.}
#' \item{isOutbound}{boolean, }
#' \item{jobType}{string, The job type of this import.}
#' \item{syncs}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, 
#' \item source object, 
#' \item destination object, 
#' \item advancedOptions object, 
#' }}
#' \item{state}{string, }
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{user}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{runningAs}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{nextRunAt}{string, The time of the next scheduled run.}
#' \item{timeZone}{string, The time zone of this import.}
#' \item{hidden}{boolean, The hidden status of the object.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' @export
imports_put_archive <- function(id, status) {

  args <- as.list(match.call())[-1]
  path <- "/imports/{id}/archive"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(status = status)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' List 
#' @param type string optional. If specified, return imports of these types. It accepts a comma-separated list, possible values are 'AutoImport', 'DbSync', 'Salesforce', 'GdocImport'.
#' @param author string optional. If specified, return imports from this author. It accepts a comma-separated list of author ids.
#' @param destination string optional. If specified, returns imports with one of these destinations. It accepts a comma-separated list of remote host ids.
#' @param status string optional. If specified, returns imports with one of these statuses. It accepts a comma-separated list, possible values are 'running', 'failed', 'succeeded', 'idle', 'scheduled'.
#' @param hidden boolean optional. If specified to be true, returns hidden objects. Defaults to false, returning non-hidden objects.
#' @param archived string optional. The archival status of the requested object(s).
#' @param limit integer optional. Number of results to return. Defaults to 20. Maximum allowed is 50.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to updated_at. Must be one of: updated_at, name, created_at, last_run.updated_at.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{name}{string, The name of the import.}
#' \item{syncType}{string, The type of sync to perform; one of Dbsync, AutoImport, SilverpopDataImport, SilverpopContactImport, GdocImport, GdocExport, and Salesforce.}
#' \item{source}{object, A list containing the following elements: 
#' \itemize{
#' \item remoteHostId integer, 
#' \item credentialId integer, 
#' \item additionalCredentials array, Array that holds additional credentials used for specific imports. For salesforce imports, the first and only element is the client credential id.
#' \item name string, 
#' }}
#' \item{destination}{object, A list containing the following elements: 
#' \itemize{
#' \item remoteHostId integer, 
#' \item credentialId integer, 
#' \item additionalCredentials array, Array that holds additional credentials used for specific imports. For salesforce imports, the first and only element is the client credential id.
#' \item name string, 
#' }}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{id}{integer, The ID for the import.}
#' \item{isOutbound}{boolean, }
#' \item{jobType}{string, The job type of this import.}
#' \item{state}{string, }
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{user}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{timeZone}{string, The time zone of this import.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' @export
imports_list <- function(type = NULL, author = NULL, destination = NULL, status = NULL, hidden = NULL, archived = NULL, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/imports/"
  path_params  <- list()
  query_params <- list(type = type, author = author, destination = destination, status = status, hidden = hidden, archived = archived, limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Create a new import configuration
#' @param name string required. The name of the import.
#' @param sync_type string required. The type of sync to perform; one of Dbsync, AutoImport, SilverpopDataImport, SilverpopContactImport, GdocImport, GdocExport, and Salesforce.
#' @param is_outbound boolean required. 
#' @param source object optional. A list containing the following elements: 
#' \itemize{
#' \item remoteHostId integer, 
#' \item credentialId integer, 
#' \item additionalCredentials array, Array that holds additional credentials used for specific imports. For salesforce imports, the first and only element is the client credential id.
#' }
#' @param destination object optional. A list containing the following elements: 
#' \itemize{
#' \item remoteHostId integer, 
#' \item credentialId integer, 
#' \item additionalCredentials array, Array that holds additional credentials used for specific imports. For salesforce imports, the first and only element is the client credential id.
#' }
#' @param schedule object optional. A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }
#' @param notifications object optional. A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }
#' @param parent_id integer optional. Parent id to trigger this import from
#' @param next_run_at string optional. The time of the next scheduled run.
#' @param time_zone string optional. The time zone of this import.
#' @param hidden boolean optional. The hidden status of the object.
#' 
#' @return  A list containing the following elements:
#' \item{name}{string, The name of the import.}
#' \item{syncType}{string, The type of sync to perform; one of Dbsync, AutoImport, SilverpopDataImport, SilverpopContactImport, GdocImport, GdocExport, and Salesforce.}
#' \item{source}{object, A list containing the following elements: 
#' \itemize{
#' \item remoteHostId integer, 
#' \item credentialId integer, 
#' \item additionalCredentials array, Array that holds additional credentials used for specific imports. For salesforce imports, the first and only element is the client credential id.
#' \item name string, 
#' }}
#' \item{destination}{object, A list containing the following elements: 
#' \itemize{
#' \item remoteHostId integer, 
#' \item credentialId integer, 
#' \item additionalCredentials array, Array that holds additional credentials used for specific imports. For salesforce imports, the first and only element is the client credential id.
#' \item name string, 
#' }}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{notifications}{object, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{parentId}{integer, Parent id to trigger this import from}
#' \item{id}{integer, The ID for the import.}
#' \item{isOutbound}{boolean, }
#' \item{jobType}{string, The job type of this import.}
#' \item{syncs}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, 
#' \item source object, 
#' \item destination object, 
#' \item advancedOptions object, 
#' }}
#' \item{state}{string, }
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{user}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{runningAs}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{nextRunAt}{string, The time of the next scheduled run.}
#' \item{timeZone}{string, The time zone of this import.}
#' \item{hidden}{boolean, The hidden status of the object.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' @export
imports_post <- function(name, sync_type, is_outbound, source = NULL, destination = NULL, schedule = NULL, notifications = NULL, parent_id = NULL, next_run_at = NULL, time_zone = NULL, hidden = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/imports/"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(name = name, syncType = sync_type, isOutbound = is_outbound, source = source, destination = destination, schedule = schedule, notifications = notifications, parentId = parent_id, nextRunAt = next_run_at, timeZone = time_zone, hidden = hidden)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' Initate an import of a tabular file into the platform
#' @param schema string required. The schema of the destination table.
#' @param name string required. The name of the destination table.
#' @param remote_host_id integer required. The id of the destination database host.
#' @param credential_id integer required. The id of the credentials to be used when performing the database import.
#' @param max_errors integer optional. The maximum number of rows with errors to remove from the import before failing.
#' @param existing_table_rows string optional. The behaviour if a table with the requested name already exists.  One of "fail", "truncate", "append", or "drop".Defaults to "fail".
#' @param diststyle string optional. The diststyle to use for the table. One of "even", "all", or "key".
#' @param distkey string optional. The column to use as the distkey for the table.
#' @param sortkey1 string optional. The column to use as the sort key for the table.
#' @param sortkey2 string optional. The second column in a compound sortkey for the table.
#' @param column_delimiter string optional. The column delimiter of the file. If column_delimiter is null or omitted, it will be auto-detected. Valid arguments are "comma", "tab", and "pipe".
#' @param first_row_is_header boolean optional. A boolean value indicating whether or not the first row is a header row. If first_row_is_header is null or omitted, it will be auto-detected.
#' @param multipart boolean optional. If true, the upload URI will require a `multipart/form-data` POST request. Defaults to false.
#' @param hidden boolean optional. The hidden status of the object.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The id of the import.}
#' \item{uploadUri}{string, The URI which may be used to upload a tabular file for import. You must use this URI to upload the file you wish imported and then inform the Civis API when your upload is complete using the URI given by the runUri field of this reponse.}
#' \item{runUri}{string, The URI to POST to once the file upload is complete. After uploading the file using the URI given in the uploadUri attribute of the reponse, POST to this URI to initiate the import of your uploaded file into the platform.}
#' \item{uploadFields}{object, If multipart was set to true, these fields should be included in the multipart upload.}
#' @export
imports_post_files <- function(schema, name, remote_host_id, credential_id, max_errors = NULL, existing_table_rows = NULL, diststyle = NULL, distkey = NULL, sortkey1 = NULL, sortkey2 = NULL, column_delimiter = NULL, first_row_is_header = NULL, multipart = NULL, hidden = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/imports/files"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(schema = schema, name = name, remoteHostId = remote_host_id, credentialId = credential_id, maxErrors = max_errors, existingTableRows = existing_table_rows, diststyle = diststyle, distkey = distkey, sortkey1 = sortkey1, sortkey2 = sortkey2, columnDelimiter = column_delimiter, firstRowIsHeader = first_row_is_header, multipart = multipart, hidden = hidden)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' Start a run
#' @param id integer required. The ID of the import.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the run.}
#' \item{importId}{integer, The ID of the import.}
#' \item{state}{string, The state of the run, one of 'queued' 'running' 'succeeded' 'failed' or 'cancelled'.}
#' \item{isCancelRequested}{boolean, True if run cancel requested, else false.}
#' \item{startedAt}{string, The time the last run started at.}
#' \item{finishedAt}{string, The time the last run completed.}
#' \item{error}{string, The error, if any, returned by the run.}
#' @export
imports_post_files_runs <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/imports/files/{id}/runs"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' List runs for the given import
#' @param id integer required. The ID of the import.
#' @param limit integer optional. Number of results to return. Defaults to 20. Maximum allowed is 100.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to id. Must be one of: id.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of the run.}
#' \item{importId}{integer, The ID of the import.}
#' \item{state}{string, The state of the run, one of 'queued' 'running' 'succeeded' 'failed' or 'cancelled'.}
#' \item{isCancelRequested}{boolean, True if run cancel requested, else false.}
#' \item{startedAt}{string, The time the last run started at.}
#' \item{finishedAt}{string, The time the last run completed.}
#' \item{error}{string, The error, if any, returned by the run.}
#' @export
imports_list_files_runs <- function(id, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/imports/files/{id}/runs"
  path_params  <- list(id = id)
  query_params <- list(limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Check status of a run
#' @param id integer required. The ID of the import.
#' @param run_id integer required. The ID of the run.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the run.}
#' \item{importId}{integer, The ID of the import.}
#' \item{state}{string, The state of the run, one of 'queued' 'running' 'succeeded' 'failed' or 'cancelled'.}
#' \item{isCancelRequested}{boolean, True if run cancel requested, else false.}
#' \item{startedAt}{string, The time the last run started at.}
#' \item{finishedAt}{string, The time the last run completed.}
#' \item{error}{string, The error, if any, returned by the run.}
#' @export
imports_get_files_runs <- function(id, run_id) {

  args <- as.list(match.call())[-1]
  path <- "/imports/files/{id}/runs/{run_id}"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Cancel a run
#' @param id integer required. The ID of the import.
#' @param run_id integer required. The ID of the run.
#' 
#' @return  An empty HTTP response
#' @export
imports_delete_files_runs <- function(id, run_id) {

  args <- as.list(match.call())[-1]
  path <- "/imports/files/{id}/runs/{run_id}"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get the logs for a run
#' @param id integer required. The ID of the import.
#' @param run_id integer required. The ID of the run.
#' @param last_id integer optional. The ID of the last log message received. Log entries with this ID value or lower will be omitted.Logs are sorted by ID if this value is provided, and are otherwise sorted by createdAt.
#' @param limit integer optional. The maximum number of log messages to return. Default of 10000.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of the log.}
#' \item{createdAt}{string, The time the log was created.}
#' \item{message}{string, The log message.}
#' \item{level}{string, The level of the log. One of unknown,fatal,error,warn,info,debug.}
#' @export
imports_list_files_runs_logs <- function(id, run_id, last_id = NULL, limit = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/imports/files/{id}/runs/{run_id}/logs"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list(last_id = last_id, limit = limit)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get the logs for a run
#' @param id integer required. The ID of the import.
#' @param run_id integer required. The ID of the run.
#' @param last_id integer optional. The ID of the last log message received. Log entries with this ID value or lower will be omitted.Logs are sorted by ID if this value is provided, and are otherwise sorted by createdAt.
#' @param limit integer optional. The maximum number of log messages to return. Default of 10000.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of the log.}
#' \item{createdAt}{string, The time the log was created.}
#' \item{message}{string, The log message.}
#' \item{level}{string, The level of the log. One of unknown,fatal,error,warn,info,debug.}
#' @export
imports_list_runs_logs <- function(id, run_id, last_id = NULL, limit = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/imports/{id}/runs/{run_id}/logs"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list(last_id = last_id, limit = limit)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' List batch imports
#' @param hidden boolean optional. If specified to be true, returns hidden objects. Defaults to false, returning non-hidden objects.
#' @param limit integer optional. Number of results to return. Defaults to 20. Maximum allowed is 50.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to updated_at. Must be one of: updated_at, created_at.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID for the import.}
#' \item{schema}{string, The destination schema name. This schema must already exist in Redshift.}
#' \item{table}{string, The destination table name, without the schema prefix. This table must already exist in Redshift.}
#' \item{remoteHostId}{integer, The ID of the destination database host.}
#' \item{state}{string, The state of the run; one of "queued", "running", "succeeded", "failed", or "cancelled".}
#' \item{startedAt}{string, The time the last run started at.}
#' \item{finishedAt}{string, The time the last run completed.}
#' \item{error}{string, The error returned by the run, if any.}
#' @export
imports_list_batches <- function(hidden = NULL, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/imports/batches"
  path_params  <- list()
  query_params <- list(hidden = hidden, limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Upload multiple files to Redshift
#' @param file_ids array required. The file IDs for the import.
#' @param schema string required. The destination schema name. This schema must already exist in Redshift.
#' @param table string required. The destination table name, without the schema prefix. This table must already exist in Redshift.
#' @param remote_host_id integer required. The ID of the destination database host.
#' @param credential_id integer required. The ID of the credentials to be used when performing the database import.
#' @param column_delimiter string optional. The column delimiter for the file. Valid arguments are "comma", "tab", and "pipe". If unspecified, defaults to "comma".
#' @param first_row_is_header boolean optional. A boolean value indicating whether or not the first row is a header row. If unspecified, defaults to false.
#' @param compression string optional. The type of compression. Valid arguments are "gzip", "zip", and "none". If unspecified, defaults to "gzip".
#' @param hidden boolean optional. The hidden status of the object.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the import.}
#' \item{schema}{string, The destination schema name. This schema must already exist in Redshift.}
#' \item{table}{string, The destination table name, without the schema prefix. This table must already exist in Redshift.}
#' \item{remoteHostId}{integer, The ID of the destination database host.}
#' \item{state}{string, The state of the run; one of "queued", "running", "succeeded", "failed", or "cancelled".}
#' \item{startedAt}{string, The time the last run started at.}
#' \item{finishedAt}{string, The time the last run completed.}
#' \item{error}{string, The error returned by the run, if any.}
#' \item{hidden}{boolean, The hidden status of the object.}
#' @export
imports_post_batches <- function(file_ids, schema, table, remote_host_id, credential_id, column_delimiter = NULL, first_row_is_header = NULL, compression = NULL, hidden = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/imports/batches"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(fileIds = file_ids, schema = schema, table = table, remoteHostId = remote_host_id, credentialId = credential_id, columnDelimiter = column_delimiter, firstRowIsHeader = first_row_is_header, compression = compression, hidden = hidden)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get details about a batch import
#' @param id integer required. The ID for the import.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the import.}
#' \item{schema}{string, The destination schema name. This schema must already exist in Redshift.}
#' \item{table}{string, The destination table name, without the schema prefix. This table must already exist in Redshift.}
#' \item{remoteHostId}{integer, The ID of the destination database host.}
#' \item{state}{string, The state of the run; one of "queued", "running", "succeeded", "failed", or "cancelled".}
#' \item{startedAt}{string, The time the last run started at.}
#' \item{finishedAt}{string, The time the last run completed.}
#' \item{error}{string, The error returned by the run, if any.}
#' \item{hidden}{boolean, The hidden status of the object.}
#' @export
imports_get_batches <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/imports/batches/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get details about an import
#' @param id integer required. The ID for the import.
#' 
#' @return  A list containing the following elements:
#' \item{name}{string, The name of the import.}
#' \item{syncType}{string, The type of sync to perform; one of Dbsync, AutoImport, SilverpopDataImport, SilverpopContactImport, GdocImport, GdocExport, and Salesforce.}
#' \item{source}{object, A list containing the following elements: 
#' \itemize{
#' \item remoteHostId integer, 
#' \item credentialId integer, 
#' \item additionalCredentials array, Array that holds additional credentials used for specific imports. For salesforce imports, the first and only element is the client credential id.
#' \item name string, 
#' }}
#' \item{destination}{object, A list containing the following elements: 
#' \itemize{
#' \item remoteHostId integer, 
#' \item credentialId integer, 
#' \item additionalCredentials array, Array that holds additional credentials used for specific imports. For salesforce imports, the first and only element is the client credential id.
#' \item name string, 
#' }}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{notifications}{object, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{parentId}{integer, Parent id to trigger this import from}
#' \item{id}{integer, The ID for the import.}
#' \item{isOutbound}{boolean, }
#' \item{jobType}{string, The job type of this import.}
#' \item{syncs}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, 
#' \item source object, 
#' \item destination object, 
#' \item advancedOptions object, 
#' }}
#' \item{state}{string, }
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{user}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{runningAs}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{nextRunAt}{string, The time of the next scheduled run.}
#' \item{timeZone}{string, The time zone of this import.}
#' \item{hidden}{boolean, The hidden status of the object.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' @export
imports_get <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/imports/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update an import
#' @param id integer required. The ID for the import.
#' @param name string required. The name of the import.
#' @param sync_type string required. The type of sync to perform; one of Dbsync, AutoImport, SilverpopDataImport, SilverpopContactImport, GdocImport, GdocExport, and Salesforce.
#' @param is_outbound boolean required. 
#' @param source object optional. A list containing the following elements: 
#' \itemize{
#' \item remoteHostId integer, 
#' \item credentialId integer, 
#' \item additionalCredentials array, Array that holds additional credentials used for specific imports. For salesforce imports, the first and only element is the client credential id.
#' }
#' @param destination object optional. A list containing the following elements: 
#' \itemize{
#' \item remoteHostId integer, 
#' \item credentialId integer, 
#' \item additionalCredentials array, Array that holds additional credentials used for specific imports. For salesforce imports, the first and only element is the client credential id.
#' }
#' @param schedule object optional. A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }
#' @param notifications object optional. A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }
#' @param parent_id integer optional. Parent id to trigger this import from
#' @param next_run_at string optional. The time of the next scheduled run.
#' @param time_zone string optional. The time zone of this import.
#' 
#' @return  A list containing the following elements:
#' \item{name}{string, The name of the import.}
#' \item{syncType}{string, The type of sync to perform; one of Dbsync, AutoImport, SilverpopDataImport, SilverpopContactImport, GdocImport, GdocExport, and Salesforce.}
#' \item{source}{object, A list containing the following elements: 
#' \itemize{
#' \item remoteHostId integer, 
#' \item credentialId integer, 
#' \item additionalCredentials array, Array that holds additional credentials used for specific imports. For salesforce imports, the first and only element is the client credential id.
#' \item name string, 
#' }}
#' \item{destination}{object, A list containing the following elements: 
#' \itemize{
#' \item remoteHostId integer, 
#' \item credentialId integer, 
#' \item additionalCredentials array, Array that holds additional credentials used for specific imports. For salesforce imports, the first and only element is the client credential id.
#' \item name string, 
#' }}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{notifications}{object, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{parentId}{integer, Parent id to trigger this import from}
#' \item{id}{integer, The ID for the import.}
#' \item{isOutbound}{boolean, }
#' \item{jobType}{string, The job type of this import.}
#' \item{syncs}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, 
#' \item source object, 
#' \item destination object, 
#' \item advancedOptions object, 
#' }}
#' \item{state}{string, }
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{user}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{runningAs}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{nextRunAt}{string, The time of the next scheduled run.}
#' \item{timeZone}{string, The time zone of this import.}
#' \item{hidden}{boolean, The hidden status of the object.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' @export
imports_put <- function(id, name, sync_type, is_outbound, source = NULL, destination = NULL, schedule = NULL, notifications = NULL, parent_id = NULL, next_run_at = NULL, time_zone = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/imports/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(name = name, syncType = sync_type, isOutbound = is_outbound, source = source, destination = destination, schedule = schedule, notifications = notifications, parentId = parent_id, nextRunAt = next_run_at, timeZone = time_zone)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get the run history of this import
#' @param id integer required. 
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, }
#' \item{state}{string, }
#' \item{createdAt}{string, The time that the run was queued.}
#' \item{startedAt}{string, The time that the run started.}
#' \item{finishedAt}{string, The time that the run completed.}
#' \item{error}{string, The error message for this run, if present.}
#' @export
imports_list_runs <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/imports/{id}/runs"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Run an import
#' @param id integer required. The ID of the import to run.
#' 
#' @return  A list containing the following elements:
#' \item{runId}{integer, The ID of the new run triggered.}
#' @export
imports_post_runs <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/imports/{id}/runs"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' Cancel a run
#' @param id integer required. The ID of the job.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the run.}
#' \item{state}{string, The state of the run, one of 'queued', 'running' or 'cancelled'.}
#' \item{isCancelRequested}{boolean, True if run cancel requested, else false.}
#' @export
imports_post_cancel <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/imports/{id}/cancel"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' Create a sync
#' @param id integer required. 
#' @param source object required. A list containing the following elements: 
#' \itemize{
#' \item path string, The path of the dataset to sync from; for a database source, schema.tablename. If you are doing a Google Sheet export, this can be blank. This is a legacy parameter, it is recommended you use one of the following: databaseTable, file, googleWorksheet, salesforce, silverpop
#' \item databaseTable object, 
#' \item file object, 
#' \item googleWorksheet object, 
#' \item salesforce object, 
#' \item silverpop object, 
#' }
#' @param destination object required. A list containing the following elements: 
#' \itemize{
#' \item path string, The schema.tablename to sync to. If you are doing a Google Sheet export, this is the spreadsheet and sheet name separated by a period. i.e. if you have a spreadsheet named "MySpreadsheet" and a sheet called "Sheet1" this field would be "MySpreadsheet.Sheet1". This is a legacy parameter, it is recommended you use one of the following: databaseTable, googleWorksheet
#' \item databaseTable object, 
#' \item googleWorksheet object, 
#' }
#' @param advanced_options object optional. A list containing the following elements: 
#' \itemize{
#' \item maxErrors integer, 
#' \item existingTableRows string, 
#' \item diststyle string, 
#' \item distkey string, 
#' \item sortkey1 string, 
#' \item sortkey2 string, 
#' \item columnDelimiter string, 
#' \item columnOverrides object, Hash used for overriding auto-detected names and types, with keys being the index of the column being overridden.
#' \item identityColumn string, 
#' \item rowChunkSize integer, 
#' \item wipeDestinationTable boolean, 
#' \item truncateLongLines boolean, 
#' \item invalidCharReplacement string, 
#' \item verifyTableRowCounts boolean, 
#' \item partitionColumnName string, 
#' \item partitionSchemaName string, 
#' \item partitionTableName string, 
#' \item partitionTablePartitionColumnMinName string, 
#' \item partitionTablePartitionColumnMaxName string, 
#' \item lastModifiedColumn string, 
#' \item mysqlCatalogMatchesSchema boolean, 
#' \item chunkingMethod string, The method used to break the data into smaller chunks for transfer.  The value can be set to sorted_by_identity_columns or if not set the chunking method will be choosen automatically.
#' \item firstRowIsHeader boolean, 
#' \item exportAction string, The kind of export action you want to have the export execute. Set to "newsprsht" if you want a new worksheet inside a new spreadsheet. Set to "newwksht" if you want a new worksheet inside an existing spreadsheet. Set to "updatewksht" if you want to overwrite an existing worksheet inside an existing spreadsheet. Set to "appendwksht" if you want to append to the end of an existing worksheet inside an existing spreadsheet. 
#' \item sqlQuery string, If you are doing a Google Sheet export, this is your SQL query.
#' \item contactLists string, 
#' \item soqlQuery string, 
#' }
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, }
#' \item{source}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of the table or file, if available.
#' \item path string, The path of the dataset to sync from; for a database source, schema.tablename. If you are doing a Google Sheet export, this can be blank. This is a legacy parameter, it is recommended you use one of the following: databaseTable, file, googleWorksheet, salesforce, silverpop
#' \item databaseTable object, 
#' \item file object, 
#' \item googleWorksheet object, 
#' \item salesforce object, 
#' \item silverpop object, 
#' }}
#' \item{destination}{object, A list containing the following elements: 
#' \itemize{
#' \item path string, The schema.tablename to sync to. If you are doing a Google Sheet export, this is the spreadsheet and sheet name separated by a period. i.e. if you have a spreadsheet named "MySpreadsheet" and a sheet called "Sheet1" this field would be "MySpreadsheet.Sheet1". This is a legacy parameter, it is recommended you use one of the following: databaseTable, googleWorksheet
#' \item databaseTable object, 
#' \item googleWorksheet object, 
#' }}
#' \item{advancedOptions}{object, A list containing the following elements: 
#' \itemize{
#' \item maxErrors integer, 
#' \item existingTableRows string, 
#' \item diststyle string, 
#' \item distkey string, 
#' \item sortkey1 string, 
#' \item sortkey2 string, 
#' \item columnDelimiter string, 
#' \item columnOverrides object, Hash used for overriding auto-detected names and types, with keys being the index of the column being overridden.
#' \item identityColumn string, 
#' \item rowChunkSize integer, 
#' \item wipeDestinationTable boolean, 
#' \item truncateLongLines boolean, 
#' \item invalidCharReplacement string, 
#' \item verifyTableRowCounts boolean, 
#' \item partitionColumnName string, 
#' \item partitionSchemaName string, 
#' \item partitionTableName string, 
#' \item partitionTablePartitionColumnMinName string, 
#' \item partitionTablePartitionColumnMaxName string, 
#' \item lastModifiedColumn string, 
#' \item mysqlCatalogMatchesSchema boolean, 
#' \item chunkingMethod string, The method used to break the data into smaller chunks for transfer.  The value can be set to sorted_by_identity_columns or if not set the chunking method will be choosen automatically.
#' \item firstRowIsHeader boolean, 
#' \item exportAction string, The kind of export action you want to have the export execute. Set to "newsprsht" if you want a new worksheet inside a new spreadsheet. Set to "newwksht" if you want a new worksheet inside an existing spreadsheet. Set to "updatewksht" if you want to overwrite an existing worksheet inside an existing spreadsheet. Set to "appendwksht" if you want to append to the end of an existing worksheet inside an existing spreadsheet. 
#' \item sqlQuery string, If you are doing a Google Sheet export, this is your SQL query.
#' \item contactLists string, 
#' \item soqlQuery string, 
#' }}
#' @export
imports_post_syncs <- function(id, source, destination, advanced_options = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/imports/{id}/syncs"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(source = source, destination = destination, advancedOptions = advanced_options)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update a sync
#' @param id integer required. The ID of the import to fetch.
#' @param sync_id integer required. The ID of the sync to fetch.
#' @param source object required. A list containing the following elements: 
#' \itemize{
#' \item path string, The path of the dataset to sync from; for a database source, schema.tablename. If you are doing a Google Sheet export, this can be blank. This is a legacy parameter, it is recommended you use one of the following: databaseTable, file, googleWorksheet, salesforce, silverpop
#' \item databaseTable object, 
#' \item file object, 
#' \item googleWorksheet object, 
#' \item salesforce object, 
#' \item silverpop object, 
#' }
#' @param destination object required. A list containing the following elements: 
#' \itemize{
#' \item path string, The schema.tablename to sync to. If you are doing a Google Sheet export, this is the spreadsheet and sheet name separated by a period. i.e. if you have a spreadsheet named "MySpreadsheet" and a sheet called "Sheet1" this field would be "MySpreadsheet.Sheet1". This is a legacy parameter, it is recommended you use one of the following: databaseTable, googleWorksheet
#' \item databaseTable object, 
#' \item googleWorksheet object, 
#' }
#' @param advanced_options object optional. A list containing the following elements: 
#' \itemize{
#' \item maxErrors integer, 
#' \item existingTableRows string, 
#' \item diststyle string, 
#' \item distkey string, 
#' \item sortkey1 string, 
#' \item sortkey2 string, 
#' \item columnDelimiter string, 
#' \item columnOverrides object, Hash used for overriding auto-detected names and types, with keys being the index of the column being overridden.
#' \item identityColumn string, 
#' \item rowChunkSize integer, 
#' \item wipeDestinationTable boolean, 
#' \item truncateLongLines boolean, 
#' \item invalidCharReplacement string, 
#' \item verifyTableRowCounts boolean, 
#' \item partitionColumnName string, 
#' \item partitionSchemaName string, 
#' \item partitionTableName string, 
#' \item partitionTablePartitionColumnMinName string, 
#' \item partitionTablePartitionColumnMaxName string, 
#' \item lastModifiedColumn string, 
#' \item mysqlCatalogMatchesSchema boolean, 
#' \item chunkingMethod string, The method used to break the data into smaller chunks for transfer.  The value can be set to sorted_by_identity_columns or if not set the chunking method will be choosen automatically.
#' \item firstRowIsHeader boolean, 
#' \item exportAction string, The kind of export action you want to have the export execute. Set to "newsprsht" if you want a new worksheet inside a new spreadsheet. Set to "newwksht" if you want a new worksheet inside an existing spreadsheet. Set to "updatewksht" if you want to overwrite an existing worksheet inside an existing spreadsheet. Set to "appendwksht" if you want to append to the end of an existing worksheet inside an existing spreadsheet. 
#' \item sqlQuery string, If you are doing a Google Sheet export, this is your SQL query.
#' \item contactLists string, 
#' \item soqlQuery string, 
#' }
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, }
#' \item{source}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of the table or file, if available.
#' \item path string, The path of the dataset to sync from; for a database source, schema.tablename. If you are doing a Google Sheet export, this can be blank. This is a legacy parameter, it is recommended you use one of the following: databaseTable, file, googleWorksheet, salesforce, silverpop
#' \item databaseTable object, 
#' \item file object, 
#' \item googleWorksheet object, 
#' \item salesforce object, 
#' \item silverpop object, 
#' }}
#' \item{destination}{object, A list containing the following elements: 
#' \itemize{
#' \item path string, The schema.tablename to sync to. If you are doing a Google Sheet export, this is the spreadsheet and sheet name separated by a period. i.e. if you have a spreadsheet named "MySpreadsheet" and a sheet called "Sheet1" this field would be "MySpreadsheet.Sheet1". This is a legacy parameter, it is recommended you use one of the following: databaseTable, googleWorksheet
#' \item databaseTable object, 
#' \item googleWorksheet object, 
#' }}
#' \item{advancedOptions}{object, A list containing the following elements: 
#' \itemize{
#' \item maxErrors integer, 
#' \item existingTableRows string, 
#' \item diststyle string, 
#' \item distkey string, 
#' \item sortkey1 string, 
#' \item sortkey2 string, 
#' \item columnDelimiter string, 
#' \item columnOverrides object, Hash used for overriding auto-detected names and types, with keys being the index of the column being overridden.
#' \item identityColumn string, 
#' \item rowChunkSize integer, 
#' \item wipeDestinationTable boolean, 
#' \item truncateLongLines boolean, 
#' \item invalidCharReplacement string, 
#' \item verifyTableRowCounts boolean, 
#' \item partitionColumnName string, 
#' \item partitionSchemaName string, 
#' \item partitionTableName string, 
#' \item partitionTablePartitionColumnMinName string, 
#' \item partitionTablePartitionColumnMaxName string, 
#' \item lastModifiedColumn string, 
#' \item mysqlCatalogMatchesSchema boolean, 
#' \item chunkingMethod string, The method used to break the data into smaller chunks for transfer.  The value can be set to sorted_by_identity_columns or if not set the chunking method will be choosen automatically.
#' \item firstRowIsHeader boolean, 
#' \item exportAction string, The kind of export action you want to have the export execute. Set to "newsprsht" if you want a new worksheet inside a new spreadsheet. Set to "newwksht" if you want a new worksheet inside an existing spreadsheet. Set to "updatewksht" if you want to overwrite an existing worksheet inside an existing spreadsheet. Set to "appendwksht" if you want to append to the end of an existing worksheet inside an existing spreadsheet. 
#' \item sqlQuery string, If you are doing a Google Sheet export, this is your SQL query.
#' \item contactLists string, 
#' \item soqlQuery string, 
#' }}
#' @export
imports_put_syncs <- function(id, sync_id, source, destination, advanced_options = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/imports/{id}/syncs/{sync_id}"
  path_params  <- list(id = id, sync_id = sync_id)
  query_params <- list()
  body_params  <- list(source = source, destination = destination, advancedOptions = advanced_options)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Archive a sync (deprecated, use the /archive endpoint instead)
#' @param id integer required. The ID of the import to fetch.
#' @param sync_id integer required. The ID of the sync to fetch.
#' 
#' @return  An empty HTTP response
#' @export
imports_delete_syncs <- function(id, sync_id) {

  args <- as.list(match.call())[-1]
  path <- "/imports/{id}/syncs/{sync_id}"
  path_params  <- list(id = id, sync_id = sync_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update the archive status of this sync
#' @param id integer required. The ID of the import to fetch.
#' @param sync_id integer required. The ID of the sync to fetch.
#' @param status boolean optional. The desired archived status of the sync.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, }
#' \item{source}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of the table or file, if available.
#' \item path string, The path of the dataset to sync from; for a database source, schema.tablename. If you are doing a Google Sheet export, this can be blank. This is a legacy parameter, it is recommended you use one of the following: databaseTable, file, googleWorksheet, salesforce, silverpop
#' \item databaseTable object, 
#' \item file object, 
#' \item googleWorksheet object, 
#' \item salesforce object, 
#' \item silverpop object, 
#' }}
#' \item{destination}{object, A list containing the following elements: 
#' \itemize{
#' \item path string, The schema.tablename to sync to. If you are doing a Google Sheet export, this is the spreadsheet and sheet name separated by a period. i.e. if you have a spreadsheet named "MySpreadsheet" and a sheet called "Sheet1" this field would be "MySpreadsheet.Sheet1". This is a legacy parameter, it is recommended you use one of the following: databaseTable, googleWorksheet
#' \item databaseTable object, 
#' \item googleWorksheet object, 
#' }}
#' \item{advancedOptions}{object, A list containing the following elements: 
#' \itemize{
#' \item maxErrors integer, 
#' \item existingTableRows string, 
#' \item diststyle string, 
#' \item distkey string, 
#' \item sortkey1 string, 
#' \item sortkey2 string, 
#' \item columnDelimiter string, 
#' \item columnOverrides object, Hash used for overriding auto-detected names and types, with keys being the index of the column being overridden.
#' \item identityColumn string, 
#' \item rowChunkSize integer, 
#' \item wipeDestinationTable boolean, 
#' \item truncateLongLines boolean, 
#' \item invalidCharReplacement string, 
#' \item verifyTableRowCounts boolean, 
#' \item partitionColumnName string, 
#' \item partitionSchemaName string, 
#' \item partitionTableName string, 
#' \item partitionTablePartitionColumnMinName string, 
#' \item partitionTablePartitionColumnMaxName string, 
#' \item lastModifiedColumn string, 
#' \item mysqlCatalogMatchesSchema boolean, 
#' \item chunkingMethod string, The method used to break the data into smaller chunks for transfer.  The value can be set to sorted_by_identity_columns or if not set the chunking method will be choosen automatically.
#' \item firstRowIsHeader boolean, 
#' \item exportAction string, The kind of export action you want to have the export execute. Set to "newsprsht" if you want a new worksheet inside a new spreadsheet. Set to "newwksht" if you want a new worksheet inside an existing spreadsheet. Set to "updatewksht" if you want to overwrite an existing worksheet inside an existing spreadsheet. Set to "appendwksht" if you want to append to the end of an existing worksheet inside an existing spreadsheet. 
#' \item sqlQuery string, If you are doing a Google Sheet export, this is your SQL query.
#' \item contactLists string, 
#' \item soqlQuery string, 
#' }}
#' @export
imports_put_syncs_archive <- function(id, sync_id, status = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/imports/{id}/syncs/{sync_id}/archive"
  path_params  <- list(id = id, sync_id = sync_id)
  query_params <- list()
  body_params  <- list(status = status)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' List 
#' @param state string optional. The job's state. One or more of queued, running, succeeded, failed, and cancelled. Specify multiple values as a comma-separated list (e.g., "A,B").
#' @param type string optional. The job's type. Specify multiple values as a comma-separated list (e.g., "A,B").
#' @param q string optional. Query string to search on the id, name, and job type
#' @param permission string optional. A permissions string, one of "read", "write", or "manage". Lists only jobs for which the current user has that permission.
#' @param scheduled boolean optional. If the object is scheduled.
#' @param hidden boolean optional. If specified to be true, returns hidden objects. Defaults to false, returning non-hidden objects.
#' @param archived string optional. The archival status of the requested object(s).
#' @param limit integer optional. Number of results to return. Defaults to its maximum of 50.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to updated_at. Must be one of: updated_at.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, }
#' \item{name}{string, }
#' \item{type}{string, }
#' \item{fromTemplateId}{integer, }
#' \item{state}{string, Whether the job is idle, queued, running, cancelled, or failed.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' @export
jobs_list <- function(state = NULL, type = NULL, q = NULL, permission = NULL, scheduled = NULL, hidden = NULL, archived = NULL, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/jobs/"
  path_params  <- list()
  query_params <- list(state = state, type = type, q = q, permission = permission, scheduled = scheduled, hidden = hidden, archived = archived, limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Show basic job info
#' @param id integer required. The ID for this job.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, }
#' \item{name}{string, }
#' \item{type}{string, }
#' \item{fromTemplateId}{integer, }
#' \item{state}{string, Whether the job is idle, queued, running, cancelled, or failed.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{runs}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{hidden}{boolean, The hidden status of the object.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' @export
jobs_get <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/jobs/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Generate and retrieve trigger email address
#' @param id integer required. The ID for this job.
#' 
#' @return  A list containing the following elements:
#' \item{triggerEmail}{string, Email address which may be used to trigger this job to run.}
#' @export
jobs_post_trigger_email <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/jobs/{id}/trigger_email"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' Show chain of parents as a list that this job triggers from
#' @param id integer required. The ID for this job.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, }
#' \item{name}{string, }
#' \item{type}{string, }
#' \item{fromTemplateId}{integer, }
#' \item{state}{string, Whether the job is idle, queued, running, cancelled, or failed.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{runs}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{hidden}{boolean, The hidden status of the object.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' @export
jobs_list_parents <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/jobs/{id}/parents"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Show nested tree of children that this job triggers
#' @param id integer required. The ID for this job.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, }
#' \item{name}{string, }
#' \item{type}{string, }
#' \item{fromTemplateId}{integer, }
#' \item{state}{string, }
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{runs}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{children}{array, }
#' @export
jobs_list_children <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/jobs/{id}/children"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Run a job
#' @param id integer required. The ID for this job.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, }
#' \item{state}{string, }
#' \item{createdAt}{string, The time that the run was queued.}
#' \item{startedAt}{string, The time that the run started.}
#' \item{finishedAt}{string, The time that the run completed.}
#' \item{error}{string, The error message for this run, if present.}
#' @export
jobs_post_runs <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/jobs/{id}/runs"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' Check status of a job
#' @param id integer required. The ID of the Job.
#' @param run_id integer required. The ID of the Run.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, }
#' \item{state}{string, }
#' \item{createdAt}{string, The time that the run was queued.}
#' \item{startedAt}{string, The time that the run started.}
#' \item{finishedAt}{string, The time that the run completed.}
#' \item{error}{string, The error message for this run, if present.}
#' @export
jobs_get_runs <- function(id, run_id) {

  args <- as.list(match.call())[-1]
  path <- "/jobs/{id}/runs/{run_id}"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' List the workflows a job belongs to
#' @param id integer required. 
#' @param archived string optional. The archival status of the requested object(s).
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID for this workflow.}
#' \item{name}{string, The name of this workflow.}
#' \item{description}{string, A description of the workflow.}
#' \item{valid}{boolean, The validity of the workflow definition.}
#' \item{fileId}{string, The file id for the s3 file containing the workflow configuration.}
#' \item{user}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The state of the workflow. State is "running" if any execution is running, otherwise reflects most recent execution state.}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{timeZone}{string, The time zone of this workflow.}
#' \item{nextExecutionAt}{string, The time of the next scheduled execution.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' @export
jobs_list_workflows <- function(id, archived = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/jobs/{id}/workflows"
  path_params  <- list(id = id)
  query_params <- list(archived = archived)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' List users and groups permissioned on this object
#' @param id integer required. The ID of the object.
#' 
#' @return  An array containing the following fields:
#' \item{readers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
jobs_list_shares <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/jobs/{id}/shares"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions users have on this object
#' @param id integer required. ID of the resource to be shared
#' @param user_ids array required. An array of one or more user IDs
#' @param permission_level string required. Options are: "read", "write", or "manage"
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
jobs_put_shares_users <- function(id, user_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/jobs/{id}/shares/users"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(userIds = user_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a user has on this object
#' @param id integer required. ID of the resource to be revoked
#' @param user_id integer required. ID of the user
#' 
#' @return  An empty HTTP response
#' @export
jobs_delete_shares_users <- function(id, user_id) {

  args <- as.list(match.call())[-1]
  path <- "/jobs/{id}/shares/users/{user_id}"
  path_params  <- list(id = id, user_id = user_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions groups has on this object
#' @param id integer required. ID of the resource to be shared
#' @param group_ids array required. An array of one or more group IDs
#' @param permission_level string required. Options are: "read", "write", or "manage"
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
jobs_put_shares_groups <- function(id, group_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/jobs/{id}/shares/groups"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(groupIds = group_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a group has on this object
#' @param id integer required. ID of the resource to be revoked
#' @param group_id integer required. ID of the group
#' 
#' @return  An empty HTTP response
#' @export
jobs_delete_shares_groups <- function(id, group_id) {

  args <- as.list(match.call())[-1]
  path <- "/jobs/{id}/shares/groups/{group_id}"
  path_params  <- list(id = id, group_id = group_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' List the projects a Job belongs to
#' @param id integer required. The ID of the resource.
#' @param hidden boolean optional. If specified to be true, returns hidden objects. Defaults to false, returning non-hidden objects.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID for this project.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{name}{string, The name of this project.}
#' \item{description}{string, A description of the project}
#' \item{users}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{autoShare}{boolean, }
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{archived}{string, The archival status of the requested object(s).}
#' @export
jobs_list_projects <- function(id, hidden = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/jobs/{id}/projects"
  path_params  <- list(id = id)
  query_params <- list(hidden = hidden)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Add a Job to a project
#' @param id integer required. ID of the resource
#' @param project_id integer required. The ID of the project
#' 
#' @return  An empty HTTP response
#' @export
jobs_put_projects <- function(id, project_id) {

  args <- as.list(match.call())[-1]
  path <- "/jobs/{id}/projects/{project_id}"
  path_params  <- list(id = id, project_id = project_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Remove a Job from a project
#' @param id integer required. ID of the resource
#' @param project_id integer required. The ID of the project
#' 
#' @return  An empty HTTP response
#' @export
jobs_delete_projects <- function(id, project_id) {

  args <- as.list(match.call())[-1]
  path <- "/jobs/{id}/projects/{project_id}"
  path_params  <- list(id = id, project_id = project_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' List match targets
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of the match target}
#' \item{name}{string, The name of the match target}
#' \item{table}{string, The name of the DynamoDB table}
#' \item{encryptionKeyCredentialId}{integer, The ID of the encryption key credential.}
#' @export
match_targets_list <- function() {

  args <- as.list(match.call())[-1]
  path <- "/match_targets/"
  path_params  <- list()
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Create a new match target
#' @param name string required. The name of the match target
#' @param table string required. The name of the DynamoDB table
#' @param id integer optional. The ID of the match target
#' @param region string optional. The AWS region of the DynamoDB table. By default, us-east-1.
#' @param encryption_key_credential_id integer optional. The ID of the encryption key credential.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the match target}
#' \item{name}{string, The name of the match target}
#' \item{table}{string, The name of the DynamoDB table}
#' \item{region}{string, The AWS region of the DynamoDB table. By default, us-east-1.}
#' \item{encryptionKeyCredentialId}{integer, The ID of the encryption key credential.}
#' @export
match_targets_post <- function(name, table, id = NULL, region = NULL, encryption_key_credential_id = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/match_targets/"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(name = name, table = table, id = id, region = region, encryptionKeyCredentialId = encryption_key_credential_id)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' Show Match Target info
#' @param id integer required. The ID of the match target
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the match target}
#' \item{name}{string, The name of the match target}
#' \item{table}{string, The name of the DynamoDB table}
#' \item{region}{string, The AWS region of the DynamoDB table. By default, us-east-1.}
#' \item{encryptionKeyCredentialId}{integer, The ID of the encryption key credential.}
#' @export
match_targets_get <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/match_targets/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update a match target
#' @param id integer required. The ID of the match target
#' @param name string optional. The name of the match target
#' @param table string optional. The name of the DynamoDB table
#' @param region string optional. The AWS region of the DynamoDB table. By default, us-east-1.
#' @param encryption_key_credential_id integer optional. The ID of the encryption key credential.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the match target}
#' \item{name}{string, The name of the match target}
#' \item{table}{string, The name of the DynamoDB table}
#' \item{region}{string, The AWS region of the DynamoDB table. By default, us-east-1.}
#' \item{encryptionKeyCredentialId}{integer, The ID of the encryption key credential.}
#' @export
match_targets_patch <- function(id, name = NULL, table = NULL, region = NULL, encryption_key_credential_id = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/match_targets/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(name = name, table = table, region = region, encryptionKeyCredentialId = encryption_key_credential_id)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("patch", path, path_params, query_params, body_params)

  return(resp)

 }


#' List users and groups permissioned on this object
#' @param id integer required. The ID of the object.
#' 
#' @return  An array containing the following fields:
#' \item{readers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
media_list_spot_orders_shares <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/media/spot_orders/{id}/shares"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions users have on this object
#' @param id integer required. ID of the resource to be shared
#' @param user_ids array required. An array of one or more user IDs
#' @param permission_level string required. Options are: "read", "write", or "manage"
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
media_put_spot_orders_shares_users <- function(id, user_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/media/spot_orders/{id}/shares/users"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(userIds = user_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a user has on this object
#' @param id integer required. ID of the resource to be revoked
#' @param user_id integer required. ID of the user
#' 
#' @return  An empty HTTP response
#' @export
media_delete_spot_orders_shares_users <- function(id, user_id) {

  args <- as.list(match.call())[-1]
  path <- "/media/spot_orders/{id}/shares/users/{user_id}"
  path_params  <- list(id = id, user_id = user_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions groups has on this object
#' @param id integer required. ID of the resource to be shared
#' @param group_ids array required. An array of one or more group IDs
#' @param permission_level string required. Options are: "read", "write", or "manage"
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
media_put_spot_orders_shares_groups <- function(id, group_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/media/spot_orders/{id}/shares/groups"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(groupIds = group_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a group has on this object
#' @param id integer required. ID of the resource to be revoked
#' @param group_id integer required. ID of the group
#' 
#' @return  An empty HTTP response
#' @export
media_delete_spot_orders_shares_groups <- function(id, group_id) {

  args <- as.list(match.call())[-1]
  path <- "/media/spot_orders/{id}/shares/groups/{group_id}"
  path_params  <- list(id = id, group_id = group_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update the archive status of this object
#' @param id integer required. The ID of the object.
#' @param status boolean required. The desired archived status of the object.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the spot order.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{csvS3Uri}{string, S3 URI for the spot order CSV file.}
#' \item{jsonS3Uri}{string, S3 URI for the spot order JSON file.}
#' \item{xmlArchiveS3Uri}{string, S3 URI for the spot order XML archive.}
#' \item{lastTransformJobId}{integer, ID of the spot order transformation job.}
#' @export
media_put_spot_orders_archive <- function(id, status) {

  args <- as.list(match.call())[-1]
  path <- "/media/spot_orders/{id}/archive"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(status = status)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' List users and groups permissioned on this object
#' @param id integer required. The ID of the object.
#' 
#' @return  An array containing the following fields:
#' \item{readers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
media_list_optimizations_shares <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/media/optimizations/{id}/shares"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions users have on this object
#' @param id integer required. ID of the resource to be shared
#' @param user_ids array required. An array of one or more user IDs
#' @param permission_level string required. Options are: "read", "write", or "manage"
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
media_put_optimizations_shares_users <- function(id, user_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/media/optimizations/{id}/shares/users"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(userIds = user_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a user has on this object
#' @param id integer required. ID of the resource to be revoked
#' @param user_id integer required. ID of the user
#' 
#' @return  An empty HTTP response
#' @export
media_delete_optimizations_shares_users <- function(id, user_id) {

  args <- as.list(match.call())[-1]
  path <- "/media/optimizations/{id}/shares/users/{user_id}"
  path_params  <- list(id = id, user_id = user_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions groups has on this object
#' @param id integer required. ID of the resource to be shared
#' @param group_ids array required. An array of one or more group IDs
#' @param permission_level string required. Options are: "read", "write", or "manage"
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
media_put_optimizations_shares_groups <- function(id, group_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/media/optimizations/{id}/shares/groups"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(groupIds = group_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a group has on this object
#' @param id integer required. ID of the resource to be revoked
#' @param group_id integer required. ID of the group
#' 
#' @return  An empty HTTP response
#' @export
media_delete_optimizations_shares_groups <- function(id, group_id) {

  args <- as.list(match.call())[-1]
  path <- "/media/optimizations/{id}/shares/groups/{group_id}"
  path_params  <- list(id = id, group_id = group_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update the archive status of this object
#' @param id integer required. The ID of the object.
#' @param status boolean required. The desired archived status of the object.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The optimization ID.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{name}{string, The name of the optimization.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{finishedAt}{string, The end time of the last run.}
#' \item{state}{string, The state of the last run.}
#' \item{lastRunId}{integer, The ID of the last run.}
#' \item{spotOrderId}{integer, The ID for the spot order produced by the optimization.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{reportLink}{string, A link to the visual report for the optimization.}
#' \item{spotOrderLink}{string, A link to the json version of the spot order.}
#' \item{fileLinks}{array, Links to the csv and xml versions of the spot order.}
#' \item{runs}{array, An array containing the following fields: 
#' \itemize{
#' \item marketId integer, The market ID.
#' \item startDate string, The start date for the media run.
#' \item endDate string, The end date for the media run.
#' \item forceCpm boolean, Whether to force optimization to use CPM data even if partition data is available.
#' \item syscodes array, The syscodes for the media run.
#' \item rateCards array, The ratecards for the media run.
#' \item constraints array, The constraints for the media run.
#' }}
#' \item{programs}{array, An array of programs that the Civis Media Optimizer either exclude or limit to.An error will be thrown if exclude_programs is not also set.}
#' \item{networks}{array, An array of networks that the Civis Media Optimizer either exclude or limit to.An error will be thrown if exclude_networks is not also set.}
#' \item{excludePrograms}{boolean, If Civis Media Optimizer should exclude the programs in the programs parameter.If this value is set to false, it will make the optimization limit itself to the programs supplied through the programs parameter.An error will be thrown if programs is not also set.}
#' \item{excludeNetworks}{boolean, If Civis Media Optimizer should exclude the networks in the networks parameter.If this value is set to false, it will make the optimization limit itself to the networks supplied through the networks.An error will be thrown if networks is not also set.}
#' \item{timeSlotPercentages}{object, The maximum amount of the budget spent on that particular day of the week, daypart, or specific time slot for broadcast and cable.}
#' @export
media_put_optimizations_archive <- function(id, status) {

  args <- as.list(match.call())[-1]
  path <- "/media/optimizations/{id}/archive"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(status = status)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' List users and groups permissioned on this object
#' @param id integer required. The ID of the object.
#' 
#' @return  An array containing the following fields:
#' \item{readers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
media_list_ratecards_shares <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/media/ratecards/{id}/shares"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions users have on this object
#' @param id integer required. ID of the resource to be shared
#' @param user_ids array required. An array of one or more user IDs
#' @param permission_level string required. Options are: "read", "write", or "manage"
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
media_put_ratecards_shares_users <- function(id, user_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/media/ratecards/{id}/shares/users"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(userIds = user_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a user has on this object
#' @param id integer required. ID of the resource to be revoked
#' @param user_id integer required. ID of the user
#' 
#' @return  An empty HTTP response
#' @export
media_delete_ratecards_shares_users <- function(id, user_id) {

  args <- as.list(match.call())[-1]
  path <- "/media/ratecards/{id}/shares/users/{user_id}"
  path_params  <- list(id = id, user_id = user_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions groups has on this object
#' @param id integer required. ID of the resource to be shared
#' @param group_ids array required. An array of one or more group IDs
#' @param permission_level string required. Options are: "read", "write", or "manage"
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
media_put_ratecards_shares_groups <- function(id, group_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/media/ratecards/{id}/shares/groups"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(groupIds = group_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a group has on this object
#' @param id integer required. ID of the resource to be revoked
#' @param group_id integer required. ID of the group
#' 
#' @return  An empty HTTP response
#' @export
media_delete_ratecards_shares_groups <- function(id, group_id) {

  args <- as.list(match.call())[-1]
  path <- "/media/ratecards/{id}/shares/groups/{group_id}"
  path_params  <- list(id = id, group_id = group_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update the archive status of this object
#' @param id integer required. The ID of the object.
#' @param status boolean required. The desired archived status of the object.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ratecard ID.}
#' \item{filename}{string, Name of the ratecard file.}
#' \item{startOn}{string, First day to which the ratecard applies.}
#' \item{endOn}{string, Last day to which the ratecard applies.}
#' \item{dmaNumber}{integer, Number of the DMA associated with the ratecard.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' @export
media_put_ratecards_archive <- function(id, status) {

  args <- as.list(match.call())[-1]
  path <- "/media/ratecards/{id}/archive"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(status = status)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' List all optimizations
#' @param archived string optional. The archival status of the requested object(s).
#' @param limit integer optional. Number of results to return. Defaults to 20. Maximum allowed is 50.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to created_at. Must be one of: created_at, author, name.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The optimization ID.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{name}{string, The name of the optimization.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{finishedAt}{string, The end time of the last run.}
#' \item{state}{string, The state of the last run.}
#' \item{lastRunId}{integer, The ID of the last run.}
#' \item{spotOrderId}{integer, The ID for the spot order produced by the optimization.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' @export
media_list_optimizations <- function(archived = NULL, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/media/optimizations"
  path_params  <- list()
  query_params <- list(archived = archived, limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Create a new optimization
#' @param runs array required. An array containing the following fields: 
#' \itemize{
#' \item marketId integer, The market ID.
#' \item startDate string, The start date for the media run.
#' \item endDate string, The end date for the media run.
#' \item forceCpm boolean, Whether to force optimization to use CPM data even if partition data is available.
#' \item syscodes array, The syscodes for the media run.
#' \item rateCards array, The ratecards for the media run.
#' \item constraints array, The constraints for the media run.
#' }
#' @param name string optional. The name of the optimization.
#' @param programs array optional. An array of programs that the Civis Media Optimizer either exclude or limit to.An error will be thrown if exclude_programs is not also set.
#' @param networks array optional. An array of networks that the Civis Media Optimizer either exclude or limit to.An error will be thrown if exclude_networks is not also set.
#' @param exclude_programs boolean optional. If Civis Media Optimizer should exclude the programs in the programs parameter.If this value is set to false, it will make the optimization limit itself to the programs supplied through the programs parameter.An error will be thrown if programs is not also set.
#' @param exclude_networks boolean optional. If Civis Media Optimizer should exclude the networks in the networks parameter.If this value is set to false, it will make the optimization limit itself to the networks supplied through the networks.An error will be thrown if networks is not also set.
#' @param time_slot_percentages object optional. The maximum amount of the budget spent on that particular day of the week, daypart, or specific time slot for broadcast and cable.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The optimization ID.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{name}{string, The name of the optimization.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{finishedAt}{string, The end time of the last run.}
#' \item{state}{string, The state of the last run.}
#' \item{lastRunId}{integer, The ID of the last run.}
#' \item{spotOrderId}{integer, The ID for the spot order produced by the optimization.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{reportLink}{string, A link to the visual report for the optimization.}
#' \item{spotOrderLink}{string, A link to the json version of the spot order.}
#' \item{fileLinks}{array, Links to the csv and xml versions of the spot order.}
#' \item{runs}{array, An array containing the following fields: 
#' \itemize{
#' \item marketId integer, The market ID.
#' \item startDate string, The start date for the media run.
#' \item endDate string, The end date for the media run.
#' \item forceCpm boolean, Whether to force optimization to use CPM data even if partition data is available.
#' \item syscodes array, The syscodes for the media run.
#' \item rateCards array, The ratecards for the media run.
#' \item constraints array, The constraints for the media run.
#' }}
#' \item{programs}{array, An array of programs that the Civis Media Optimizer either exclude or limit to.An error will be thrown if exclude_programs is not also set.}
#' \item{networks}{array, An array of networks that the Civis Media Optimizer either exclude or limit to.An error will be thrown if exclude_networks is not also set.}
#' \item{excludePrograms}{boolean, If Civis Media Optimizer should exclude the programs in the programs parameter.If this value is set to false, it will make the optimization limit itself to the programs supplied through the programs parameter.An error will be thrown if programs is not also set.}
#' \item{excludeNetworks}{boolean, If Civis Media Optimizer should exclude the networks in the networks parameter.If this value is set to false, it will make the optimization limit itself to the networks supplied through the networks.An error will be thrown if networks is not also set.}
#' \item{timeSlotPercentages}{object, The maximum amount of the budget spent on that particular day of the week, daypart, or specific time slot for broadcast and cable.}
#' @export
media_post_optimizations <- function(runs, name = NULL, programs = NULL, networks = NULL, exclude_programs = NULL, exclude_networks = NULL, time_slot_percentages = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/media/optimizations"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(runs = runs, name = name, programs = programs, networks = networks, excludePrograms = exclude_programs, excludeNetworks = exclude_networks, timeSlotPercentages = time_slot_percentages)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' Show a single optimization
#' @param id integer required. The optimization ID.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The optimization ID.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{name}{string, The name of the optimization.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{finishedAt}{string, The end time of the last run.}
#' \item{state}{string, The state of the last run.}
#' \item{lastRunId}{integer, The ID of the last run.}
#' \item{spotOrderId}{integer, The ID for the spot order produced by the optimization.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{reportLink}{string, A link to the visual report for the optimization.}
#' \item{spotOrderLink}{string, A link to the json version of the spot order.}
#' \item{fileLinks}{array, Links to the csv and xml versions of the spot order.}
#' \item{runs}{array, An array containing the following fields: 
#' \itemize{
#' \item marketId integer, The market ID.
#' \item startDate string, The start date for the media run.
#' \item endDate string, The end date for the media run.
#' \item forceCpm boolean, Whether to force optimization to use CPM data even if partition data is available.
#' \item syscodes array, The syscodes for the media run.
#' \item rateCards array, The ratecards for the media run.
#' \item constraints array, The constraints for the media run.
#' }}
#' \item{programs}{array, An array of programs that the Civis Media Optimizer either exclude or limit to.An error will be thrown if exclude_programs is not also set.}
#' \item{networks}{array, An array of networks that the Civis Media Optimizer either exclude or limit to.An error will be thrown if exclude_networks is not also set.}
#' \item{excludePrograms}{boolean, If Civis Media Optimizer should exclude the programs in the programs parameter.If this value is set to false, it will make the optimization limit itself to the programs supplied through the programs parameter.An error will be thrown if programs is not also set.}
#' \item{excludeNetworks}{boolean, If Civis Media Optimizer should exclude the networks in the networks parameter.If this value is set to false, it will make the optimization limit itself to the networks supplied through the networks.An error will be thrown if networks is not also set.}
#' \item{timeSlotPercentages}{object, The maximum amount of the budget spent on that particular day of the week, daypart, or specific time slot for broadcast and cable.}
#' @export
media_get_optimizations <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/media/optimizations/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Edit an existing optimization
#' @param id integer required. The optimization ID.
#' @param name string optional. The name of the optimization.
#' @param runs array optional. An array containing the following fields: 
#' \itemize{
#' \item marketId integer, The market ID.
#' \item startDate string, The start date for the media run.
#' \item endDate string, The end date for the media run.
#' \item forceCpm boolean, Whether to force optimization to use CPM data even if partition data is available.
#' \item syscodes array, The syscodes for the media run.
#' \item rateCards array, The ratecards for the media run.
#' \item constraints array, The constraints for the media run.
#' }
#' @param programs array optional. An array of programs that the Civis Media Optimizer either exclude or limit to.An error will be thrown if exclude_programs is not also set.
#' @param networks array optional. An array of networks that the Civis Media Optimizer either exclude or limit to.An error will be thrown if exclude_networks is not also set.
#' @param exclude_programs boolean optional. If Civis Media Optimizer should exclude the programs in the programs parameter.If this value is set to false, it will make the optimization limit itself to the programs supplied through the programs parameter.An error will be thrown if programs is not also set.
#' @param exclude_networks boolean optional. If Civis Media Optimizer should exclude the networks in the networks parameter.If this value is set to false, it will make the optimization limit itself to the networks supplied through the networks.An error will be thrown if networks is not also set.
#' @param time_slot_percentages object optional. The maximum amount of the budget spent on that particular day of the week, daypart, or specific time slot for broadcast and cable.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The optimization ID.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{name}{string, The name of the optimization.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{finishedAt}{string, The end time of the last run.}
#' \item{state}{string, The state of the last run.}
#' \item{lastRunId}{integer, The ID of the last run.}
#' \item{spotOrderId}{integer, The ID for the spot order produced by the optimization.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{reportLink}{string, A link to the visual report for the optimization.}
#' \item{spotOrderLink}{string, A link to the json version of the spot order.}
#' \item{fileLinks}{array, Links to the csv and xml versions of the spot order.}
#' \item{runs}{array, An array containing the following fields: 
#' \itemize{
#' \item marketId integer, The market ID.
#' \item startDate string, The start date for the media run.
#' \item endDate string, The end date for the media run.
#' \item forceCpm boolean, Whether to force optimization to use CPM data even if partition data is available.
#' \item syscodes array, The syscodes for the media run.
#' \item rateCards array, The ratecards for the media run.
#' \item constraints array, The constraints for the media run.
#' }}
#' \item{programs}{array, An array of programs that the Civis Media Optimizer either exclude or limit to.An error will be thrown if exclude_programs is not also set.}
#' \item{networks}{array, An array of networks that the Civis Media Optimizer either exclude or limit to.An error will be thrown if exclude_networks is not also set.}
#' \item{excludePrograms}{boolean, If Civis Media Optimizer should exclude the programs in the programs parameter.If this value is set to false, it will make the optimization limit itself to the programs supplied through the programs parameter.An error will be thrown if programs is not also set.}
#' \item{excludeNetworks}{boolean, If Civis Media Optimizer should exclude the networks in the networks parameter.If this value is set to false, it will make the optimization limit itself to the networks supplied through the networks.An error will be thrown if networks is not also set.}
#' \item{timeSlotPercentages}{object, The maximum amount of the budget spent on that particular day of the week, daypart, or specific time slot for broadcast and cable.}
#' @export
media_patch_optimizations <- function(id, name = NULL, runs = NULL, programs = NULL, networks = NULL, exclude_programs = NULL, exclude_networks = NULL, time_slot_percentages = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/media/optimizations/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(name = name, runs = runs, programs = programs, networks = networks, excludePrograms = exclude_programs, excludeNetworks = exclude_networks, timeSlotPercentages = time_slot_percentages)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("patch", path, path_params, query_params, body_params)

  return(resp)

 }


#' Clone an existing optimization
#' @param id integer required. The optimization ID.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The optimization ID.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{name}{string, The name of the optimization.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{finishedAt}{string, The end time of the last run.}
#' \item{state}{string, The state of the last run.}
#' \item{lastRunId}{integer, The ID of the last run.}
#' \item{spotOrderId}{integer, The ID for the spot order produced by the optimization.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{reportLink}{string, A link to the visual report for the optimization.}
#' \item{spotOrderLink}{string, A link to the json version of the spot order.}
#' \item{fileLinks}{array, Links to the csv and xml versions of the spot order.}
#' \item{runs}{array, An array containing the following fields: 
#' \itemize{
#' \item marketId integer, The market ID.
#' \item startDate string, The start date for the media run.
#' \item endDate string, The end date for the media run.
#' \item forceCpm boolean, Whether to force optimization to use CPM data even if partition data is available.
#' \item syscodes array, The syscodes for the media run.
#' \item rateCards array, The ratecards for the media run.
#' \item constraints array, The constraints for the media run.
#' }}
#' \item{programs}{array, An array of programs that the Civis Media Optimizer either exclude or limit to.An error will be thrown if exclude_programs is not also set.}
#' \item{networks}{array, An array of networks that the Civis Media Optimizer either exclude or limit to.An error will be thrown if exclude_networks is not also set.}
#' \item{excludePrograms}{boolean, If Civis Media Optimizer should exclude the programs in the programs parameter.If this value is set to false, it will make the optimization limit itself to the programs supplied through the programs parameter.An error will be thrown if programs is not also set.}
#' \item{excludeNetworks}{boolean, If Civis Media Optimizer should exclude the networks in the networks parameter.If this value is set to false, it will make the optimization limit itself to the networks supplied through the networks.An error will be thrown if networks is not also set.}
#' \item{timeSlotPercentages}{object, The maximum amount of the budget spent on that particular day of the week, daypart, or specific time slot for broadcast and cable.}
#' @export
media_post_optimizations_clone <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/media/optimizations/{id}/clone"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' Start a run
#' @param id integer required. The ID of the optimization.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the run.}
#' \item{optimizationId}{integer, The ID of the optimization.}
#' \item{state}{string, The state of the run, one of 'queued' 'running' 'succeeded' 'failed' or 'cancelled'.}
#' \item{isCancelRequested}{boolean, True if run cancel requested, else false.}
#' \item{startedAt}{string, The time the last run started at.}
#' \item{finishedAt}{string, The time the last run completed.}
#' \item{error}{string, The error, if any, returned by the run.}
#' @export
media_post_optimizations_runs <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/media/optimizations/{id}/runs"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' List runs for the given optimization
#' @param id integer required. The ID of the optimization.
#' @param limit integer optional. Number of results to return. Defaults to 20. Maximum allowed is 100.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to id. Must be one of: id.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of the run.}
#' \item{optimizationId}{integer, The ID of the optimization.}
#' \item{state}{string, The state of the run, one of 'queued' 'running' 'succeeded' 'failed' or 'cancelled'.}
#' \item{isCancelRequested}{boolean, True if run cancel requested, else false.}
#' \item{startedAt}{string, The time the last run started at.}
#' \item{finishedAt}{string, The time the last run completed.}
#' \item{error}{string, The error, if any, returned by the run.}
#' @export
media_list_optimizations_runs <- function(id, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/media/optimizations/{id}/runs"
  path_params  <- list(id = id)
  query_params <- list(limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Check status of a run
#' @param id integer required. The ID of the optimization.
#' @param run_id integer required. The ID of the run.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the run.}
#' \item{optimizationId}{integer, The ID of the optimization.}
#' \item{state}{string, The state of the run, one of 'queued' 'running' 'succeeded' 'failed' or 'cancelled'.}
#' \item{isCancelRequested}{boolean, True if run cancel requested, else false.}
#' \item{startedAt}{string, The time the last run started at.}
#' \item{finishedAt}{string, The time the last run completed.}
#' \item{error}{string, The error, if any, returned by the run.}
#' @export
media_get_optimizations_runs <- function(id, run_id) {

  args <- as.list(match.call())[-1]
  path <- "/media/optimizations/{id}/runs/{run_id}"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Cancel a run
#' @param id integer required. The ID of the optimization.
#' @param run_id integer required. The ID of the run.
#' 
#' @return  An empty HTTP response
#' @export
media_delete_optimizations_runs <- function(id, run_id) {

  args <- as.list(match.call())[-1]
  path <- "/media/optimizations/{id}/runs/{run_id}"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get the logs for a run
#' @param id integer required. The ID of the optimization.
#' @param run_id integer required. The ID of the run.
#' @param last_id integer optional. The ID of the last log message received. Log entries with this ID value or lower will be omitted.Logs are sorted by ID if this value is provided, and are otherwise sorted by createdAt.
#' @param limit integer optional. The maximum number of log messages to return. Default of 10000.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of the log.}
#' \item{createdAt}{string, The time the log was created.}
#' \item{message}{string, The log message.}
#' \item{level}{string, The level of the log. One of unknown,fatal,error,warn,info,debug.}
#' @export
media_list_optimizations_runs_logs <- function(id, run_id, last_id = NULL, limit = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/media/optimizations/{id}/runs/{run_id}/logs"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list(last_id = last_id, limit = limit)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' List all spot orders
#' @param id integer optional. The ID for the spot order.
#' @param archived string optional. The archival status of the requested object(s).
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID for the spot order.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' @export
media_list_spot_orders <- function(id = NULL, archived = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/media/spot_orders"
  path_params  <- list()
  query_params <- list(id = id, archived = archived)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Create a spot order
#' @param body string optional. CSV body of a spot order.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the spot order.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{csvS3Uri}{string, S3 URI for the spot order CSV file.}
#' \item{jsonS3Uri}{string, S3 URI for the spot order JSON file.}
#' \item{xmlArchiveS3Uri}{string, S3 URI for the spot order XML archive.}
#' \item{lastTransformJobId}{integer, ID of the spot order transformation job.}
#' @export
media_post_spot_orders <- function(body = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/media/spot_orders"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(body = body)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' Show a single spot order
#' @param id integer required. The ID for the spot order.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the spot order.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{csvS3Uri}{string, S3 URI for the spot order CSV file.}
#' \item{jsonS3Uri}{string, S3 URI for the spot order JSON file.}
#' \item{xmlArchiveS3Uri}{string, S3 URI for the spot order XML archive.}
#' \item{lastTransformJobId}{integer, ID of the spot order transformation job.}
#' @export
media_get_spot_orders <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/media/spot_orders/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Edit the specified spot order
#' @param id integer required. The ID for the spot order.
#' @param body string optional. CSV body of a spot order.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the spot order.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{csvS3Uri}{string, S3 URI for the spot order CSV file.}
#' \item{jsonS3Uri}{string, S3 URI for the spot order JSON file.}
#' \item{xmlArchiveS3Uri}{string, S3 URI for the spot order XML archive.}
#' \item{lastTransformJobId}{integer, ID of the spot order transformation job.}
#' @export
media_put_spot_orders <- function(id, body = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/media/spot_orders/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(body = body)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' List all ratecards
#' @param archived string optional. The archival status of the requested object(s).
#' @param filename string optional. If specified, will be used to filter the ratecards returned. Substring matching is supported with "\%" and "*" wildcards (e.g., "filename=\%ratecard\%" will return both "ratecard 1" and "my ratecard").
#' @param dma_number integer optional. If specified, will be used to filter the ratecards by DMA.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ratecard ID.}
#' \item{filename}{string, Name of the ratecard file.}
#' \item{startOn}{string, First day to which the ratecard applies.}
#' \item{endOn}{string, Last day to which the ratecard applies.}
#' \item{dmaNumber}{integer, Number of the DMA associated with the ratecard.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' @export
media_list_ratecards <- function(archived = NULL, filename = NULL, dma_number = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/media/ratecards"
  path_params  <- list()
  query_params <- list(archived = archived, filename = filename, dma_number = dma_number)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Create a Ratecard
#' @param filename string required. Name of the ratecard file.
#' @param start_on string required. First day to which the ratecard applies.
#' @param end_on string required. Last day to which the ratecard applies.
#' @param dma_number integer required. Number of the DMA associated with the ratecard.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ratecard ID.}
#' \item{filename}{string, Name of the ratecard file.}
#' \item{startOn}{string, First day to which the ratecard applies.}
#' \item{endOn}{string, Last day to which the ratecard applies.}
#' \item{dmaNumber}{integer, Number of the DMA associated with the ratecard.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' @export
media_post_ratecards <- function(filename, start_on, end_on, dma_number) {

  args <- as.list(match.call())[-1]
  path <- "/media/ratecards"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(filename = filename, startOn = start_on, endOn = end_on, dmaNumber = dma_number)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get a Ratecard
#' @param id integer required. 
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ratecard ID.}
#' \item{filename}{string, Name of the ratecard file.}
#' \item{startOn}{string, First day to which the ratecard applies.}
#' \item{endOn}{string, Last day to which the ratecard applies.}
#' \item{dmaNumber}{integer, Number of the DMA associated with the ratecard.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' @export
media_get_ratecards <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/media/ratecards/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Replace all attributes of this Ratecard
#' @param id integer required. The ratecard ID.
#' @param filename string required. Name of the ratecard file.
#' @param start_on string required. First day to which the ratecard applies.
#' @param end_on string required. Last day to which the ratecard applies.
#' @param dma_number integer required. Number of the DMA associated with the ratecard.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ratecard ID.}
#' \item{filename}{string, Name of the ratecard file.}
#' \item{startOn}{string, First day to which the ratecard applies.}
#' \item{endOn}{string, Last day to which the ratecard applies.}
#' \item{dmaNumber}{integer, Number of the DMA associated with the ratecard.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' @export
media_put_ratecards <- function(id, filename, start_on, end_on, dma_number) {

  args <- as.list(match.call())[-1]
  path <- "/media/ratecards/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(filename = filename, startOn = start_on, endOn = end_on, dmaNumber = dma_number)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update some attributes of this Ratecard
#' @param id integer required. The ratecard ID.
#' @param filename string optional. Name of the ratecard file.
#' @param start_on string optional. First day to which the ratecard applies.
#' @param end_on string optional. Last day to which the ratecard applies.
#' @param dma_number integer optional. Number of the DMA associated with the ratecard.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ratecard ID.}
#' \item{filename}{string, Name of the ratecard file.}
#' \item{startOn}{string, First day to which the ratecard applies.}
#' \item{endOn}{string, Last day to which the ratecard applies.}
#' \item{dmaNumber}{integer, Number of the DMA associated with the ratecard.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' @export
media_patch_ratecards <- function(id, filename = NULL, start_on = NULL, end_on = NULL, dma_number = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/media/ratecards/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(filename = filename, startOn = start_on, endOn = end_on, dmaNumber = dma_number)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("patch", path, path_params, query_params, body_params)

  return(resp)

 }


#' List all Designated Market Areas
#' @param name string optional. If specified, will be used to filter the DMAs returned. Substring matching is supported with "\%" and "*" wildcards (e.g., "name=\%region\%" will return both "region1" and "my region").
#' @param number integer optional. If specified, will be used to filter the DMAS by number.
#' 
#' @return  An array containing the following fields:
#' \item{name}{string, Name for the DMA region.}
#' \item{number}{integer, Identifier number for a DMA.}
#' @export
media_list_dmas <- function(name = NULL, number = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/media/dmas"
  path_params  <- list()
  query_params <- list(name = name, number = number)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' List all Media Targets
#' @param name string optional. The name of the target.
#' @param identifier string optional. A unique identifier for this target.
#' @param data_source string optional. The source of viewership data for this target.
#' 
#' @return  An array containing the following fields:
#' \item{name}{string, The name of the target.}
#' \item{identifier}{string, A unique identifier for this target.}
#' \item{dataSource}{string, The source of viewership data for this target.}
#' @export
media_list_targets <- function(name = NULL, identifier = NULL, data_source = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/media/targets"
  path_params  <- list()
  query_params <- list(name = name, identifier = identifier, data_source = data_source)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' List all available model types
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of the model type.}
#' \item{algorithm}{string, The name of the algorithm used to train the model.}
#' \item{dvType}{string, The type of dependent variable predicted by the model.}
#' \item{fintAllowed}{boolean, Whether this model type supports searching for interaction terms.}
#' @export
models_list_types <- function() {

  args <- as.list(match.call())[-1]
  path <- "/models/types"
  path_params  <- list()
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Create new configuration for a model
#' @param table_name string optional. The qualified name of the table containing the training set from which to build the model.
#' @param database_id integer optional. The ID of the database holding the training set table used to build the model.
#' @param credential_id integer optional. The ID of the credential used to read the target table. Defaults to the user's default credential.
#' @param model_name string optional. The name of the model.
#' @param description string optional. A description of the model.
#' @param interaction_terms boolean optional. Whether to search for interaction terms.
#' @param box_cox_transformation boolean optional. Whether to transform data so that it assumes a normal distribution. Valid only with continuous models.
#' @param model_type_id integer optional. The ID of the model's type.
#' @param primary_key string optional. The unique ID (primary key) of the training dataset.
#' @param dependent_variable string optional. The dependent variable of the training dataset.
#' @param dependent_variable_order array optional. The order of dependent variables, especially useful for Ordinal Modeling.
#' @param excluded_columns array optional. A list of columns which will be considered ineligible to be independent variables.
#' @param limiting_sql string optional. A custom SQL WHERE clause used to filter the rows used to build the model. (e.g., "id > 105").
#' @param active_build_id integer optional. The ID of the current active build, the build used to score predictions.
#' @param cross_validation_parameters object optional. Cross validation parameter grid for tree methods, e.g. {"n_estimators": [100, 200, 500], "learning_rate": [0.01, 0.1], "max_depth": [2, 3]}.
#' @param number_of_folds integer optional. Number of folds for cross validation. Default value is 5.
#' @param notifications object optional. A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }
#' @param schedule object optional. A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }
#' @param parent_id integer optional. The ID of the parent job that will trigger this model.
#' @param time_zone string optional. The time zone of this model.
#' @param hidden boolean optional. The hidden status of the object.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the model.}
#' \item{tableName}{string, The qualified name of the table containing the training set from which to build the model.}
#' \item{databaseId}{integer, The ID of the database holding the training set table used to build the model.}
#' \item{credentialId}{integer, The ID of the credential used to read the target table. Defaults to the user's default credential.}
#' \item{modelName}{string, The name of the model.}
#' \item{description}{string, A description of the model.}
#' \item{interactionTerms}{boolean, Whether to search for interaction terms.}
#' \item{boxCoxTransformation}{boolean, Whether to transform data so that it assumes a normal distribution. Valid only with continuous models.}
#' \item{modelTypeId}{integer, The ID of the model's type.}
#' \item{primaryKey}{string, The unique ID (primary key) of the training dataset.}
#' \item{dependentVariable}{string, The dependent variable of the training dataset.}
#' \item{dependentVariableOrder}{array, The order of dependent variables, especially useful for Ordinal Modeling.}
#' \item{excludedColumns}{array, A list of columns which will be considered ineligible to be independent variables.}
#' \item{limitingSQL}{string, A custom SQL WHERE clause used to filter the rows used to build the model. (e.g., "id > 105").}
#' \item{activeBuildId}{integer, The ID of the current active build, the build used to score predictions.}
#' \item{crossValidationParameters}{object, Cross validation parameter grid for tree methods, e.g. {"n_estimators": [100, 200, 500], "learning_rate": [0.01, 0.1], "max_depth": [2, 3]}.}
#' \item{numberOfFolds}{integer, Number of folds for cross validation. Default value is 5.}
#' \item{notifications}{object, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{parentId}{integer, The ID of the parent job that will trigger this model.}
#' \item{runningAs}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{timeZone}{string, The time zone of this model.}
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{hidden}{boolean, The hidden status of the object.}
#' \item{user}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{createdAt}{string, The time the model was created.}
#' \item{updatedAt}{string, The time the model was updated.}
#' \item{currentBuildState}{string, The status of the current model build. One of "succeeded", "failed", "queued", or "running,"or "idle", if no build has been attempted.}
#' \item{currentBuildException}{string, Exception message, if applicable, of the current model build.}
#' \item{builds}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of the model build.
#' \item name string, The name of the model build.
#' \item createdAt string, The time the model build was created.
#' \item description string, A description of the model build.
#' \item rootMeanSquaredError number, A key metric for continuous models. Nil for other model types.
#' \item rSquaredError number, A key metric for continuous models. Nil for other model types.
#' \item rocAuc number, A key metric for binary, multinomial, and ordinal models. Nil for other model types.
#' }}
#' \item{predictions}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of the model to which to apply the prediction.
#' \item tableName string, The qualified name of the table on which to apply the predictive model.
#' \item primaryKey array, The primary key or composite keys of the table being predicted.
#' \item limitingSQL string, A SQL WHERE clause used to scope the rows to be predicted.
#' \item outputTable string, The qualified name of the table to be created which will contain the model's predictions.
#' \item schedule object, 
#' \item state string, The status of the prediction. One of: "succeeded", "failed", "queued", or "running,"or "idle", if no build has been attempted.
#' }}
#' \item{lastOutputLocation}{string, The output JSON for the last build.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' @export
models_post <- function(table_name = NULL, database_id = NULL, credential_id = NULL, model_name = NULL, description = NULL, interaction_terms = NULL, box_cox_transformation = NULL, model_type_id = NULL, primary_key = NULL, dependent_variable = NULL, dependent_variable_order = NULL, excluded_columns = NULL, limiting_sql = NULL, active_build_id = NULL, cross_validation_parameters = NULL, number_of_folds = NULL, notifications = NULL, schedule = NULL, parent_id = NULL, time_zone = NULL, hidden = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/models/"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(tableName = table_name, databaseId = database_id, credentialId = credential_id, modelName = model_name, description = description, interactionTerms = interaction_terms, boxCoxTransformation = box_cox_transformation, modelTypeId = model_type_id, primaryKey = primary_key, dependentVariable = dependent_variable, dependentVariableOrder = dependent_variable_order, excludedColumns = excluded_columns, limitingSQL = limiting_sql, activeBuildId = active_build_id, crossValidationParameters = cross_validation_parameters, numberOfFolds = number_of_folds, notifications = notifications, schedule = schedule, parentId = parent_id, timeZone = time_zone, hidden = hidden)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' List 
#' @param model_name string optional. If specified, will be used to filter the models returned. Substring matching is supported. (e.g., "modelName=model" will return both "model1" and "my model").
#' @param training_table_name string optional. If specified, will be used to filter the models returned by the training dataset table name. Substring matching is supported. (e.g., "trainingTableName=table" will return both "table1" and "my_table").
#' @param dependent_variable string optional. If specified, will be used to filter the models returned by the dependent variable column name. Substring matching is supported. (e.g., "dependentVariable=predictor" will return both "predictor" and "my predictor").
#' @param author string optional. If specified, return models from this author. It accepts a comma-separated list of author ids.
#' @param status string optional. If specified, returns models with one of these statuses. It accepts a comma-separated list, possible values are 'running', 'failed', 'succeeded', 'idle', 'scheduled'.
#' @param hidden boolean optional. If specified to be true, returns hidden objects. Defaults to false, returning non-hidden objects.
#' @param archived string optional. The archival status of the requested object(s).
#' @param limit integer optional. Number of results to return. Defaults to its maximum of 50.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to updated_at. Must be one of: updated_at, name, created_at, last_run.updated_at.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of the model.}
#' \item{tableName}{string, The qualified name of the table containing the training set from which to build the model.}
#' \item{databaseId}{integer, The ID of the database holding the training set table used to build the model.}
#' \item{credentialId}{integer, The ID of the credential used to read the target table. Defaults to the user's default credential.}
#' \item{modelName}{string, The name of the model.}
#' \item{description}{string, A description of the model.}
#' \item{interactionTerms}{boolean, Whether to search for interaction terms.}
#' \item{boxCoxTransformation}{boolean, Whether to transform data so that it assumes a normal distribution. Valid only with continuous models.}
#' \item{modelTypeId}{integer, The ID of the model's type.}
#' \item{primaryKey}{string, The unique ID (primary key) of the training dataset.}
#' \item{dependentVariable}{string, The dependent variable of the training dataset.}
#' \item{dependentVariableOrder}{array, The order of dependent variables, especially useful for Ordinal Modeling.}
#' \item{excludedColumns}{array, A list of columns which will be considered ineligible to be independent variables.}
#' \item{limitingSQL}{string, A custom SQL WHERE clause used to filter the rows used to build the model. (e.g., "id > 105").}
#' \item{crossValidationParameters}{object, Cross validation parameter grid for tree methods, e.g. {"n_estimators": [100, 200, 500], "learning_rate": [0.01, 0.1], "max_depth": [2, 3]}.}
#' \item{numberOfFolds}{integer, Number of folds for cross validation. Default value is 5.}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{parentId}{integer, The ID of the parent job that will trigger this model.}
#' \item{timeZone}{string, The time zone of this model.}
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{user}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{createdAt}{string, The time the model was created.}
#' \item{updatedAt}{string, The time the model was updated.}
#' \item{currentBuildState}{string, The status of the current model build. One of "succeeded", "failed", "queued", or "running,"or "idle", if no build has been attempted.}
#' \item{currentBuildException}{string, Exception message, if applicable, of the current model build.}
#' \item{builds}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of the model build.
#' \item name string, The name of the model build.
#' \item createdAt string, The time the model build was created.
#' \item description string, A description of the model build.
#' \item rootMeanSquaredError number, A key metric for continuous models. Nil for other model types.
#' \item rSquaredError number, A key metric for continuous models. Nil for other model types.
#' \item rocAuc number, A key metric for binary, multinomial, and ordinal models. Nil for other model types.
#' }}
#' \item{predictions}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of the model to which to apply the prediction.
#' \item tableName string, The qualified name of the table on which to apply the predictive model.
#' \item primaryKey array, The primary key or composite keys of the table being predicted.
#' \item limitingSQL string, A SQL WHERE clause used to scope the rows to be predicted.
#' \item outputTable string, The qualified name of the table to be created which will contain the model's predictions.
#' \item state string, The status of the prediction. One of: "succeeded", "failed", "queued", or "running,"or "idle", if no build has been attempted.
#' }}
#' \item{lastOutputLocation}{string, The output JSON for the last build.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' @export
models_list <- function(model_name = NULL, training_table_name = NULL, dependent_variable = NULL, author = NULL, status = NULL, hidden = NULL, archived = NULL, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/models/"
  path_params  <- list()
  query_params <- list(model_name = model_name, training_table_name = training_table_name, dependent_variable = dependent_variable, author = author, status = status, hidden = hidden, archived = archived, limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update model configuration
#' @param id integer required. The ID of the model.
#' @param table_name string optional. The qualified name of the table containing the training set from which to build the model.
#' @param database_id integer optional. The ID of the database holding the training set table used to build the model.
#' @param credential_id integer optional. The ID of the credential used to read the target table. Defaults to the user's default credential.
#' @param model_name string optional. The name of the model.
#' @param description string optional. A description of the model.
#' @param interaction_terms boolean optional. Whether to search for interaction terms.
#' @param box_cox_transformation boolean optional. Whether to transform data so that it assumes a normal distribution. Valid only with continuous models.
#' @param model_type_id integer optional. The ID of the model's type.
#' @param primary_key string optional. The unique ID (primary key) of the training dataset.
#' @param dependent_variable string optional. The dependent variable of the training dataset.
#' @param dependent_variable_order array optional. The order of dependent variables, especially useful for Ordinal Modeling.
#' @param excluded_columns array optional. A list of columns which will be considered ineligible to be independent variables.
#' @param limiting_sql string optional. A custom SQL WHERE clause used to filter the rows used to build the model. (e.g., "id > 105").
#' @param active_build_id integer optional. The ID of the current active build, the build used to score predictions.
#' @param cross_validation_parameters object optional. Cross validation parameter grid for tree methods, e.g. {"n_estimators": [100, 200, 500], "learning_rate": [0.01, 0.1], "max_depth": [2, 3]}.
#' @param number_of_folds integer optional. Number of folds for cross validation. Default value is 5.
#' @param notifications object optional. A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }
#' @param schedule object optional. A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }
#' @param parent_id integer optional. The ID of the parent job that will trigger this model.
#' @param time_zone string optional. The time zone of this model.
#' 
#' @return  An empty HTTP response
#' @export
models_patch <- function(id, table_name = NULL, database_id = NULL, credential_id = NULL, model_name = NULL, description = NULL, interaction_terms = NULL, box_cox_transformation = NULL, model_type_id = NULL, primary_key = NULL, dependent_variable = NULL, dependent_variable_order = NULL, excluded_columns = NULL, limiting_sql = NULL, active_build_id = NULL, cross_validation_parameters = NULL, number_of_folds = NULL, notifications = NULL, schedule = NULL, parent_id = NULL, time_zone = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/models/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(tableName = table_name, databaseId = database_id, credentialId = credential_id, modelName = model_name, description = description, interactionTerms = interaction_terms, boxCoxTransformation = box_cox_transformation, modelTypeId = model_type_id, primaryKey = primary_key, dependentVariable = dependent_variable, dependentVariableOrder = dependent_variable_order, excludedColumns = excluded_columns, limitingSQL = limiting_sql, activeBuildId = active_build_id, crossValidationParameters = cross_validation_parameters, numberOfFolds = number_of_folds, notifications = notifications, schedule = schedule, parentId = parent_id, timeZone = time_zone)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("patch", path, path_params, query_params, body_params)

  return(resp)

 }


#' Retrieve model configuration
#' @param id integer required. The ID of the model.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the model.}
#' \item{tableName}{string, The qualified name of the table containing the training set from which to build the model.}
#' \item{databaseId}{integer, The ID of the database holding the training set table used to build the model.}
#' \item{credentialId}{integer, The ID of the credential used to read the target table. Defaults to the user's default credential.}
#' \item{modelName}{string, The name of the model.}
#' \item{description}{string, A description of the model.}
#' \item{interactionTerms}{boolean, Whether to search for interaction terms.}
#' \item{boxCoxTransformation}{boolean, Whether to transform data so that it assumes a normal distribution. Valid only with continuous models.}
#' \item{modelTypeId}{integer, The ID of the model's type.}
#' \item{primaryKey}{string, The unique ID (primary key) of the training dataset.}
#' \item{dependentVariable}{string, The dependent variable of the training dataset.}
#' \item{dependentVariableOrder}{array, The order of dependent variables, especially useful for Ordinal Modeling.}
#' \item{excludedColumns}{array, A list of columns which will be considered ineligible to be independent variables.}
#' \item{limitingSQL}{string, A custom SQL WHERE clause used to filter the rows used to build the model. (e.g., "id > 105").}
#' \item{activeBuildId}{integer, The ID of the current active build, the build used to score predictions.}
#' \item{crossValidationParameters}{object, Cross validation parameter grid for tree methods, e.g. {"n_estimators": [100, 200, 500], "learning_rate": [0.01, 0.1], "max_depth": [2, 3]}.}
#' \item{numberOfFolds}{integer, Number of folds for cross validation. Default value is 5.}
#' \item{notifications}{object, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{parentId}{integer, The ID of the parent job that will trigger this model.}
#' \item{runningAs}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{timeZone}{string, The time zone of this model.}
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{hidden}{boolean, The hidden status of the object.}
#' \item{user}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{createdAt}{string, The time the model was created.}
#' \item{updatedAt}{string, The time the model was updated.}
#' \item{currentBuildState}{string, The status of the current model build. One of "succeeded", "failed", "queued", or "running,"or "idle", if no build has been attempted.}
#' \item{currentBuildException}{string, Exception message, if applicable, of the current model build.}
#' \item{builds}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of the model build.
#' \item name string, The name of the model build.
#' \item createdAt string, The time the model build was created.
#' \item description string, A description of the model build.
#' \item rootMeanSquaredError number, A key metric for continuous models. Nil for other model types.
#' \item rSquaredError number, A key metric for continuous models. Nil for other model types.
#' \item rocAuc number, A key metric for binary, multinomial, and ordinal models. Nil for other model types.
#' }}
#' \item{predictions}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of the model to which to apply the prediction.
#' \item tableName string, The qualified name of the table on which to apply the predictive model.
#' \item primaryKey array, The primary key or composite keys of the table being predicted.
#' \item limitingSQL string, A SQL WHERE clause used to scope the rows to be predicted.
#' \item outputTable string, The qualified name of the table to be created which will contain the model's predictions.
#' \item schedule object, 
#' \item state string, The status of the prediction. One of: "succeeded", "failed", "queued", or "running,"or "idle", if no build has been attempted.
#' }}
#' \item{lastOutputLocation}{string, The output JSON for the last build.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' @export
models_get <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/models/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Start a build
#' @param id integer required. The ID of the model.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the model build.}
#' \item{state}{string, The state of the model build.one of 'queued' 'running' 'succeeded' 'failed' or 'cancelled'.}
#' \item{error}{string, The error, if any, returned by the build.}
#' \item{name}{string, The name of the model build.}
#' \item{createdAt}{string, The time the model build was created.}
#' \item{description}{string, A description of the model build.}
#' \item{rootMeanSquaredError}{number, A key metric for continuous models. Nil for other model types.}
#' \item{rSquaredError}{number, A key metric for continuous models. Nil for other model types.}
#' \item{rocAuc}{number, A key metric for binary, multinomial, and ordinal models. Nil for other model types.}
#' \item{transformationMetadata}{string, A string representing the full JSON output of the metadata for transformation of column names}
#' \item{output}{string, A string representing the JSON output for the specified build. Only present when smaller than 10KB in size.}
#' \item{outputLocation}{string, A URL representing the location of the full JSON output for the specified build.The URL link will be valid for 5 minutes.}
#' @export
models_post_builds <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/models/{id}/builds"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' List builds for the given model
#' @param id integer required. The ID of the model.
#' @param limit integer optional. Number of results to return. Defaults to 20. Maximum allowed is 100.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to id. Must be one of: id.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of the model build.}
#' \item{state}{string, The state of the model build.one of 'queued' 'running' 'succeeded' 'failed' or 'cancelled'.}
#' \item{error}{string, The error, if any, returned by the build.}
#' \item{name}{string, The name of the model build.}
#' \item{createdAt}{string, The time the model build was created.}
#' \item{description}{string, A description of the model build.}
#' \item{rootMeanSquaredError}{number, A key metric for continuous models. Nil for other model types.}
#' \item{rSquaredError}{number, A key metric for continuous models. Nil for other model types.}
#' \item{rocAuc}{number, A key metric for binary, multinomial, and ordinal models. Nil for other model types.}
#' \item{transformationMetadata}{string, A string representing the full JSON output of the metadata for transformation of column names}
#' \item{output}{string, A string representing the JSON output for the specified build. Only present when smaller than 10KB in size.}
#' \item{outputLocation}{string, A URL representing the location of the full JSON output for the specified build.The URL link will be valid for 5 minutes.}
#' @export
models_list_builds <- function(id, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/models/{id}/builds"
  path_params  <- list(id = id)
  query_params <- list(limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Check status of a build
#' @param id integer required. The ID of the model.
#' @param build_id integer required. The ID of the build.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the model build.}
#' \item{state}{string, The state of the model build.one of 'queued' 'running' 'succeeded' 'failed' or 'cancelled'.}
#' \item{error}{string, The error, if any, returned by the build.}
#' \item{name}{string, The name of the model build.}
#' \item{createdAt}{string, The time the model build was created.}
#' \item{description}{string, A description of the model build.}
#' \item{rootMeanSquaredError}{number, A key metric for continuous models. Nil for other model types.}
#' \item{rSquaredError}{number, A key metric for continuous models. Nil for other model types.}
#' \item{rocAuc}{number, A key metric for binary, multinomial, and ordinal models. Nil for other model types.}
#' \item{transformationMetadata}{string, A string representing the full JSON output of the metadata for transformation of column names}
#' \item{output}{string, A string representing the JSON output for the specified build. Only present when smaller than 10KB in size.}
#' \item{outputLocation}{string, A URL representing the location of the full JSON output for the specified build.The URL link will be valid for 5 minutes.}
#' @export
models_get_builds <- function(id, build_id) {

  args <- as.list(match.call())[-1]
  path <- "/models/{id}/builds/{build_id}"
  path_params  <- list(id = id, build_id = build_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Cancel a build
#' @param id integer required. The ID of the model.
#' @param build_id integer required. The ID of the build.
#' 
#' @return  An empty HTTP response
#' @export
models_delete_builds <- function(id, build_id) {

  args <- as.list(match.call())[-1]
  path <- "/models/{id}/builds/{build_id}"
  path_params  <- list(id = id, build_id = build_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get the logs for a build
#' @param id integer required. The ID of the model.
#' @param build_id integer required. The ID of the build.
#' @param last_id integer optional. The ID of the last log message received. Log entries with this ID value or lower will be omitted.Logs are sorted by ID if this value is provided, and are otherwise sorted by createdAt.
#' @param limit integer optional. The maximum number of log messages to return. Default of 10000.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of the log.}
#' \item{createdAt}{string, The time the log was created.}
#' \item{message}{string, The log message.}
#' \item{level}{string, The level of the log. One of unknown,fatal,error,warn,info,debug.}
#' @export
models_list_builds_logs <- function(id, build_id, last_id = NULL, limit = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/models/{id}/builds/{build_id}/logs"
  path_params  <- list(id = id, build_id = build_id)
  query_params <- list(last_id = last_id, limit = limit)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' List users and groups permissioned on this object
#' @param id integer required. The ID of the object.
#' 
#' @return  An array containing the following fields:
#' \item{readers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
models_list_shares <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/models/{id}/shares"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions users have on this object
#' @param id integer required. ID of the resource to be shared
#' @param user_ids array required. An array of one or more user IDs
#' @param permission_level string required. Options are: "read", "write", or "manage"
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
models_put_shares_users <- function(id, user_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/models/{id}/shares/users"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(userIds = user_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a user has on this object
#' @param id integer required. ID of the resource to be revoked
#' @param user_id integer required. ID of the user
#' 
#' @return  An empty HTTP response
#' @export
models_delete_shares_users <- function(id, user_id) {

  args <- as.list(match.call())[-1]
  path <- "/models/{id}/shares/users/{user_id}"
  path_params  <- list(id = id, user_id = user_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions groups has on this object
#' @param id integer required. ID of the resource to be shared
#' @param group_ids array required. An array of one or more group IDs
#' @param permission_level string required. Options are: "read", "write", or "manage"
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
models_put_shares_groups <- function(id, group_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/models/{id}/shares/groups"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(groupIds = group_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a group has on this object
#' @param id integer required. ID of the resource to be revoked
#' @param group_id integer required. ID of the group
#' 
#' @return  An empty HTTP response
#' @export
models_delete_shares_groups <- function(id, group_id) {

  args <- as.list(match.call())[-1]
  path <- "/models/{id}/shares/groups/{group_id}"
  path_params  <- list(id = id, group_id = group_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' List the projects a models belongs to
#' @param id integer required. The ID of the resource.
#' @param hidden boolean optional. If specified to be true, returns hidden objects. Defaults to false, returning non-hidden objects.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID for this project.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{name}{string, The name of this project.}
#' \item{description}{string, A description of the project}
#' \item{users}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{autoShare}{boolean, }
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{archived}{string, The archival status of the requested object(s).}
#' @export
models_list_projects <- function(id, hidden = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/models/{id}/projects"
  path_params  <- list(id = id)
  query_params <- list(hidden = hidden)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Add a models to a project
#' @param id integer required. ID of the resource
#' @param project_id integer required. The ID of the project
#' 
#' @return  An empty HTTP response
#' @export
models_put_projects <- function(id, project_id) {

  args <- as.list(match.call())[-1]
  path <- "/models/{id}/projects/{project_id}"
  path_params  <- list(id = id, project_id = project_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Remove a models from a project
#' @param id integer required. ID of the resource
#' @param project_id integer required. The ID of the project
#' 
#' @return  An empty HTTP response
#' @export
models_delete_projects <- function(id, project_id) {

  args <- as.list(match.call())[-1]
  path <- "/models/{id}/projects/{project_id}"
  path_params  <- list(id = id, project_id = project_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update the archive status of this object
#' @param id integer required. The ID of the object.
#' @param status boolean required. The desired archived status of the object.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the model.}
#' \item{tableName}{string, The qualified name of the table containing the training set from which to build the model.}
#' \item{databaseId}{integer, The ID of the database holding the training set table used to build the model.}
#' \item{credentialId}{integer, The ID of the credential used to read the target table. Defaults to the user's default credential.}
#' \item{modelName}{string, The name of the model.}
#' \item{description}{string, A description of the model.}
#' \item{interactionTerms}{boolean, Whether to search for interaction terms.}
#' \item{boxCoxTransformation}{boolean, Whether to transform data so that it assumes a normal distribution. Valid only with continuous models.}
#' \item{modelTypeId}{integer, The ID of the model's type.}
#' \item{primaryKey}{string, The unique ID (primary key) of the training dataset.}
#' \item{dependentVariable}{string, The dependent variable of the training dataset.}
#' \item{dependentVariableOrder}{array, The order of dependent variables, especially useful for Ordinal Modeling.}
#' \item{excludedColumns}{array, A list of columns which will be considered ineligible to be independent variables.}
#' \item{limitingSQL}{string, A custom SQL WHERE clause used to filter the rows used to build the model. (e.g., "id > 105").}
#' \item{activeBuildId}{integer, The ID of the current active build, the build used to score predictions.}
#' \item{crossValidationParameters}{object, Cross validation parameter grid for tree methods, e.g. {"n_estimators": [100, 200, 500], "learning_rate": [0.01, 0.1], "max_depth": [2, 3]}.}
#' \item{numberOfFolds}{integer, Number of folds for cross validation. Default value is 5.}
#' \item{notifications}{object, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{parentId}{integer, The ID of the parent job that will trigger this model.}
#' \item{runningAs}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{timeZone}{string, The time zone of this model.}
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{hidden}{boolean, The hidden status of the object.}
#' \item{user}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{createdAt}{string, The time the model was created.}
#' \item{updatedAt}{string, The time the model was updated.}
#' \item{currentBuildState}{string, The status of the current model build. One of "succeeded", "failed", "queued", or "running,"or "idle", if no build has been attempted.}
#' \item{currentBuildException}{string, Exception message, if applicable, of the current model build.}
#' \item{builds}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of the model build.
#' \item name string, The name of the model build.
#' \item createdAt string, The time the model build was created.
#' \item description string, A description of the model build.
#' \item rootMeanSquaredError number, A key metric for continuous models. Nil for other model types.
#' \item rSquaredError number, A key metric for continuous models. Nil for other model types.
#' \item rocAuc number, A key metric for binary, multinomial, and ordinal models. Nil for other model types.
#' }}
#' \item{predictions}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of the model to which to apply the prediction.
#' \item tableName string, The qualified name of the table on which to apply the predictive model.
#' \item primaryKey array, The primary key or composite keys of the table being predicted.
#' \item limitingSQL string, A SQL WHERE clause used to scope the rows to be predicted.
#' \item outputTable string, The qualified name of the table to be created which will contain the model's predictions.
#' \item schedule object, 
#' \item state string, The status of the prediction. One of: "succeeded", "failed", "queued", or "running,"or "idle", if no build has been attempted.
#' }}
#' \item{lastOutputLocation}{string, The output JSON for the last build.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' @export
models_put_archive <- function(id, status) {

  args <- as.list(match.call())[-1]
  path <- "/models/{id}/archive"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(status = status)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Add a table on which to apply the predictive model
#' @param id integer required. The ID of the model to which to apply the prediction.
#' @param table_name string required. The qualified name of the table on which to apply the predictive model.
#' @param primary_key array required. The primary key or composite keys of the table being predicted.
#' @param limiting_sql string optional. A SQL WHERE clause used to scope the rows to be predicted.
#' @param output_table string optional. The qualified name of the table to be created which will contain the model's predictions.
#' @param schedule object optional. A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the model to which to apply the prediction.}
#' \item{tableName}{string, The qualified name of the table on which to apply the predictive model.}
#' \item{primaryKey}{array, The primary key or composite keys of the table being predicted.}
#' \item{limitingSQL}{string, A SQL WHERE clause used to scope the rows to be predicted.}
#' \item{outputTable}{string, The qualified name of the table to be created which will contain the model's predictions.}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{state}{string, The status of the prediction. One of: "succeeded", "failed", "queued", or "running,"or "idle", if no build has been attempted.}
#' @export
models_put_predictions <- function(id, table_name, primary_key, limiting_sql = NULL, output_table = NULL, schedule = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/models/{id}/predictions"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(tableName = table_name, primaryKey = primary_key, limitingSQL = limiting_sql, outputTable = output_table, schedule = schedule)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Show the model build schedule
#' @param id integer required. The ID of the model associated with this schedule.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the model associated with this schedule.}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' @export
models_list_schedules <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/models/{id}/schedules"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Schedule the model build
#' @param id integer required. The ID of the model associated with this schedule.
#' @param schedule object required. A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the model associated with this schedule.}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' @export
models_put_schedules <- function(id, schedule) {

  args <- as.list(match.call())[-1]
  path <- "/models/{id}/schedules"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(schedule = schedule)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' List Notebooks
#' @param hidden boolean optional. If specified to be true, returns hidden objects. Defaults to false, returning non-hidden objects.
#' @param archived string optional. The archival status of the requested object(s).
#' @param author string optional. If specified, return imports from this author. It accepts a comma-separated list of author IDs.
#' @param status string optional. If specified, returns notebooks with one of these statuses. It accepts a comma-separated list, possible values are 'running', 'pending', 'idle'.
#' @param limit integer optional. Number of results to return. Defaults to 20. Maximum allowed is 50.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to updated_at. Must be one of: updated_at, name, created_at.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID for this notebook.}
#' \item{name}{string, The name of this notebook.}
#' \item{language}{string, The kernel language of this notebook.}
#' \item{description}{string, The description of this notebook.}
#' \item{user}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{mostRecentDeployment}{object, A list containing the following elements: 
#' \itemize{
#' \item deploymentId integer, The ID for this deployment.
#' \item userId integer, The ID of the owner.
#' \item host string, Domain of the deployment.
#' \item name string, Name of the deployment.
#' \item dockerImageName string, The name of the docker image to pull from DockerHub.
#' \item dockerImageTag string, The tag of the docker image to pull from DockerHub (default: latest).
#' \item instanceType string, The EC2 instance type requested for the deployment.
#' \item memory integer, The memory allocated to the deployment.
#' \item cpu integer, The cpu allocated to the deployment.
#' \item state string, The state of the deployment.
#' \item stateMessage string, A detailed description of the state.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item published boolean, 
#' \item notebookId integer, The ID of owning Notebook
#' }}
#' \item{archived}{string, The archival status of the requested object(s).}
#' @export
notebooks_list <- function(hidden = NULL, archived = NULL, author = NULL, status = NULL, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/"
  path_params  <- list()
  query_params <- list(hidden = hidden, archived = archived, author = author, status = status, limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Create a Notebook
#' @param name string optional. The name of this notebook.
#' @param language string optional. The kernel language of this notebook.
#' @param description string optional. The description of this notebook.
#' @param file_id string optional. The file ID for the S3 file containing the .ipynb file.
#' @param requirements_file_id string optional. The file ID for the S3 file containing the requirements.txt file.
#' @param requirements string optional. The requirements txt file.
#' @param docker_image_name string optional. The name of the docker image to pull from DockerHub.
#' @param docker_image_tag string optional. The tag of the docker image to pull from DockerHub (default: latest).
#' @param instance_type string optional. The EC2 instance type to deploy to.
#' @param memory integer optional. The amount of memory allocated to the notebook.
#' @param cpu integer optional. The amount of cpu allocated to the the notebook.
#' @param credentials array optional. A list of credential IDs to pass to the notebook.
#' @param environment_variables object optional. Environment variables to be passed into the Notebook.
#' @param idle_timeout integer optional. How long the notebook will stay alive without any kernel activity.
#' @param git_repo_url string optional. The url of the git repository
#' @param git_ref string optional. The git reference if git repo is specified
#' @param git_path string optional. The path to the .ipynb file in the git repo that will be started up on notebook launch
#' @param hidden boolean optional. The hidden status of the object.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for this notebook.}
#' \item{name}{string, The name of this notebook.}
#' \item{language}{string, The kernel language of this notebook.}
#' \item{description}{string, The description of this notebook.}
#' \item{notebookUrl}{string, Time-limited URL to get the .ipynb file for this notebook.}
#' \item{notebookPreviewUrl}{string, Time-limited URL to get the .htm preview file for this notebook.}
#' \item{requirementsUrl}{string, Time-limited URL to get the requirements.txt file for this notebook.}
#' \item{fileId}{string, The file ID for the S3 file containing the .ipynb file.}
#' \item{requirementsFileId}{string, The file ID for the S3 file containing the requirements.txt file.}
#' \item{user}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{dockerImageName}{string, The name of the docker image to pull from DockerHub.}
#' \item{dockerImageTag}{string, The tag of the docker image to pull from DockerHub (default: latest).}
#' \item{instanceType}{string, The EC2 instance type to deploy to.}
#' \item{memory}{integer, The amount of memory allocated to the notebook.}
#' \item{cpu}{integer, The amount of cpu allocated to the the notebook.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{mostRecentDeployment}{object, A list containing the following elements: 
#' \itemize{
#' \item deploymentId integer, The ID for this deployment.
#' \item userId integer, The ID of the owner.
#' \item host string, Domain of the deployment.
#' \item name string, Name of the deployment.
#' \item dockerImageName string, The name of the docker image to pull from DockerHub.
#' \item dockerImageTag string, The tag of the docker image to pull from DockerHub (default: latest).
#' \item displayUrl string, URL that displays the deployed platform object and includes auth token.
#' \item instanceType string, The EC2 instance type requested for the deployment.
#' \item memory integer, The memory allocated to the deployment.
#' \item cpu integer, The cpu allocated to the deployment.
#' \item state string, The state of the deployment.
#' \item stateMessage string, A detailed description of the state.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item published boolean, 
#' \item notebookId integer, The ID of owning Notebook
#' }}
#' \item{credentials}{array, A list of credential IDs to pass to the notebook.}
#' \item{environmentVariables}{object, Environment variables to be passed into the Notebook.}
#' \item{idleTimeout}{integer, How long the notebook will stay alive without any kernel activity.}
#' \item{gitRepoId}{integer, The ID of the git repository.}
#' \item{gitRepoUrl}{string, The url of the git repository}
#' \item{gitRef}{string, The git reference if git repo is specified}
#' \item{gitPath}{string, The path to the .ipynb file in the git repo that will be started up on notebook launch}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{hidden}{boolean, The hidden status of the object.}
#' @export
notebooks_post <- function(name = NULL, language = NULL, description = NULL, file_id = NULL, requirements_file_id = NULL, requirements = NULL, docker_image_name = NULL, docker_image_tag = NULL, instance_type = NULL, memory = NULL, cpu = NULL, credentials = NULL, environment_variables = NULL, idle_timeout = NULL, git_repo_url = NULL, git_ref = NULL, git_path = NULL, hidden = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(name = name, language = language, description = description, fileId = file_id, requirementsFileId = requirements_file_id, requirements = requirements, dockerImageName = docker_image_name, dockerImageTag = docker_image_tag, instanceType = instance_type, memory = memory, cpu = cpu, credentials = credentials, environmentVariables = environment_variables, idleTimeout = idle_timeout, gitRepoUrl = git_repo_url, gitRef = git_ref, gitPath = git_path, hidden = hidden)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get a Notebook
#' @param id integer required. 
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for this notebook.}
#' \item{name}{string, The name of this notebook.}
#' \item{language}{string, The kernel language of this notebook.}
#' \item{description}{string, The description of this notebook.}
#' \item{notebookUrl}{string, Time-limited URL to get the .ipynb file for this notebook.}
#' \item{notebookPreviewUrl}{string, Time-limited URL to get the .htm preview file for this notebook.}
#' \item{requirementsUrl}{string, Time-limited URL to get the requirements.txt file for this notebook.}
#' \item{fileId}{string, The file ID for the S3 file containing the .ipynb file.}
#' \item{requirementsFileId}{string, The file ID for the S3 file containing the requirements.txt file.}
#' \item{user}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{dockerImageName}{string, The name of the docker image to pull from DockerHub.}
#' \item{dockerImageTag}{string, The tag of the docker image to pull from DockerHub (default: latest).}
#' \item{instanceType}{string, The EC2 instance type to deploy to.}
#' \item{memory}{integer, The amount of memory allocated to the notebook.}
#' \item{cpu}{integer, The amount of cpu allocated to the the notebook.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{mostRecentDeployment}{object, A list containing the following elements: 
#' \itemize{
#' \item deploymentId integer, The ID for this deployment.
#' \item userId integer, The ID of the owner.
#' \item host string, Domain of the deployment.
#' \item name string, Name of the deployment.
#' \item dockerImageName string, The name of the docker image to pull from DockerHub.
#' \item dockerImageTag string, The tag of the docker image to pull from DockerHub (default: latest).
#' \item displayUrl string, URL that displays the deployed platform object and includes auth token.
#' \item instanceType string, The EC2 instance type requested for the deployment.
#' \item memory integer, The memory allocated to the deployment.
#' \item cpu integer, The cpu allocated to the deployment.
#' \item state string, The state of the deployment.
#' \item stateMessage string, A detailed description of the state.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item published boolean, 
#' \item notebookId integer, The ID of owning Notebook
#' }}
#' \item{credentials}{array, A list of credential IDs to pass to the notebook.}
#' \item{environmentVariables}{object, Environment variables to be passed into the Notebook.}
#' \item{idleTimeout}{integer, How long the notebook will stay alive without any kernel activity.}
#' \item{gitRepoId}{integer, The ID of the git repository.}
#' \item{gitRepoUrl}{string, The url of the git repository}
#' \item{gitRef}{string, The git reference if git repo is specified}
#' \item{gitPath}{string, The path to the .ipynb file in the git repo that will be started up on notebook launch}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{hidden}{boolean, The hidden status of the object.}
#' @export
notebooks_get <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Replace all attributes of this Notebook
#' @param id integer required. The ID for this notebook.
#' @param name string optional. The name of this notebook.
#' @param language string optional. The kernel language of this notebook.
#' @param description string optional. The description of this notebook.
#' @param file_id string optional. The file ID for the S3 file containing the .ipynb file.
#' @param requirements_file_id string optional. The file ID for the S3 file containing the requirements.txt file.
#' @param requirements string optional. The requirements txt file.
#' @param docker_image_name string optional. The name of the docker image to pull from DockerHub.
#' @param docker_image_tag string optional. The tag of the docker image to pull from DockerHub (default: latest).
#' @param instance_type string optional. The EC2 instance type to deploy to.
#' @param memory integer optional. The amount of memory allocated to the notebook.
#' @param cpu integer optional. The amount of cpu allocated to the the notebook.
#' @param credentials array optional. A list of credential IDs to pass to the notebook.
#' @param environment_variables object optional. Environment variables to be passed into the Notebook.
#' @param idle_timeout integer optional. How long the notebook will stay alive without any kernel activity.
#' @param git_repo_url string optional. The url of the git repository
#' @param git_ref string optional. The git reference if git repo is specified
#' @param git_path string optional. The path to the .ipynb file in the git repo that will be started up on notebook launch
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for this notebook.}
#' \item{name}{string, The name of this notebook.}
#' \item{language}{string, The kernel language of this notebook.}
#' \item{description}{string, The description of this notebook.}
#' \item{notebookUrl}{string, Time-limited URL to get the .ipynb file for this notebook.}
#' \item{notebookPreviewUrl}{string, Time-limited URL to get the .htm preview file for this notebook.}
#' \item{requirementsUrl}{string, Time-limited URL to get the requirements.txt file for this notebook.}
#' \item{fileId}{string, The file ID for the S3 file containing the .ipynb file.}
#' \item{requirementsFileId}{string, The file ID for the S3 file containing the requirements.txt file.}
#' \item{user}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{dockerImageName}{string, The name of the docker image to pull from DockerHub.}
#' \item{dockerImageTag}{string, The tag of the docker image to pull from DockerHub (default: latest).}
#' \item{instanceType}{string, The EC2 instance type to deploy to.}
#' \item{memory}{integer, The amount of memory allocated to the notebook.}
#' \item{cpu}{integer, The amount of cpu allocated to the the notebook.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{mostRecentDeployment}{object, A list containing the following elements: 
#' \itemize{
#' \item deploymentId integer, The ID for this deployment.
#' \item userId integer, The ID of the owner.
#' \item host string, Domain of the deployment.
#' \item name string, Name of the deployment.
#' \item dockerImageName string, The name of the docker image to pull from DockerHub.
#' \item dockerImageTag string, The tag of the docker image to pull from DockerHub (default: latest).
#' \item displayUrl string, URL that displays the deployed platform object and includes auth token.
#' \item instanceType string, The EC2 instance type requested for the deployment.
#' \item memory integer, The memory allocated to the deployment.
#' \item cpu integer, The cpu allocated to the deployment.
#' \item state string, The state of the deployment.
#' \item stateMessage string, A detailed description of the state.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item published boolean, 
#' \item notebookId integer, The ID of owning Notebook
#' }}
#' \item{credentials}{array, A list of credential IDs to pass to the notebook.}
#' \item{environmentVariables}{object, Environment variables to be passed into the Notebook.}
#' \item{idleTimeout}{integer, How long the notebook will stay alive without any kernel activity.}
#' \item{gitRepoId}{integer, The ID of the git repository.}
#' \item{gitRepoUrl}{string, The url of the git repository}
#' \item{gitRef}{string, The git reference if git repo is specified}
#' \item{gitPath}{string, The path to the .ipynb file in the git repo that will be started up on notebook launch}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{hidden}{boolean, The hidden status of the object.}
#' @export
notebooks_put <- function(id, name = NULL, language = NULL, description = NULL, file_id = NULL, requirements_file_id = NULL, requirements = NULL, docker_image_name = NULL, docker_image_tag = NULL, instance_type = NULL, memory = NULL, cpu = NULL, credentials = NULL, environment_variables = NULL, idle_timeout = NULL, git_repo_url = NULL, git_ref = NULL, git_path = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(name = name, language = language, description = description, fileId = file_id, requirementsFileId = requirements_file_id, requirements = requirements, dockerImageName = docker_image_name, dockerImageTag = docker_image_tag, instanceType = instance_type, memory = memory, cpu = cpu, credentials = credentials, environmentVariables = environment_variables, idleTimeout = idle_timeout, gitRepoUrl = git_repo_url, gitRef = git_ref, gitPath = git_path)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update some attributes of this Notebook
#' @param id integer required. The ID for this notebook.
#' @param name string optional. The name of this notebook.
#' @param language string optional. The kernel language of this notebook.
#' @param description string optional. The description of this notebook.
#' @param file_id string optional. The file ID for the S3 file containing the .ipynb file.
#' @param requirements_file_id string optional. The file ID for the S3 file containing the requirements.txt file.
#' @param requirements string optional. The requirements txt file.
#' @param docker_image_name string optional. The name of the docker image to pull from DockerHub.
#' @param docker_image_tag string optional. The tag of the docker image to pull from DockerHub (default: latest).
#' @param instance_type string optional. The EC2 instance type to deploy to.
#' @param memory integer optional. The amount of memory allocated to the notebook.
#' @param cpu integer optional. The amount of cpu allocated to the the notebook.
#' @param credentials array optional. A list of credential IDs to pass to the notebook.
#' @param environment_variables object optional. Environment variables to be passed into the Notebook.
#' @param idle_timeout integer optional. How long the notebook will stay alive without any kernel activity.
#' @param git_repo_url string optional. The url of the git repository
#' @param git_ref string optional. The git reference if git repo is specified
#' @param git_path string optional. The path to the .ipynb file in the git repo that will be started up on notebook launch
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for this notebook.}
#' \item{name}{string, The name of this notebook.}
#' \item{language}{string, The kernel language of this notebook.}
#' \item{description}{string, The description of this notebook.}
#' \item{notebookUrl}{string, Time-limited URL to get the .ipynb file for this notebook.}
#' \item{notebookPreviewUrl}{string, Time-limited URL to get the .htm preview file for this notebook.}
#' \item{requirementsUrl}{string, Time-limited URL to get the requirements.txt file for this notebook.}
#' \item{fileId}{string, The file ID for the S3 file containing the .ipynb file.}
#' \item{requirementsFileId}{string, The file ID for the S3 file containing the requirements.txt file.}
#' \item{user}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{dockerImageName}{string, The name of the docker image to pull from DockerHub.}
#' \item{dockerImageTag}{string, The tag of the docker image to pull from DockerHub (default: latest).}
#' \item{instanceType}{string, The EC2 instance type to deploy to.}
#' \item{memory}{integer, The amount of memory allocated to the notebook.}
#' \item{cpu}{integer, The amount of cpu allocated to the the notebook.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{mostRecentDeployment}{object, A list containing the following elements: 
#' \itemize{
#' \item deploymentId integer, The ID for this deployment.
#' \item userId integer, The ID of the owner.
#' \item host string, Domain of the deployment.
#' \item name string, Name of the deployment.
#' \item dockerImageName string, The name of the docker image to pull from DockerHub.
#' \item dockerImageTag string, The tag of the docker image to pull from DockerHub (default: latest).
#' \item displayUrl string, URL that displays the deployed platform object and includes auth token.
#' \item instanceType string, The EC2 instance type requested for the deployment.
#' \item memory integer, The memory allocated to the deployment.
#' \item cpu integer, The cpu allocated to the deployment.
#' \item state string, The state of the deployment.
#' \item stateMessage string, A detailed description of the state.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item published boolean, 
#' \item notebookId integer, The ID of owning Notebook
#' }}
#' \item{credentials}{array, A list of credential IDs to pass to the notebook.}
#' \item{environmentVariables}{object, Environment variables to be passed into the Notebook.}
#' \item{idleTimeout}{integer, How long the notebook will stay alive without any kernel activity.}
#' \item{gitRepoId}{integer, The ID of the git repository.}
#' \item{gitRepoUrl}{string, The url of the git repository}
#' \item{gitRef}{string, The git reference if git repo is specified}
#' \item{gitPath}{string, The path to the .ipynb file in the git repo that will be started up on notebook launch}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{hidden}{boolean, The hidden status of the object.}
#' @export
notebooks_patch <- function(id, name = NULL, language = NULL, description = NULL, file_id = NULL, requirements_file_id = NULL, requirements = NULL, docker_image_name = NULL, docker_image_tag = NULL, instance_type = NULL, memory = NULL, cpu = NULL, credentials = NULL, environment_variables = NULL, idle_timeout = NULL, git_repo_url = NULL, git_ref = NULL, git_path = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(name = name, language = language, description = description, fileId = file_id, requirementsFileId = requirements_file_id, requirements = requirements, dockerImageName = docker_image_name, dockerImageTag = docker_image_tag, instanceType = instance_type, memory = memory, cpu = cpu, credentials = credentials, environmentVariables = environment_variables, idleTimeout = idle_timeout, gitRepoUrl = git_repo_url, gitRef = git_ref, gitPath = git_path)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("patch", path, path_params, query_params, body_params)

  return(resp)

 }


#' Archive a Notebook (deprecated, use archiving endpoints instead)
#' @param id integer required. 
#' 
#' @return  An empty HTTP response
#' @export
notebooks_delete <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get URLs to update notebook
#' @param id integer required. 
#' 
#' @return  A list containing the following elements:
#' \item{updateUrl}{string, Time-limited URL to PUT new contents of the .ipynb file for this notebook.}
#' \item{updatePreviewUrl}{string, Time-limited URL to PUT new contents of the .htm preview file for this notebook.}
#' @export
notebooks_list_update_links <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/{id}/update-links"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Clone this Notebook
#' @param id integer required. 
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for this notebook.}
#' \item{name}{string, The name of this notebook.}
#' \item{language}{string, The kernel language of this notebook.}
#' \item{description}{string, The description of this notebook.}
#' \item{notebookUrl}{string, Time-limited URL to get the .ipynb file for this notebook.}
#' \item{notebookPreviewUrl}{string, Time-limited URL to get the .htm preview file for this notebook.}
#' \item{requirementsUrl}{string, Time-limited URL to get the requirements.txt file for this notebook.}
#' \item{fileId}{string, The file ID for the S3 file containing the .ipynb file.}
#' \item{requirementsFileId}{string, The file ID for the S3 file containing the requirements.txt file.}
#' \item{user}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{dockerImageName}{string, The name of the docker image to pull from DockerHub.}
#' \item{dockerImageTag}{string, The tag of the docker image to pull from DockerHub (default: latest).}
#' \item{instanceType}{string, The EC2 instance type to deploy to.}
#' \item{memory}{integer, The amount of memory allocated to the notebook.}
#' \item{cpu}{integer, The amount of cpu allocated to the the notebook.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{mostRecentDeployment}{object, A list containing the following elements: 
#' \itemize{
#' \item deploymentId integer, The ID for this deployment.
#' \item userId integer, The ID of the owner.
#' \item host string, Domain of the deployment.
#' \item name string, Name of the deployment.
#' \item dockerImageName string, The name of the docker image to pull from DockerHub.
#' \item dockerImageTag string, The tag of the docker image to pull from DockerHub (default: latest).
#' \item displayUrl string, URL that displays the deployed platform object and includes auth token.
#' \item instanceType string, The EC2 instance type requested for the deployment.
#' \item memory integer, The memory allocated to the deployment.
#' \item cpu integer, The cpu allocated to the deployment.
#' \item state string, The state of the deployment.
#' \item stateMessage string, A detailed description of the state.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item published boolean, 
#' \item notebookId integer, The ID of owning Notebook
#' }}
#' \item{credentials}{array, A list of credential IDs to pass to the notebook.}
#' \item{environmentVariables}{object, Environment variables to be passed into the Notebook.}
#' \item{idleTimeout}{integer, How long the notebook will stay alive without any kernel activity.}
#' \item{gitRepoId}{integer, The ID of the git repository.}
#' \item{gitRepoUrl}{string, The url of the git repository}
#' \item{gitRef}{string, The git reference if git repo is specified}
#' \item{gitPath}{string, The path to the .ipynb file in the git repo that will be started up on notebook launch}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{hidden}{boolean, The hidden status of the object.}
#' @export
notebooks_post_clone <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/{id}/clone"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' List users and groups permissioned on this object
#' @param id integer required. The ID of the object.
#' 
#' @return  An array containing the following fields:
#' \item{readers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
notebooks_list_shares <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/{id}/shares"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions users have on this object
#' @param id integer required. ID of the resource to be shared
#' @param user_ids array required. An array of one or more user IDs
#' @param permission_level string required. Options are: "read", "write", or "manage"
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
notebooks_put_shares_users <- function(id, user_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/{id}/shares/users"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(userIds = user_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a user has on this object
#' @param id integer required. ID of the resource to be revoked
#' @param user_id integer required. ID of the user
#' 
#' @return  An empty HTTP response
#' @export
notebooks_delete_shares_users <- function(id, user_id) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/{id}/shares/users/{user_id}"
  path_params  <- list(id = id, user_id = user_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions groups has on this object
#' @param id integer required. ID of the resource to be shared
#' @param group_ids array required. An array of one or more group IDs
#' @param permission_level string required. Options are: "read", "write", or "manage"
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
notebooks_put_shares_groups <- function(id, group_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/{id}/shares/groups"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(groupIds = group_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a group has on this object
#' @param id integer required. ID of the resource to be revoked
#' @param group_id integer required. ID of the group
#' 
#' @return  An empty HTTP response
#' @export
notebooks_delete_shares_groups <- function(id, group_id) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/{id}/shares/groups/{group_id}"
  path_params  <- list(id = id, group_id = group_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update the archive status of this object
#' @param id integer required. The ID of the object.
#' @param status boolean required. The desired archived status of the object.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for this notebook.}
#' \item{name}{string, The name of this notebook.}
#' \item{language}{string, The kernel language of this notebook.}
#' \item{description}{string, The description of this notebook.}
#' \item{notebookUrl}{string, Time-limited URL to get the .ipynb file for this notebook.}
#' \item{notebookPreviewUrl}{string, Time-limited URL to get the .htm preview file for this notebook.}
#' \item{requirementsUrl}{string, Time-limited URL to get the requirements.txt file for this notebook.}
#' \item{fileId}{string, The file ID for the S3 file containing the .ipynb file.}
#' \item{requirementsFileId}{string, The file ID for the S3 file containing the requirements.txt file.}
#' \item{user}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{dockerImageName}{string, The name of the docker image to pull from DockerHub.}
#' \item{dockerImageTag}{string, The tag of the docker image to pull from DockerHub (default: latest).}
#' \item{instanceType}{string, The EC2 instance type to deploy to.}
#' \item{memory}{integer, The amount of memory allocated to the notebook.}
#' \item{cpu}{integer, The amount of cpu allocated to the the notebook.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{mostRecentDeployment}{object, A list containing the following elements: 
#' \itemize{
#' \item deploymentId integer, The ID for this deployment.
#' \item userId integer, The ID of the owner.
#' \item host string, Domain of the deployment.
#' \item name string, Name of the deployment.
#' \item dockerImageName string, The name of the docker image to pull from DockerHub.
#' \item dockerImageTag string, The tag of the docker image to pull from DockerHub (default: latest).
#' \item displayUrl string, URL that displays the deployed platform object and includes auth token.
#' \item instanceType string, The EC2 instance type requested for the deployment.
#' \item memory integer, The memory allocated to the deployment.
#' \item cpu integer, The cpu allocated to the deployment.
#' \item state string, The state of the deployment.
#' \item stateMessage string, A detailed description of the state.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item published boolean, 
#' \item notebookId integer, The ID of owning Notebook
#' }}
#' \item{credentials}{array, A list of credential IDs to pass to the notebook.}
#' \item{environmentVariables}{object, Environment variables to be passed into the Notebook.}
#' \item{idleTimeout}{integer, How long the notebook will stay alive without any kernel activity.}
#' \item{gitRepoId}{integer, The ID of the git repository.}
#' \item{gitRepoUrl}{string, The url of the git repository}
#' \item{gitRef}{string, The git reference if git repo is specified}
#' \item{gitPath}{string, The path to the .ipynb file in the git repo that will be started up on notebook launch}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{hidden}{boolean, The hidden status of the object.}
#' @export
notebooks_put_archive <- function(id, status) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/{id}/archive"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(status = status)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' List the projects a Notebook belongs to
#' @param id integer required. The ID of the resource.
#' @param hidden boolean optional. If specified to be true, returns hidden objects. Defaults to false, returning non-hidden objects.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID for this project.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{name}{string, The name of this project.}
#' \item{description}{string, A description of the project}
#' \item{users}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{autoShare}{boolean, }
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{archived}{string, The archival status of the requested object(s).}
#' @export
notebooks_list_projects <- function(id, hidden = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/{id}/projects"
  path_params  <- list(id = id)
  query_params <- list(hidden = hidden)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Add a Notebook to a project
#' @param id integer required. ID of the resource
#' @param project_id integer required. The ID of the project
#' 
#' @return  An empty HTTP response
#' @export
notebooks_put_projects <- function(id, project_id) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/{id}/projects/{project_id}"
  path_params  <- list(id = id, project_id = project_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Remove a Notebook from a project
#' @param id integer required. ID of the resource
#' @param project_id integer required. The ID of the project
#' 
#' @return  An empty HTTP response
#' @export
notebooks_delete_projects <- function(id, project_id) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/{id}/projects/{project_id}"
  path_params  <- list(id = id, project_id = project_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' List deployments for a Notebook
#' @param notebook_id integer required. The ID of the owning Notebook
#' @param deployment_id integer optional. The ID for this deployment
#' @param limit integer optional. Number of results to return. Defaults to 20. Maximum allowed is 50.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to created_at. Must be one of: created_at.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{deploymentId}{integer, The ID for this deployment.}
#' \item{userId}{integer, The ID of the owner.}
#' \item{host}{string, Domain of the deployment.}
#' \item{name}{string, Name of the deployment.}
#' \item{dockerImageName}{string, The name of the docker image to pull from DockerHub.}
#' \item{dockerImageTag}{string, The tag of the docker image to pull from DockerHub (default: latest).}
#' \item{instanceType}{string, The EC2 instance type requested for the deployment.}
#' \item{memory}{integer, The memory allocated to the deployment.}
#' \item{cpu}{integer, The cpu allocated to the deployment.}
#' \item{state}{string, The state of the deployment.}
#' \item{stateMessage}{string, A detailed description of the state.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{published}{boolean, }
#' \item{notebookId}{integer, The ID of owning Notebook}
#' @export
notebooks_list_deployments <- function(notebook_id, deployment_id = NULL, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/{notebook_id}/deployments"
  path_params  <- list(notebook_id = notebook_id)
  query_params <- list(deployment_id = deployment_id, limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Deploy a Notebook
#' @param notebook_id integer required. The ID of the owning Notebook
#' @param deployment_id integer optional. The ID for this deployment
#' @param published boolean optional. 
#' 
#' @return  A list containing the following elements:
#' \item{deploymentId}{integer, The ID for this deployment.}
#' \item{userId}{integer, The ID of the owner.}
#' \item{host}{string, Domain of the deployment.}
#' \item{name}{string, Name of the deployment.}
#' \item{dockerImageName}{string, The name of the docker image to pull from DockerHub.}
#' \item{dockerImageTag}{string, The tag of the docker image to pull from DockerHub (default: latest).}
#' \item{displayUrl}{string, URL that displays the deployed platform object and includes auth token.}
#' \item{instanceType}{string, The EC2 instance type requested for the deployment.}
#' \item{memory}{integer, The memory allocated to the deployment.}
#' \item{cpu}{integer, The cpu allocated to the deployment.}
#' \item{state}{string, The state of the deployment.}
#' \item{stateMessage}{string, A detailed description of the state.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{published}{boolean, }
#' \item{notebookId}{integer, The ID of owning Notebook}
#' @export
notebooks_post_deployments <- function(notebook_id, deployment_id = NULL, published = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/{notebook_id}/deployments"
  path_params  <- list(notebook_id = notebook_id)
  query_params <- list()
  body_params  <- list(deploymentId = deployment_id, published = published)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get details about a Notebook deployment
#' @param notebook_id integer required. The ID of the owning Notebook
#' @param deployment_id integer required. The ID for this deployment
#' 
#' @return  A list containing the following elements:
#' \item{deploymentId}{integer, The ID for this deployment.}
#' \item{userId}{integer, The ID of the owner.}
#' \item{host}{string, Domain of the deployment.}
#' \item{name}{string, Name of the deployment.}
#' \item{dockerImageName}{string, The name of the docker image to pull from DockerHub.}
#' \item{dockerImageTag}{string, The tag of the docker image to pull from DockerHub (default: latest).}
#' \item{displayUrl}{string, URL that displays the deployed platform object and includes auth token.}
#' \item{instanceType}{string, The EC2 instance type requested for the deployment.}
#' \item{memory}{integer, The memory allocated to the deployment.}
#' \item{cpu}{integer, The cpu allocated to the deployment.}
#' \item{state}{string, The state of the deployment.}
#' \item{stateMessage}{string, A detailed description of the state.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{published}{boolean, }
#' \item{notebookId}{integer, The ID of owning Notebook}
#' @export
notebooks_get_deployments <- function(notebook_id, deployment_id) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/{notebook_id}/deployments/{deployment_id}"
  path_params  <- list(notebook_id = notebook_id, deployment_id = deployment_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Delete a Notebook deployment
#' @param notebook_id integer required. The ID of the owning Notebook
#' @param deployment_id integer required. The ID for this deployment
#' 
#' @return  An empty HTTP response
#' @export
notebooks_delete_deployments <- function(notebook_id, deployment_id) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/{notebook_id}/deployments/{deployment_id}"
  path_params  <- list(notebook_id = notebook_id, deployment_id = deployment_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get the git metadata attached to this Notebook
#' @param id integer required. The ID of the object.
#' 
#' @return  A list containing the following elements:
#' \item{gitRef}{string, A git reference specifying an unambiguous version of the file. Can be a branch name, or the full or shortened SHA of a commit.}
#' \item{gitBranch}{string, The git branch that the file is on.}
#' \item{gitPath}{string, The path of the file in the repository.}
#' \item{gitRepo}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID for this git repository.
#' \item repoUrl string, The URL for this git repository.
#' \item createdAt string, 
#' \item updatedAt string, 
#' }}
#' @export
notebooks_list_git <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/{id}/git"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Attach this Notebook to a git repo/file
#' @param id integer required. The ID of the object.
#' @param git_ref string optional. A git reference specifying an unambiguous version of the file. Can be a branch name, or the full or shortened SHA of a commit.
#' @param git_branch string optional. The git branch that the file is on.
#' @param git_path string optional. The path of the file in the repository.
#' @param git_repo_url string optional. The URL of the git repository.
#' 
#' @return  A list containing the following elements:
#' \item{gitRef}{string, A git reference specifying an unambiguous version of the file. Can be a branch name, or the full or shortened SHA of a commit.}
#' \item{gitBranch}{string, The git branch that the file is on.}
#' \item{gitPath}{string, The path of the file in the repository.}
#' \item{gitRepo}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID for this git repository.
#' \item repoUrl string, The URL for this git repository.
#' \item createdAt string, 
#' \item updatedAt string, 
#' }}
#' @export
notebooks_put_git <- function(id, git_ref = NULL, git_branch = NULL, git_path = NULL, git_repo_url = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/{id}/git"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(gitRef = git_ref, gitBranch = git_branch, gitPath = git_path, gitRepoUrl = git_repo_url)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get the git commits for this Notebook
#' @param id integer required. The ID of the object.
#' 
#' @return  A list containing the following elements:
#' \item{commitHash}{string, The SHA of the commit.}
#' \item{authorName}{string, The name of the commit's author.}
#' \item{date}{string, The commit's timestamp.}
#' \item{message}{string, The commit message.}
#' @export
notebooks_list_git_commits <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/{id}/git/commits"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Commit and push a new version of the file
#' @param id integer required. The ID of the object.
#' @param content string required. The contents to commit to the file.
#' @param message string required. A commit message describing the changes being made.
#' @param file_hash string required. The full SHA of the file being replaced.
#' 
#' @return  A list containing the following elements:
#' \item{content}{string, The file's contents.}
#' \item{type}{string, The file's type.}
#' \item{size}{integer, The file's size.}
#' \item{fileHash}{string, The SHA of the file.}
#' @export
notebooks_post_git_commits <- function(id, content, message, file_hash) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/{id}/git/commits"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(content = content, message = message, fileHash = file_hash)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get file contents at commit_hash
#' @param id integer required. The ID of the object.
#' @param commit_hash string required. The SHA (full or shortened) of the desired git commit.
#' 
#' @return  A list containing the following elements:
#' \item{content}{string, The file's contents.}
#' \item{type}{string, The file's type.}
#' \item{size}{integer, The file's size.}
#' \item{fileHash}{string, The SHA of the file.}
#' @export
notebooks_get_git_commits <- function(id, commit_hash) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/{id}/git/commits/{commit_hash}"
  path_params  <- list(id = id, commit_hash = commit_hash)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Receive a stream of notifications as they come in
#' @param last_event_id string optional. allows browser to keep track of last event fired
#' @param r string optional. specifies retry/reconnect timeout
#' @param mock string optional. used for testing
#' 
#' @return  An empty HTTP response
#' @export
notifications_list <- function(last_event_id = NULL, r = NULL, mock = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/notifications/"
  path_params  <- list()
  query_params <- list(last_event_id = last_event_id, r = r, mock = mock)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' List the ontology of column names Civis uses
#' @param subset string optional. A subset of fields to return.
#' 
#' @return  An array containing the following fields:
#' \item{key}{string, }
#' \item{title}{string, }
#' \item{desc}{string, A description of this field.}
#' \item{aliases}{array, }
#' @export
ontology_list <- function(subset = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/ontology/"
  path_params  <- list()
  query_params <- list(subset = subset)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' List predictions
#' @param model_id integer optional. If specified, only return predictions associated with this model ID.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of the prediction.}
#' \item{modelId}{integer, The ID of the model used for this prediction.}
#' \item{scoredTableId}{integer, The ID of the source table for this prediction.}
#' \item{scoredTableName}{string, The name of the source table for this prediction.}
#' \item{outputTableName}{string, The name of the output table for this prediction.}
#' \item{state}{string, The state of the last run of this prediction.}
#' \item{error}{string, The error, if any, of the last run of this prediction.}
#' \item{startedAt}{string, The start time of the last run of this prediction.}
#' \item{finishedAt}{string, The end time of the last run of this prediction.}
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' @export
predictions_list <- function(model_id = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/predictions/"
  path_params  <- list()
  query_params <- list(model_id = model_id)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Show the specified prediction
#' @param id integer required. The ID of the prediction.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the prediction.}
#' \item{modelId}{integer, The ID of the model used for this prediction.}
#' \item{scoredTableId}{integer, The ID of the source table for this prediction.}
#' \item{scoredTableName}{string, The name of the source table for this prediction.}
#' \item{outputTableName}{string, The name of the output table for this prediction.}
#' \item{state}{string, The state of the last run of this prediction.}
#' \item{error}{string, The error, if any, of the last run of this prediction.}
#' \item{startedAt}{string, The start time of the last run of this prediction.}
#' \item{finishedAt}{string, The end time of the last run of this prediction.}
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{scoredTables}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of the table with created predictions.
#' \item schema string, The schema of table with created predictions.
#' \item name string, The name of table with created predictions.
#' \item createdAt string, The time when the table with created predictions was created.
#' \item scoreStats array, An array of metrics on the created predictions.
#' }}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{limitingSQL}{string, A SQL WHERE clause used to scope the rows to be predicted.}
#' \item{primaryKey}{array, The primary key or composite keys of the table being predicted.}
#' @export
predictions_get <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/predictions/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update a prediction
#' @param id integer required. The ID of the prediction.
#' @param output_table_name string optional. The name of the output table for this prediction.
#' @param limiting_sql string optional. A SQL WHERE clause used to scope the rows to be predicted.
#' @param primary_key array optional. The primary key or composite keys of the table being predicted.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the prediction.}
#' \item{modelId}{integer, The ID of the model used for this prediction.}
#' \item{scoredTableId}{integer, The ID of the source table for this prediction.}
#' \item{scoredTableName}{string, The name of the source table for this prediction.}
#' \item{outputTableName}{string, The name of the output table for this prediction.}
#' \item{state}{string, The state of the last run of this prediction.}
#' \item{error}{string, The error, if any, of the last run of this prediction.}
#' \item{startedAt}{string, The start time of the last run of this prediction.}
#' \item{finishedAt}{string, The end time of the last run of this prediction.}
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{scoredTables}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of the table with created predictions.
#' \item schema string, The schema of table with created predictions.
#' \item name string, The name of table with created predictions.
#' \item createdAt string, The time when the table with created predictions was created.
#' \item scoreStats array, An array of metrics on the created predictions.
#' }}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{limitingSQL}{string, A SQL WHERE clause used to scope the rows to be predicted.}
#' \item{primaryKey}{array, The primary key or composite keys of the table being predicted.}
#' @export
predictions_patch <- function(id, output_table_name = NULL, limiting_sql = NULL, primary_key = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/predictions/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(outputTableName = output_table_name, limitingSQL = limiting_sql, primaryKey = primary_key)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("patch", path, path_params, query_params, body_params)

  return(resp)

 }


#' Show the prediction schedule
#' @param id integer required. ID of the prediction associated with this schedule.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, ID of the prediction associated with this schedule.}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{scoreOnModelBuild}{boolean, Whether the prediction will run after a rebuild of the associated model.}
#' @export
predictions_list_schedules <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/predictions/{id}/schedules"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Schedule the prediction
#' @param id integer required. ID of the prediction associated with this schedule.
#' @param schedule object optional. A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }
#' @param score_on_model_build boolean optional. Whether the prediction will run after a rebuild of the associated model.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, ID of the prediction associated with this schedule.}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{scoreOnModelBuild}{boolean, Whether the prediction will run after a rebuild of the associated model.}
#' @export
predictions_put_schedules <- function(id, schedule = NULL, score_on_model_build = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/predictions/{id}/schedules"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(schedule = schedule, scoreOnModelBuild = score_on_model_build)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Start a run
#' @param id integer required. The ID of the prediction.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the prediction run.}
#' \item{predictionId}{integer, The ID of the prediction.}
#' \item{state}{string, The state of the prediction run.}
#' \item{exception}{string, The exception, if any, returned by the prediction run.}
#' \item{name}{string, The name of table created by this predictions run.}
#' \item{createdAt}{string, The time when the table with created predictions was created.}
#' \item{scoreStats}{array, An array containing the following fields: 
#' \itemize{
#' \item scoreName string, The name of the score.
#' \item histogram array, The histogram of the distribution of scores.
#' \item avgScore number, The average score.
#' \item minScore number, The minimum score.
#' \item maxScore number, The maximum score.
#' }}
#' @export
predictions_post_runs <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/predictions/{id}/runs"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' List runs for the given prediction
#' @param id integer required. The ID of the prediction.
#' @param limit integer optional. Number of results to return. Defaults to 20. Maximum allowed is 100.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to id. Must be one of: id.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of the prediction run.}
#' \item{predictionId}{integer, The ID of the prediction.}
#' \item{state}{string, The state of the prediction run.}
#' \item{exception}{string, The exception, if any, returned by the prediction run.}
#' \item{name}{string, The name of table created by this predictions run.}
#' \item{createdAt}{string, The time when the table with created predictions was created.}
#' \item{scoreStats}{array, An array containing the following fields: 
#' \itemize{
#' \item scoreName string, The name of the score.
#' \item histogram array, The histogram of the distribution of scores.
#' \item avgScore number, The average score.
#' \item minScore number, The minimum score.
#' \item maxScore number, The maximum score.
#' }}
#' @export
predictions_list_runs <- function(id, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/predictions/{id}/runs"
  path_params  <- list(id = id)
  query_params <- list(limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Check status of a run
#' @param id integer required. The ID of the prediction.
#' @param run_id integer required. The ID of the run.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the prediction run.}
#' \item{predictionId}{integer, The ID of the prediction.}
#' \item{state}{string, The state of the prediction run.}
#' \item{exception}{string, The exception, if any, returned by the prediction run.}
#' \item{name}{string, The name of table created by this predictions run.}
#' \item{createdAt}{string, The time when the table with created predictions was created.}
#' \item{scoreStats}{array, An array containing the following fields: 
#' \itemize{
#' \item scoreName string, The name of the score.
#' \item histogram array, The histogram of the distribution of scores.
#' \item avgScore number, The average score.
#' \item minScore number, The minimum score.
#' \item maxScore number, The maximum score.
#' }}
#' @export
predictions_get_runs <- function(id, run_id) {

  args <- as.list(match.call())[-1]
  path <- "/predictions/{id}/runs/{run_id}"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Cancel a run
#' @param id integer required. The ID of the prediction.
#' @param run_id integer required. The ID of the run.
#' 
#' @return  An empty HTTP response
#' @export
predictions_delete_runs <- function(id, run_id) {

  args <- as.list(match.call())[-1]
  path <- "/predictions/{id}/runs/{run_id}"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get the logs for a run
#' @param id integer required. The ID of the prediction.
#' @param run_id integer required. The ID of the run.
#' @param last_id integer optional. The ID of the last log message received. Log entries with this ID value or lower will be omitted.Logs are sorted by ID if this value is provided, and are otherwise sorted by createdAt.
#' @param limit integer optional. The maximum number of log messages to return. Default of 10000.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of the log.}
#' \item{createdAt}{string, The time the log was created.}
#' \item{message}{string, The log message.}
#' \item{level}{string, The level of the log. One of unknown,fatal,error,warn,info,debug.}
#' @export
predictions_list_runs_logs <- function(id, run_id, last_id = NULL, limit = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/predictions/{id}/runs/{run_id}/logs"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list(last_id = last_id, limit = limit)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' List projects
#' @param author string optional. If specified, return projects owned by this author. It accepts a comma-separated list of author ids.
#' @param permission string optional. A permissions string, one of "read", "write", or "manage". Lists only projects for which the current user has that permission.
#' @param hidden boolean optional. If specified to be true, returns hidden objects. Defaults to false, returning non-hidden objects.
#' @param archived string optional. The archival status of the requested object(s).
#' @param limit integer optional. Number of results to return. Defaults to 20. Maximum allowed is 1000.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to updated_at. Must be one of: updated_at, name, created_at.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID for this project.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{name}{string, The name of this project.}
#' \item{description}{string, A description of the project}
#' \item{users}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{autoShare}{boolean, }
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{archived}{string, The archival status of the requested object(s).}
#' @export
projects_list <- function(author = NULL, permission = NULL, hidden = NULL, archived = NULL, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/projects/"
  path_params  <- list()
  query_params <- list(author = author, permission = permission, hidden = hidden, archived = archived, limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Create a project
#' @param name string required. The name of this project.
#' @param description string required. A description of the project
#' @param note string optional. Notes for the project
#' @param hidden boolean optional. The hidden status of the object.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for this project.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{name}{string, The name of this project.}
#' \item{description}{string, A description of the project}
#' \item{users}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{autoShare}{boolean, }
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{tables}{array, An array containing the following fields: 
#' \itemize{
#' \item schema string, 
#' \item name string, 
#' \item rowCount integer, 
#' \item columnCount integer, 
#' \item createdAt string, 
#' \item updatedAt string, 
#' }}
#' \item{surveys}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The object ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' }}
#' \item{scripts}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The object ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item type string, 
#' \item finishedAt string, 
#' \item state string, 
#' }}
#' \item{imports}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The object ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item type string, 
#' \item finishedAt string, 
#' \item state string, 
#' }}
#' \item{models}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The object ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item state string, 
#' }}
#' \item{notebooks}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The object ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item currentDeploymentId integer, 
#' }}
#' \item{services}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The object ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item currentDeploymentId integer, 
#' }}
#' \item{workflows}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The object ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item state string, 
#' }}
#' \item{reports}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The object ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item state string, 
#' }}
#' \item{scriptTemplates}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The object ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' }}
#' \item{files}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The object ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item fileName string, 
#' \item fileSize integer, 
#' \item expired boolean, 
#' }}
#' \item{appInstances}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The object ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item slug string, 
#' }}
#' \item{allObjects}{array, An array containing the following fields: 
#' \itemize{
#' \item projectId integer, 
#' \item objectId integer, 
#' \item objectType string, 
#' \item fcoType string, 
#' \item subType string, 
#' \item name string, 
#' \item icon string, 
#' \item author string, 
#' \item archived string, The archival status of the requested object(s).
#' }}
#' \item{note}{string, }
#' \item{hidden}{boolean, The hidden status of the object.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' @export
projects_post <- function(name, description, note = NULL, hidden = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/projects/"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(name = name, description = description, note = note, hidden = hidden)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get a detailed view of a project and the objects in it
#' @param project_id integer required. 
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for this project.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{name}{string, The name of this project.}
#' \item{description}{string, A description of the project}
#' \item{users}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{autoShare}{boolean, }
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{tables}{array, An array containing the following fields: 
#' \itemize{
#' \item schema string, 
#' \item name string, 
#' \item rowCount integer, 
#' \item columnCount integer, 
#' \item createdAt string, 
#' \item updatedAt string, 
#' }}
#' \item{surveys}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The object ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' }}
#' \item{scripts}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The object ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item type string, 
#' \item finishedAt string, 
#' \item state string, 
#' }}
#' \item{imports}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The object ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item type string, 
#' \item finishedAt string, 
#' \item state string, 
#' }}
#' \item{models}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The object ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item state string, 
#' }}
#' \item{notebooks}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The object ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item currentDeploymentId integer, 
#' }}
#' \item{services}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The object ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item currentDeploymentId integer, 
#' }}
#' \item{workflows}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The object ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item state string, 
#' }}
#' \item{reports}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The object ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item state string, 
#' }}
#' \item{scriptTemplates}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The object ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' }}
#' \item{files}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The object ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item fileName string, 
#' \item fileSize integer, 
#' \item expired boolean, 
#' }}
#' \item{appInstances}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The object ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item slug string, 
#' }}
#' \item{allObjects}{array, An array containing the following fields: 
#' \itemize{
#' \item projectId integer, 
#' \item objectId integer, 
#' \item objectType string, 
#' \item fcoType string, 
#' \item subType string, 
#' \item name string, 
#' \item icon string, 
#' \item author string, 
#' \item archived string, The archival status of the requested object(s).
#' }}
#' \item{note}{string, }
#' \item{hidden}{boolean, The hidden status of the object.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' @export
projects_get <- function(project_id) {

  args <- as.list(match.call())[-1]
  path <- "/projects/{project_id}"
  path_params  <- list(project_id = project_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update a project
#' @param project_id integer required. 
#' @param name string optional. The name of this project.
#' @param description string optional. A description of the project
#' @param note string optional. Notes for the project
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for this project.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{name}{string, The name of this project.}
#' \item{description}{string, A description of the project}
#' \item{users}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{autoShare}{boolean, }
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{tables}{array, An array containing the following fields: 
#' \itemize{
#' \item schema string, 
#' \item name string, 
#' \item rowCount integer, 
#' \item columnCount integer, 
#' \item createdAt string, 
#' \item updatedAt string, 
#' }}
#' \item{surveys}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The object ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' }}
#' \item{scripts}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The object ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item type string, 
#' \item finishedAt string, 
#' \item state string, 
#' }}
#' \item{imports}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The object ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item type string, 
#' \item finishedAt string, 
#' \item state string, 
#' }}
#' \item{models}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The object ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item state string, 
#' }}
#' \item{notebooks}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The object ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item currentDeploymentId integer, 
#' }}
#' \item{services}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The object ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item currentDeploymentId integer, 
#' }}
#' \item{workflows}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The object ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item state string, 
#' }}
#' \item{reports}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The object ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item state string, 
#' }}
#' \item{scriptTemplates}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The object ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' }}
#' \item{files}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The object ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item fileName string, 
#' \item fileSize integer, 
#' \item expired boolean, 
#' }}
#' \item{appInstances}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The object ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item slug string, 
#' }}
#' \item{allObjects}{array, An array containing the following fields: 
#' \itemize{
#' \item projectId integer, 
#' \item objectId integer, 
#' \item objectType string, 
#' \item fcoType string, 
#' \item subType string, 
#' \item name string, 
#' \item icon string, 
#' \item author string, 
#' \item archived string, The archival status of the requested object(s).
#' }}
#' \item{note}{string, }
#' \item{hidden}{boolean, The hidden status of the object.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' @export
projects_put <- function(project_id, name = NULL, description = NULL, note = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/projects/{project_id}"
  path_params  <- list(project_id = project_id)
  query_params <- list()
  body_params  <- list(name = name, description = description, note = note)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Archive a project (deprecated, use the /archive endpoint instead)
#' @param project_id integer required. 
#' 
#' @return  An empty HTTP response
#' @export
projects_delete <- function(project_id) {

  args <- as.list(match.call())[-1]
  path <- "/projects/{project_id}"
  path_params  <- list(project_id = project_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' List users and groups permissioned on this object
#' @param id integer required. The ID of the object.
#' 
#' @return  An array containing the following fields:
#' \item{readers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
projects_list_shares <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/projects/{id}/shares"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions users have on this object
#' @param id integer required. ID of the resource to be shared
#' @param user_ids array required. An array of one or more user IDs
#' @param permission_level string required. Options are: "read", "write", or "manage"
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
projects_put_shares_users <- function(id, user_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/projects/{id}/shares/users"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(userIds = user_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a user has on this object
#' @param id integer required. ID of the resource to be revoked
#' @param user_id integer required. ID of the user
#' 
#' @return  An empty HTTP response
#' @export
projects_delete_shares_users <- function(id, user_id) {

  args <- as.list(match.call())[-1]
  path <- "/projects/{id}/shares/users/{user_id}"
  path_params  <- list(id = id, user_id = user_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions groups has on this object
#' @param id integer required. ID of the resource to be shared
#' @param group_ids array required. An array of one or more group IDs
#' @param permission_level string required. Options are: "read", "write", or "manage"
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
projects_put_shares_groups <- function(id, group_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/projects/{id}/shares/groups"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(groupIds = group_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a group has on this object
#' @param id integer required. ID of the resource to be revoked
#' @param group_id integer required. ID of the group
#' 
#' @return  An empty HTTP response
#' @export
projects_delete_shares_groups <- function(id, group_id) {

  args <- as.list(match.call())[-1]
  path <- "/projects/{id}/shares/groups/{group_id}"
  path_params  <- list(id = id, group_id = group_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update the archive status of this object
#' @param id integer required. The ID of the object.
#' @param status boolean required. The desired archived status of the object.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for this project.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{name}{string, The name of this project.}
#' \item{description}{string, A description of the project}
#' \item{users}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{autoShare}{boolean, }
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{tables}{array, An array containing the following fields: 
#' \itemize{
#' \item schema string, 
#' \item name string, 
#' \item rowCount integer, 
#' \item columnCount integer, 
#' \item createdAt string, 
#' \item updatedAt string, 
#' }}
#' \item{surveys}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The object ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' }}
#' \item{scripts}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The object ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item type string, 
#' \item finishedAt string, 
#' \item state string, 
#' }}
#' \item{imports}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The object ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item type string, 
#' \item finishedAt string, 
#' \item state string, 
#' }}
#' \item{models}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The object ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item state string, 
#' }}
#' \item{notebooks}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The object ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item currentDeploymentId integer, 
#' }}
#' \item{services}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The object ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item currentDeploymentId integer, 
#' }}
#' \item{workflows}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The object ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item state string, 
#' }}
#' \item{reports}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The object ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item state string, 
#' }}
#' \item{scriptTemplates}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The object ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' }}
#' \item{files}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The object ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item fileName string, 
#' \item fileSize integer, 
#' \item expired boolean, 
#' }}
#' \item{appInstances}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The object ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item slug string, 
#' }}
#' \item{allObjects}{array, An array containing the following fields: 
#' \itemize{
#' \item projectId integer, 
#' \item objectId integer, 
#' \item objectType string, 
#' \item fcoType string, 
#' \item subType string, 
#' \item name string, 
#' \item icon string, 
#' \item author string, 
#' \item archived string, The archival status of the requested object(s).
#' }}
#' \item{note}{string, }
#' \item{hidden}{boolean, The hidden status of the object.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' @export
projects_put_archive <- function(id, status) {

  args <- as.list(match.call())[-1]
  path <- "/projects/{id}/archive"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(status = status)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' List 
#' @param database_id integer optional. The database ID.
#' @param author_id integer optional. The author of the query.
#' @param created_before string optional. An upper bound for the creation date of the query.
#' @param exclude_results boolean optional. If true, does not return cached query results.
#' @param hidden boolean optional. If specified to be true, returns hidden objects. Defaults to false, returning non-hidden objects.
#' @param limit integer optional. Number of results to return. Defaults to 20. Maximum allowed is 50.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to created_at. Must be one of: created_at.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The query ID.}
#' \item{database}{integer, The database ID.}
#' \item{sql}{string, The SQL to execute.}
#' \item{credential}{integer, The credential ID.}
#' \item{resultRows}{array, A preview of rows returned by the query.}
#' \item{resultColumns}{array, A preview of columns returned by the query.}
#' \item{scriptId}{integer, The ID of the script associated with this query.}
#' \item{exception}{string, Exception returned from the query, null if the query was a success.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{finishedAt}{string, The end time of the last run.}
#' \item{state}{string, The state of the last run.}
#' \item{lastRunId}{integer, The ID of the last run.}
#' \item{previewRows}{integer, The number of rows to save from the query's result (maximum: 100).}
#' \item{startedAt}{string, The start time of the last run.}
#' \item{reportId}{integer, The ID of the report associated with this query.}
#' @export
queries_list <- function(database_id = NULL, author_id = NULL, created_before = NULL, exclude_results = NULL, hidden = NULL, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/queries/"
  path_params  <- list()
  query_params <- list(database_id = database_id, author_id = author_id, created_before = created_before, exclude_results = exclude_results, hidden = hidden, limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Execute a query
#' @param database integer required. The database ID.
#' @param sql string required. The SQL to execute.
#' @param preview_rows integer required. The number of rows to save from the query's result (maximum: 100).
#' @param credential integer optional. The credential ID.
#' @param hidden boolean optional. The hidden status of the object.
#' @param interactive boolean optional. Deprecated and not used.
#' @param include_header boolean optional. Whether the CSV output should include a header row [default: true].
#' @param compression string optional. The type of compression. One of gzip or zip, or none [default: gzip].
#' @param column_delimiter string optional. The delimiter to use. One of comma or tab, or pipe [default: comma].
#' @param unquoted boolean optional. If true, will not quote fields.
#' @param filename_prefix string optional. The output filename prefix.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The query ID.}
#' \item{database}{integer, The database ID.}
#' \item{sql}{string, The SQL to execute.}
#' \item{credential}{integer, The credential ID.}
#' \item{resultRows}{array, A preview of rows returned by the query.}
#' \item{resultColumns}{array, A preview of columns returned by the query.}
#' \item{scriptId}{integer, The ID of the script associated with this query.}
#' \item{exception}{string, Exception returned from the query, null if the query was a success.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{finishedAt}{string, The end time of the last run.}
#' \item{state}{string, The state of the last run.}
#' \item{lastRunId}{integer, The ID of the last run.}
#' \item{hidden}{boolean, The hidden status of the object.}
#' \item{interactive}{boolean, Deprecated and not used.}
#' \item{previewRows}{integer, The number of rows to save from the query's result (maximum: 100).}
#' \item{includeHeader}{boolean, Whether the CSV output should include a header row [default: true].}
#' \item{compression}{string, The type of compression. One of gzip or zip, or none [default: gzip].}
#' \item{columnDelimiter}{string, The delimiter to use. One of comma or tab, or pipe [default: comma].}
#' \item{unquoted}{boolean, If true, will not quote fields.}
#' \item{filenamePrefix}{string, The output filename prefix.}
#' \item{startedAt}{string, The start time of the last run.}
#' \item{reportId}{integer, The ID of the report associated with this query.}
#' @export
queries_post <- function(database, sql, preview_rows, credential = NULL, hidden = NULL, interactive = NULL, include_header = NULL, compression = NULL, column_delimiter = NULL, unquoted = NULL, filename_prefix = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/queries/"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(database = database, sql = sql, previewRows = preview_rows, credential = credential, hidden = hidden, interactive = interactive, includeHeader = include_header, compression = compression, columnDelimiter = column_delimiter, unquoted = unquoted, filenamePrefix = filename_prefix)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' Start a run
#' @param id integer required. The ID of the query.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the run.}
#' \item{queryId}{integer, The ID of the query.}
#' \item{state}{string, The state of the run, one of 'queued' 'running' 'succeeded' 'failed' or 'cancelled'.}
#' \item{isCancelRequested}{boolean, True if run cancel requested, else false.}
#' \item{startedAt}{string, The time the last run started at.}
#' \item{finishedAt}{string, The time the last run completed.}
#' \item{error}{string, The error, if any, returned by the run.}
#' @export
queries_post_runs <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/queries/{id}/runs"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' List runs for the given query
#' @param id integer required. The ID of the query.
#' @param limit integer optional. Number of results to return. Defaults to 20. Maximum allowed is 100.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to id. Must be one of: id.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of the run.}
#' \item{queryId}{integer, The ID of the query.}
#' \item{state}{string, The state of the run, one of 'queued' 'running' 'succeeded' 'failed' or 'cancelled'.}
#' \item{isCancelRequested}{boolean, True if run cancel requested, else false.}
#' \item{startedAt}{string, The time the last run started at.}
#' \item{finishedAt}{string, The time the last run completed.}
#' \item{error}{string, The error, if any, returned by the run.}
#' @export
queries_list_runs <- function(id, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/queries/{id}/runs"
  path_params  <- list(id = id)
  query_params <- list(limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Check status of a run
#' @param id integer required. The ID of the query.
#' @param run_id integer required. The ID of the run.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the run.}
#' \item{queryId}{integer, The ID of the query.}
#' \item{state}{string, The state of the run, one of 'queued' 'running' 'succeeded' 'failed' or 'cancelled'.}
#' \item{isCancelRequested}{boolean, True if run cancel requested, else false.}
#' \item{startedAt}{string, The time the last run started at.}
#' \item{finishedAt}{string, The time the last run completed.}
#' \item{error}{string, The error, if any, returned by the run.}
#' @export
queries_get_runs <- function(id, run_id) {

  args <- as.list(match.call())[-1]
  path <- "/queries/{id}/runs/{run_id}"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Cancel a run
#' @param id integer required. The ID of the query.
#' @param run_id integer required. The ID of the run.
#' 
#' @return  An empty HTTP response
#' @export
queries_delete_runs <- function(id, run_id) {

  args <- as.list(match.call())[-1]
  path <- "/queries/{id}/runs/{run_id}"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get the logs for a run
#' @param id integer required. The ID of the query.
#' @param run_id integer required. The ID of the run.
#' @param last_id integer optional. The ID of the last log message received. Log entries with this ID value or lower will be omitted.Logs are sorted by ID if this value is provided, and are otherwise sorted by createdAt.
#' @param limit integer optional. The maximum number of log messages to return. Default of 10000.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of the log.}
#' \item{createdAt}{string, The time the log was created.}
#' \item{message}{string, The log message.}
#' \item{level}{string, The level of the log. One of unknown,fatal,error,warn,info,debug.}
#' @export
queries_list_runs_logs <- function(id, run_id, last_id = NULL, limit = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/queries/{id}/runs/{run_id}/logs"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list(last_id = last_id, limit = limit)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update the query's associated script
#' @param id integer required. The query ID.
#' @param script_id integer required. The ID of the script associated with this query.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The query ID.}
#' \item{database}{integer, The database ID.}
#' \item{sql}{string, The SQL to execute.}
#' \item{credential}{integer, The credential ID.}
#' \item{resultRows}{array, A preview of rows returned by the query.}
#' \item{resultColumns}{array, A preview of columns returned by the query.}
#' \item{scriptId}{integer, The ID of the script associated with this query.}
#' \item{exception}{string, Exception returned from the query, null if the query was a success.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{finishedAt}{string, The end time of the last run.}
#' \item{state}{string, The state of the last run.}
#' \item{lastRunId}{integer, The ID of the last run.}
#' \item{hidden}{boolean, The hidden status of the object.}
#' \item{name}{string, The name of the query.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{startedAt}{string, The start time of the last run.}
#' \item{reportId}{integer, The ID of the report associated with this query.}
#' @export
queries_put_scripts <- function(id, script_id) {

  args <- as.list(match.call())[-1]
  path <- "/queries/{id}/scripts/{script_id}"
  path_params  <- list(id = id, script_id = script_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get details about a query
#' @param id integer required. The query ID.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The query ID.}
#' \item{database}{integer, The database ID.}
#' \item{sql}{string, The SQL to execute.}
#' \item{credential}{integer, The credential ID.}
#' \item{resultRows}{array, A preview of rows returned by the query.}
#' \item{resultColumns}{array, A preview of columns returned by the query.}
#' \item{scriptId}{integer, The ID of the script associated with this query.}
#' \item{exception}{string, Exception returned from the query, null if the query was a success.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{finishedAt}{string, The end time of the last run.}
#' \item{state}{string, The state of the last run.}
#' \item{lastRunId}{integer, The ID of the last run.}
#' \item{hidden}{boolean, The hidden status of the object.}
#' \item{name}{string, The name of the query.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{startedAt}{string, The start time of the last run.}
#' \item{reportId}{integer, The ID of the report associated with this query.}
#' @export
queries_get <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/queries/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' List the remote hosts
#' @param type string optional. The type of remote host. One of: 
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of the remote host.}
#' \item{name}{string, The name of the remote host.}
#' \item{type}{string, The type of remote host. One of: }
#' \item{url}{string, The URL for remote host.}
#' @export
remote_hosts_list <- function(type = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/remote_hosts/"
  path_params  <- list()
  query_params <- list(type = type)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Create a new remote host
#' @param name string required. The human readable name for the remote host.
#' @param url string required. The URL to your host.
#' @param type string required. The type of remote host. One of: 
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the remote host.}
#' \item{name}{string, The name of the remote host.}
#' \item{type}{string, The type of remote host. One of: }
#' \item{url}{string, The URL for remote host.}
#' @export
remote_hosts_post <- function(name, url, type) {

  args <- as.list(match.call())[-1]
  path <- "/remote_hosts/"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(name = name, url = url, type = type)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' Authenticate against a remote host using either a credential or a user name and password
#' @param id integer required. The ID of the remote host.
#' @param credential_id integer optional. The credential ID.
#' @param username string optional. The user name for remote host.
#' @param password string optional. The password for remote host.
#' 
#' @return  An empty HTTP response
#' @export
remote_hosts_post_authenticate <- function(id, credential_id = NULL, username = NULL, password = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/remote_hosts/{id}/authenticate"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(credentialId = credential_id, username = username, password = password)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' List data sets available from a remote host
#' @param id integer required. The ID of the remote host.
#' @param credential_id integer optional. The credential ID.
#' @param username string optional. The user name for remote host.
#' @param password string optional. The password for remote host.
#' @param q string optional. The query string for data set.
#' @param s boolean optional. If true will only return schemas, otherwise, the results will be the full path.
#' 
#' @return  An array containing the following fields:
#' \item{name}{string, The path to a data_set.}
#' \item{fullPath}{boolean, Boolean that indicates whether further querying needs to be done before the table can be selected.}
#' @export
remote_hosts_list_data_sets <- function(id, credential_id = NULL, username = NULL, password = NULL, q = NULL, s = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/remote_hosts/{id}/data_sets"
  path_params  <- list(id = id)
  query_params <- list(credential_id = credential_id, username = username, password = password, q = q, s = s)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' List 
#' @param type string optional. If specified, return report of these types. It accepts a comma-separated list, possible values are 'tableau' or 'other'.
#' @param author string optional. If specified, return reports from this author. It accepts a comma-separated list of author ids.
#' @param template_id integer optional. If specified, return reports using the provided Template.
#' @param hidden boolean optional. If specified to be true, returns hidden objects. Defaults to false, returning non-hidden objects.
#' @param archived string optional. The archival status of the requested object(s).
#' @param limit integer optional. Number of results to return. Defaults to 20. Maximum allowed is 50.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to updated_at. Must be one of: updated_at, name, created_at.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of this report.}
#' \item{name}{string, The name of the report.}
#' \item{user}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{projects}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID for the project.
#' \item name string, The name of the project.
#' }}
#' \item{state}{string, The status of the report's last run.}
#' \item{finishedAt}{string, The time that the report's last run finished.}
#' \item{vizUpdatedAt}{string, The time that the report's visualization was last updated.}
#' \item{script}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID for the script.
#' \item name string, The name of the script.
#' \item sql string, The raw SQL query for the script.
#' }}
#' \item{jobPath}{string, The link to details of the job that backs this report.}
#' \item{tableauId}{integer, }
#' \item{type}{string, }
#' \item{templateId}{integer, The ID of the template used for this report.}
#' \item{authThumbnailUrl}{string, URL for a thumbnail of the report.}
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{archived}{string, The archival status of the requested object(s).}
#' @export
results_list <- function(type = NULL, author = NULL, template_id = NULL, hidden = NULL, archived = NULL, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/results/"
  path_params  <- list()
  query_params <- list(type = type, author = author, template_id = template_id, hidden = hidden, archived = archived, limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Create a report
#' @param script_id integer optional. The ID of the job (a script or a query) used to create this report.
#' @param name string optional. The name of the report.
#' @param code_body string optional. The code for the report visualization.
#' @param app_state object optional. Any application state blob for this report.
#' @param provide_api_key boolean optional. Allow the report to provide an API key to front-end code.
#' @param template_id integer optional. The ID of the template used for this report.
#' @param hidden boolean optional. The hidden status of the object.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this report.}
#' \item{name}{string, The name of the report.}
#' \item{user}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{projects}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID for the project.
#' \item name string, The name of the project.
#' }}
#' \item{state}{string, The status of the report's last run.}
#' \item{finishedAt}{string, The time that the report's last run finished.}
#' \item{vizUpdatedAt}{string, The time that the report's visualization was last updated.}
#' \item{script}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID for the script.
#' \item name string, The name of the script.
#' \item sql string, The raw SQL query for the script.
#' }}
#' \item{jobPath}{string, The link to details of the job that backs this report.}
#' \item{tableauId}{integer, }
#' \item{type}{string, }
#' \item{templateId}{integer, The ID of the template used for this report.}
#' \item{authThumbnailUrl}{string, URL for a thumbnail of the report.}
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{hidden}{boolean, The hidden status of the object.}
#' \item{authDataUrl}{string, }
#' \item{authCodeUrl}{string, }
#' \item{config}{string, Any configuration metadata for this report.}
#' \item{validOutputFile}{boolean, Whether the job (a script or a query) that backs the report currently has a valid output file.}
#' \item{provideAPIKey}{boolean, Whether the report requests an API Key from the report viewer.}
#' \item{apiKey}{string, A Civis API key that can be used by this report.}
#' \item{apiKeyId}{integer, The ID of the API key. Can be used for auditing API use by this report.}
#' \item{appState}{object, Any application state blob for this report.}
#' \item{useViewersTableauUsername}{boolean, Apply user level filtering on Tableau reports.}
#' @export
results_post <- function(script_id = NULL, name = NULL, code_body = NULL, app_state = NULL, provide_api_key = NULL, template_id = NULL, hidden = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/results/"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(scriptId = script_id, name = name, codeBody = code_body, appState = app_state, provideAPIKey = provide_api_key, templateId = template_id, hidden = hidden)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' List 
#' @param type string optional. If specified, return report of these types. It accepts a comma-separated list, possible values are 'tableau' or 'other'.
#' @param author string optional. If specified, return reports from this author. It accepts a comma-separated list of author ids.
#' @param template_id integer optional. If specified, return reports using the provided Template.
#' @param hidden boolean optional. If specified to be true, returns hidden objects. Defaults to false, returning non-hidden objects.
#' @param archived string optional. The archival status of the requested object(s).
#' @param limit integer optional. Number of results to return. Defaults to 20. Maximum allowed is 50.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to updated_at. Must be one of: updated_at, name, created_at.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of this report.}
#' \item{name}{string, The name of the report.}
#' \item{user}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{projects}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID for the project.
#' \item name string, The name of the project.
#' }}
#' \item{state}{string, The status of the report's last run.}
#' \item{finishedAt}{string, The time that the report's last run finished.}
#' \item{vizUpdatedAt}{string, The time that the report's visualization was last updated.}
#' \item{script}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID for the script.
#' \item name string, The name of the script.
#' \item sql string, The raw SQL query for the script.
#' }}
#' \item{jobPath}{string, The link to details of the job that backs this report.}
#' \item{tableauId}{integer, }
#' \item{type}{string, }
#' \item{templateId}{integer, The ID of the template used for this report.}
#' \item{authThumbnailUrl}{string, URL for a thumbnail of the report.}
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{archived}{string, The archival status of the requested object(s).}
#' @export
reports_list <- function(type = NULL, author = NULL, template_id = NULL, hidden = NULL, archived = NULL, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/reports/"
  path_params  <- list()
  query_params <- list(type = type, author = author, template_id = template_id, hidden = hidden, archived = archived, limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Create a report
#' @param script_id integer optional. The ID of the job (a script or a query) used to create this report.
#' @param name string optional. The name of the report.
#' @param code_body string optional. The code for the report visualization.
#' @param app_state object optional. Any application state blob for this report.
#' @param provide_api_key boolean optional. Allow the report to provide an API key to front-end code.
#' @param template_id integer optional. The ID of the template used for this report.
#' @param hidden boolean optional. The hidden status of the object.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this report.}
#' \item{name}{string, The name of the report.}
#' \item{user}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{projects}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID for the project.
#' \item name string, The name of the project.
#' }}
#' \item{state}{string, The status of the report's last run.}
#' \item{finishedAt}{string, The time that the report's last run finished.}
#' \item{vizUpdatedAt}{string, The time that the report's visualization was last updated.}
#' \item{script}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID for the script.
#' \item name string, The name of the script.
#' \item sql string, The raw SQL query for the script.
#' }}
#' \item{jobPath}{string, The link to details of the job that backs this report.}
#' \item{tableauId}{integer, }
#' \item{type}{string, }
#' \item{templateId}{integer, The ID of the template used for this report.}
#' \item{authThumbnailUrl}{string, URL for a thumbnail of the report.}
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{hidden}{boolean, The hidden status of the object.}
#' \item{authDataUrl}{string, }
#' \item{authCodeUrl}{string, }
#' \item{config}{string, Any configuration metadata for this report.}
#' \item{validOutputFile}{boolean, Whether the job (a script or a query) that backs the report currently has a valid output file.}
#' \item{provideAPIKey}{boolean, Whether the report requests an API Key from the report viewer.}
#' \item{apiKey}{string, A Civis API key that can be used by this report.}
#' \item{apiKeyId}{integer, The ID of the API key. Can be used for auditing API use by this report.}
#' \item{appState}{object, Any application state blob for this report.}
#' \item{useViewersTableauUsername}{boolean, Apply user level filtering on Tableau reports.}
#' @export
reports_post <- function(script_id = NULL, name = NULL, code_body = NULL, app_state = NULL, provide_api_key = NULL, template_id = NULL, hidden = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/reports/"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(scriptId = script_id, name = name, codeBody = code_body, appState = app_state, provideAPIKey = provide_api_key, templateId = template_id, hidden = hidden)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get the git metadata attached to this Report
#' @param id integer required. The ID of the object.
#' 
#' @return  A list containing the following elements:
#' \item{gitRef}{string, A git reference specifying an unambiguous version of the file. Can be a branch name, or the full or shortened SHA of a commit.}
#' \item{gitBranch}{string, The git branch that the file is on.}
#' \item{gitPath}{string, The path of the file in the repository.}
#' \item{gitRepo}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID for this git repository.
#' \item repoUrl string, The URL for this git repository.
#' \item createdAt string, 
#' \item updatedAt string, 
#' }}
#' @export
results_list_git <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/results/{id}/git"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Attach this Report to a git repo/file
#' @param id integer required. The ID of the object.
#' @param git_ref string optional. A git reference specifying an unambiguous version of the file. Can be a branch name, or the full or shortened SHA of a commit.
#' @param git_branch string optional. The git branch that the file is on.
#' @param git_path string optional. The path of the file in the repository.
#' @param git_repo_url string optional. The URL of the git repository.
#' 
#' @return  A list containing the following elements:
#' \item{gitRef}{string, A git reference specifying an unambiguous version of the file. Can be a branch name, or the full or shortened SHA of a commit.}
#' \item{gitBranch}{string, The git branch that the file is on.}
#' \item{gitPath}{string, The path of the file in the repository.}
#' \item{gitRepo}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID for this git repository.
#' \item repoUrl string, The URL for this git repository.
#' \item createdAt string, 
#' \item updatedAt string, 
#' }}
#' @export
results_put_git <- function(id, git_ref = NULL, git_branch = NULL, git_path = NULL, git_repo_url = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/results/{id}/git"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(gitRef = git_ref, gitBranch = git_branch, gitPath = git_path, gitRepoUrl = git_repo_url)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get the git metadata attached to this Report
#' @param id integer required. The ID of the object.
#' 
#' @return  A list containing the following elements:
#' \item{gitRef}{string, A git reference specifying an unambiguous version of the file. Can be a branch name, or the full or shortened SHA of a commit.}
#' \item{gitBranch}{string, The git branch that the file is on.}
#' \item{gitPath}{string, The path of the file in the repository.}
#' \item{gitRepo}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID for this git repository.
#' \item repoUrl string, The URL for this git repository.
#' \item createdAt string, 
#' \item updatedAt string, 
#' }}
#' @export
reports_list_git <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/reports/{id}/git"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Attach this Report to a git repo/file
#' @param id integer required. The ID of the object.
#' @param git_ref string optional. A git reference specifying an unambiguous version of the file. Can be a branch name, or the full or shortened SHA of a commit.
#' @param git_branch string optional. The git branch that the file is on.
#' @param git_path string optional. The path of the file in the repository.
#' @param git_repo_url string optional. The URL of the git repository.
#' 
#' @return  A list containing the following elements:
#' \item{gitRef}{string, A git reference specifying an unambiguous version of the file. Can be a branch name, or the full or shortened SHA of a commit.}
#' \item{gitBranch}{string, The git branch that the file is on.}
#' \item{gitPath}{string, The path of the file in the repository.}
#' \item{gitRepo}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID for this git repository.
#' \item repoUrl string, The URL for this git repository.
#' \item createdAt string, 
#' \item updatedAt string, 
#' }}
#' @export
reports_put_git <- function(id, git_ref = NULL, git_branch = NULL, git_path = NULL, git_repo_url = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/reports/{id}/git"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(gitRef = git_ref, gitBranch = git_branch, gitPath = git_path, gitRepoUrl = git_repo_url)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get the git commits for this Report
#' @param id integer required. The ID of the object.
#' 
#' @return  A list containing the following elements:
#' \item{commitHash}{string, The SHA of the commit.}
#' \item{authorName}{string, The name of the commit's author.}
#' \item{date}{string, The commit's timestamp.}
#' \item{message}{string, The commit message.}
#' @export
results_list_git_commits <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/results/{id}/git/commits"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Commit and push a new version of the file
#' @param id integer required. The ID of the object.
#' @param content string required. The contents to commit to the file.
#' @param message string required. A commit message describing the changes being made.
#' @param file_hash string required. The full SHA of the file being replaced.
#' 
#' @return  A list containing the following elements:
#' \item{content}{string, The file's contents.}
#' \item{type}{string, The file's type.}
#' \item{size}{integer, The file's size.}
#' \item{fileHash}{string, The SHA of the file.}
#' @export
results_post_git_commits <- function(id, content, message, file_hash) {

  args <- as.list(match.call())[-1]
  path <- "/results/{id}/git/commits"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(content = content, message = message, fileHash = file_hash)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get the git commits for this Report
#' @param id integer required. The ID of the object.
#' 
#' @return  A list containing the following elements:
#' \item{commitHash}{string, The SHA of the commit.}
#' \item{authorName}{string, The name of the commit's author.}
#' \item{date}{string, The commit's timestamp.}
#' \item{message}{string, The commit message.}
#' @export
reports_list_git_commits <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/reports/{id}/git/commits"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Commit and push a new version of the file
#' @param id integer required. The ID of the object.
#' @param content string required. The contents to commit to the file.
#' @param message string required. A commit message describing the changes being made.
#' @param file_hash string required. The full SHA of the file being replaced.
#' 
#' @return  A list containing the following elements:
#' \item{content}{string, The file's contents.}
#' \item{type}{string, The file's type.}
#' \item{size}{integer, The file's size.}
#' \item{fileHash}{string, The SHA of the file.}
#' @export
reports_post_git_commits <- function(id, content, message, file_hash) {

  args <- as.list(match.call())[-1]
  path <- "/reports/{id}/git/commits"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(content = content, message = message, fileHash = file_hash)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get file contents at commit_hash
#' @param id integer required. The ID of the object.
#' @param commit_hash string required. The SHA (full or shortened) of the desired git commit.
#' 
#' @return  A list containing the following elements:
#' \item{content}{string, The file's contents.}
#' \item{type}{string, The file's type.}
#' \item{size}{integer, The file's size.}
#' \item{fileHash}{string, The SHA of the file.}
#' @export
results_get_git_commits <- function(id, commit_hash) {

  args <- as.list(match.call())[-1]
  path <- "/results/{id}/git/commits/{commit_hash}"
  path_params  <- list(id = id, commit_hash = commit_hash)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get file contents at commit_hash
#' @param id integer required. The ID of the object.
#' @param commit_hash string required. The SHA (full or shortened) of the desired git commit.
#' 
#' @return  A list containing the following elements:
#' \item{content}{string, The file's contents.}
#' \item{type}{string, The file's type.}
#' \item{size}{integer, The file's size.}
#' \item{fileHash}{string, The SHA of the file.}
#' @export
reports_get_git_commits <- function(id, commit_hash) {

  args <- as.list(match.call())[-1]
  path <- "/reports/{id}/git/commits/{commit_hash}"
  path_params  <- list(id = id, commit_hash = commit_hash)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Show a single report
#' @param id integer required. The ID of this report.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this report.}
#' \item{name}{string, The name of the report.}
#' \item{user}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{projects}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID for the project.
#' \item name string, The name of the project.
#' }}
#' \item{state}{string, The status of the report's last run.}
#' \item{finishedAt}{string, The time that the report's last run finished.}
#' \item{vizUpdatedAt}{string, The time that the report's visualization was last updated.}
#' \item{script}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID for the script.
#' \item name string, The name of the script.
#' \item sql string, The raw SQL query for the script.
#' }}
#' \item{jobPath}{string, The link to details of the job that backs this report.}
#' \item{tableauId}{integer, }
#' \item{type}{string, }
#' \item{templateId}{integer, The ID of the template used for this report.}
#' \item{authThumbnailUrl}{string, URL for a thumbnail of the report.}
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{hidden}{boolean, The hidden status of the object.}
#' \item{authDataUrl}{string, }
#' \item{authCodeUrl}{string, }
#' \item{config}{string, Any configuration metadata for this report.}
#' \item{validOutputFile}{boolean, Whether the job (a script or a query) that backs the report currently has a valid output file.}
#' \item{provideAPIKey}{boolean, Whether the report requests an API Key from the report viewer.}
#' \item{apiKey}{string, A Civis API key that can be used by this report.}
#' \item{apiKeyId}{integer, The ID of the API key. Can be used for auditing API use by this report.}
#' \item{appState}{object, Any application state blob for this report.}
#' \item{useViewersTableauUsername}{boolean, Apply user level filtering on Tableau reports.}
#' @export
results_get <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/results/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update a report
#' @param id integer required. The ID of the report to modify.
#' @param name string optional. The name of the report.
#' @param script_id integer optional. The ID of the job (a script or a query) used to create this report.
#' @param code_body string optional. The code for the report visualization.
#' @param config string optional. 
#' @param app_state object optional. The application state blob for this report.
#' @param provide_api_key boolean optional. Allow the report to provide an API key to front-end code.
#' @param template_id integer optional. The ID of the template used for this report. If null is passed, no template will back this report. Changes to the backing template will reset the report appState.
#' @param use_viewers_tableau_username boolean optional. Apply user level filtering on Tableau reports.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this report.}
#' \item{name}{string, The name of the report.}
#' \item{user}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{projects}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID for the project.
#' \item name string, The name of the project.
#' }}
#' \item{state}{string, The status of the report's last run.}
#' \item{finishedAt}{string, The time that the report's last run finished.}
#' \item{vizUpdatedAt}{string, The time that the report's visualization was last updated.}
#' \item{script}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID for the script.
#' \item name string, The name of the script.
#' \item sql string, The raw SQL query for the script.
#' }}
#' \item{jobPath}{string, The link to details of the job that backs this report.}
#' \item{tableauId}{integer, }
#' \item{type}{string, }
#' \item{templateId}{integer, The ID of the template used for this report.}
#' \item{authThumbnailUrl}{string, URL for a thumbnail of the report.}
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{hidden}{boolean, The hidden status of the object.}
#' \item{authDataUrl}{string, }
#' \item{authCodeUrl}{string, }
#' \item{config}{string, Any configuration metadata for this report.}
#' \item{validOutputFile}{boolean, Whether the job (a script or a query) that backs the report currently has a valid output file.}
#' \item{provideAPIKey}{boolean, Whether the report requests an API Key from the report viewer.}
#' \item{apiKey}{string, A Civis API key that can be used by this report.}
#' \item{apiKeyId}{integer, The ID of the API key. Can be used for auditing API use by this report.}
#' \item{appState}{object, Any application state blob for this report.}
#' \item{useViewersTableauUsername}{boolean, Apply user level filtering on Tableau reports.}
#' @export
results_patch <- function(id, name = NULL, script_id = NULL, code_body = NULL, config = NULL, app_state = NULL, provide_api_key = NULL, template_id = NULL, use_viewers_tableau_username = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/results/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(name = name, scriptId = script_id, codeBody = code_body, config = config, appState = app_state, provideAPIKey = provide_api_key, templateId = template_id, useViewersTableauUsername = use_viewers_tableau_username)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("patch", path, path_params, query_params, body_params)

  return(resp)

 }


#' Show a single report
#' @param id integer required. The ID of this report.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this report.}
#' \item{name}{string, The name of the report.}
#' \item{user}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{projects}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID for the project.
#' \item name string, The name of the project.
#' }}
#' \item{state}{string, The status of the report's last run.}
#' \item{finishedAt}{string, The time that the report's last run finished.}
#' \item{vizUpdatedAt}{string, The time that the report's visualization was last updated.}
#' \item{script}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID for the script.
#' \item name string, The name of the script.
#' \item sql string, The raw SQL query for the script.
#' }}
#' \item{jobPath}{string, The link to details of the job that backs this report.}
#' \item{tableauId}{integer, }
#' \item{type}{string, }
#' \item{templateId}{integer, The ID of the template used for this report.}
#' \item{authThumbnailUrl}{string, URL for a thumbnail of the report.}
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{hidden}{boolean, The hidden status of the object.}
#' \item{authDataUrl}{string, }
#' \item{authCodeUrl}{string, }
#' \item{config}{string, Any configuration metadata for this report.}
#' \item{validOutputFile}{boolean, Whether the job (a script or a query) that backs the report currently has a valid output file.}
#' \item{provideAPIKey}{boolean, Whether the report requests an API Key from the report viewer.}
#' \item{apiKey}{string, A Civis API key that can be used by this report.}
#' \item{apiKeyId}{integer, The ID of the API key. Can be used for auditing API use by this report.}
#' \item{appState}{object, Any application state blob for this report.}
#' \item{useViewersTableauUsername}{boolean, Apply user level filtering on Tableau reports.}
#' @export
reports_get <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/reports/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update a report
#' @param id integer required. The ID of the report to modify.
#' @param name string optional. The name of the report.
#' @param script_id integer optional. The ID of the job (a script or a query) used to create this report.
#' @param code_body string optional. The code for the report visualization.
#' @param config string optional. 
#' @param app_state object optional. The application state blob for this report.
#' @param provide_api_key boolean optional. Allow the report to provide an API key to front-end code.
#' @param template_id integer optional. The ID of the template used for this report. If null is passed, no template will back this report. Changes to the backing template will reset the report appState.
#' @param use_viewers_tableau_username boolean optional. Apply user level filtering on Tableau reports.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this report.}
#' \item{name}{string, The name of the report.}
#' \item{user}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{projects}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID for the project.
#' \item name string, The name of the project.
#' }}
#' \item{state}{string, The status of the report's last run.}
#' \item{finishedAt}{string, The time that the report's last run finished.}
#' \item{vizUpdatedAt}{string, The time that the report's visualization was last updated.}
#' \item{script}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID for the script.
#' \item name string, The name of the script.
#' \item sql string, The raw SQL query for the script.
#' }}
#' \item{jobPath}{string, The link to details of the job that backs this report.}
#' \item{tableauId}{integer, }
#' \item{type}{string, }
#' \item{templateId}{integer, The ID of the template used for this report.}
#' \item{authThumbnailUrl}{string, URL for a thumbnail of the report.}
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{hidden}{boolean, The hidden status of the object.}
#' \item{authDataUrl}{string, }
#' \item{authCodeUrl}{string, }
#' \item{config}{string, Any configuration metadata for this report.}
#' \item{validOutputFile}{boolean, Whether the job (a script or a query) that backs the report currently has a valid output file.}
#' \item{provideAPIKey}{boolean, Whether the report requests an API Key from the report viewer.}
#' \item{apiKey}{string, A Civis API key that can be used by this report.}
#' \item{apiKeyId}{integer, The ID of the API key. Can be used for auditing API use by this report.}
#' \item{appState}{object, Any application state blob for this report.}
#' \item{useViewersTableauUsername}{boolean, Apply user level filtering on Tableau reports.}
#' @export
reports_patch <- function(id, name = NULL, script_id = NULL, code_body = NULL, config = NULL, app_state = NULL, provide_api_key = NULL, template_id = NULL, use_viewers_tableau_username = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/reports/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(name = name, scriptId = script_id, codeBody = code_body, config = config, appState = app_state, provideAPIKey = provide_api_key, templateId = template_id, useViewersTableauUsername = use_viewers_tableau_username)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("patch", path, path_params, query_params, body_params)

  return(resp)

 }


#' Grant this report the ability to perform Civis platform API operations on your behalf
#' @param id integer required. The ID of this report.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this report.}
#' \item{name}{string, The name of the report.}
#' \item{user}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{projects}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID for the project.
#' \item name string, The name of the project.
#' }}
#' \item{state}{string, The status of the report's last run.}
#' \item{finishedAt}{string, The time that the report's last run finished.}
#' \item{vizUpdatedAt}{string, The time that the report's visualization was last updated.}
#' \item{script}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID for the script.
#' \item name string, The name of the script.
#' \item sql string, The raw SQL query for the script.
#' }}
#' \item{jobPath}{string, The link to details of the job that backs this report.}
#' \item{tableauId}{integer, }
#' \item{type}{string, }
#' \item{templateId}{integer, The ID of the template used for this report.}
#' \item{authThumbnailUrl}{string, URL for a thumbnail of the report.}
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{hidden}{boolean, The hidden status of the object.}
#' \item{authDataUrl}{string, }
#' \item{authCodeUrl}{string, }
#' \item{config}{string, Any configuration metadata for this report.}
#' \item{validOutputFile}{boolean, Whether the job (a script or a query) that backs the report currently has a valid output file.}
#' \item{provideAPIKey}{boolean, Whether the report requests an API Key from the report viewer.}
#' \item{apiKey}{string, A Civis API key that can be used by this report.}
#' \item{apiKeyId}{integer, The ID of the API key. Can be used for auditing API use by this report.}
#' \item{appState}{object, Any application state blob for this report.}
#' \item{useViewersTableauUsername}{boolean, Apply user level filtering on Tableau reports.}
#' @export
results_post_grants <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/results/{id}/grants"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke permission for this report to perform Civis platform API operations on your behalf
#' @param id integer required. The ID of this report.
#' 
#' @return  An empty HTTP response
#' @export
results_delete_grants <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/results/{id}/grants"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' Grant this report the ability to perform Civis platform API operations on your behalf
#' @param id integer required. The ID of this report.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this report.}
#' \item{name}{string, The name of the report.}
#' \item{user}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{projects}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID for the project.
#' \item name string, The name of the project.
#' }}
#' \item{state}{string, The status of the report's last run.}
#' \item{finishedAt}{string, The time that the report's last run finished.}
#' \item{vizUpdatedAt}{string, The time that the report's visualization was last updated.}
#' \item{script}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID for the script.
#' \item name string, The name of the script.
#' \item sql string, The raw SQL query for the script.
#' }}
#' \item{jobPath}{string, The link to details of the job that backs this report.}
#' \item{tableauId}{integer, }
#' \item{type}{string, }
#' \item{templateId}{integer, The ID of the template used for this report.}
#' \item{authThumbnailUrl}{string, URL for a thumbnail of the report.}
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{hidden}{boolean, The hidden status of the object.}
#' \item{authDataUrl}{string, }
#' \item{authCodeUrl}{string, }
#' \item{config}{string, Any configuration metadata for this report.}
#' \item{validOutputFile}{boolean, Whether the job (a script or a query) that backs the report currently has a valid output file.}
#' \item{provideAPIKey}{boolean, Whether the report requests an API Key from the report viewer.}
#' \item{apiKey}{string, A Civis API key that can be used by this report.}
#' \item{apiKeyId}{integer, The ID of the API key. Can be used for auditing API use by this report.}
#' \item{appState}{object, Any application state blob for this report.}
#' \item{useViewersTableauUsername}{boolean, Apply user level filtering on Tableau reports.}
#' @export
reports_post_grants <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/reports/{id}/grants"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke permission for this report to perform Civis platform API operations on your behalf
#' @param id integer required. The ID of this report.
#' 
#' @return  An empty HTTP response
#' @export
reports_delete_grants <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/reports/{id}/grants"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get details about the report's snapshot automation settings
#' @param id integer required. The ID of this report.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this report.}
#' \item{state}{string, The status of the job's last run.}
#' \item{finishedAt}{string, The time that the job's last run finished.}
#' \item{sendEmailOnCompletion}{boolean, Whether the job will send emails on completion.}
#' \item{emailTemplate}{string, Custom email template.}
#' \item{recipientEmailAddresses}{string, Email addresses to send report to, comma separated.}
#' \item{emailSubject}{string, Subject for Email.}
#' \item{height}{integer, The height of the cropped snapshot image in screen pixels. The default value is 900 pixels. Minimum value is 600 pixels.}
#' \item{width}{integer, The width of the cropped snapshot image in screen pixels. The default value is 1440 pixels. Minimum value is 600 pixels.}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{parentId}{integer, The ID of the parent job that will trigger this snapshot.}
#' @export
results_list_snapshots <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/results/{id}/snapshots"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Generate and optionally email a snapshot of the specified report
#' @param id integer required. The ID of this report.
#' @param state string optional. The status of the job's last run.
#' @param finished_at string optional. The time that the job's last run finished.
#' @param send_email_on_completion boolean optional. Whether the job will send emails on completion.
#' @param email_template string optional. Custom email template.
#' @param recipient_email_addresses string optional. Email addresses to send report to, comma separated.
#' @param email_subject string optional. Subject for Email.
#' @param height integer optional. The height of the cropped snapshot image in screen pixels. The default value is 900 pixels. Minimum value is 600 pixels.
#' @param width integer optional. The width of the cropped snapshot image in screen pixels. The default value is 1440 pixels. Minimum value is 600 pixels.
#' @param schedule object optional. A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }
#' @param parent_id integer optional. The ID of the parent job that will trigger this snapshot.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this report.}
#' \item{state}{string, The status of the job's last run.}
#' \item{finishedAt}{string, The time that the job's last run finished.}
#' \item{sendEmailOnCompletion}{boolean, Whether the job will send emails on completion.}
#' \item{emailTemplate}{string, Custom email template.}
#' \item{recipientEmailAddresses}{string, Email addresses to send report to, comma separated.}
#' \item{emailSubject}{string, Subject for Email.}
#' \item{height}{integer, The height of the cropped snapshot image in screen pixels. The default value is 900 pixels. Minimum value is 600 pixels.}
#' \item{width}{integer, The width of the cropped snapshot image in screen pixels. The default value is 1440 pixels. Minimum value is 600 pixels.}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{parentId}{integer, The ID of the parent job that will trigger this snapshot.}
#' @export
results_post_snapshots <- function(id, state = NULL, finished_at = NULL, send_email_on_completion = NULL, email_template = NULL, recipient_email_addresses = NULL, email_subject = NULL, height = NULL, width = NULL, schedule = NULL, parent_id = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/results/{id}/snapshots"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(state = state, finishedAt = finished_at, sendEmailOnCompletion = send_email_on_completion, emailTemplate = email_template, recipientEmailAddresses = recipient_email_addresses, emailSubject = email_subject, height = height, width = width, schedule = schedule, parentId = parent_id)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update the report's snapshot automation settings
#' @param id integer required. The ID of this report.
#' @param state string optional. The status of the job's last run.
#' @param finished_at string optional. The time that the job's last run finished.
#' @param send_email_on_completion boolean optional. Whether the job will send emails on completion.
#' @param email_template string optional. Custom email template.
#' @param recipient_email_addresses string optional. Email addresses to send report to, comma separated.
#' @param email_subject string optional. Subject for Email.
#' @param height integer optional. The height of the cropped snapshot image in screen pixels. The default value is 900 pixels. Minimum value is 600 pixels.
#' @param width integer optional. The width of the cropped snapshot image in screen pixels. The default value is 1440 pixels. Minimum value is 600 pixels.
#' @param schedule object optional. A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }
#' @param parent_id integer optional. The ID of the parent job that will trigger this snapshot.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this report.}
#' \item{state}{string, The status of the job's last run.}
#' \item{finishedAt}{string, The time that the job's last run finished.}
#' \item{sendEmailOnCompletion}{boolean, Whether the job will send emails on completion.}
#' \item{emailTemplate}{string, Custom email template.}
#' \item{recipientEmailAddresses}{string, Email addresses to send report to, comma separated.}
#' \item{emailSubject}{string, Subject for Email.}
#' \item{height}{integer, The height of the cropped snapshot image in screen pixels. The default value is 900 pixels. Minimum value is 600 pixels.}
#' \item{width}{integer, The width of the cropped snapshot image in screen pixels. The default value is 1440 pixels. Minimum value is 600 pixels.}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{parentId}{integer, The ID of the parent job that will trigger this snapshot.}
#' @export
results_patch_snapshots <- function(id, state = NULL, finished_at = NULL, send_email_on_completion = NULL, email_template = NULL, recipient_email_addresses = NULL, email_subject = NULL, height = NULL, width = NULL, schedule = NULL, parent_id = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/results/{id}/snapshots"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(state = state, finishedAt = finished_at, sendEmailOnCompletion = send_email_on_completion, emailTemplate = email_template, recipientEmailAddresses = recipient_email_addresses, emailSubject = email_subject, height = height, width = width, schedule = schedule, parentId = parent_id)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("patch", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get details about the report's snapshot automation settings
#' @param id integer required. The ID of this report.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this report.}
#' \item{state}{string, The status of the job's last run.}
#' \item{finishedAt}{string, The time that the job's last run finished.}
#' \item{sendEmailOnCompletion}{boolean, Whether the job will send emails on completion.}
#' \item{emailTemplate}{string, Custom email template.}
#' \item{recipientEmailAddresses}{string, Email addresses to send report to, comma separated.}
#' \item{emailSubject}{string, Subject for Email.}
#' \item{height}{integer, The height of the cropped snapshot image in screen pixels. The default value is 900 pixels. Minimum value is 600 pixels.}
#' \item{width}{integer, The width of the cropped snapshot image in screen pixels. The default value is 1440 pixels. Minimum value is 600 pixels.}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{parentId}{integer, The ID of the parent job that will trigger this snapshot.}
#' @export
reports_list_snapshots <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/reports/{id}/snapshots"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Generate and optionally email a snapshot of the specified report
#' @param id integer required. The ID of this report.
#' @param state string optional. The status of the job's last run.
#' @param finished_at string optional. The time that the job's last run finished.
#' @param send_email_on_completion boolean optional. Whether the job will send emails on completion.
#' @param email_template string optional. Custom email template.
#' @param recipient_email_addresses string optional. Email addresses to send report to, comma separated.
#' @param email_subject string optional. Subject for Email.
#' @param height integer optional. The height of the cropped snapshot image in screen pixels. The default value is 900 pixels. Minimum value is 600 pixels.
#' @param width integer optional. The width of the cropped snapshot image in screen pixels. The default value is 1440 pixels. Minimum value is 600 pixels.
#' @param schedule object optional. A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }
#' @param parent_id integer optional. The ID of the parent job that will trigger this snapshot.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this report.}
#' \item{state}{string, The status of the job's last run.}
#' \item{finishedAt}{string, The time that the job's last run finished.}
#' \item{sendEmailOnCompletion}{boolean, Whether the job will send emails on completion.}
#' \item{emailTemplate}{string, Custom email template.}
#' \item{recipientEmailAddresses}{string, Email addresses to send report to, comma separated.}
#' \item{emailSubject}{string, Subject for Email.}
#' \item{height}{integer, The height of the cropped snapshot image in screen pixels. The default value is 900 pixels. Minimum value is 600 pixels.}
#' \item{width}{integer, The width of the cropped snapshot image in screen pixels. The default value is 1440 pixels. Minimum value is 600 pixels.}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{parentId}{integer, The ID of the parent job that will trigger this snapshot.}
#' @export
reports_post_snapshots <- function(id, state = NULL, finished_at = NULL, send_email_on_completion = NULL, email_template = NULL, recipient_email_addresses = NULL, email_subject = NULL, height = NULL, width = NULL, schedule = NULL, parent_id = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/reports/{id}/snapshots"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(state = state, finishedAt = finished_at, sendEmailOnCompletion = send_email_on_completion, emailTemplate = email_template, recipientEmailAddresses = recipient_email_addresses, emailSubject = email_subject, height = height, width = width, schedule = schedule, parentId = parent_id)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update the report's snapshot automation settings
#' @param id integer required. The ID of this report.
#' @param state string optional. The status of the job's last run.
#' @param finished_at string optional. The time that the job's last run finished.
#' @param send_email_on_completion boolean optional. Whether the job will send emails on completion.
#' @param email_template string optional. Custom email template.
#' @param recipient_email_addresses string optional. Email addresses to send report to, comma separated.
#' @param email_subject string optional. Subject for Email.
#' @param height integer optional. The height of the cropped snapshot image in screen pixels. The default value is 900 pixels. Minimum value is 600 pixels.
#' @param width integer optional. The width of the cropped snapshot image in screen pixels. The default value is 1440 pixels. Minimum value is 600 pixels.
#' @param schedule object optional. A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }
#' @param parent_id integer optional. The ID of the parent job that will trigger this snapshot.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this report.}
#' \item{state}{string, The status of the job's last run.}
#' \item{finishedAt}{string, The time that the job's last run finished.}
#' \item{sendEmailOnCompletion}{boolean, Whether the job will send emails on completion.}
#' \item{emailTemplate}{string, Custom email template.}
#' \item{recipientEmailAddresses}{string, Email addresses to send report to, comma separated.}
#' \item{emailSubject}{string, Subject for Email.}
#' \item{height}{integer, The height of the cropped snapshot image in screen pixels. The default value is 900 pixels. Minimum value is 600 pixels.}
#' \item{width}{integer, The width of the cropped snapshot image in screen pixels. The default value is 1440 pixels. Minimum value is 600 pixels.}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{parentId}{integer, The ID of the parent job that will trigger this snapshot.}
#' @export
reports_patch_snapshots <- function(id, state = NULL, finished_at = NULL, send_email_on_completion = NULL, email_template = NULL, recipient_email_addresses = NULL, email_subject = NULL, height = NULL, width = NULL, schedule = NULL, parent_id = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/reports/{id}/snapshots"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(state = state, finishedAt = finished_at, sendEmailOnCompletion = send_email_on_completion, emailTemplate = email_template, recipientEmailAddresses = recipient_email_addresses, emailSubject = email_subject, height = height, width = width, schedule = schedule, parentId = parent_id)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("patch", path, path_params, query_params, body_params)

  return(resp)

 }


#' List users and groups permissioned on this object
#' @param id integer required. The ID of the object.
#' 
#' @return  An array containing the following fields:
#' \item{readers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
results_list_shares <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/results/{id}/shares"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' List users and groups permissioned on this object
#' @param id integer required. The ID of the object.
#' 
#' @return  An array containing the following fields:
#' \item{readers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
reports_list_shares <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/reports/{id}/shares"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions users have on this object
#' @param id integer required. ID of the resource to be shared
#' @param user_ids array required. An array of one or more user IDs
#' @param permission_level string required. Options are: "read", "write", or "manage"
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
results_put_shares_users <- function(id, user_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/results/{id}/shares/users"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(userIds = user_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions users have on this object
#' @param id integer required. ID of the resource to be shared
#' @param user_ids array required. An array of one or more user IDs
#' @param permission_level string required. Options are: "read", "write", or "manage"
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
reports_put_shares_users <- function(id, user_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/reports/{id}/shares/users"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(userIds = user_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a user has on this object
#' @param id integer required. ID of the resource to be revoked
#' @param user_id integer required. ID of the user
#' 
#' @return  An empty HTTP response
#' @export
results_delete_shares_users <- function(id, user_id) {

  args <- as.list(match.call())[-1]
  path <- "/results/{id}/shares/users/{user_id}"
  path_params  <- list(id = id, user_id = user_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a user has on this object
#' @param id integer required. ID of the resource to be revoked
#' @param user_id integer required. ID of the user
#' 
#' @return  An empty HTTP response
#' @export
reports_delete_shares_users <- function(id, user_id) {

  args <- as.list(match.call())[-1]
  path <- "/reports/{id}/shares/users/{user_id}"
  path_params  <- list(id = id, user_id = user_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions groups has on this object
#' @param id integer required. ID of the resource to be shared
#' @param group_ids array required. An array of one or more group IDs
#' @param permission_level string required. Options are: "read", "write", or "manage"
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
results_put_shares_groups <- function(id, group_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/results/{id}/shares/groups"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(groupIds = group_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions groups has on this object
#' @param id integer required. ID of the resource to be shared
#' @param group_ids array required. An array of one or more group IDs
#' @param permission_level string required. Options are: "read", "write", or "manage"
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
reports_put_shares_groups <- function(id, group_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/reports/{id}/shares/groups"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(groupIds = group_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a group has on this object
#' @param id integer required. ID of the resource to be revoked
#' @param group_id integer required. ID of the group
#' 
#' @return  An empty HTTP response
#' @export
results_delete_shares_groups <- function(id, group_id) {

  args <- as.list(match.call())[-1]
  path <- "/results/{id}/shares/groups/{group_id}"
  path_params  <- list(id = id, group_id = group_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a group has on this object
#' @param id integer required. ID of the resource to be revoked
#' @param group_id integer required. ID of the group
#' 
#' @return  An empty HTTP response
#' @export
reports_delete_shares_groups <- function(id, group_id) {

  args <- as.list(match.call())[-1]
  path <- "/reports/{id}/shares/groups/{group_id}"
  path_params  <- list(id = id, group_id = group_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' List the projects a Report belongs to
#' @param id integer required. The ID of the resource.
#' @param hidden boolean optional. If specified to be true, returns hidden objects. Defaults to false, returning non-hidden objects.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID for this project.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{name}{string, The name of this project.}
#' \item{description}{string, A description of the project}
#' \item{users}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{autoShare}{boolean, }
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{archived}{string, The archival status of the requested object(s).}
#' @export
results_list_projects <- function(id, hidden = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/results/{id}/projects"
  path_params  <- list(id = id)
  query_params <- list(hidden = hidden)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' List the projects a Report belongs to
#' @param id integer required. The ID of the resource.
#' @param hidden boolean optional. If specified to be true, returns hidden objects. Defaults to false, returning non-hidden objects.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID for this project.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{name}{string, The name of this project.}
#' \item{description}{string, A description of the project}
#' \item{users}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{autoShare}{boolean, }
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{archived}{string, The archival status of the requested object(s).}
#' @export
reports_list_projects <- function(id, hidden = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/reports/{id}/projects"
  path_params  <- list(id = id)
  query_params <- list(hidden = hidden)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Add a Report to a project
#' @param id integer required. ID of the resource
#' @param project_id integer required. The ID of the project
#' 
#' @return  An empty HTTP response
#' @export
results_put_projects <- function(id, project_id) {

  args <- as.list(match.call())[-1]
  path <- "/results/{id}/projects/{project_id}"
  path_params  <- list(id = id, project_id = project_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Remove a Report from a project
#' @param id integer required. ID of the resource
#' @param project_id integer required. The ID of the project
#' 
#' @return  An empty HTTP response
#' @export
results_delete_projects <- function(id, project_id) {

  args <- as.list(match.call())[-1]
  path <- "/results/{id}/projects/{project_id}"
  path_params  <- list(id = id, project_id = project_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' Add a Report to a project
#' @param id integer required. ID of the resource
#' @param project_id integer required. The ID of the project
#' 
#' @return  An empty HTTP response
#' @export
reports_put_projects <- function(id, project_id) {

  args <- as.list(match.call())[-1]
  path <- "/reports/{id}/projects/{project_id}"
  path_params  <- list(id = id, project_id = project_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Remove a Report from a project
#' @param id integer required. ID of the resource
#' @param project_id integer required. The ID of the project
#' 
#' @return  An empty HTTP response
#' @export
reports_delete_projects <- function(id, project_id) {

  args <- as.list(match.call())[-1]
  path <- "/reports/{id}/projects/{project_id}"
  path_params  <- list(id = id, project_id = project_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update the archive status of this object
#' @param id integer required. The ID of the object.
#' @param status boolean required. The desired archived status of the object.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this report.}
#' \item{name}{string, The name of the report.}
#' \item{user}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{projects}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID for the project.
#' \item name string, The name of the project.
#' }}
#' \item{state}{string, The status of the report's last run.}
#' \item{finishedAt}{string, The time that the report's last run finished.}
#' \item{vizUpdatedAt}{string, The time that the report's visualization was last updated.}
#' \item{script}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID for the script.
#' \item name string, The name of the script.
#' \item sql string, The raw SQL query for the script.
#' }}
#' \item{jobPath}{string, The link to details of the job that backs this report.}
#' \item{tableauId}{integer, }
#' \item{type}{string, }
#' \item{templateId}{integer, The ID of the template used for this report.}
#' \item{authThumbnailUrl}{string, URL for a thumbnail of the report.}
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{hidden}{boolean, The hidden status of the object.}
#' \item{authDataUrl}{string, }
#' \item{authCodeUrl}{string, }
#' \item{config}{string, Any configuration metadata for this report.}
#' \item{validOutputFile}{boolean, Whether the job (a script or a query) that backs the report currently has a valid output file.}
#' \item{provideAPIKey}{boolean, Whether the report requests an API Key from the report viewer.}
#' \item{apiKey}{string, A Civis API key that can be used by this report.}
#' \item{apiKeyId}{integer, The ID of the API key. Can be used for auditing API use by this report.}
#' \item{appState}{object, Any application state blob for this report.}
#' \item{useViewersTableauUsername}{boolean, Apply user level filtering on Tableau reports.}
#' @export
results_put_archive <- function(id, status) {

  args <- as.list(match.call())[-1]
  path <- "/results/{id}/archive"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(status = status)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update the archive status of this object
#' @param id integer required. The ID of the object.
#' @param status boolean required. The desired archived status of the object.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this report.}
#' \item{name}{string, The name of the report.}
#' \item{user}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{projects}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID for the project.
#' \item name string, The name of the project.
#' }}
#' \item{state}{string, The status of the report's last run.}
#' \item{finishedAt}{string, The time that the report's last run finished.}
#' \item{vizUpdatedAt}{string, The time that the report's visualization was last updated.}
#' \item{script}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID for the script.
#' \item name string, The name of the script.
#' \item sql string, The raw SQL query for the script.
#' }}
#' \item{jobPath}{string, The link to details of the job that backs this report.}
#' \item{tableauId}{integer, }
#' \item{type}{string, }
#' \item{templateId}{integer, The ID of the template used for this report.}
#' \item{authThumbnailUrl}{string, URL for a thumbnail of the report.}
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{hidden}{boolean, The hidden status of the object.}
#' \item{authDataUrl}{string, }
#' \item{authCodeUrl}{string, }
#' \item{config}{string, Any configuration metadata for this report.}
#' \item{validOutputFile}{boolean, Whether the job (a script or a query) that backs the report currently has a valid output file.}
#' \item{provideAPIKey}{boolean, Whether the report requests an API Key from the report viewer.}
#' \item{apiKey}{string, A Civis API key that can be used by this report.}
#' \item{apiKeyId}{integer, The ID of the API key. Can be used for auditing API use by this report.}
#' \item{appState}{object, Any application state blob for this report.}
#' \item{useViewersTableauUsername}{boolean, Apply user level filtering on Tableau reports.}
#' @export
reports_put_archive <- function(id, status) {

  args <- as.list(match.call())[-1]
  path <- "/reports/{id}/archive"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(status = status)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Show a single service report
#' @param id integer required. The ID of this report.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this report.}
#' \item{name}{string, The name of the report.}
#' \item{user}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{host}{string, The host for the service report}
#' \item{displayUrl}{string, The URL to display the service report.}
#' \item{serviceId}{integer, The id of the backing service}
#' @export
results_get_services <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/results/services/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update some attributes of this service report
#' @param id integer required. The ID of this report.
#' @param name string optional. The name of the service report.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this report.}
#' \item{name}{string, The name of the report.}
#' \item{user}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{host}{string, The host for the service report}
#' \item{displayUrl}{string, The URL to display the service report.}
#' \item{serviceId}{integer, The id of the backing service}
#' @export
results_patch_services <- function(id, name = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/results/services/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(name = name)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("patch", path, path_params, query_params, body_params)

  return(resp)

 }


#' Show a single service report
#' @param id integer required. The ID of this report.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this report.}
#' \item{name}{string, The name of the report.}
#' \item{user}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{host}{string, The host for the service report}
#' \item{displayUrl}{string, The URL to display the service report.}
#' \item{serviceId}{integer, The id of the backing service}
#' @export
reports_get_services <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/reports/services/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update some attributes of this service report
#' @param id integer required. The ID of this report.
#' @param name string optional. The name of the service report.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this report.}
#' \item{name}{string, The name of the report.}
#' \item{user}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{host}{string, The host for the service report}
#' \item{displayUrl}{string, The URL to display the service report.}
#' \item{serviceId}{integer, The id of the backing service}
#' @export
reports_patch_services <- function(id, name = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/reports/services/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(name = name)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("patch", path, path_params, query_params, body_params)

  return(resp)

 }


#' List users and groups permissioned on this object
#' @param id integer required. The ID of the object.
#' 
#' @return  An array containing the following fields:
#' \item{readers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
results_list_services_shares <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/results/services/{id}/shares"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' List users and groups permissioned on this object
#' @param id integer required. The ID of the object.
#' 
#' @return  An array containing the following fields:
#' \item{readers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
reports_list_services_shares <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/reports/services/{id}/shares"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions users have on this object
#' @param id integer required. ID of the resource to be shared
#' @param user_ids array required. An array of one or more user IDs
#' @param permission_level string required. Options are: "read", "write", or "manage"
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
results_put_services_shares_users <- function(id, user_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/results/services/{id}/shares/users"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(userIds = user_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions users have on this object
#' @param id integer required. ID of the resource to be shared
#' @param user_ids array required. An array of one or more user IDs
#' @param permission_level string required. Options are: "read", "write", or "manage"
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
reports_put_services_shares_users <- function(id, user_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/reports/services/{id}/shares/users"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(userIds = user_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a user has on this object
#' @param id integer required. ID of the resource to be revoked
#' @param user_id integer required. ID of the user
#' 
#' @return  An empty HTTP response
#' @export
results_delete_services_shares_users <- function(id, user_id) {

  args <- as.list(match.call())[-1]
  path <- "/results/services/{id}/shares/users/{user_id}"
  path_params  <- list(id = id, user_id = user_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a user has on this object
#' @param id integer required. ID of the resource to be revoked
#' @param user_id integer required. ID of the user
#' 
#' @return  An empty HTTP response
#' @export
reports_delete_services_shares_users <- function(id, user_id) {

  args <- as.list(match.call())[-1]
  path <- "/reports/services/{id}/shares/users/{user_id}"
  path_params  <- list(id = id, user_id = user_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions groups has on this object
#' @param id integer required. ID of the resource to be shared
#' @param group_ids array required. An array of one or more group IDs
#' @param permission_level string required. Options are: "read", "write", or "manage"
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
results_put_services_shares_groups <- function(id, group_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/results/services/{id}/shares/groups"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(groupIds = group_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions groups has on this object
#' @param id integer required. ID of the resource to be shared
#' @param group_ids array required. An array of one or more group IDs
#' @param permission_level string required. Options are: "read", "write", or "manage"
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{object, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
reports_put_services_shares_groups <- function(id, group_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/reports/services/{id}/shares/groups"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(groupIds = group_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a group has on this object
#' @param id integer required. ID of the resource to be revoked
#' @param group_id integer required. ID of the group
#' 
#' @return  An empty HTTP response
#' @export
results_delete_services_shares_groups <- function(id, group_id) {

  args <- as.list(match.call())[-1]
  path <- "/results/services/{id}/shares/groups/{group_id}"
  path_params  <- list(id = id, group_id = group_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a group has on this object
#' @param id integer required. ID of the resource to be revoked
#' @param group_id integer required. ID of the group
#' 
#' @return  An empty HTTP response
#' @export
reports_delete_services_shares_groups <- function(id, group_id) {

  args <- as.list(match.call())[-1]
  path <- "/reports/services/{id}/shares/groups/{group_id}"
  path_params  <- list(id = id, group_id = group_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' List available script types
#' 
#' @return  An array containing the following fields:
#' \item{name}{string, The name of the type.}
#' @export
scripts_list_types <- function() {

  args <- as.list(match.call())[-1]
  path <- "/scripts/types"
  path_params  <- list()
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get the run history and outputs of this script
#' @param id integer required. The ID for the script.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this run.}
#' \item{sqlId}{integer, The ID of this sql.}
#' \item{state}{string, The state of this run.}
#' \item{isCancelRequested}{boolean, True if run cancel requested, else false.}
#' \item{finishedAt}{string, The time that this run finished.}
#' \item{error}{string, The error message for this run, if present.}
#' \item{output}{array, An array containing the following fields: 
#' \itemize{
#' \item outputName string, The name of the output file.
#' \item fileId integer, The unique ID of the output file.
#' \item path string, The temporary link to download this output file, valid for 36 hours.
#' }}
#' @export
scripts_list_history <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/scripts/{id}/history"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Create a script
#' @param name string required. The name of the script.
#' @param remote_host_id integer required. The database ID.
#' @param credential_id integer required. The credential ID.
#' @param sql string required. The raw SQL query for the script.
#' @param params array optional. An array containing the following fields: 
#' \itemize{
#' \item name string, The variable's name as used within your code.
#' \item label string, The label to present to users when asking them for the value.
#' \item description string, A short sentence or fragment describing this parameter to the end user.
#' \item type string, The type of parameter. Valid options: string, multi_line_string, integer, float, bool, file, database, credential_aws, credential_redshift, or credential_custom
#' \item required boolean, Whether this param is required.
#' \item value string, The value you would like to set this param to. Setting this value makes this parameter a fixed param.
#' \item default string, If an argument for this parameter is not defined, it will use this default value. Use true, True, t, y, yes, or 1 for true bool's or false, False, f, n, no, or 0 for false bool's. Cannot be used for parameters that are required or a credential type.
#' \item allowedValues array, The possible values this parameter can take, effectively making this an enumerable parameter. Allowed values is an array of hashes of the following format: `{label: 'Import', 'value': 'import'}`
#' }
#' @param arguments object optional. Dictionary of name/value pairs to use to run this script. Only settable if this script has defined params.
#' @param template_script_id integer optional. The ID of the template script, if any.  A script cannot both have a template script and be a template for other scripts.
#' @param notifications object optional. A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }
#' @param hidden boolean optional. The hidden status of the object.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the script.}
#' \item{name}{string, The name of the script.}
#' \item{type}{string, The type of the script (e.g SQL, Container, Python, R, JavaScript)}
#' \item{createdAt}{string, The time this script was created.}
#' \item{updatedAt}{string, The time the script was last updated.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the script's last run.}
#' \item{finishedAt}{string, The time that the script's last run finished.}
#' \item{category}{string, The category of the script.}
#' \item{projects}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID for the project.
#' \item name string, The name of the project.
#' }}
#' \item{parentId}{integer, The ID of the parent job that will trigger this script}
#' \item{userContext}{string, "runner" or "author", who to execute the script as when run as a template.}
#' \item{params}{array, An array containing the following fields: 
#' \itemize{
#' \item name string, The variable's name as used within your code.
#' \item label string, The label to present to users when asking them for the value.
#' \item description string, A short sentence or fragment describing this parameter to the end user.
#' \item type string, The type of parameter. Valid options: string, multi_line_string, integer, float, bool, file, database, credential_aws, credential_redshift, or credential_custom
#' \item required boolean, Whether this param is required.
#' \item value string, The value you would like to set this param to. Setting this value makes this parameter a fixed param.
#' \item default string, If an argument for this parameter is not defined, it will use this default value. Use true, True, t, y, yes, or 1 for true bool's or false, False, f, n, no, or 0 for false bool's. Cannot be used for parameters that are required or a credential type.
#' \item allowedValues array, The possible values this parameter can take, effectively making this an enumerable parameter. Allowed values is an array of hashes of the following format: `{label: 'Import', 'value': 'import'}`
#' }}
#' \item{arguments}{object, Dictionary of name/value pairs to use to run this script. Only settable if this script has defined params.}
#' \item{isTemplate}{boolean, Whether others scripts use this one as a template.}
#' \item{publishedAsTemplateId}{integer, The ID of the template that this script is backing.}
#' \item{fromTemplateId}{integer, The ID of the template this script uses, if any.}
#' \item{templateDependentsCount}{integer, How many other scripts use this one as a template.}
#' \item{templateScriptName}{string, The name of the template script.}
#' \item{links}{object, A list containing the following elements: 
#' \itemize{
#' \item details string, The details link to get more information about the script.
#' \item runs string, The runs link to get the run information list for this script.
#' }}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{notifications}{object, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{runningAs}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{nextRunAt}{string, The time of the next scheduled run.}
#' \item{timeZone}{string, The time zone of this script.}
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{hidden}{boolean, The hidden status of the object.}
#' \item{targetProjectId}{integer, Target project to which script outputs will be added.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{templateScriptId}{integer, The ID of the template script, if any.}
#' @export
scripts_post <- function(name, remote_host_id, credential_id, sql, params = NULL, arguments = NULL, template_script_id = NULL, notifications = NULL, hidden = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/scripts/"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(name = name, remoteHostId = remote_host_id, credentialId = credential_id, sql = sql, params = params, arguments = arguments, templateScriptId = template_script_id, notifications = notifications, hidden = hidden)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' List scripts
#' @param type string optional. If specified, return objects of these types. The valid types are sql, python3, javascript, r, and containers.
#' @param category string optional. A job category for filtering scripts. Must be one of script, import, export, and enhancement.
#' @param author string optional. If specified, return objects from this author. Must use user IDs. A comma separated list of IDs is also accepted to return objects from multiple authors.
#' @param status string optional. If specified, returns objects with one of these statuses. It accepts a comma-separated list, possible values are 'running', 'failed', 'succeeded', 'idle', 'scheduled'.
#' @param hidden boolean optional. If specified to be true, returns hidden objects. Defaults to false, returning non-hidden objects.
#' @param archived string optional. The archival status of the requested object(s).
#' @param limit integer optional. Number of results to return. Defaults to 20. Maximum allowed is 50.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to updated_at. Must be one of: updated_at, name, created_at, last_run.updated_at.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID for the script.}
#' \item{name}{string, The name of the script.}
#' \item{type}{string, The type of the script (e.g SQL, Container, Python, R, JavaScript)}
#' \item{createdAt}{string, The time this script was created.}
#' \item{updatedAt}{string, The time the script was last updated.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the script's last run.}
#' \item{finishedAt}{string, The time that the script's last run finished.}
#' \item{projects}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID for the project.
#' \item name string, The name of the project.
#' }}
#' \item{parentId}{integer, The ID of the parent job that will trigger this script}
#' \item{isTemplate}{boolean, Whether others scripts use this one as a template.}
#' \item{fromTemplateId}{integer, The ID of the template this script uses, if any.}
#' \item{links}{object, A list containing the following elements: 
#' \itemize{
#' \item details string, The details link to get more information about the script.
#' \item runs string, The runs link to get the run information list for this script.
#' }}
#' \item{timeZone}{string, The time zone of this script.}
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{templateScriptId}{integer, The ID of the template script, if any.}
#' @export
scripts_list <- function(type = NULL, category = NULL, author = NULL, status = NULL, hidden = NULL, archived = NULL, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/scripts/"
  path_params  <- list()
  query_params <- list(type = type, category = category, author = author, status = status, hidden = hidden, archived = archived, limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update a script
#' @param id integer required. The ID for the script.
#' @param name string optional. The name of the script.
#' @param sql string optional. The raw SQL query for the script.
#' @param params array optional. An array containing the following fields: 
#' \itemize{
#' \item name string, The variable's name as used within your code.
#' \item label string, The label to present to users when asking them for the value.
#' \item description string, A short sentence or fragment describing this parameter to the end user.
#' \item type string, The type of parameter. Valid options: string, multi_line_string, integer, float, bool, file, database, credential_aws, credential_redshift, or credential_custom
#' \item required boolean, Whether this param is required.
#' \item value string, The value you would like to set this param to. Setting this value makes this parameter a fixed param.
#' \item default string, If an argument for this parameter is not defined, it will use this default value. Use true, True, t, y, yes, or 1 for true bool's or false, False, f, n, no, or 0 for false bool's. Cannot be used for parameters that are required or a credential type.
#' \item allowedValues array, The possible values this parameter can take, effectively making this an enumerable parameter. Allowed values is an array of hashes of the following format: `{label: 'Import', 'value': 'import'}`
#' }
#' @param arguments object optional. Dictionary of name/value pairs to use to run this script. Only settable if this script has defined params.
#' @param template_script_id integer optional. The ID of the template script, if any.  A script cannot both have a template script and be a template for other scripts.
#' @param schedule object optional. A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }
#' @param notifications object optional. A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }
#' @param parent_id integer optional. The ID of the parent job that will trigger this script
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the script.}
#' \item{name}{string, The name of the script.}
#' \item{type}{string, The type of script.}
#' \item{createdAt}{string, The time this script was created.}
#' \item{updatedAt}{string, The time this script was last updated.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the script's last run.}
#' \item{finishedAt}{string, The time that the script's last run finished.}
#' \item{category}{string, The category of the script.}
#' \item{projects}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID for the project.
#' \item name string, The name of the project.
#' }}
#' \item{parentId}{integer, The ID of the parent job that will trigger this script}
#' \item{userContext}{string, "runner" or "author", who to execute the script as when run as a template.}
#' \item{params}{array, An array containing the following fields: 
#' \itemize{
#' \item name string, The variable's name as used within your code.
#' \item label string, The label to present to users when asking them for the value.
#' \item description string, A short sentence or fragment describing this parameter to the end user.
#' \item type string, The type of parameter. Valid options: string, multi_line_string, integer, float, bool, file, database, credential_aws, credential_redshift, or credential_custom
#' \item required boolean, Whether this param is required.
#' \item value string, The value you would like to set this param to. Setting this value makes this parameter a fixed param.
#' \item default string, If an argument for this parameter is not defined, it will use this default value. Use true, True, t, y, yes, or 1 for true bool's or false, False, f, n, no, or 0 for false bool's. Cannot be used for parameters that are required or a credential type.
#' \item allowedValues array, The possible values this parameter can take, effectively making this an enumerable parameter. Allowed values is an array of hashes of the following format: `{label: 'Import', 'value': 'import'}`
#' }}
#' \item{arguments}{object, Dictionary of name/value pairs to use to run this script. Only settable if this script has defined params.}
#' \item{isTemplate}{boolean, Whether others scripts use this one as a template.}
#' \item{publishedAsTemplateId}{integer, The ID of the template that this script is backing.}
#' \item{fromTemplateId}{integer, The ID of the template this script uses, if any.}
#' \item{templateDependentsCount}{integer, How many other scripts use this one as a template.}
#' \item{templateScriptName}{string, The name of the template script.}
#' \item{links}{object, A list containing the following elements: 
#' \itemize{
#' \item details string, The details link to get more information about the script.
#' \item runs string, The runs link to get the run information list for this script.
#' }}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{notifications}{object, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{runningAs}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{nextRunAt}{string, The time of the next scheduled run.}
#' \item{timeZone}{string, The time zone of this script.}
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{hidden}{boolean, The hidden status of the object.}
#' \item{targetProjectId}{integer, Target project to which script outputs will be added.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{sql}{string, The raw SQL query for the script.}
#' \item{expandedArguments}{object, Expanded arguments for use in injecting into different environments.}
#' \item{templateScriptId}{integer, The ID of the template script, if any.}
#' @export
scripts_patch <- function(id, name = NULL, sql = NULL, params = NULL, arguments = NULL, template_script_id = NULL, schedule = NULL, notifications = NULL, parent_id = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/scripts/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(name = name, sql = sql, params = params, arguments = arguments, templateScriptId = template_script_id, schedule = schedule, notifications = notifications, parentId = parent_id)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("patch", path, path_params, query_params, body_params)

  return(resp)

 }


#' Archive a script (deprecated, use archive endpoints instead)
#' @param id integer required. The ID for the script.
#' 
#' @return  An empty HTTP response
#' @export
scripts_delete <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/scripts/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get details about a script
#' @param id integer required. The ID for the script.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the script.}
#' \item{name}{string, The name of the script.}
#' \item{type}{string, The type of script.}
#' \item{createdAt}{string, The time this script was created.}
#' \item{updatedAt}{string, The time this script was last updated.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the script's last run.}
#' \item{finishedAt}{string, The time that the script's last run finished.}
#' \item{category}{string, The category of the script.}
#' \item{projects}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID for the project.
#' \item name string, The name of the project.
#' }}
#' \item{parentId}{integer, The ID of the parent job that will trigger this script}
#' \item{userContext}{string, "runner" or "author", who to execute the script as when run as a template.}
#' \item{params}{array, An array containing the following fields: 
#' \itemize{
#' \item name string, The variable's name as used within your code.
#' \item label string, The label to present to users when asking them for the value.
#' \item description string, A short sentence or fragment describing this parameter to the end user.
#' \item type string, The type of parameter. Valid options: string, multi_line_string, integer, float, bool, file, database, credential_aws, credential_redshift, or credential_custom
#' \item required boolean, Whether this param is required.
#' \item value string, The value you would like to set this param to. Setting this value makes this parameter a fixed param.
#' \item default string, If an argument for this parameter is not defined, it will use this default value. Use true, True, t, y, yes, or 1 for true bool's or false, False, f, n, no, or 0 for false bool's. Cannot be used for parameters that are required or a credential type.
#' \item allowedValues array, The possible values this parameter can take, effectively making this an enumerable parameter. Allowed values is an array of hashes of the following format: `{label: 'Import', 'value': 'import'}`
#' }}
#' \item{arguments}{object, Dictionary of name/value pairs to use to run this script. Only settable if this script has defined params.}
#' \item{isTemplate}{boolean, Whether others scripts use this one as a template.}
#' \item{publishedAsTemplateId}{integer, The ID of the template that this script is backing.}
#' \item{fromTemplateId}{integer, The ID of the template this script uses, if any.}
#' \item{templateDependentsCount}{integer, How many other scripts use this one as a template.}
#' \item{templateScriptName}{string, The name of the template script.}
#' \item{links}{object, A list containing the following elements: 
#' \itemize{
#' \item details string, The details link to get more information about the script.
#' \item runs string, The runs link to get the run information list for this script.
#' }}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{notifications}{object, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{runningAs}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{nextRunAt}{string, The time of the next scheduled run.}
#' \item{timeZone}{string, The time zone of this script.}
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{hidden}{boolean, The hidden status of the object.}
#' \item{targetProjectId}{integer, Target project to which script outputs will be added.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{sql}{string, The raw SQL query for the script.}
#' \item{expandedArguments}{object, Expanded arguments for use in injecting into different environments.}
#' \item{templateScriptId}{integer, The ID of the template script, if any.}
#' @export
scripts_get <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/scripts/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Run a script
#' @param id integer required. The ID for the script.
#' 
#' @return  An empty HTTP response
#' @export
scripts_post_run <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/scripts/{id}/run"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' Cancel a run
#' @param id integer required. The ID of the job.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the run.}
#' \item{state}{string, The state of the run, one of 'queued', 'running' or 'cancelled'.}
#' \item{isCancelRequested}{boolean, True if run cancel requested, else false.}
#' @export
scripts_post_cancel <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/scripts/{id}/cancel"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' Create a container
#' @param required_resources object required. A list containing the following elements: 
#' \itemize{
#' \item cpu integer, The number of CPU shares to allocate for the container. Each core has 1024 shares.
#' \item memory integer, The amount of RAM to allocate for the container (in MiB).
#' \item diskSpace number, The amount of disk space, in GB, to allocate for the container. This space will be used to hold the git repo configured for the container and anything your container writes to /tmp or /data. Fractional values (e.g. 0.25) are supported.
#' \item wholeInstance boolean, Whether or not to use the entire instance. If true, cpu, memory, and disk space are not required and will be set to an instance's max.
#' }
#' @param docker_command string required. The command to run on the container. Will be run via sh as: ["sh", "-c", dockerCommand]
#' @param docker_image_name string required. The name of the docker image to pull from DockerHub.
#' @param name string optional. The name of the container.
#' @param parent_id integer optional. The ID of the parent job that will trigger this script
#' @param user_context string optional. "runner" or "author", who to execute the script as when run as a template.
#' @param params array optional. An array containing the following fields: 
#' \itemize{
#' \item name string, The variable's name as used within your code.
#' \item label string, The label to present to users when asking them for the value.
#' \item description string, A short sentence or fragment describing this parameter to the end user.
#' \item type string, The type of parameter. Valid options: string, multi_line_string, integer, float, bool, file, database, credential_aws, credential_redshift, or credential_custom
#' \item required boolean, Whether this param is required.
#' \item value string, The value you would like to set this param to. Setting this value makes this parameter a fixed param.
#' \item default string, If an argument for this parameter is not defined, it will use this default value. Use true, True, t, y, yes, or 1 for true bool's or false, False, f, n, no, or 0 for false bool's. Cannot be used for parameters that are required or a credential type.
#' \item allowedValues array, The possible values this parameter can take, effectively making this an enumerable parameter. Allowed values is an array of hashes of the following format: `{label: 'Import', 'value': 'import'}`
#' }
#' @param arguments object optional. Dictionary of name/value pairs to use to run this script. Only settable if this script has defined params.
#' @param schedule object optional. A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }
#' @param notifications object optional. A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }
#' @param repo_http_uri string optional. The location of a github repo to clone into the container, e.g. github.com/my-user/my-repo.git.
#' @param repo_ref string optional. The tag or branch of the github repo to clone into the container.
#' @param remote_host_credential_id integer optional. The id of the database credentials to pass into the environment of the container.
#' @param git_credential_id integer optional. The id of the git credential to be used when checking out the specified git repo. If not supplied, the first git credential you've submitted will be used. Unnecessary if no git repo is specified or the git repo is public.
#' @param docker_image_tag string optional. The tag of the docker image to pull from DockerHub (default: latest).
#' @param cancel_timeout integer optional. The amount of time (in seconds) to wait before forcibly terminating the script. When the script is cancelled, it is first sent a TERM signal. If the script is still running after the timeout, it is sent a KILL signal. Defaults to 0.
#' @param time_zone string optional. The time zone of this script.
#' @param hidden boolean optional. The hidden status of the object.
#' @param target_project_id integer optional. Target project to which script outputs will be added.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the script.}
#' \item{name}{string, The name of the container.}
#' \item{type}{string, The type of the script (e.g Container)}
#' \item{createdAt}{string, The time this script was created.}
#' \item{updatedAt}{string, The time the script was last updated.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the script's last run.}
#' \item{finishedAt}{string, The time that the script's last run finished.}
#' \item{category}{string, The category of the script.}
#' \item{projects}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID for the project.
#' \item name string, The name of the project.
#' }}
#' \item{parentId}{integer, The ID of the parent job that will trigger this script}
#' \item{userContext}{string, "runner" or "author", who to execute the script as when run as a template.}
#' \item{params}{array, An array containing the following fields: 
#' \itemize{
#' \item name string, The variable's name as used within your code.
#' \item label string, The label to present to users when asking them for the value.
#' \item description string, A short sentence or fragment describing this parameter to the end user.
#' \item type string, The type of parameter. Valid options: string, multi_line_string, integer, float, bool, file, database, credential_aws, credential_redshift, or credential_custom
#' \item required boolean, Whether this param is required.
#' \item value string, The value you would like to set this param to. Setting this value makes this parameter a fixed param.
#' \item default string, If an argument for this parameter is not defined, it will use this default value. Use true, True, t, y, yes, or 1 for true bool's or false, False, f, n, no, or 0 for false bool's. Cannot be used for parameters that are required or a credential type.
#' \item allowedValues array, The possible values this parameter can take, effectively making this an enumerable parameter. Allowed values is an array of hashes of the following format: `{label: 'Import', 'value': 'import'}`
#' }}
#' \item{arguments}{object, Dictionary of name/value pairs to use to run this script. Only settable if this script has defined params.}
#' \item{isTemplate}{boolean, Whether others scripts use this one as a template.}
#' \item{templateDependentsCount}{integer, How many other scripts use this one as a template.}
#' \item{publishedAsTemplateId}{integer, The ID of the template that this script is backing.}
#' \item{fromTemplateId}{integer, The ID of the template script.}
#' \item{templateScriptName}{string, The name of the template script.}
#' \item{links}{object, A list containing the following elements: 
#' \itemize{
#' \item details string, The details link to get more information about the script.
#' \item runs string, The runs link to get the run information list for this script.
#' }}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{notifications}{object, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{runningAs}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{requiredResources}{object, A list containing the following elements: 
#' \itemize{
#' \item cpu integer, The number of CPU shares to allocate for the container. Each core has 1024 shares.
#' \item memory integer, The amount of RAM to allocate for the container (in MiB).
#' \item diskSpace number, The amount of disk space, in GB, to allocate for the container. This space will be used to hold the git repo configured for the container and anything your container writes to /tmp or /data. Fractional values (e.g. 0.25) are supported.
#' }}
#' \item{repoHttpUri}{string, The location of a github repo to clone into the container, e.g. github.com/my-user/my-repo.git.}
#' \item{repoRef}{string, The tag or branch of the github repo to clone into the container.}
#' \item{remoteHostCredentialId}{integer, The id of the database credentials to pass into the environment of the container.}
#' \item{gitCredentialId}{integer, The id of the git credential to be used when checking out the specified git repo. If not supplied, the first git credential you've submitted will be used. Unnecessary if no git repo is specified or the git repo is public.}
#' \item{dockerCommand}{string, The command to run on the container. Will be run via sh as: ["sh", "-c", dockerCommand]}
#' \item{dockerImageName}{string, The name of the docker image to pull from DockerHub.}
#' \item{dockerImageTag}{string, The tag of the docker image to pull from DockerHub (default: latest).}
#' \item{cancelTimeout}{integer, The amount of time (in seconds) to wait before forcibly terminating the script. When the script is cancelled, it is first sent a TERM signal. If the script is still running after the timeout, it is sent a KILL signal. Defaults to 0.}
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{timeZone}{string, The time zone of this script.}
#' \item{hidden}{boolean, The hidden status of the object.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{targetProjectId}{integer, Target project to which script outputs will be added.}
#' @export
scripts_post_containers <- function(required_resources, docker_command, docker_image_name, name = NULL, parent_id = NULL, user_context = NULL, params = NULL, arguments = NULL, schedule = NULL, notifications = NULL, repo_http_uri = NULL, repo_ref = NULL, remote_host_credential_id = NULL, git_credential_id = NULL, docker_image_tag = NULL, cancel_timeout = NULL, time_zone = NULL, hidden = NULL, target_project_id = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/scripts/containers"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(requiredResources = required_resources, dockerCommand = docker_command, dockerImageName = docker_image_name, name = name, parentId = parent_id, userContext = user_context, params = params, arguments = arguments, schedule = schedule, notifications = notifications, repoHttpUri = repo_http_uri, repoRef = repo_ref, remoteHostCredentialId = remote_host_credential_id, gitCredentialId = git_credential_id, dockerImageTag = docker_image_tag, cancelTimeout = cancel_timeout, timeZone = time_zone, hidden = hidden, targetProjectId = target_project_id)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' View a container
#' @param id integer required. The ID for the script.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the script.}
#' \item{name}{string, The name of the container.}
#' \item{type}{string, The type of the script (e.g Container)}
#' \item{createdAt}{string, The time this script was created.}
#' \item{updatedAt}{string, The time the script was last updated.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the script's last run.}
#' \item{finishedAt}{string, The time that the script's last run finished.}
#' \item{category}{string, The category of the script.}
#' \item{projects}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID for the project.
#' \item name string, The name of the project.
#' }}
#' \item{parentId}{integer, The ID of the parent job that will trigger this script}
#' \item{userContext}{string, "runner" or "author", who to execute the script as when run as a template.}
#' \item{params}{array, An array containing the following fields: 
#' \itemize{
#' \item name string, The variable's name as used within your code.
#' \item label string, The label to present to users when asking them for the value.
#' \item description string, A short sentence or fragment describing this parameter to the end user.
#' \item type string, The type of parameter. Valid options: string, multi_line_string, integer, float, bool, file, database, credential_aws, credential_redshift, or credential_custom
#' \item required boolean, Whether this param is required.
#' \item value string, The value you would like to set this param to. Setting this value makes this parameter a fixed param.
#' \item default string, If an argument for this parameter is not defined, it will use this default value. Use true, True, t, y, yes, or 1 for true bool's or false, False, f, n, no, or 0 for false bool's. Cannot be used for parameters that are required or a credential type.
#' \item allowedValues array, The possible values this parameter can take, effectively making this an enumerable parameter. Allowed values is an array of hashes of the following format: `{label: 'Import', 'value': 'import'}`
#' }}
#' \item{arguments}{object, Dictionary of name/value pairs to use to run this script. Only settable if this script has defined params.}
#' \item{isTemplate}{boolean, Whether others scripts use this one as a template.}
#' \item{templateDependentsCount}{integer, How many other scripts use this one as a template.}
#' \item{publishedAsTemplateId}{integer, The ID of the template that this script is backing.}
#' \item{fromTemplateId}{integer, The ID of the template script.}
#' \item{templateScriptName}{string, The name of the template script.}
#' \item{links}{object, A list containing the following elements: 
#' \itemize{
#' \item details string, The details link to get more information about the script.
#' \item runs string, The runs link to get the run information list for this script.
#' }}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{notifications}{object, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{runningAs}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{requiredResources}{object, A list containing the following elements: 
#' \itemize{
#' \item cpu integer, The number of CPU shares to allocate for the container. Each core has 1024 shares.
#' \item memory integer, The amount of RAM to allocate for the container (in MiB).
#' \item diskSpace number, The amount of disk space, in GB, to allocate for the container. This space will be used to hold the git repo configured for the container and anything your container writes to /tmp or /data. Fractional values (e.g. 0.25) are supported.
#' }}
#' \item{repoHttpUri}{string, The location of a github repo to clone into the container, e.g. github.com/my-user/my-repo.git.}
#' \item{repoRef}{string, The tag or branch of the github repo to clone into the container.}
#' \item{remoteHostCredentialId}{integer, The id of the database credentials to pass into the environment of the container.}
#' \item{gitCredentialId}{integer, The id of the git credential to be used when checking out the specified git repo. If not supplied, the first git credential you've submitted will be used. Unnecessary if no git repo is specified or the git repo is public.}
#' \item{dockerCommand}{string, The command to run on the container. Will be run via sh as: ["sh", "-c", dockerCommand]}
#' \item{dockerImageName}{string, The name of the docker image to pull from DockerHub.}
#' \item{dockerImageTag}{string, The tag of the docker image to pull from DockerHub (default: latest).}
#' \item{cancelTimeout}{integer, The amount of time (in seconds) to wait before forcibly terminating the script. When the script is cancelled, it is first sent a TERM signal. If the script is still running after the timeout, it is sent a KILL signal. Defaults to 0.}
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{timeZone}{string, The time zone of this script.}
#' \item{hidden}{boolean, The hidden status of the object.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{targetProjectId}{integer, Target project to which script outputs will be added.}
#' @export
scripts_get_containers <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/scripts/containers/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Edit a container
#' @param id integer required. The ID for the script.
#' @param required_resources object required. A list containing the following elements: 
#' \itemize{
#' \item cpu integer, The number of CPU shares to allocate for the container. Each core has 1024 shares.
#' \item memory integer, The amount of RAM to allocate for the container (in MiB).
#' \item diskSpace number, The amount of disk space, in GB, to allocate for the container. This space will be used to hold the git repo configured for the container and anything your container writes to /tmp or /data. Fractional values (e.g. 0.25) are supported.
#' \item wholeInstance boolean, Whether or not to use the entire instance. If true, cpu, memory, and disk space are not required and will be set to an instance's max.
#' }
#' @param docker_command string required. The command to run on the container. Will be run via sh as: ["sh", "-c", dockerCommand]
#' @param docker_image_name string required. The name of the docker image to pull from DockerHub.
#' @param name string optional. The name of the container.
#' @param parent_id integer optional. The ID of the parent job that will trigger this script
#' @param user_context string optional. "runner" or "author", who to execute the script as when run as a template.
#' @param params array optional. An array containing the following fields: 
#' \itemize{
#' \item name string, The variable's name as used within your code.
#' \item label string, The label to present to users when asking them for the value.
#' \item description string, A short sentence or fragment describing this parameter to the end user.
#' \item type string, The type of parameter. Valid options: string, multi_line_string, integer, float, bool, file, database, credential_aws, credential_redshift, or credential_custom
#' \item required boolean, Whether this param is required.
#' \item value string, The value you would like to set this param to. Setting this value makes this parameter a fixed param.
#' \item default string, If an argument for this parameter is not defined, it will use this default value. Use true, True, t, y, yes, or 1 for true bool's or false, False, f, n, no, or 0 for false bool's. Cannot be used for parameters that are required or a credential type.
#' \item allowedValues array, The possible values this parameter can take, effectively making this an enumerable parameter. Allowed values is an array of hashes of the following format: `{label: 'Import', 'value': 'import'}`
#' }
#' @param arguments object optional. Dictionary of name/value pairs to use to run this script. Only settable if this script has defined params.
#' @param schedule object optional. A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }
#' @param notifications object optional. A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }
#' @param repo_http_uri string optional. The location of a github repo to clone into the container, e.g. github.com/my-user/my-repo.git.
#' @param repo_ref string optional. The tag or branch of the github repo to clone into the container.
#' @param remote_host_credential_id integer optional. The id of the database credentials to pass into the environment of the container.
#' @param git_credential_id integer optional. The id of the git credential to be used when checking out the specified git repo. If not supplied, the first git credential you've submitted will be used. Unnecessary if no git repo is specified or the git repo is public.
#' @param docker_image_tag string optional. The tag of the docker image to pull from DockerHub (default: latest).
#' @param cancel_timeout integer optional. The amount of time (in seconds) to wait before forcibly terminating the script. When the script is cancelled, it is first sent a TERM signal. If the script is still running after the timeout, it is sent a KILL signal. Defaults to 0.
#' @param time_zone string optional. The time zone of this script.
#' @param target_project_id integer optional. Target project to which script outputs will be added.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the script.}
#' \item{name}{string, The name of the container.}
#' \item{type}{string, The type of the script (e.g Container)}
#' \item{createdAt}{string, The time this script was created.}
#' \item{updatedAt}{string, The time the script was last updated.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the script's last run.}
#' \item{finishedAt}{string, The time that the script's last run finished.}
#' \item{category}{string, The category of the script.}
#' \item{projects}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID for the project.
#' \item name string, The name of the project.
#' }}
#' \item{parentId}{integer, The ID of the parent job that will trigger this script}
#' \item{userContext}{string, "runner" or "author", who to execute the script as when run as a template.}
#' \item{params}{array, An array containing the following fields: 
#' \itemize{
#' \item name string, The variable's name as used within your code.
#' \item label string, The label to present to users when asking them for the value.
#' \item description string, A short sentence or fragment describing this parameter to the end user.
#' \item type string, The type of parameter. Valid options: string, multi_line_string, integer, float, bool, file, database, credential_aws, credential_redshift, or credential_custom
#' \item required boolean, Whether this param is required.
#' \item value string, The value you would like to set this param to. Setting this value makes this parameter a fixed param.
#' \item default string, If an argument for this parameter is not defined, it will use this default value. Use true, True, t, y, yes, or 1 for true bool's or false, False, f, n, no, or 0 for false bool's. Cannot be used for parameters that are required or a credential type.
#' \item allowedValues array, The possible values this parameter can take, effectively making this an enumerable parameter. Allowed values is an array of hashes of the following format: `{label: 'Import', 'value': 'import'}`
#' }}
#' \item{arguments}{object, Dictionary of name/value pairs to use to run this script. Only settable if this script has defined params.}
#' \item{isTemplate}{boolean, Whether others scripts use this one as a template.}
#' \item{templateDependentsCount}{integer, How many other scripts use this one as a template.}
#' \item{publishedAsTemplateId}{integer, The ID of the template that this script is backing.}
#' \item{fromTemplateId}{integer, The ID of the template script.}
#' \item{templateScriptName}{string, The name of the template script.}
#' \item{links}{object, A list containing the following elements: 
#' \itemize{
#' \item details string, The details link to get more information about the script.
#' \item runs string, The runs link to get the run information list for this script.
#' }}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{notifications}{object, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{runningAs}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{requiredResources}{object, A list containing the following elements: 
#' \itemize{
#' \item cpu integer, The number of CPU shares to allocate for the container. Each core has 1024 shares.
#' \item memory integer, The amount of RAM to allocate for the container (in MiB).
#' \item diskSpace number, The amount of disk space, in GB, to allocate for the container. This space will be used to hold the git repo configured for the container and anything your container writes to /tmp or /data. Fractional values (e.g. 0.25) are supported.
#' }}
#' \item{repoHttpUri}{string, The location of a github repo to clone into the container, e.g. github.com/my-user/my-repo.git.}
#' \item{repoRef}{string, The tag or branch of the github repo to clone into the container.}
#' \item{remoteHostCredentialId}{integer, The id of the database credentials to pass into the environment of the container.}
#' \item{gitCredentialId}{integer, The id of the git credential to be used when checking out the specified git repo. If not supplied, the first git credential you've submitted will be used. Unnecessary if no git repo is specified or the git repo is public.}
#' \item{dockerCommand}{string, The command to run on the container. Will be run via sh as: ["sh", "-c", dockerCommand]}
#' \item{dockerImageName}{string, The name of the docker image to pull from DockerHub.}
#' \item{dockerImageTag}{string, The tag of the docker image to pull from DockerHub (default: latest).}
#' \item{cancelTimeout}{integer, The amount of time (in seconds) to wait before forcibly terminating the script. When the script is cancelled, it is first sent a TERM signal. If the script is still running after the timeout, it is sent a KILL signal. Defaults to 0.}
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{timeZone}{string, The time zone of this script.}
#' \item{hidden}{boolean, The hidden status of the object.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{targetProjectId}{integer, Target project to which script outputs will be added.}
#' @export
scripts_put_containers <- function(id, required_resources, docker_command, docker_image_name, name = NULL, parent_id = NULL, user_context = NULL, params = NULL, arguments = NULL, schedule = NULL, notifications = NULL, repo_http_uri = NULL, repo_ref = NULL, remote_host_credential_id = NULL, git_credential_id = NULL, docker_image_tag = NULL, cancel_timeout = NULL, time_zone = NULL, target_project_id = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/scripts/containers/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(requiredResources = required_resources, dockerCommand = docker_command, dockerImageName = docker_image_name, name = name, parentId = parent_id, userContext = user_context, params = params, arguments = arguments, schedule = schedule, notifications = notifications, repoHttpUri = repo_http_uri, repoRef = repo_ref, remoteHostCredentialId = remote_host_credential_id, gitCredentialId = git_credential_id, dockerImageTag = docker_image_tag, cancelTimeout = cancel_timeout, timeZone = time_zone, targetProjectId = target_project_id)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update a container
#' @param id integer required. The ID for the script.
#' @param name string optional. The name of the container.
#' @param parent_id integer optional. The ID of the parent job that will trigger this script
#' @param user_context string optional. "runner" or "author", who to execute the script as when run as a template.
#' @param params array optional. An array containing the following fields: 
#' \itemize{
#' \item name string, The variable's name as used within your code.
#' \item label string, The label to present to users when asking them for the value.
#' \item description string, A short sentence or fragment describing this parameter to the end user.
#' \item type string, The type of parameter. Valid options: string, multi_line_string, integer, float, bool, file, database, credential_aws, credential_redshift, or credential_custom
#' \item required boolean, Whether this param is required.
#' \item value string, The value you would like to set this param to. Setting this value makes this parameter a fixed param.
#' \item default string, If an argument for this parameter is not defined, it will use this default value. Use true, True, t, y, yes, or 1 for true bool's or false, False, f, n, no, or 0 for false bool's. Cannot be used for parameters that are required or a credential type.
#' \item allowedValues array, The possible values this parameter can take, effectively making this an enumerable parameter. Allowed values is an array of hashes of the following format: `{label: 'Import', 'value': 'import'}`
#' }
#' @param arguments object optional. Dictionary of name/value pairs to use to run this script. Only settable if this script has defined params.
#' @param schedule object optional. A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }
#' @param notifications object optional. A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }
#' @param required_resources object optional. A list containing the following elements: 
#' \itemize{
#' \item cpu integer, The number of CPU shares to allocate for the container. Each core has 1024 shares.
#' \item memory integer, The amount of RAM to allocate for the container (in MiB).
#' \item diskSpace number, The amount of disk space, in GB, to allocate for the container. This space will be used to hold the git repo configured for the container and anything your container writes to /tmp or /data. Fractional values (e.g. 0.25) are supported.
#' \item wholeInstance boolean, Whether or not to use the entire instance. If true, cpu, memory, and disk space are not required and will be set to an instance's max.
#' }
#' @param repo_http_uri string optional. The location of a github repo to clone into the container, e.g. github.com/my-user/my-repo.git.
#' @param repo_ref string optional. The tag or branch of the github repo to clone into the container.
#' @param remote_host_credential_id integer optional. The id of the database credentials to pass into the environment of the container.
#' @param git_credential_id integer optional. The id of the git credential to be used when checking out the specified git repo. If not supplied, the first git credential you've submitted will be used. Unnecessary if no git repo is specified or the git repo is public.
#' @param docker_command string optional. The command to run on the container. Will be run via sh as: ["sh", "-c", dockerCommand]
#' @param docker_image_name string optional. The name of the docker image to pull from DockerHub.
#' @param docker_image_tag string optional. The tag of the docker image to pull from DockerHub (default: latest).
#' @param cancel_timeout integer optional. The amount of time (in seconds) to wait before forcibly terminating the script. When the script is cancelled, it is first sent a TERM signal. If the script is still running after the timeout, it is sent a KILL signal. Defaults to 0.
#' @param time_zone string optional. The time zone of this script.
#' @param target_project_id integer optional. Target project to which script outputs will be added.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the script.}
#' \item{name}{string, The name of the container.}
#' \item{type}{string, The type of the script (e.g Container)}
#' \item{createdAt}{string, The time this script was created.}
#' \item{updatedAt}{string, The time the script was last updated.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the script's last run.}
#' \item{finishedAt}{string, The time that the script's last run finished.}
#' \item{category}{string, The category of the script.}
#' \item{projects}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID for the project.
#' \item name string, The name of the project.
#' }}
#' \item{parentId}{integer, The ID of the parent job that will trigger this script}
#' \item{userContext}{string, "runner" or "author", who to execute the script as when run as a template.}
#' \item{params}{array, An array containing the following fields: 
#' \itemize{
#' \item name string, The variable's name as used within your code.
#' \item label string, The label to present to users when asking them for the value.
#' \item description string, A short sentence or fragment describing this parameter to the end user.
#' \item type string, The type of parameter. Valid options: string, multi_line_string, integer, float, bool, file, database, credential_aws, credential_redshift, or credential_custom
#' \item required boolean, Whether this param is required.
#' \item value string, The value you would like to set this param to. Setting this value makes this parameter a fixed param.
#' \item default string, If an argument for this parameter is not defined, it will use this default value. Use true, True, t, y, yes, or 1 for true bool's or false, False, f, n, no, or 0 for false bool's. Cannot be used for parameters that are required or a credential type.
#' \item allowedValues array, The possible values this parameter can take, effectively making this an enumerable parameter. Allowed values is an array of hashes of the following format: `{label: 'Import', 'value': 'import'}`
#' }}
#' \item{arguments}{object, Dictionary of name/value pairs to use to run this script. Only settable if this script has defined params.}
#' \item{isTemplate}{boolean, Whether others scripts use this one as a template.}
#' \item{templateDependentsCount}{integer, How many other scripts use this one as a template.}
#' \item{publishedAsTemplateId}{integer, The ID of the template that this script is backing.}
#' \item{fromTemplateId}{integer, The ID of the template script.}
#' \item{templateScriptName}{string, The name of the template script.}
#' \item{links}{object, A list containing the following elements: 
#' \itemize{
#' \item details string, The details link to get more information about the script.
#' \item runs string, The runs link to get the run information list for this script.
#' }}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{notifications}{object, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{runningAs}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{requiredResources}{object, A list containing the following elements: 
#' \itemize{
#' \item cpu integer, The number of CPU shares to allocate for the container. Each core has 1024 shares.
#' \item memory integer, The amount of RAM to allocate for the container (in MiB).
#' \item diskSpace number, The amount of disk space, in GB, to allocate for the container. This space will be used to hold the git repo configured for the container and anything your container writes to /tmp or /data. Fractional values (e.g. 0.25) are supported.
#' }}
#' \item{repoHttpUri}{string, The location of a github repo to clone into the container, e.g. github.com/my-user/my-repo.git.}
#' \item{repoRef}{string, The tag or branch of the github repo to clone into the container.}
#' \item{remoteHostCredentialId}{integer, The id of the database credentials to pass into the environment of the container.}
#' \item{gitCredentialId}{integer, The id of the git credential to be used when checking out the specified git repo. If not supplied, the first git credential you've submitted will be used. Unnecessary if no git repo is specified or the git repo is public.}
#' \item{dockerCommand}{string, The command to run on the container. Will be run via sh as: ["sh", "-c", dockerCommand]}
#' \item{dockerImageName}{string, The name of the docker image to pull from DockerHub.}
#' \item{dockerImageTag}{string, The tag of the docker image to pull from DockerHub (default: latest).}
#' \item{cancelTimeout}{integer, The amount of time (in seconds) to wait before forcibly terminating the script. When the script is cancelled, it is first sent a TERM signal. If the script is still running after the timeout, it is sent a KILL signal. Defaults to 0.}
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{timeZone}{string, The time zone of this script.}
#' \item{hidden}{boolean, The hidden status of the object.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{targetProjectId}{integer, Target project to which script outputs will be added.}
#' @export
scripts_patch_containers <- function(id, name = NULL, parent_id = NULL, user_context = NULL, params = NULL, arguments = NULL, schedule = NULL, notifications = NULL, required_resources = NULL, repo_http_uri = NULL, repo_ref = NULL, remote_host_credential_id = NULL, git_credential_id = NULL, docker_command = NULL, docker_image_name = NULL, docker_image_tag = NULL, cancel_timeout = NULL, time_zone = NULL, target_project_id = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/scripts/containers/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(name = name, parentId = parent_id, userContext = user_context, params = params, arguments = arguments, schedule = schedule, notifications = notifications, requiredResources = required_resources, repoHttpUri = repo_http_uri, repoRef = repo_ref, remoteHostCredentialId = remote_host_credential_id, gitCredentialId = git_credential_id, dockerCommand = docker_command, dockerImageName = docker_image_name, dockerImageTag = docker_image_tag, cancelTimeout = cancel_timeout, timeZone = time_zone, targetProjectId = target_project_id)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("patch", path, path_params, query_params, body_params)

  return(resp)

 }


#' Archive a container (deprecated, use archive endpoints)
#' @param id integer required. The ID for the script.
#' 
#' @return  An empty HTTP response
#' @export
scripts_delete_containers <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/scripts/containers/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' Add log messages
#' @param id integer required. The ID of the script.
#' @param run_id integer required. The ID of the script run.
#' @param message string optional. The log message to store.
#' @param level string optional. The log level of this message [default: info]
#' @param messages array optional. An array containing the following fields: 
#' \itemize{
#' \item message string, The log message to store.
#' \item level string, The log level of this message [default: info]
#' \item createdAt string, 
#' }
#' @param child_job_id integer optional. The ID of the child job the message came from.
#' 
#' @return  An empty HTTP response
#' @export
scripts_post_containers_runs_logs <- function(id, run_id, message = NULL, level = NULL, messages = NULL, child_job_id = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/scripts/containers/{id}/runs/{run_id}/logs"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list()
  body_params  <- list(message = message, level = level, messages = messages, childJobId = child_job_id)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get the logs for a run
#' @param id integer required. The ID of the container.
#' @param run_id integer required. The ID of the run.
#' @param last_id integer optional. The ID of the last log message received. Log entries with this ID value or lower will be omitted.Logs are sorted by ID if this value is provided, and are otherwise sorted by createdAt.
#' @param limit integer optional. The maximum number of log messages to return. Default of 10000.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of the log.}
#' \item{createdAt}{string, The time the log was created.}
#' \item{message}{string, The log message.}
#' \item{level}{string, The level of the log. One of unknown,fatal,error,warn,info,debug.}
#' @export
scripts_list_containers_runs_logs <- function(id, run_id, last_id = NULL, limit = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/scripts/containers/{id}/runs/{run_id}/logs"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list(last_id = last_id, limit = limit)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Create a SQL script
#' @param name string required. The name of the script.
#' @param sql string required. The raw SQL query for the script.
#' @param remote_host_id integer required. The remote host ID that this script will connect to.
#' @param credential_id integer required. The credential that this script will use.
#' @param parent_id integer optional. The ID of the parent job that will trigger this script
#' @param user_context string optional. "runner" or "author", who to execute the script as when run as a template.
#' @param params array optional. An array containing the following fields: 
#' \itemize{
#' \item name string, The variable's name as used within your code.
#' \item label string, The label to present to users when asking them for the value.
#' \item description string, A short sentence or fragment describing this parameter to the end user.
#' \item type string, The type of parameter. Valid options: string, multi_line_string, integer, float, bool, file, database, credential_aws, credential_redshift, or credential_custom
#' \item required boolean, Whether this param is required.
#' \item value string, The value you would like to set this param to. Setting this value makes this parameter a fixed param.
#' \item default string, If an argument for this parameter is not defined, it will use this default value. Use true, True, t, y, yes, or 1 for true bool's or false, False, f, n, no, or 0 for false bool's. Cannot be used for parameters that are required or a credential type.
#' \item allowedValues array, The possible values this parameter can take, effectively making this an enumerable parameter. Allowed values is an array of hashes of the following format: `{label: 'Import', 'value': 'import'}`
#' }
#' @param arguments object optional. Dictionary of name/value pairs to use to run this script. Only settable if this script has defined params.
#' @param schedule object optional. A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }
#' @param notifications object optional. A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }
#' @param next_run_at string optional. The time of the next scheduled run.
#' @param time_zone string optional. The time zone of this script.
#' @param hidden boolean optional. The hidden status of the object.
#' @param target_project_id integer optional. Target project to which script outputs will be added.
#' @param csv_settings object optional. A list containing the following elements: 
#' \itemize{
#' \item includeHeader boolean, Whether or not to include headers in the output data. Default: true
#' \item compression string, The type of compression to use, if any, one of "none", "zip", or "gzip". Default: gzip
#' \item columnDelimiter string, Which delimiter to use, one of "comma", "tab", or "pipe". Default: comma
#' \item unquoted boolean, Whether or not to quote fields. Default: false
#' \item forceMultifile boolean, Whether or not the csv should be split into multiple files. Default: false
#' \item filenamePrefix string, A user specified filename prefix for the output file to have. Default: null
#' }
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the script.}
#' \item{name}{string, The name of the script.}
#' \item{type}{string, The type of the script (e.g SQL, Container, Python, R, JavaScript)}
#' \item{createdAt}{string, The time this script was created.}
#' \item{updatedAt}{string, The time the script was last updated.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the script's last run.}
#' \item{finishedAt}{string, The time that the script's last run finished.}
#' \item{category}{string, The category of the script.}
#' \item{projects}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID for the project.
#' \item name string, The name of the project.
#' }}
#' \item{parentId}{integer, The ID of the parent job that will trigger this script}
#' \item{userContext}{string, "runner" or "author", who to execute the script as when run as a template.}
#' \item{params}{array, An array containing the following fields: 
#' \itemize{
#' \item name string, The variable's name as used within your code.
#' \item label string, The label to present to users when asking them for the value.
#' \item description string, A short sentence or fragment describing this parameter to the end user.
#' \item type string, The type of parameter. Valid options: string, multi_line_string, integer, float, bool, file, database, credential_aws, credential_redshift, or credential_custom
#' \item required boolean, Whether this param is required.
#' \item value string, The value you would like to set this param to. Setting this value makes this parameter a fixed param.
#' \item default string, If an argument for this parameter is not defined, it will use this default value. Use true, True, t, y, yes, or 1 for true bool's or false, False, f, n, no, or 0 for false bool's. Cannot be used for parameters that are required or a credential type.
#' \item allowedValues array, The possible values this parameter can take, effectively making this an enumerable parameter. Allowed values is an array of hashes of the following format: `{label: 'Import', 'value': 'import'}`
#' }}
#' \item{arguments}{object, Dictionary of name/value pairs to use to run this script. Only settable if this script has defined params.}
#' \item{isTemplate}{boolean, Whether others scripts use this one as a template.}
#' \item{publishedAsTemplateId}{integer, The ID of the template that this script is backing.}
#' \item{fromTemplateId}{integer, The ID of the template this script uses, if any.}
#' \item{templateDependentsCount}{integer, How many other scripts use this one as a template.}
#' \item{templateScriptName}{string, The name of the template script.}
#' \item{links}{object, A list containing the following elements: 
#' \itemize{
#' \item details string, The details link to get more information about the script.
#' \item runs string, The runs link to get the run information list for this script.
#' }}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{notifications}{object, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{runningAs}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{nextRunAt}{string, The time of the next scheduled run.}
#' \item{timeZone}{string, The time zone of this script.}
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{hidden}{boolean, The hidden status of the object.}
#' \item{targetProjectId}{integer, Target project to which script outputs will be added.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{sql}{string, The raw SQL query for the script.}
#' \item{expandedArguments}{object, Expanded arguments for use in injecting into different environments.}
#' \item{remoteHostId}{integer, The remote host ID that this script will connect to.}
#' \item{credentialId}{integer, The credential that this script will use.}
#' \item{codePreview}{string, The code that this script will run with arguments inserted.}
#' \item{csvSettings}{object, A list containing the following elements: 
#' \itemize{
#' \item includeHeader boolean, Whether or not to include headers in the output data. Default: true
#' \item compression string, The type of compression to use, if any, one of "none", "zip", or "gzip". Default: gzip
#' \item columnDelimiter string, Which delimiter to use, one of "comma", "tab", or "pipe". Default: comma
#' \item unquoted boolean, Whether or not to quote fields. Default: false
#' \item forceMultifile boolean, Whether or not the csv should be split into multiple files. Default: false
#' \item filenamePrefix string, A user specified filename prefix for the output file to have. Default: null
#' }}
#' @export
scripts_post_sql <- function(name, sql, remote_host_id, credential_id, parent_id = NULL, user_context = NULL, params = NULL, arguments = NULL, schedule = NULL, notifications = NULL, next_run_at = NULL, time_zone = NULL, hidden = NULL, target_project_id = NULL, csv_settings = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/scripts/sql"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(name = name, sql = sql, remoteHostId = remote_host_id, credentialId = credential_id, parentId = parent_id, userContext = user_context, params = params, arguments = arguments, schedule = schedule, notifications = notifications, nextRunAt = next_run_at, timeZone = time_zone, hidden = hidden, targetProjectId = target_project_id, csvSettings = csv_settings)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get a SQL script
#' @param id integer required. 
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the script.}
#' \item{name}{string, The name of the script.}
#' \item{type}{string, The type of the script (e.g SQL, Container, Python, R, JavaScript)}
#' \item{createdAt}{string, The time this script was created.}
#' \item{updatedAt}{string, The time the script was last updated.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the script's last run.}
#' \item{finishedAt}{string, The time that the script's last run finished.}
#' \item{category}{string, The category of the script.}
#' \item{projects}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID for the project.
#' \item name string, The name of the project.
#' }}
#' \item{parentId}{integer, The ID of the parent job that will trigger this script}
#' \item{userContext}{string, "runner" or "author", who to execute the script as when run as a template.}
#' \item{params}{array, An array containing the following fields: 
#' \itemize{
#' \item name string, The variable's name as used within your code.
#' \item label string, The label to present to users when asking them for the value.
#' \item description string, A short sentence or fragment describing this parameter to the end user.
#' \item type string, The type of parameter. Valid options: string, multi_line_string, integer, float, bool, file, database, credential_aws, credential_redshift, or credential_custom
#' \item required boolean, Whether this param is required.
#' \item value string, The value you would like to set this param to. Setting this value makes this parameter a fixed param.
#' \item default string, If an argument for this parameter is not defined, it will use this default value. Use true, True, t, y, yes, or 1 for true bool's or false, False, f, n, no, or 0 for false bool's. Cannot be used for parameters that are required or a credential type.
#' \item allowedValues array, The possible values this parameter can take, effectively making this an enumerable parameter. Allowed values is an array of hashes of the following format: `{label: 'Import', 'value': 'import'}`
#' }}
#' \item{arguments}{object, Dictionary of name/value pairs to use to run this script. Only settable if this script has defined params.}
#' \item{isTemplate}{boolean, Whether others scripts use this one as a template.}
#' \item{publishedAsTemplateId}{integer, The ID of the template that this script is backing.}
#' \item{fromTemplateId}{integer, The ID of the template this script uses, if any.}
#' \item{templateDependentsCount}{integer, How many other scripts use this one as a template.}
#' \item{templateScriptName}{string, The name of the template script.}
#' \item{links}{object, A list containing the following elements: 
#' \itemize{
#' \item details string, The details link to get more information about the script.
#' \item runs string, The runs link to get the run information list for this script.
#' }}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{notifications}{object, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{runningAs}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{nextRunAt}{string, The time of the next scheduled run.}
#' \item{timeZone}{string, The time zone of this script.}
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{hidden}{boolean, The hidden status of the object.}
#' \item{targetProjectId}{integer, Target project to which script outputs will be added.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{sql}{string, The raw SQL query for the script.}
#' \item{expandedArguments}{object, Expanded arguments for use in injecting into different environments.}
#' \item{remoteHostId}{integer, The remote host ID that this script will connect to.}
#' \item{credentialId}{integer, The credential that this script will use.}
#' \item{codePreview}{string, The code that this script will run with arguments inserted.}
#' \item{csvSettings}{object, A list containing the following elements: 
#' \itemize{
#' \item includeHeader boolean, Whether or not to include headers in the output data. Default: true
#' \item compression string, The type of compression to use, if any, one of "none", "zip", or "gzip". Default: gzip
#' \item columnDelimiter string, Which delimiter to use, one of "comma", "tab", or "pipe". Default: comma
#' \item unquoted boolean, Whether or not to quote fields. Default: false
#' \item forceMultifile boolean, Whether or not the csv should be split into multiple files. Default: false
#' \item filenamePrefix string, A user specified filename prefix for the output file to have. Default: null
#' }}
#' @export
scripts_get_sql <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/scripts/sql/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Replace all attributes of this SQL script
#' @param id integer required. The ID for the script.
#' @param name string required. The name of the script.
#' @param sql string required. The raw SQL query for the script.
#' @param remote_host_id integer required. The remote host ID that this script will connect to.
#' @param credential_id integer required. The credential that this script will use.
#' @param parent_id integer optional. The ID of the parent job that will trigger this script
#' @param user_context string optional. "runner" or "author", who to execute the script as when run as a template.
#' @param params array optional. An array containing the following fields: 
#' \itemize{
#' \item name string, The variable's name as used within your code.
#' \item label string, The label to present to users when asking them for the value.
#' \item description string, A short sentence or fragment describing this parameter to the end user.
#' \item type string, The type of parameter. Valid options: string, multi_line_string, integer, float, bool, file, database, credential_aws, credential_redshift, or credential_custom
#' \item required boolean, Whether this param is required.
#' \item value string, The value you would like to set this param to. Setting this value makes this parameter a fixed param.
#' \item default string, If an argument for this parameter is not defined, it will use this default value. Use true, True, t, y, yes, or 1 for true bool's or false, False, f, n, no, or 0 for false bool's. Cannot be used for parameters that are required or a credential type.
#' \item allowedValues array, The possible values this parameter can take, effectively making this an enumerable parameter. Allowed values is an array of hashes of the following format: `{label: 'Import', 'value': 'import'}`
#' }
#' @param arguments object optional. Dictionary of name/value pairs to use to run this script. Only settable if this script has defined params.
#' @param schedule object optional. A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }
#' @param notifications object optional. A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }
#' @param next_run_at string optional. The time of the next scheduled run.
#' @param time_zone string optional. The time zone of this script.
#' @param target_project_id integer optional. Target project to which script outputs will be added.
#' @param csv_settings object optional. A list containing the following elements: 
#' \itemize{
#' \item includeHeader boolean, Whether or not to include headers in the output data. Default: true
#' \item compression string, The type of compression to use, if any, one of "none", "zip", or "gzip". Default: gzip
#' \item columnDelimiter string, Which delimiter to use, one of "comma", "tab", or "pipe". Default: comma
#' \item unquoted boolean, Whether or not to quote fields. Default: false
#' \item forceMultifile boolean, Whether or not the csv should be split into multiple files. Default: false
#' \item filenamePrefix string, A user specified filename prefix for the output file to have. Default: null
#' }
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the script.}
#' \item{name}{string, The name of the script.}
#' \item{type}{string, The type of the script (e.g SQL, Container, Python, R, JavaScript)}
#' \item{createdAt}{string, The time this script was created.}
#' \item{updatedAt}{string, The time the script was last updated.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the script's last run.}
#' \item{finishedAt}{string, The time that the script's last run finished.}
#' \item{category}{string, The category of the script.}
#' \item{projects}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID for the project.
#' \item name string, The name of the project.
#' }}
#' \item{parentId}{integer, The ID of the parent job that will trigger this script}
#' \item{userContext}{string, "runner" or "author", who to execute the script as when run as a template.}
#' \item{params}{array, An array containing the following fields: 
#' \itemize{
#' \item name string, The variable's name as used within your code.
#' \item label string, The label to present to users when asking them for the value.
#' \item description string, A short sentence or fragment describing this parameter to the end user.
#' \item type string, The type of parameter. Valid options: string, multi_line_string, integer, float, bool, file, database, credential_aws, credential_redshift, or credential_custom
#' \item required boolean, Whether this param is required.
#' \item value string, The value you would like to set this param to. Setting this value makes this parameter a fixed param.
#' \item default string, If an argument for this parameter is not defined, it will use this default value. Use true, True, t, y, yes, or 1 for true bool's or false, False, f, n, no, or 0 for false bool's. Cannot be used for parameters that are required or a credential type.
#' \item allowedValues array, The possible values this parameter can take, effectively making this an enumerable parameter. Allowed values is an array of hashes of the following format: `{label: 'Import', 'value': 'import'}`
#' }}
#' \item{arguments}{object, Dictionary of name/value pairs to use to run this script. Only settable if this script has defined params.}
#' \item{isTemplate}{boolean, Whether others scripts use this one as a template.}
#' \item{publishedAsTemplateId}{integer, The ID of the template that this script is backing.}
#' \item{fromTemplateId}{integer, The ID of the template this script uses, if any.}
#' \item{templateDependentsCount}{integer, How many other scripts use this one as a template.}
#' \item{templateScriptName}{string, The name of the template script.}
#' \item{links}{object, A list containing the following elements: 
#' \itemize{
#' \item details string, The details link to get more information about the script.
#' \item runs string, The runs link to get the run information list for this script.
#' }}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{notifications}{object, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{runningAs}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{nextRunAt}{string, The time of the next scheduled run.}
#' \item{timeZone}{string, The time zone of this script.}
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{hidden}{boolean, The hidden status of the object.}
#' \item{targetProjectId}{integer, Target project to which script outputs will be added.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{sql}{string, The raw SQL query for the script.}
#' \item{expandedArguments}{object, Expanded arguments for use in injecting into different environments.}
#' \item{remoteHostId}{integer, The remote host ID that this script will connect to.}
#' \item{credentialId}{integer, The credential that this script will use.}
#' \item{codePreview}{string, The code that this script will run with arguments inserted.}
#' \item{csvSettings}{object, A list containing the following elements: 
#' \itemize{
#' \item includeHeader boolean, Whether or not to include headers in the output data. Default: true
#' \item compression string, The type of compression to use, if any, one of "none", "zip", or "gzip". Default: gzip
#' \item columnDelimiter string, Which delimiter to use, one of "comma", "tab", or "pipe". Default: comma
#' \item unquoted boolean, Whether or not to quote fields. Default: false
#' \item forceMultifile boolean, Whether or not the csv should be split into multiple files. Default: false
#' \item filenamePrefix string, A user specified filename prefix for the output file to have. Default: null
#' }}
#' @export
scripts_put_sql <- function(id, name, sql, remote_host_id, credential_id, parent_id = NULL, user_context = NULL, params = NULL, arguments = NULL, schedule = NULL, notifications = NULL, next_run_at = NULL, time_zone = NULL, target_project_id = NULL, csv_settings = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/scripts/sql/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(name = name, sql = sql, remoteHostId = remote_host_id, credentialId = credential_id, parentId = parent_id, userContext = user_context, params = params, arguments = arguments, schedule = schedule, notifications = notifications, nextRunAt = next_run_at, timeZone = time_zone, targetProjectId = target_project_id, csvSettings = csv_settings)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update some attributes of this SQL script
#' @param id integer required. The ID for the script.
#' @param name string optional. The name of the script.
#' @param parent_id integer optional. The ID of the parent job that will trigger this script
#' @param user_context string optional. "runner" or "author", who to execute the script as when run as a template.
#' @param params array optional. An array containing the following fields: 
#' \itemize{
#' \item name string, The variable's name as used within your code.
#' \item label string, The label to present to users when asking them for the value.
#' \item description string, A short sentence or fragment describing this parameter to the end user.
#' \item type string, The type of parameter. Valid options: string, multi_line_string, integer, float, bool, file, database, credential_aws, credential_redshift, or credential_custom
#' \item required boolean, Whether this param is required.
#' \item value string, The value you would like to set this param to. Setting this value makes this parameter a fixed param.
#' \item default string, If an argument for this parameter is not defined, it will use this default value. Use true, True, t, y, yes, or 1 for true bool's or false, False, f, n, no, or 0 for false bool's. Cannot be used for parameters that are required or a credential type.
#' \item allowedValues array, The possible values this parameter can take, effectively making this an enumerable parameter. Allowed values is an array of hashes of the following format: `{label: 'Import', 'value': 'import'}`
#' }
#' @param arguments object optional. Dictionary of name/value pairs to use to run this script. Only settable if this script has defined params.
#' @param schedule object optional. A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }
#' @param notifications object optional. A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }
#' @param next_run_at string optional. The time of the next scheduled run.
#' @param time_zone string optional. The time zone of this script.
#' @param target_project_id integer optional. Target project to which script outputs will be added.
#' @param sql string optional. The raw SQL query for the script.
#' @param remote_host_id integer optional. The remote host ID that this script will connect to.
#' @param credential_id integer optional. The credential that this script will use.
#' @param csv_settings object optional. A list containing the following elements: 
#' \itemize{
#' \item includeHeader boolean, Whether or not to include headers in the output data. Default: true
#' \item compression string, The type of compression to use, if any, one of "none", "zip", or "gzip". Default: gzip
#' \item columnDelimiter string, Which delimiter to use, one of "comma", "tab", or "pipe". Default: comma
#' \item unquoted boolean, Whether or not to quote fields. Default: false
#' \item forceMultifile boolean, Whether or not the csv should be split into multiple files. Default: false
#' \item filenamePrefix string, A user specified filename prefix for the output file to have. Default: null
#' }
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the script.}
#' \item{name}{string, The name of the script.}
#' \item{type}{string, The type of the script (e.g SQL, Container, Python, R, JavaScript)}
#' \item{createdAt}{string, The time this script was created.}
#' \item{updatedAt}{string, The time the script was last updated.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the script's last run.}
#' \item{finishedAt}{string, The time that the script's last run finished.}
#' \item{category}{string, The category of the script.}
#' \item{projects}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID for the project.
#' \item name string, The name of the project.
#' }}
#' \item{parentId}{integer, The ID of the parent job that will trigger this script}
#' \item{userContext}{string, "runner" or "author", who to execute the script as when run as a template.}
#' \item{params}{array, An array containing the following fields: 
#' \itemize{
#' \item name string, The variable's name as used within your code.
#' \item label string, The label to present to users when asking them for the value.
#' \item description string, A short sentence or fragment describing this parameter to the end user.
#' \item type string, The type of parameter. Valid options: string, multi_line_string, integer, float, bool, file, database, credential_aws, credential_redshift, or credential_custom
#' \item required boolean, Whether this param is required.
#' \item value string, The value you would like to set this param to. Setting this value makes this parameter a fixed param.
#' \item default string, If an argument for this parameter is not defined, it will use this default value. Use true, True, t, y, yes, or 1 for true bool's or false, False, f, n, no, or 0 for false bool's. Cannot be used for parameters that are required or a credential type.
#' \item allowedValues array, The possible values this parameter can take, effectively making this an enumerable parameter. Allowed values is an array of hashes of the following format: `{label: 'Import', 'value': 'import'}`
#' }}
#' \item{arguments}{object, Dictionary of name/value pairs to use to run this script. Only settable if this script has defined params.}
#' \item{isTemplate}{boolean, Whether others scripts use this one as a template.}
#' \item{publishedAsTemplateId}{integer, The ID of the template that this script is backing.}
#' \item{fromTemplateId}{integer, The ID of the template this script uses, if any.}
#' \item{templateDependentsCount}{integer, How many other scripts use this one as a template.}
#' \item{templateScriptName}{string, The name of the template script.}
#' \item{links}{object, A list containing the following elements: 
#' \itemize{
#' \item details string, The details link to get more information about the script.
#' \item runs string, The runs link to get the run information list for this script.
#' }}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{notifications}{object, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{runningAs}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{nextRunAt}{string, The time of the next scheduled run.}
#' \item{timeZone}{string, The time zone of this script.}
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{hidden}{boolean, The hidden status of the object.}
#' \item{targetProjectId}{integer, Target project to which script outputs will be added.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{sql}{string, The raw SQL query for the script.}
#' \item{expandedArguments}{object, Expanded arguments for use in injecting into different environments.}
#' \item{remoteHostId}{integer, The remote host ID that this script will connect to.}
#' \item{credentialId}{integer, The credential that this script will use.}
#' \item{codePreview}{string, The code that this script will run with arguments inserted.}
#' \item{csvSettings}{object, A list containing the following elements: 
#' \itemize{
#' \item includeHeader boolean, Whether or not to include headers in the output data. Default: true
#' \item compression string, The type of compression to use, if any, one of "none", "zip", or "gzip". Default: gzip
#' \item columnDelimiter string, Which delimiter to use, one of "comma", "tab", or "pipe". Default: comma
#' \item unquoted boolean, Whether or not to quote fields. Default: false
#' \item forceMultifile boolean, Whether or not the csv should be split into multiple files. Default: false
#' \item filenamePrefix string, A user specified filename prefix for the output file to have. Default: null
#' }}
#' @export
scripts_patch_sql <- function(id, name = NULL, parent_id = NULL, user_context = NULL, params = NULL, arguments = NULL, schedule = NULL, notifications = NULL, next_run_at = NULL, time_zone = NULL, target_project_id = NULL, sql = NULL, remote_host_id = NULL, credential_id = NULL, csv_settings = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/scripts/sql/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(name = name, parentId = parent_id, userContext = user_context, params = params, arguments = arguments, schedule = schedule, notifications = notifications, nextRunAt = next_run_at, timeZone = time_zone, targetProjectId = target_project_id, sql = sql, remoteHostId = remote_host_id, credentialId = credential_id, csvSettings = csv_settings)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("patch", path, path_params, query_params, body_params)

  return(resp)

 }


#' Archive a SQL script (deprecated, use archiving endpoints instead)
#' @param id integer required. 
#' 
#' @return  An empty HTTP response
#' @export
scripts_delete_sql <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/scripts/sql/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' Create a Python Script
#' @param name string required. The name of the script.
#' @param source string required. The body/text of the script.
#' @param parent_id integer optional. The ID of the parent job that will trigger this script
#' @param user_context string optional. "runner" or "author", who to execute the script as when run as a template.
#' @param params array optional. An array containing the following fields: 
#' \itemize{
#' \item name string, The variable's name as used within your code.
#' \item label string, The label to present to users when asking them for the value.
#' \item description string, A short sentence or fragment describing this parameter to the end user.
#' \item type string, The type of parameter. Valid options: string, multi_line_string, integer, float, bool, file, database, credential_aws, credential_redshift, or credential_custom
#' \item required boolean, Whether this param is required.
#' \item value string, The value you would like to set this param to. Setting this value makes this parameter a fixed param.
#' \item default string, If an argument for this parameter is not defined, it will use this default value. Use true, True, t, y, yes, or 1 for true bool's or false, False, f, n, no, or 0 for false bool's. Cannot be used for parameters that are required or a credential type.
#' \item allowedValues array, The possible values this parameter can take, effectively making this an enumerable parameter. Allowed values is an array of hashes of the following format: `{label: 'Import', 'value': 'import'}`
#' }
#' @param arguments object optional. Dictionary of name/value pairs to use to run this script. Only settable if this script has defined params.
#' @param schedule object optional. A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }
#' @param notifications object optional. A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }
#' @param next_run_at string optional. The time of the next scheduled run.
#' @param time_zone string optional. The time zone of this script.
#' @param hidden boolean optional. The hidden status of the object.
#' @param target_project_id integer optional. Target project to which script outputs will be added.
#' @param required_resources object optional. A list containing the following elements: 
#' \itemize{
#' \item cpu integer, The number of CPU shares to allocate for the container. Each core has 1024 shares. Must be at least 2 shares.
#' \item memory integer, The amount of RAM to allocate for the container (in MiB). Must be at least 4 MiB.
#' \item diskSpace number, The amount of disk space, in GB, to allocate for the container. This space will be used to hold the git repo configured for the container and anything your container writes to /tmp or /data. Fractional values (e.g. 0.25) are supported.
#' }
#' @param cancel_timeout integer optional. The amount of time (in seconds) to wait before forcibly terminating the script. When the script is cancelled, it is first sent a TERM signal. If the script is still running after the timeout, it is sent a KILL signal. Defaults to 0.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the script.}
#' \item{name}{string, The name of the script.}
#' \item{type}{string, The type of the script (e.g SQL, Container, Python, R, JavaScript)}
#' \item{createdAt}{string, The time this script was created.}
#' \item{updatedAt}{string, The time the script was last updated.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the script's last run.}
#' \item{finishedAt}{string, The time that the script's last run finished.}
#' \item{category}{string, The category of the script.}
#' \item{projects}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID for the project.
#' \item name string, The name of the project.
#' }}
#' \item{parentId}{integer, The ID of the parent job that will trigger this script}
#' \item{userContext}{string, "runner" or "author", who to execute the script as when run as a template.}
#' \item{params}{array, An array containing the following fields: 
#' \itemize{
#' \item name string, The variable's name as used within your code.
#' \item label string, The label to present to users when asking them for the value.
#' \item description string, A short sentence or fragment describing this parameter to the end user.
#' \item type string, The type of parameter. Valid options: string, multi_line_string, integer, float, bool, file, database, credential_aws, credential_redshift, or credential_custom
#' \item required boolean, Whether this param is required.
#' \item value string, The value you would like to set this param to. Setting this value makes this parameter a fixed param.
#' \item default string, If an argument for this parameter is not defined, it will use this default value. Use true, True, t, y, yes, or 1 for true bool's or false, False, f, n, no, or 0 for false bool's. Cannot be used for parameters that are required or a credential type.
#' \item allowedValues array, The possible values this parameter can take, effectively making this an enumerable parameter. Allowed values is an array of hashes of the following format: `{label: 'Import', 'value': 'import'}`
#' }}
#' \item{arguments}{object, Dictionary of name/value pairs to use to run this script. Only settable if this script has defined params.}
#' \item{isTemplate}{boolean, Whether others scripts use this one as a template.}
#' \item{publishedAsTemplateId}{integer, The ID of the template that this script is backing.}
#' \item{fromTemplateId}{integer, The ID of the template this script uses, if any.}
#' \item{templateDependentsCount}{integer, How many other scripts use this one as a template.}
#' \item{templateScriptName}{string, The name of the template script.}
#' \item{links}{object, A list containing the following elements: 
#' \itemize{
#' \item details string, The details link to get more information about the script.
#' \item runs string, The runs link to get the run information list for this script.
#' }}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{notifications}{object, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{runningAs}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{nextRunAt}{string, The time of the next scheduled run.}
#' \item{timeZone}{string, The time zone of this script.}
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{hidden}{boolean, The hidden status of the object.}
#' \item{targetProjectId}{integer, Target project to which script outputs will be added.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{requiredResources}{object, A list containing the following elements: 
#' \itemize{
#' \item cpu integer, The number of CPU shares to allocate for the container. Each core has 1024 shares. Must be at least 2 shares.
#' \item memory integer, The amount of RAM to allocate for the container (in MiB). Must be at least 4 MiB.
#' \item diskSpace number, The amount of disk space, in GB, to allocate for the container. This space will be used to hold the git repo configured for the container and anything your container writes to /tmp or /data. Fractional values (e.g. 0.25) are supported.
#' }}
#' \item{source}{string, The body/text of the script.}
#' \item{cancelTimeout}{integer, The amount of time (in seconds) to wait before forcibly terminating the script. When the script is cancelled, it is first sent a TERM signal. If the script is still running after the timeout, it is sent a KILL signal. Defaults to 0.}
#' @export
scripts_post_python3 <- function(name, source, parent_id = NULL, user_context = NULL, params = NULL, arguments = NULL, schedule = NULL, notifications = NULL, next_run_at = NULL, time_zone = NULL, hidden = NULL, target_project_id = NULL, required_resources = NULL, cancel_timeout = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/scripts/python3"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(name = name, source = source, parentId = parent_id, userContext = user_context, params = params, arguments = arguments, schedule = schedule, notifications = notifications, nextRunAt = next_run_at, timeZone = time_zone, hidden = hidden, targetProjectId = target_project_id, requiredResources = required_resources, cancelTimeout = cancel_timeout)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get a Python Script
#' @param id integer required. 
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the script.}
#' \item{name}{string, The name of the script.}
#' \item{type}{string, The type of the script (e.g SQL, Container, Python, R, JavaScript)}
#' \item{createdAt}{string, The time this script was created.}
#' \item{updatedAt}{string, The time the script was last updated.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the script's last run.}
#' \item{finishedAt}{string, The time that the script's last run finished.}
#' \item{category}{string, The category of the script.}
#' \item{projects}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID for the project.
#' \item name string, The name of the project.
#' }}
#' \item{parentId}{integer, The ID of the parent job that will trigger this script}
#' \item{userContext}{string, "runner" or "author", who to execute the script as when run as a template.}
#' \item{params}{array, An array containing the following fields: 
#' \itemize{
#' \item name string, The variable's name as used within your code.
#' \item label string, The label to present to users when asking them for the value.
#' \item description string, A short sentence or fragment describing this parameter to the end user.
#' \item type string, The type of parameter. Valid options: string, multi_line_string, integer, float, bool, file, database, credential_aws, credential_redshift, or credential_custom
#' \item required boolean, Whether this param is required.
#' \item value string, The value you would like to set this param to. Setting this value makes this parameter a fixed param.
#' \item default string, If an argument for this parameter is not defined, it will use this default value. Use true, True, t, y, yes, or 1 for true bool's or false, False, f, n, no, or 0 for false bool's. Cannot be used for parameters that are required or a credential type.
#' \item allowedValues array, The possible values this parameter can take, effectively making this an enumerable parameter. Allowed values is an array of hashes of the following format: `{label: 'Import', 'value': 'import'}`
#' }}
#' \item{arguments}{object, Dictionary of name/value pairs to use to run this script. Only settable if this script has defined params.}
#' \item{isTemplate}{boolean, Whether others scripts use this one as a template.}
#' \item{publishedAsTemplateId}{integer, The ID of the template that this script is backing.}
#' \item{fromTemplateId}{integer, The ID of the template this script uses, if any.}
#' \item{templateDependentsCount}{integer, How many other scripts use this one as a template.}
#' \item{templateScriptName}{string, The name of the template script.}
#' \item{links}{object, A list containing the following elements: 
#' \itemize{
#' \item details string, The details link to get more information about the script.
#' \item runs string, The runs link to get the run information list for this script.
#' }}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{notifications}{object, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{runningAs}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{nextRunAt}{string, The time of the next scheduled run.}
#' \item{timeZone}{string, The time zone of this script.}
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{hidden}{boolean, The hidden status of the object.}
#' \item{targetProjectId}{integer, Target project to which script outputs will be added.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{requiredResources}{object, A list containing the following elements: 
#' \itemize{
#' \item cpu integer, The number of CPU shares to allocate for the container. Each core has 1024 shares. Must be at least 2 shares.
#' \item memory integer, The amount of RAM to allocate for the container (in MiB). Must be at least 4 MiB.
#' \item diskSpace number, The amount of disk space, in GB, to allocate for the container. This space will be used to hold the git repo configured for the container and anything your container writes to /tmp or /data. Fractional values (e.g. 0.25) are supported.
#' }}
#' \item{source}{string, The body/text of the script.}
#' \item{cancelTimeout}{integer, The amount of time (in seconds) to wait before forcibly terminating the script. When the script is cancelled, it is first sent a TERM signal. If the script is still running after the timeout, it is sent a KILL signal. Defaults to 0.}
#' @export
scripts_get_python3 <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/scripts/python3/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Replace all attributes of this Python Script
#' @param id integer required. The ID for the script.
#' @param name string required. The name of the script.
#' @param source string required. The body/text of the script.
#' @param parent_id integer optional. The ID of the parent job that will trigger this script
#' @param user_context string optional. "runner" or "author", who to execute the script as when run as a template.
#' @param params array optional. An array containing the following fields: 
#' \itemize{
#' \item name string, The variable's name as used within your code.
#' \item label string, The label to present to users when asking them for the value.
#' \item description string, A short sentence or fragment describing this parameter to the end user.
#' \item type string, The type of parameter. Valid options: string, multi_line_string, integer, float, bool, file, database, credential_aws, credential_redshift, or credential_custom
#' \item required boolean, Whether this param is required.
#' \item value string, The value you would like to set this param to. Setting this value makes this parameter a fixed param.
#' \item default string, If an argument for this parameter is not defined, it will use this default value. Use true, True, t, y, yes, or 1 for true bool's or false, False, f, n, no, or 0 for false bool's. Cannot be used for parameters that are required or a credential type.
#' \item allowedValues array, The possible values this parameter can take, effectively making this an enumerable parameter. Allowed values is an array of hashes of the following format: `{label: 'Import', 'value': 'import'}`
#' }
#' @param arguments object optional. Dictionary of name/value pairs to use to run this script. Only settable if this script has defined params.
#' @param schedule object optional. A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }
#' @param notifications object optional. A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }
#' @param next_run_at string optional. The time of the next scheduled run.
#' @param time_zone string optional. The time zone of this script.
#' @param target_project_id integer optional. Target project to which script outputs will be added.
#' @param required_resources object optional. A list containing the following elements: 
#' \itemize{
#' \item cpu integer, The number of CPU shares to allocate for the container. Each core has 1024 shares. Must be at least 2 shares.
#' \item memory integer, The amount of RAM to allocate for the container (in MiB). Must be at least 4 MiB.
#' \item diskSpace number, The amount of disk space, in GB, to allocate for the container. This space will be used to hold the git repo configured for the container and anything your container writes to /tmp or /data. Fractional values (e.g. 0.25) are supported.
#' }
#' @param cancel_timeout integer optional. The amount of time (in seconds) to wait before forcibly terminating the script. When the script is cancelled, it is first sent a TERM signal. If the script is still running after the timeout, it is sent a KILL signal. Defaults to 0.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the script.}
#' \item{name}{string, The name of the script.}
#' \item{type}{string, The type of the script (e.g SQL, Container, Python, R, JavaScript)}
#' \item{createdAt}{string, The time this script was created.}
#' \item{updatedAt}{string, The time the script was last updated.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the script's last run.}
#' \item{finishedAt}{string, The time that the script's last run finished.}
#' \item{category}{string, The category of the script.}
#' \item{projects}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID for the project.
#' \item name string, The name of the project.
#' }}
#' \item{parentId}{integer, The ID of the parent job that will trigger this script}
#' \item{userContext}{string, "runner" or "author", who to execute the script as when run as a template.}
#' \item{params}{array, An array containing the following fields: 
#' \itemize{
#' \item name string, The variable's name as used within your code.
#' \item label string, The label to present to users when asking them for the value.
#' \item description string, A short sentence or fragment describing this parameter to the end user.
#' \item type string, The type of parameter. Valid options: string, multi_line_string, integer, float, bool, file, database, credential_aws, credential_redshift, or credential_custom
#' \item required boolean, Whether this param is required.
#' \item value string, The value you would like to set this param to. Setting this value makes this parameter a fixed param.
#' \item default string, If an argument for this parameter is not defined, it will use this default value. Use true, True, t, y, yes, or 1 for true bool's or false, False, f, n, no, or 0 for false bool's. Cannot be used for parameters that are required or a credential type.
#' \item allowedValues array, The possible values this parameter can take, effectively making this an enumerable parameter. Allowed values is an array of hashes of the following format: `{label: 'Import', 'value': 'import'}`
#' }}
#' \item{arguments}{object, Dictionary of name/value pairs to use to run this script. Only settable if this script has defined params.}
#' \item{isTemplate}{boolean, Whether others scripts use this one as a template.}
#' \item{publishedAsTemplateId}{integer, The ID of the template that this script is backing.}
#' \item{fromTemplateId}{integer, The ID of the template this script uses, if any.}
#' \item{templateDependentsCount}{integer, How many other scripts use this one as a template.}
#' \item{templateScriptName}{string, The name of the template script.}
#' \item{links}{object, A list containing the following elements: 
#' \itemize{
#' \item details string, The details link to get more information about the script.
#' \item runs string, The runs link to get the run information list for this script.
#' }}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{notifications}{object, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{runningAs}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{nextRunAt}{string, The time of the next scheduled run.}
#' \item{timeZone}{string, The time zone of this script.}
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{hidden}{boolean, The hidden status of the object.}
#' \item{targetProjectId}{integer, Target project to which script outputs will be added.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{requiredResources}{object, A list containing the following elements: 
#' \itemize{
#' \item cpu integer, The number of CPU shares to allocate for the container. Each core has 1024 shares. Must be at least 2 shares.
#' \item memory integer, The amount of RAM to allocate for the container (in MiB). Must be at least 4 MiB.
#' \item diskSpace number, The amount of disk space, in GB, to allocate for the container. This space will be used to hold the git repo configured for the container and anything your container writes to /tmp or /data. Fractional values (e.g. 0.25) are supported.
#' }}
#' \item{source}{string, The body/text of the script.}
#' \item{cancelTimeout}{integer, The amount of time (in seconds) to wait before forcibly terminating the script. When the script is cancelled, it is first sent a TERM signal. If the script is still running after the timeout, it is sent a KILL signal. Defaults to 0.}
#' @export
scripts_put_python3 <- function(id, name, source, parent_id = NULL, user_context = NULL, params = NULL, arguments = NULL, schedule = NULL, notifications = NULL, next_run_at = NULL, time_zone = NULL, target_project_id = NULL, required_resources = NULL, cancel_timeout = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/scripts/python3/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(name = name, source = source, parentId = parent_id, userContext = user_context, params = params, arguments = arguments, schedule = schedule, notifications = notifications, nextRunAt = next_run_at, timeZone = time_zone, targetProjectId = target_project_id, requiredResources = required_resources, cancelTimeout = cancel_timeout)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update some attributes of this Python Script
#' @param id integer required. The ID for the script.
#' @param name string optional. The name of the script.
#' @param parent_id integer optional. The ID of the parent job that will trigger this script
#' @param user_context string optional. "runner" or "author", who to execute the script as when run as a template.
#' @param params array optional. An array containing the following fields: 
#' \itemize{
#' \item name string, The variable's name as used within your code.
#' \item label string, The label to present to users when asking them for the value.
#' \item description string, A short sentence or fragment describing this parameter to the end user.
#' \item type string, The type of parameter. Valid options: string, multi_line_string, integer, float, bool, file, database, credential_aws, credential_redshift, or credential_custom
#' \item required boolean, Whether this param is required.
#' \item value string, The value you would like to set this param to. Setting this value makes this parameter a fixed param.
#' \item default string, If an argument for this parameter is not defined, it will use this default value. Use true, True, t, y, yes, or 1 for true bool's or false, False, f, n, no, or 0 for false bool's. Cannot be used for parameters that are required or a credential type.
#' \item allowedValues array, The possible values this parameter can take, effectively making this an enumerable parameter. Allowed values is an array of hashes of the following format: `{label: 'Import', 'value': 'import'}`
#' }
#' @param arguments object optional. Dictionary of name/value pairs to use to run this script. Only settable if this script has defined params.
#' @param schedule object optional. A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }
#' @param notifications object optional. A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }
#' @param next_run_at string optional. The time of the next scheduled run.
#' @param time_zone string optional. The time zone of this script.
#' @param target_project_id integer optional. Target project to which script outputs will be added.
#' @param required_resources object optional. A list containing the following elements: 
#' \itemize{
#' \item cpu integer, The number of CPU shares to allocate for the container. Each core has 1024 shares. Must be at least 2 shares.
#' \item memory integer, The amount of RAM to allocate for the container (in MiB). Must be at least 4 MiB.
#' \item diskSpace number, The amount of disk space, in GB, to allocate for the container. This space will be used to hold the git repo configured for the container and anything your container writes to /tmp or /data. Fractional values (e.g. 0.25) are supported.
#' }
#' @param source string optional. The body/text of the script.
#' @param cancel_timeout integer optional. The amount of time (in seconds) to wait before forcibly terminating the script. When the script is cancelled, it is first sent a TERM signal. If the script is still running after the timeout, it is sent a KILL signal. Defaults to 0.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the script.}
#' \item{name}{string, The name of the script.}
#' \item{type}{string, The type of the script (e.g SQL, Container, Python, R, JavaScript)}
#' \item{createdAt}{string, The time this script was created.}
#' \item{updatedAt}{string, The time the script was last updated.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the script's last run.}
#' \item{finishedAt}{string, The time that the script's last run finished.}
#' \item{category}{string, The category of the script.}
#' \item{projects}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID for the project.
#' \item name string, The name of the project.
#' }}
#' \item{parentId}{integer, The ID of the parent job that will trigger this script}
#' \item{userContext}{string, "runner" or "author", who to execute the script as when run as a template.}
#' \item{params}{array, An array containing the following fields: 
#' \itemize{
#' \item name string, The variable's name as used within your code.
#' \item label string, The label to present to users when asking them for the value.
#' \item description string, A short sentence or fragment describing this parameter to the end user.
#' \item type string, The type of parameter. Valid options: string, multi_line_string, integer, float, bool, file, database, credential_aws, credential_redshift, or credential_custom
#' \item required boolean, Whether this param is required.
#' \item value string, The value you would like to set this param to. Setting this value makes this parameter a fixed param.
#' \item default string, If an argument for this parameter is not defined, it will use this default value. Use true, True, t, y, yes, or 1 for true bool's or false, False, f, n, no, or 0 for false bool's. Cannot be used for parameters that are required or a credential type.
#' \item allowedValues array, The possible values this parameter can take, effectively making this an enumerable parameter. Allowed values is an array of hashes of the following format: `{label: 'Import', 'value': 'import'}`
#' }}
#' \item{arguments}{object, Dictionary of name/value pairs to use to run this script. Only settable if this script has defined params.}
#' \item{isTemplate}{boolean, Whether others scripts use this one as a template.}
#' \item{publishedAsTemplateId}{integer, The ID of the template that this script is backing.}
#' \item{fromTemplateId}{integer, The ID of the template this script uses, if any.}
#' \item{templateDependentsCount}{integer, How many other scripts use this one as a template.}
#' \item{templateScriptName}{string, The name of the template script.}
#' \item{links}{object, A list containing the following elements: 
#' \itemize{
#' \item details string, The details link to get more information about the script.
#' \item runs string, The runs link to get the run information list for this script.
#' }}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{notifications}{object, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{runningAs}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{nextRunAt}{string, The time of the next scheduled run.}
#' \item{timeZone}{string, The time zone of this script.}
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{hidden}{boolean, The hidden status of the object.}
#' \item{targetProjectId}{integer, Target project to which script outputs will be added.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{requiredResources}{object, A list containing the following elements: 
#' \itemize{
#' \item cpu integer, The number of CPU shares to allocate for the container. Each core has 1024 shares. Must be at least 2 shares.
#' \item memory integer, The amount of RAM to allocate for the container (in MiB). Must be at least 4 MiB.
#' \item diskSpace number, The amount of disk space, in GB, to allocate for the container. This space will be used to hold the git repo configured for the container and anything your container writes to /tmp or /data. Fractional values (e.g. 0.25) are supported.
#' }}
#' \item{source}{string, The body/text of the script.}
#' \item{cancelTimeout}{integer, The amount of time (in seconds) to wait before forcibly terminating the script. When the script is cancelled, it is first sent a TERM signal. If the script is still running after the timeout, it is sent a KILL signal. Defaults to 0.}
#' @export
scripts_patch_python3 <- function(id, name = NULL, parent_id = NULL, user_context = NULL, params = NULL, arguments = NULL, schedule = NULL, notifications = NULL, next_run_at = NULL, time_zone = NULL, target_project_id = NULL, required_resources = NULL, source = NULL, cancel_timeout = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/scripts/python3/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(name = name, parentId = parent_id, userContext = user_context, params = params, arguments = arguments, schedule = schedule, notifications = notifications, nextRunAt = next_run_at, timeZone = time_zone, targetProjectId = target_project_id, requiredResources = required_resources, source = source, cancelTimeout = cancel_timeout)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("patch", path, path_params, query_params, body_params)

  return(resp)

 }


#' Archive a Python Script (deprecated, use archiving endpoints instead)
#' @param id integer required. 
#' 
#' @return  An empty HTTP response
#' @export
scripts_delete_python3 <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/scripts/python3/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' Create an R Script
#' @param name string required. The name of the script.
#' @param source string required. The body/text of the script.
#' @param parent_id integer optional. The ID of the parent job that will trigger this script
#' @param user_context string optional. "runner" or "author", who to execute the script as when run as a template.
#' @param params array optional. An array containing the following fields: 
#' \itemize{
#' \item name string, The variable's name as used within your code.
#' \item label string, The label to present to users when asking them for the value.
#' \item description string, A short sentence or fragment describing this parameter to the end user.
#' \item type string, The type of parameter. Valid options: string, multi_line_string, integer, float, bool, file, database, credential_aws, credential_redshift, or credential_custom
#' \item required boolean, Whether this param is required.
#' \item value string, The value you would like to set this param to. Setting this value makes this parameter a fixed param.
#' \item default string, If an argument for this parameter is not defined, it will use this default value. Use true, True, t, y, yes, or 1 for true bool's or false, False, f, n, no, or 0 for false bool's. Cannot be used for parameters that are required or a credential type.
#' \item allowedValues array, The possible values this parameter can take, effectively making this an enumerable parameter. Allowed values is an array of hashes of the following format: `{label: 'Import', 'value': 'import'}`
#' }
#' @param arguments object optional. Dictionary of name/value pairs to use to run this script. Only settable if this script has defined params.
#' @param schedule object optional. A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }
#' @param notifications object optional. A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }
#' @param next_run_at string optional. The time of the next scheduled run.
#' @param time_zone string optional. The time zone of this script.
#' @param hidden boolean optional. The hidden status of the object.
#' @param target_project_id integer optional. Target project to which script outputs will be added.
#' @param required_resources object optional. A list containing the following elements: 
#' \itemize{
#' \item cpu integer, The number of CPU shares to allocate for the container. Each core has 1024 shares. Must be at least 2 shares.
#' \item memory integer, The amount of RAM to allocate for the container (in MiB). Must be at least 4 MiB.
#' \item diskSpace number, The amount of disk space, in GB, to allocate for the container. This space will be used to hold the git repo configured for the container and anything your container writes to /tmp or /data. Fractional values (e.g. 0.25) are supported.
#' }
#' @param cancel_timeout integer optional. The amount of time (in seconds) to wait before forcibly terminating the script. When the script is cancelled, it is first sent a TERM signal. If the script is still running after the timeout, it is sent a KILL signal. Defaults to 0.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the script.}
#' \item{name}{string, The name of the script.}
#' \item{type}{string, The type of the script (e.g SQL, Container, Python, R, JavaScript)}
#' \item{createdAt}{string, The time this script was created.}
#' \item{updatedAt}{string, The time the script was last updated.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the script's last run.}
#' \item{finishedAt}{string, The time that the script's last run finished.}
#' \item{category}{string, The category of the script.}
#' \item{projects}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID for the project.
#' \item name string, The name of the project.
#' }}
#' \item{parentId}{integer, The ID of the parent job that will trigger this script}
#' \item{userContext}{string, "runner" or "author", who to execute the script as when run as a template.}
#' \item{params}{array, An array containing the following fields: 
#' \itemize{
#' \item name string, The variable's name as used within your code.
#' \item label string, The label to present to users when asking them for the value.
#' \item description string, A short sentence or fragment describing this parameter to the end user.
#' \item type string, The type of parameter. Valid options: string, multi_line_string, integer, float, bool, file, database, credential_aws, credential_redshift, or credential_custom
#' \item required boolean, Whether this param is required.
#' \item value string, The value you would like to set this param to. Setting this value makes this parameter a fixed param.
#' \item default string, If an argument for this parameter is not defined, it will use this default value. Use true, True, t, y, yes, or 1 for true bool's or false, False, f, n, no, or 0 for false bool's. Cannot be used for parameters that are required or a credential type.
#' \item allowedValues array, The possible values this parameter can take, effectively making this an enumerable parameter. Allowed values is an array of hashes of the following format: `{label: 'Import', 'value': 'import'}`
#' }}
#' \item{arguments}{object, Dictionary of name/value pairs to use to run this script. Only settable if this script has defined params.}
#' \item{isTemplate}{boolean, Whether others scripts use this one as a template.}
#' \item{publishedAsTemplateId}{integer, The ID of the template that this script is backing.}
#' \item{fromTemplateId}{integer, The ID of the template this script uses, if any.}
#' \item{templateDependentsCount}{integer, How many other scripts use this one as a template.}
#' \item{templateScriptName}{string, The name of the template script.}
#' \item{links}{object, A list containing the following elements: 
#' \itemize{
#' \item details string, The details link to get more information about the script.
#' \item runs string, The runs link to get the run information list for this script.
#' }}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{notifications}{object, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{runningAs}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{nextRunAt}{string, The time of the next scheduled run.}
#' \item{timeZone}{string, The time zone of this script.}
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{hidden}{boolean, The hidden status of the object.}
#' \item{targetProjectId}{integer, Target project to which script outputs will be added.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{requiredResources}{object, A list containing the following elements: 
#' \itemize{
#' \item cpu integer, The number of CPU shares to allocate for the container. Each core has 1024 shares. Must be at least 2 shares.
#' \item memory integer, The amount of RAM to allocate for the container (in MiB). Must be at least 4 MiB.
#' \item diskSpace number, The amount of disk space, in GB, to allocate for the container. This space will be used to hold the git repo configured for the container and anything your container writes to /tmp or /data. Fractional values (e.g. 0.25) are supported.
#' }}
#' \item{source}{string, The body/text of the script.}
#' \item{cancelTimeout}{integer, The amount of time (in seconds) to wait before forcibly terminating the script. When the script is cancelled, it is first sent a TERM signal. If the script is still running after the timeout, it is sent a KILL signal. Defaults to 0.}
#' @export
scripts_post_r <- function(name, source, parent_id = NULL, user_context = NULL, params = NULL, arguments = NULL, schedule = NULL, notifications = NULL, next_run_at = NULL, time_zone = NULL, hidden = NULL, target_project_id = NULL, required_resources = NULL, cancel_timeout = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/scripts/r"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(name = name, source = source, parentId = parent_id, userContext = user_context, params = params, arguments = arguments, schedule = schedule, notifications = notifications, nextRunAt = next_run_at, timeZone = time_zone, hidden = hidden, targetProjectId = target_project_id, requiredResources = required_resources, cancelTimeout = cancel_timeout)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get an R Script
#' @param id integer required. 
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the script.}
#' \item{name}{string, The name of the script.}
#' \item{type}{string, The type of the script (e.g SQL, Container, Python, R, JavaScript)}
#' \item{createdAt}{string, The time this script was created.}
#' \item{updatedAt}{string, The time the script was last updated.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the script's last run.}
#' \item{finishedAt}{string, The time that the script's last run finished.}
#' \item{category}{string, The category of the script.}
#' \item{projects}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID for the project.
#' \item name string, The name of the project.
#' }}
#' \item{parentId}{integer, The ID of the parent job that will trigger this script}
#' \item{userContext}{string, "runner" or "author", who to execute the script as when run as a template.}
#' \item{params}{array, An array containing the following fields: 
#' \itemize{
#' \item name string, The variable's name as used within your code.
#' \item label string, The label to present to users when asking them for the value.
#' \item description string, A short sentence or fragment describing this parameter to the end user.
#' \item type string, The type of parameter. Valid options: string, multi_line_string, integer, float, bool, file, database, credential_aws, credential_redshift, or credential_custom
#' \item required boolean, Whether this param is required.
#' \item value string, The value you would like to set this param to. Setting this value makes this parameter a fixed param.
#' \item default string, If an argument for this parameter is not defined, it will use this default value. Use true, True, t, y, yes, or 1 for true bool's or false, False, f, n, no, or 0 for false bool's. Cannot be used for parameters that are required or a credential type.
#' \item allowedValues array, The possible values this parameter can take, effectively making this an enumerable parameter. Allowed values is an array of hashes of the following format: `{label: 'Import', 'value': 'import'}`
#' }}
#' \item{arguments}{object, Dictionary of name/value pairs to use to run this script. Only settable if this script has defined params.}
#' \item{isTemplate}{boolean, Whether others scripts use this one as a template.}
#' \item{publishedAsTemplateId}{integer, The ID of the template that this script is backing.}
#' \item{fromTemplateId}{integer, The ID of the template this script uses, if any.}
#' \item{templateDependentsCount}{integer, How many other scripts use this one as a template.}
#' \item{templateScriptName}{string, The name of the template script.}
#' \item{links}{object, A list containing the following elements: 
#' \itemize{
#' \item details string, The details link to get more information about the script.
#' \item runs string, The runs link to get the run information list for this script.
#' }}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{notifications}{object, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{runningAs}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{nextRunAt}{string, The time of the next scheduled run.}
#' \item{timeZone}{string, The time zone of this script.}
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{hidden}{boolean, The hidden status of the object.}
#' \item{targetProjectId}{integer, Target project to which script outputs will be added.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{requiredResources}{object, A list containing the following elements: 
#' \itemize{
#' \item cpu integer, The number of CPU shares to allocate for the container. Each core has 1024 shares. Must be at least 2 shares.
#' \item memory integer, The amount of RAM to allocate for the container (in MiB). Must be at least 4 MiB.
#' \item diskSpace number, The amount of disk space, in GB, to allocate for the container. This space will be used to hold the git repo configured for the container and anything your container writes to /tmp or /data. Fractional values (e.g. 0.25) are supported.
#' }}
#' \item{source}{string, The body/text of the script.}
#' \item{cancelTimeout}{integer, The amount of time (in seconds) to wait before forcibly terminating the script. When the script is cancelled, it is first sent a TERM signal. If the script is still running after the timeout, it is sent a KILL signal. Defaults to 0.}
#' @export
scripts_get_r <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/scripts/r/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Replace all attributes of this R Script
#' @param id integer required. The ID for the script.
#' @param name string required. The name of the script.
#' @param source string required. The body/text of the script.
#' @param parent_id integer optional. The ID of the parent job that will trigger this script
#' @param user_context string optional. "runner" or "author", who to execute the script as when run as a template.
#' @param params array optional. An array containing the following fields: 
#' \itemize{
#' \item name string, The variable's name as used within your code.
#' \item label string, The label to present to users when asking them for the value.
#' \item description string, A short sentence or fragment describing this parameter to the end user.
#' \item type string, The type of parameter. Valid options: string, multi_line_string, integer, float, bool, file, database, credential_aws, credential_redshift, or credential_custom
#' \item required boolean, Whether this param is required.
#' \item value string, The value you would like to set this param to. Setting this value makes this parameter a fixed param.
#' \item default string, If an argument for this parameter is not defined, it will use this default value. Use true, True, t, y, yes, or 1 for true bool's or false, False, f, n, no, or 0 for false bool's. Cannot be used for parameters that are required or a credential type.
#' \item allowedValues array, The possible values this parameter can take, effectively making this an enumerable parameter. Allowed values is an array of hashes of the following format: `{label: 'Import', 'value': 'import'}`
#' }
#' @param arguments object optional. Dictionary of name/value pairs to use to run this script. Only settable if this script has defined params.
#' @param schedule object optional. A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }
#' @param notifications object optional. A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }
#' @param next_run_at string optional. The time of the next scheduled run.
#' @param time_zone string optional. The time zone of this script.
#' @param target_project_id integer optional. Target project to which script outputs will be added.
#' @param required_resources object optional. A list containing the following elements: 
#' \itemize{
#' \item cpu integer, The number of CPU shares to allocate for the container. Each core has 1024 shares. Must be at least 2 shares.
#' \item memory integer, The amount of RAM to allocate for the container (in MiB). Must be at least 4 MiB.
#' \item diskSpace number, The amount of disk space, in GB, to allocate for the container. This space will be used to hold the git repo configured for the container and anything your container writes to /tmp or /data. Fractional values (e.g. 0.25) are supported.
#' }
#' @param cancel_timeout integer optional. The amount of time (in seconds) to wait before forcibly terminating the script. When the script is cancelled, it is first sent a TERM signal. If the script is still running after the timeout, it is sent a KILL signal. Defaults to 0.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the script.}
#' \item{name}{string, The name of the script.}
#' \item{type}{string, The type of the script (e.g SQL, Container, Python, R, JavaScript)}
#' \item{createdAt}{string, The time this script was created.}
#' \item{updatedAt}{string, The time the script was last updated.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the script's last run.}
#' \item{finishedAt}{string, The time that the script's last run finished.}
#' \item{category}{string, The category of the script.}
#' \item{projects}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID for the project.
#' \item name string, The name of the project.
#' }}
#' \item{parentId}{integer, The ID of the parent job that will trigger this script}
#' \item{userContext}{string, "runner" or "author", who to execute the script as when run as a template.}
#' \item{params}{array, An array containing the following fields: 
#' \itemize{
#' \item name string, The variable's name as used within your code.
#' \item label string, The label to present to users when asking them for the value.
#' \item description string, A short sentence or fragment describing this parameter to the end user.
#' \item type string, The type of parameter. Valid options: string, multi_line_string, integer, float, bool, file, database, credential_aws, credential_redshift, or credential_custom
#' \item required boolean, Whether this param is required.
#' \item value string, The value you would like to set this param to. Setting this value makes this parameter a fixed param.
#' \item default string, If an argument for this parameter is not defined, it will use this default value. Use true, True, t, y, yes, or 1 for true bool's or false, False, f, n, no, or 0 for false bool's. Cannot be used for parameters that are required or a credential type.
#' \item allowedValues array, The possible values this parameter can take, effectively making this an enumerable parameter. Allowed values is an array of hashes of the following format: `{label: 'Import', 'value': 'import'}`
#' }}
#' \item{arguments}{object, Dictionary of name/value pairs to use to run this script. Only settable if this script has defined params.}
#' \item{isTemplate}{boolean, Whether others scripts use this one as a template.}
#' \item{publishedAsTemplateId}{integer, The ID of the template that this script is backing.}
#' \item{fromTemplateId}{integer, The ID of the template this script uses, if any.}
#' \item{templateDependentsCount}{integer, How many other scripts use this one as a template.}
#' \item{templateScriptName}{string, The name of the template script.}
#' \item{links}{object, A list containing the following elements: 
#' \itemize{
#' \item details string, The details link to get more information about the script.
#' \item runs string, The runs link to get the run information list for this script.
#' }}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{notifications}{object, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{runningAs}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{nextRunAt}{string, The time of the next scheduled run.}
#' \item{timeZone}{string, The time zone of this script.}
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{hidden}{boolean, The hidden status of the object.}
#' \item{targetProjectId}{integer, Target project to which script outputs will be added.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{requiredResources}{object, A list containing the following elements: 
#' \itemize{
#' \item cpu integer, The number of CPU shares to allocate for the container. Each core has 1024 shares. Must be at least 2 shares.
#' \item memory integer, The amount of RAM to allocate for the container (in MiB). Must be at least 4 MiB.
#' \item diskSpace number, The amount of disk space, in GB, to allocate for the container. This space will be used to hold the git repo configured for the container and anything your container writes to /tmp or /data. Fractional values (e.g. 0.25) are supported.
#' }}
#' \item{source}{string, The body/text of the script.}
#' \item{cancelTimeout}{integer, The amount of time (in seconds) to wait before forcibly terminating the script. When the script is cancelled, it is first sent a TERM signal. If the script is still running after the timeout, it is sent a KILL signal. Defaults to 0.}
#' @export
scripts_put_r <- function(id, name, source, parent_id = NULL, user_context = NULL, params = NULL, arguments = NULL, schedule = NULL, notifications = NULL, next_run_at = NULL, time_zone = NULL, target_project_id = NULL, required_resources = NULL, cancel_timeout = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/scripts/r/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(name = name, source = source, parentId = parent_id, userContext = user_context, params = params, arguments = arguments, schedule = schedule, notifications = notifications, nextRunAt = next_run_at, timeZone = time_zone, targetProjectId = target_project_id, requiredResources = required_resources, cancelTimeout = cancel_timeout)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update some attributes of this R Script
#' @param id integer required. The ID for the script.
#' @param name string optional. The name of the script.
#' @param parent_id integer optional. The ID of the parent job that will trigger this script
#' @param user_context string optional. "runner" or "author", who to execute the script as when run as a template.
#' @param params array optional. An array containing the following fields: 
#' \itemize{
#' \item name string, The variable's name as used within your code.
#' \item label string, The label to present to users when asking them for the value.
#' \item description string, A short sentence or fragment describing this parameter to the end user.
#' \item type string, The type of parameter. Valid options: string, multi_line_string, integer, float, bool, file, database, credential_aws, credential_redshift, or credential_custom
#' \item required boolean, Whether this param is required.
#' \item value string, The value you would like to set this param to. Setting this value makes this parameter a fixed param.
#' \item default string, If an argument for this parameter is not defined, it will use this default value. Use true, True, t, y, yes, or 1 for true bool's or false, False, f, n, no, or 0 for false bool's. Cannot be used for parameters that are required or a credential type.
#' \item allowedValues array, The possible values this parameter can take, effectively making this an enumerable parameter. Allowed values is an array of hashes of the following format: `{label: 'Import', 'value': 'import'}`
#' }
#' @param arguments object optional. Dictionary of name/value pairs to use to run this script. Only settable if this script has defined params.
#' @param schedule object optional. A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }
#' @param notifications object optional. A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }
#' @param next_run_at string optional. The time of the next scheduled run.
#' @param time_zone string optional. The time zone of this script.
#' @param target_project_id integer optional. Target project to which script outputs will be added.
#' @param required_resources object optional. A list containing the following elements: 
#' \itemize{
#' \item cpu integer, The number of CPU shares to allocate for the container. Each core has 1024 shares. Must be at least 2 shares.
#' \item memory integer, The amount of RAM to allocate for the container (in MiB). Must be at least 4 MiB.
#' \item diskSpace number, The amount of disk space, in GB, to allocate for the container. This space will be used to hold the git repo configured for the container and anything your container writes to /tmp or /data. Fractional values (e.g. 0.25) are supported.
#' }
#' @param source string optional. The body/text of the script.
#' @param cancel_timeout integer optional. The amount of time (in seconds) to wait before forcibly terminating the script. When the script is cancelled, it is first sent a TERM signal. If the script is still running after the timeout, it is sent a KILL signal. Defaults to 0.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the script.}
#' \item{name}{string, The name of the script.}
#' \item{type}{string, The type of the script (e.g SQL, Container, Python, R, JavaScript)}
#' \item{createdAt}{string, The time this script was created.}
#' \item{updatedAt}{string, The time the script was last updated.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the script's last run.}
#' \item{finishedAt}{string, The time that the script's last run finished.}
#' \item{category}{string, The category of the script.}
#' \item{projects}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID for the project.
#' \item name string, The name of the project.
#' }}
#' \item{parentId}{integer, The ID of the parent job that will trigger this script}
#' \item{userContext}{string, "runner" or "author", who to execute the script as when run as a template.}
#' \item{params}{array, An array containing the following fields: 
#' \itemize{
#' \item name string, The variable's name as used within your code.
#' \item label string, The label to present to users when asking them for the value.
#' \item description string, A short sentence or fragment describing this parameter to the end user.
#' \item type string, The type of parameter. Valid options: string, multi_line_string, integer, float, bool, file, database, credential_aws, credential_redshift, or credential_custom
#' \item required boolean, Whether this param is required.
#' \item value string, The value you would like to set this param to. Setting this value makes this parameter a fixed param.
#' \item default string, If an argument for this parameter is not defined, it will use this default value. Use true, True, t, y, yes, or 1 for true bool's or false, False, f, n, no, or 0 for false bool's. Cannot be used for parameters that are required or a credential type.
#' \item allowedValues array, The possible values this parameter can take, effectively making this an enumerable parameter. Allowed values is an array of hashes of the following format: `{label: 'Import', 'value': 'import'}`
#' }}
#' \item{arguments}{object, Dictionary of name/value pairs to use to run this script. Only settable if this script has defined params.}
#' \item{isTemplate}{boolean, Whether others scripts use this one as a template.}
#' \item{publishedAsTemplateId}{integer, The ID of the template that this script is backing.}
#' \item{fromTemplateId}{integer, The ID of the template this script uses, if any.}
#' \item{templateDependentsCount}{integer, How many other scripts use this one as a template.}
#' \item{templateScriptName}{string, The name of the template script.}
#' \item{links}{object, A list containing the following elements: 
#' \itemize{
#' \item details string, The details link to get more information about the script.
#' \item runs string, The runs link to get the run information list for this script.
#' }}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{notifications}{object, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{runningAs}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{nextRunAt}{string, The time of the next scheduled run.}
#' \item{timeZone}{string, The time zone of this script.}
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{hidden}{boolean, The hidden status of the object.}
#' \item{targetProjectId}{integer, Target project to which script outputs will be added.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{requiredResources}{object, A list containing the following elements: 
#' \itemize{
#' \item cpu integer, The number of CPU shares to allocate for the container. Each core has 1024 shares. Must be at least 2 shares.
#' \item memory integer, The amount of RAM to allocate for the container (in MiB). Must be at least 4 MiB.
#' \item diskSpace number, The amount of disk space, in GB, to allocate for the container. This space will be used to hold the git repo configured for the container and anything your container writes to /tmp or /data. Fractional values (e.g. 0.25) are supported.
#' }}
#' \item{source}{string, The body/text of the script.}
#' \item{cancelTimeout}{integer, The amount of time (in seconds) to wait before forcibly terminating the script. When the script is cancelled, it is first sent a TERM signal. If the script is still running after the timeout, it is sent a KILL signal. Defaults to 0.}
#' @export
scripts_patch_r <- function(id, name = NULL, parent_id = NULL, user_context = NULL, params = NULL, arguments = NULL, schedule = NULL, notifications = NULL, next_run_at = NULL, time_zone = NULL, target_project_id = NULL, required_resources = NULL, source = NULL, cancel_timeout = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/scripts/r/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(name = name, parentId = parent_id, userContext = user_context, params = params, arguments = arguments, schedule = schedule, notifications = notifications, nextRunAt = next_run_at, timeZone = time_zone, targetProjectId = target_project_id, requiredResources = required_resources, source = source, cancelTimeout = cancel_timeout)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("patch", path, path_params, query_params, body_params)

  return(resp)

 }


#' Archive an R Script (deprecated, use archiving endpoints instead)
#' @param id integer required. 
#' 
#' @return  An empty HTTP response
#' @export
scripts_delete_r <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/scripts/r/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' Create a JavaScript Script
#' @param name string required. The name of the script.
#' @param source string required. The body/text of the script.
#' @param remote_host_id integer required. The remote host ID that this script will connect to.
#' @param credential_id integer required. The credential that this script will use.
#' @param parent_id integer optional. The ID of the parent job that will trigger this script
#' @param user_context string optional. "runner" or "author", who to execute the script as when run as a template.
#' @param params array optional. An array containing the following fields: 
#' \itemize{
#' \item name string, The variable's name as used within your code.
#' \item label string, The label to present to users when asking them for the value.
#' \item description string, A short sentence or fragment describing this parameter to the end user.
#' \item type string, The type of parameter. Valid options: string, multi_line_string, integer, float, bool, file, database, credential_aws, credential_redshift, or credential_custom
#' \item required boolean, Whether this param is required.
#' \item value string, The value you would like to set this param to. Setting this value makes this parameter a fixed param.
#' \item default string, If an argument for this parameter is not defined, it will use this default value. Use true, True, t, y, yes, or 1 for true bool's or false, False, f, n, no, or 0 for false bool's. Cannot be used for parameters that are required or a credential type.
#' \item allowedValues array, The possible values this parameter can take, effectively making this an enumerable parameter. Allowed values is an array of hashes of the following format: `{label: 'Import', 'value': 'import'}`
#' }
#' @param arguments object optional. Dictionary of name/value pairs to use to run this script. Only settable if this script has defined params.
#' @param schedule object optional. A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }
#' @param notifications object optional. A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }
#' @param next_run_at string optional. The time of the next scheduled run.
#' @param time_zone string optional. The time zone of this script.
#' @param hidden boolean optional. The hidden status of the object.
#' @param target_project_id integer optional. Target project to which script outputs will be added.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the script.}
#' \item{name}{string, The name of the script.}
#' \item{type}{string, The type of the script (e.g SQL, Container, Python, R, JavaScript)}
#' \item{createdAt}{string, The time this script was created.}
#' \item{updatedAt}{string, The time the script was last updated.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the script's last run.}
#' \item{finishedAt}{string, The time that the script's last run finished.}
#' \item{category}{string, The category of the script.}
#' \item{projects}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID for the project.
#' \item name string, The name of the project.
#' }}
#' \item{parentId}{integer, The ID of the parent job that will trigger this script}
#' \item{userContext}{string, "runner" or "author", who to execute the script as when run as a template.}
#' \item{params}{array, An array containing the following fields: 
#' \itemize{
#' \item name string, The variable's name as used within your code.
#' \item label string, The label to present to users when asking them for the value.
#' \item description string, A short sentence or fragment describing this parameter to the end user.
#' \item type string, The type of parameter. Valid options: string, multi_line_string, integer, float, bool, file, database, credential_aws, credential_redshift, or credential_custom
#' \item required boolean, Whether this param is required.
#' \item value string, The value you would like to set this param to. Setting this value makes this parameter a fixed param.
#' \item default string, If an argument for this parameter is not defined, it will use this default value. Use true, True, t, y, yes, or 1 for true bool's or false, False, f, n, no, or 0 for false bool's. Cannot be used for parameters that are required or a credential type.
#' \item allowedValues array, The possible values this parameter can take, effectively making this an enumerable parameter. Allowed values is an array of hashes of the following format: `{label: 'Import', 'value': 'import'}`
#' }}
#' \item{arguments}{object, Dictionary of name/value pairs to use to run this script. Only settable if this script has defined params.}
#' \item{isTemplate}{boolean, Whether others scripts use this one as a template.}
#' \item{publishedAsTemplateId}{integer, The ID of the template that this script is backing.}
#' \item{fromTemplateId}{integer, The ID of the template this script uses, if any.}
#' \item{templateDependentsCount}{integer, How many other scripts use this one as a template.}
#' \item{templateScriptName}{string, The name of the template script.}
#' \item{links}{object, A list containing the following elements: 
#' \itemize{
#' \item details string, The details link to get more information about the script.
#' \item runs string, The runs link to get the run information list for this script.
#' }}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{notifications}{object, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{runningAs}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{nextRunAt}{string, The time of the next scheduled run.}
#' \item{timeZone}{string, The time zone of this script.}
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{hidden}{boolean, The hidden status of the object.}
#' \item{targetProjectId}{integer, Target project to which script outputs will be added.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{source}{string, The body/text of the script.}
#' \item{remoteHostId}{integer, The remote host ID that this script will connect to.}
#' \item{credentialId}{integer, The credential that this script will use.}
#' @export
scripts_post_javascript <- function(name, source, remote_host_id, credential_id, parent_id = NULL, user_context = NULL, params = NULL, arguments = NULL, schedule = NULL, notifications = NULL, next_run_at = NULL, time_zone = NULL, hidden = NULL, target_project_id = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/scripts/javascript"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(name = name, source = source, remoteHostId = remote_host_id, credentialId = credential_id, parentId = parent_id, userContext = user_context, params = params, arguments = arguments, schedule = schedule, notifications = notifications, nextRunAt = next_run_at, timeZone = time_zone, hidden = hidden, targetProjectId = target_project_id)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get a JavaScript Script
#' @param id integer required. 
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the script.}
#' \item{name}{string, The name of the script.}
#' \item{type}{string, The type of the script (e.g SQL, Container, Python, R, JavaScript)}
#' \item{createdAt}{string, The time this script was created.}
#' \item{updatedAt}{string, The time the script was last updated.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the script's last run.}
#' \item{finishedAt}{string, The time that the script's last run finished.}
#' \item{category}{string, The category of the script.}
#' \item{projects}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID for the project.
#' \item name string, The name of the project.
#' }}
#' \item{parentId}{integer, The ID of the parent job that will trigger this script}
#' \item{userContext}{string, "runner" or "author", who to execute the script as when run as a template.}
#' \item{params}{array, An array containing the following fields: 
#' \itemize{
#' \item name string, The variable's name as used within your code.
#' \item label string, The label to present to users when asking them for the value.
#' \item description string, A short sentence or fragment describing this parameter to the end user.
#' \item type string, The type of parameter. Valid options: string, multi_line_string, integer, float, bool, file, database, credential_aws, credential_redshift, or credential_custom
#' \item required boolean, Whether this param is required.
#' \item value string, The value you would like to set this param to. Setting this value makes this parameter a fixed param.
#' \item default string, If an argument for this parameter is not defined, it will use this default value. Use true, True, t, y, yes, or 1 for true bool's or false, False, f, n, no, or 0 for false bool's. Cannot be used for parameters that are required or a credential type.
#' \item allowedValues array, The possible values this parameter can take, effectively making this an enumerable parameter. Allowed values is an array of hashes of the following format: `{label: 'Import', 'value': 'import'}`
#' }}
#' \item{arguments}{object, Dictionary of name/value pairs to use to run this script. Only settable if this script has defined params.}
#' \item{isTemplate}{boolean, Whether others scripts use this one as a template.}
#' \item{publishedAsTemplateId}{integer, The ID of the template that this script is backing.}
#' \item{fromTemplateId}{integer, The ID of the template this script uses, if any.}
#' \item{templateDependentsCount}{integer, How many other scripts use this one as a template.}
#' \item{templateScriptName}{string, The name of the template script.}
#' \item{links}{object, A list containing the following elements: 
#' \itemize{
#' \item details string, The details link to get more information about the script.
#' \item runs string, The runs link to get the run information list for this script.
#' }}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{notifications}{object, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{runningAs}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{nextRunAt}{string, The time of the next scheduled run.}
#' \item{timeZone}{string, The time zone of this script.}
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{hidden}{boolean, The hidden status of the object.}
#' \item{targetProjectId}{integer, Target project to which script outputs will be added.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{source}{string, The body/text of the script.}
#' \item{remoteHostId}{integer, The remote host ID that this script will connect to.}
#' \item{credentialId}{integer, The credential that this script will use.}
#' @export
scripts_get_javascript <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/scripts/javascript/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Replace all attributes of this JavaScript Script
#' @param id integer required. The ID for the script.
#' @param name string required. The name of the script.
#' @param source string required. The body/text of the script.
#' @param remote_host_id integer required. The remote host ID that this script will connect to.
#' @param credential_id integer required. The credential that this script will use.
#' @param parent_id integer optional. The ID of the parent job that will trigger this script
#' @param user_context string optional. "runner" or "author", who to execute the script as when run as a template.
#' @param params array optional. An array containing the following fields: 
#' \itemize{
#' \item name string, The variable's name as used within your code.
#' \item label string, The label to present to users when asking them for the value.
#' \item description string, A short sentence or fragment describing this parameter to the end user.
#' \item type string, The type of parameter. Valid options: string, multi_line_string, integer, float, bool, file, database, credential_aws, credential_redshift, or credential_custom
#' \item required boolean, Whether this param is required.
#' \item value string, The value you would like to set this param to. Setting this value makes this parameter a fixed param.
#' \item default string, If an argument for this parameter is not defined, it will use this default value. Use true, True, t, y, yes, or 1 for true bool's or false, False, f, n, no, or 0 for false bool's. Cannot be used for parameters that are required or a credential type.
#' \item allowedValues array, The possible values this parameter can take, effectively making this an enumerable parameter. Allowed values is an array of hashes of the following format: `{label: 'Import', 'value': 'import'}`
#' }
#' @param arguments object optional. Dictionary of name/value pairs to use to run this script. Only settable if this script has defined params.
#' @param schedule object optional. A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }
#' @param notifications object optional. A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }
#' @param next_run_at string optional. The time of the next scheduled run.
#' @param time_zone string optional. The time zone of this script.
#' @param target_project_id integer optional. Target project to which script outputs will be added.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the script.}
#' \item{name}{string, The name of the script.}
#' \item{type}{string, The type of the script (e.g SQL, Container, Python, R, JavaScript)}
#' \item{createdAt}{string, The time this script was created.}
#' \item{updatedAt}{string, The time the script was last updated.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the script's last run.}
#' \item{finishedAt}{string, The time that the script's last run finished.}
#' \item{category}{string, The category of the script.}
#' \item{projects}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID for the project.
#' \item name string, The name of the project.
#' }}
#' \item{parentId}{integer, The ID of the parent job that will trigger this script}
#' \item{userContext}{string, "runner" or "author", who to execute the script as when run as a template.}
#' \item{params}{array, An array containing the following fields: 
#' \itemize{
#' \item name string, The variable's name as used within your code.
#' \item label string, The label to present to users when asking them for the value.
#' \item description string, A short sentence or fragment describing this parameter to the end user.
#' \item type string, The type of parameter. Valid options: string, multi_line_string, integer, float, bool, file, database, credential_aws, credential_redshift, or credential_custom
#' \item required boolean, Whether this param is required.
#' \item value string, The value you would like to set this param to. Setting this value makes this parameter a fixed param.
#' \item default string, If an argument for this parameter is not defined, it will use this default value. Use true, True, t, y, yes, or 1 for true bool's or false, False, f, n, no, or 0 for false bool's. Cannot be used for parameters that are required or a credential type.
#' \item allowedValues array, The possible values this parameter can take, effectively making this an enumerable parameter. Allowed values is an array of hashes of the following format: `{label: 'Import', 'value': 'import'}`
#' }}
#' \item{arguments}{object, Dictionary of name/value pairs to use to run this script. Only settable if this script has defined params.}
#' \item{isTemplate}{boolean, Whether others scripts use this one as a template.}
#' \item{publishedAsTemplateId}{integer, The ID of the template that this script is backing.}
#' \item{fromTemplateId}{integer, The ID of the template this script uses, if any.}
#' \item{templateDependentsCount}{integer, How many other scripts use this one as a template.}
#' \item{templateScriptName}{string, The name of the template script.}
#' \item{links}{object, A list containing the following elements: 
#' \itemize{
#' \item details string, The details link to get more information about the script.
#' \item runs string, The runs link to get the run information list for this script.
#' }}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{notifications}{object, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{runningAs}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{nextRunAt}{string, The time of the next scheduled run.}
#' \item{timeZone}{string, The time zone of this script.}
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{hidden}{boolean, The hidden status of the object.}
#' \item{targetProjectId}{integer, Target project to which script outputs will be added.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{source}{string, The body/text of the script.}
#' \item{remoteHostId}{integer, The remote host ID that this script will connect to.}
#' \item{credentialId}{integer, The credential that this script will use.}
#' @export
scripts_put_javascript <- function(id, name, source, remote_host_id, credential_id, parent_id = NULL, user_context = NULL, params = NULL, arguments = NULL, schedule = NULL, notifications = NULL, next_run_at = NULL, time_zone = NULL, target_project_id = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/scripts/javascript/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(name = name, source = source, remoteHostId = remote_host_id, credentialId = credential_id, parentId = parent_id, userContext = user_context, params = params, arguments = arguments, schedule = schedule, notifications = notifications, nextRunAt = next_run_at, timeZone = time_zone, targetProjectId = target_project_id)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update some attributes of this JavaScript Script
#' @param id integer required. The ID for the script.
#' @param name string optional. The name of the script.
#' @param parent_id integer optional. The ID of the parent job that will trigger this script
#' @param user_context string optional. "runner" or "author", who to execute the script as when run as a template.
#' @param params array optional. An array containing the following fields: 
#' \itemize{
#' \item name string, The variable's name as used within your code.
#' \item label string, The label to present to users when asking them for the value.
#' \item description string, A short sentence or fragment describing this parameter to the end user.
#' \item type string, The type of parameter. Valid options: string, multi_line_string, integer, float, bool, file, database, credential_aws, credential_redshift, or credential_custom
#' \item required boolean, Whether this param is required.
#' \item value string, The value you would like to set this param to. Setting this value makes this parameter a fixed param.
#' \item default string, If an argument for this parameter is not defined, it will use this default value. Use true, True, t, y, yes, or 1 for true bool's or false, False, f, n, no, or 0 for false bool's. Cannot be used for parameters that are required or a credential type.
#' \item allowedValues array, The possible values this parameter can take, effectively making this an enumerable parameter. Allowed values is an array of hashes of the following format: `{label: 'Import', 'value': 'import'}`
#' }
#' @param arguments object optional. Dictionary of name/value pairs to use to run this script. Only settable if this script has defined params.
#' @param schedule object optional. A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }
#' @param notifications object optional. A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }
#' @param next_run_at string optional. The time of the next scheduled run.
#' @param time_zone string optional. The time zone of this script.
#' @param target_project_id integer optional. Target project to which script outputs will be added.
#' @param source string optional. The body/text of the script.
#' @param remote_host_id integer optional. The remote host ID that this script will connect to.
#' @param credential_id integer optional. The credential that this script will use.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the script.}
#' \item{name}{string, The name of the script.}
#' \item{type}{string, The type of the script (e.g SQL, Container, Python, R, JavaScript)}
#' \item{createdAt}{string, The time this script was created.}
#' \item{updatedAt}{string, The time the script was last updated.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the script's last run.}
#' \item{finishedAt}{string, The time that the script's last run finished.}
#' \item{category}{string, The category of the script.}
#' \item{projects}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID for the project.
#' \item name string, The name of the project.
#' }}
#' \item{parentId}{integer, The ID of the parent job that will trigger this script}
#' \item{userContext}{string, "runner" or "author", who to execute the script as when run as a template.}
#' \item{params}{array, An array containing the following fields: 
#' \itemize{
#' \item name string, The variable's name as used within your code.
#' \item label string, The label to present to users when asking them for the value.
#' \item description string, A short sentence or fragment describing this parameter to the end user.
#' \item type string, The type of parameter. Valid options: string, multi_line_string, integer, float, bool, file, database, credential_aws, credential_redshift, or credential_custom
#' \item required boolean, Whether this param is required.
#' \item value string, The value you would like to set this param to. Setting this value makes this parameter a fixed param.
#' \item default string, If an argument for this parameter is not defined, it will use this default value. Use true, True, t, y, yes, or 1 for true bool's or false, False, f, n, no, or 0 for false bool's. Cannot be used for parameters that are required or a credential type.
#' \item allowedValues array, The possible values this parameter can take, effectively making this an enumerable parameter. Allowed values is an array of hashes of the following format: `{label: 'Import', 'value': 'import'}`
#' }}
#' \item{arguments}{object, Dictionary of name/value pairs to use to run this script. Only settable if this script has defined params.}
#' \item{isTemplate}{boolean, Whether others scripts use this one as a template.}
#' \item{publishedAsTemplateId}{integer, The ID of the template that this script is backing.}
#' \item{fromTemplateId}{integer, The ID of the template this script uses, if any.}
#' \item{templateDependentsCount}{integer, How many other scripts use this one as a template.}
#' \item{templateScriptName}{string, The name of the template script.}
#' \item{links}{object, A list containing the following elements: 
#' \itemize{
#' \item details string, The details link to get more information about the script.
#' \item runs string, The runs link to get the run information list for this script.
#' }}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{notifications}{object, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{runningAs}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{nextRunAt}{string, The time of the next scheduled run.}
#' \item{timeZone}{string, The time zone of this script.}
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{hidden}{boolean, The hidden status of the object.}
#' \item{targetProjectId}{integer, Target project to which script outputs will be added.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{source}{string, The body/text of the script.}
#' \item{remoteHostId}{integer, The remote host ID that this script will connect to.}
#' \item{credentialId}{integer, The credential that this script will use.}
#' @export
scripts_patch_javascript <- function(id, name = NULL, parent_id = NULL, user_context = NULL, params = NULL, arguments = NULL, schedule = NULL, notifications = NULL, next_run_at = NULL, time_zone = NULL, target_project_id = NULL, source = NULL, remote_host_id = NULL, credential_id = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/scripts/javascript/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(name = name, parentId = parent_id, userContext = user_context, params = params, arguments = arguments, schedule = schedule, notifications = notifications, nextRunAt = next_run_at, timeZone = time_zone, targetProjectId = target_project_id, source = source, remoteHostId = remote_host_id, credentialId = credential_id)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("patch", path, path_params, query_params, body_params)

  return(resp)

 }


#' Archive a JavaScript Script (deprecated, use archiving endpoints instead)
#' @param id integer required. 
#' 
#' @return  An empty HTTP response
#' @export
scripts_delete_javascript <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/scripts/javascript/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' List Custom Scripts
#' @param from_template_id string optional. If specified, return scripts based on the template with this ID. Specify multiple IDs as a comma-separated list.
#' @param author string optional. If specified, return objects from this author. Must use user IDs. A comma separated list of IDs is also accepted to return objects from multiple authors.
#' @param status string optional. If specified, returns objects with one of these statuses. It accepts a comma-separated list, possible values are 'running', 'failed', 'succeeded', 'idle', 'scheduled'.
#' @param hidden boolean optional. If specified to be true, returns hidden objects. Defaults to false, returning non-hidden objects.
#' @param archived string optional. The archival status of the requested object(s).
#' @param limit integer optional. Number of results to return. Defaults to its maximum of 50.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to updated_at. Must be one of: updated_at, name, created_at.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to asc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID for the script.}
#' \item{name}{string, The name of the script.}
#' \item{type}{string, The type of the script (e.g Custom)}
#' \item{createdAt}{string, The time this script was created.}
#' \item{updatedAt}{string, The time the script was last updated.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the script's last run.}
#' \item{finishedAt}{string, The time that the script's last run finished.}
#' \item{projects}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID for the project.
#' \item name string, The name of the project.
#' }}
#' \item{parentId}{integer, The ID of the parent job that will trigger this script}
#' \item{fromTemplateId}{integer, The ID of the template script.}
#' \item{timeZone}{string, The time zone of this script.}
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{archived}{string, The archival status of the requested object(s).}
#' @export
scripts_list_custom <- function(from_template_id = NULL, author = NULL, status = NULL, hidden = NULL, archived = NULL, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/scripts/custom"
  path_params  <- list()
  query_params <- list(from_template_id = from_template_id, author = author, status = status, hidden = hidden, archived = archived, limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Create a Custom Script
#' @param from_template_id integer required. The ID of the template script.
#' @param name string optional. The name of the script.
#' @param parent_id integer optional. The ID of the parent job that will trigger this script
#' @param arguments object optional. Dictionary of name/value pairs to use to run this script. Only settable if this script has defined params.
#' @param remote_host_id integer optional. The remote host ID that this script will connect to.
#' @param credential_id integer optional. The credential that this script will use.
#' @param schedule object optional. A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }
#' @param notifications object optional. A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }
#' @param time_zone string optional. The time zone of this script.
#' @param hidden boolean optional. The hidden status of the object.
#' @param target_project_id integer optional. Target project to which script outputs will be added.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the script.}
#' \item{name}{string, The name of the script.}
#' \item{type}{string, The type of the script (e.g Custom)}
#' \item{createdAt}{string, The time this script was created.}
#' \item{updatedAt}{string, The time the script was last updated.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the script's last run.}
#' \item{finishedAt}{string, The time that the script's last run finished.}
#' \item{category}{string, }
#' \item{projects}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID for the project.
#' \item name string, The name of the project.
#' }}
#' \item{parentId}{integer, The ID of the parent job that will trigger this script}
#' \item{params}{array, An array containing the following fields: 
#' \itemize{
#' \item name string, The variable's name as used within your code.
#' \item label string, The label to present to users when asking them for the value.
#' \item description string, A short sentence or fragment describing this parameter to the end user.
#' \item type string, The type of parameter. Valid options: string, multi_line_string, integer, float, bool, file, database, credential_aws, credential_redshift, or credential_custom
#' \item required boolean, Whether this param is required.
#' \item value string, The value you would like to set this param to. Setting this value makes this parameter a fixed param.
#' \item default string, If an argument for this parameter is not defined, it will use this default value. Use true, True, t, y, yes, or 1 for true bool's or false, False, f, n, no, or 0 for false bool's. Cannot be used for parameters that are required or a credential type.
#' \item allowedValues array, The possible values this parameter can take, effectively making this an enumerable parameter. Allowed values is an array of hashes of the following format: `{label: 'Import', 'value': 'import'}`
#' }}
#' \item{arguments}{object, Dictionary of name/value pairs to use to run this script. Only settable if this script has defined params.}
#' \item{isTemplate}{boolean, Whether others scripts use this one as a template.}
#' \item{publishedAsTemplateId}{integer, The ID of the template that this script is backing.}
#' \item{fromTemplateId}{integer, The ID of the template script.}
#' \item{uiReportUrl}{integer, The url of the custom HTML.}
#' \item{uiReportId}{integer, The id of the report with the custom HTML.}
#' \item{uiReportProvideAPIKey}{boolean, Whether the ui report requests an API Key from the report viewer.}
#' \item{templateScriptName}{string, The name of the template script.}
#' \item{templateNote}{string, The template's note.}
#' \item{remoteHostId}{integer, The remote host ID that this script will connect to.}
#' \item{credentialId}{integer, The credential that this script will use.}
#' \item{codePreview}{string, The code that this script will run with arguments inserted.}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{notifications}{object, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{runningAs}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{timeZone}{string, The time zone of this script.}
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{hidden}{boolean, The hidden status of the object.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{targetProjectId}{integer, Target project to which script outputs will be added.}
#' @export
scripts_post_custom <- function(from_template_id, name = NULL, parent_id = NULL, arguments = NULL, remote_host_id = NULL, credential_id = NULL, schedule = NULL, notifications = NULL, time_zone = NULL, hidden = NULL, target_project_id = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/scripts/custom"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(fromTemplateId = from_template_id, name = name, parentId = parent_id, arguments = arguments, remoteHostId = remote_host_id, credentialId = credential_id, schedule = schedule, notifications = notifications, timeZone = time_zone, hidden = hidden, targetProjectId = target_project_id)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get a Custom Script
#' @param id integer required. 
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the script.}
#' \item{name}{string, The name of the script.}
#' \item{type}{string, The type of the script (e.g Custom)}
#' \item{createdAt}{string, The time this script was created.}
#' \item{updatedAt}{string, The time the script was last updated.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the script's last run.}
#' \item{finishedAt}{string, The time that the script's last run finished.}
#' \item{category}{string, }
#' \item{projects}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID for the project.
#' \item name string, The name of the project.
#' }}
#' \item{parentId}{integer, The ID of the parent job that will trigger this script}
#' \item{params}{array, An array containing the following fields: 
#' \itemize{
#' \item name string, The variable's name as used within your code.
#' \item label string, The label to present to users when asking them for the value.
#' \item description string, A short sentence or fragment describing this parameter to the end user.
#' \item type string, The type of parameter. Valid options: string, multi_line_string, integer, float, bool, file, database, credential_aws, credential_redshift, or credential_custom
#' \item required boolean, Whether this param is required.
#' \item value string, The value you would like to set this param to. Setting this value makes this parameter a fixed param.
#' \item default string, If an argument for this parameter is not defined, it will use this default value. Use true, True, t, y, yes, or 1 for true bool's or false, False, f, n, no, or 0 for false bool's. Cannot be used for parameters that are required or a credential type.
#' \item allowedValues array, The possible values this parameter can take, effectively making this an enumerable parameter. Allowed values is an array of hashes of the following format: `{label: 'Import', 'value': 'import'}`
#' }}
#' \item{arguments}{object, Dictionary of name/value pairs to use to run this script. Only settable if this script has defined params.}
#' \item{isTemplate}{boolean, Whether others scripts use this one as a template.}
#' \item{publishedAsTemplateId}{integer, The ID of the template that this script is backing.}
#' \item{fromTemplateId}{integer, The ID of the template script.}
#' \item{uiReportUrl}{integer, The url of the custom HTML.}
#' \item{uiReportId}{integer, The id of the report with the custom HTML.}
#' \item{uiReportProvideAPIKey}{boolean, Whether the ui report requests an API Key from the report viewer.}
#' \item{templateScriptName}{string, The name of the template script.}
#' \item{templateNote}{string, The template's note.}
#' \item{remoteHostId}{integer, The remote host ID that this script will connect to.}
#' \item{credentialId}{integer, The credential that this script will use.}
#' \item{codePreview}{string, The code that this script will run with arguments inserted.}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{notifications}{object, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{runningAs}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{timeZone}{string, The time zone of this script.}
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{hidden}{boolean, The hidden status of the object.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{targetProjectId}{integer, Target project to which script outputs will be added.}
#' @export
scripts_get_custom <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/scripts/custom/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Replace all attributes of this Custom Script
#' @param id integer required. The ID for the script.
#' @param name string optional. The name of the script.
#' @param parent_id integer optional. The ID of the parent job that will trigger this script
#' @param arguments object optional. Dictionary of name/value pairs to use to run this script. Only settable if this script has defined params.
#' @param remote_host_id integer optional. The remote host ID that this script will connect to.
#' @param credential_id integer optional. The credential that this script will use.
#' @param schedule object optional. A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }
#' @param notifications object optional. A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }
#' @param time_zone string optional. The time zone of this script.
#' @param target_project_id integer optional. Target project to which script outputs will be added.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the script.}
#' \item{name}{string, The name of the script.}
#' \item{type}{string, The type of the script (e.g Custom)}
#' \item{createdAt}{string, The time this script was created.}
#' \item{updatedAt}{string, The time the script was last updated.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the script's last run.}
#' \item{finishedAt}{string, The time that the script's last run finished.}
#' \item{category}{string, }
#' \item{projects}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID for the project.
#' \item name string, The name of the project.
#' }}
#' \item{parentId}{integer, The ID of the parent job that will trigger this script}
#' \item{params}{array, An array containing the following fields: 
#' \itemize{
#' \item name string, The variable's name as used within your code.
#' \item label string, The label to present to users when asking them for the value.
#' \item description string, A short sentence or fragment describing this parameter to the end user.
#' \item type string, The type of parameter. Valid options: string, multi_line_string, integer, float, bool, file, database, credential_aws, credential_redshift, or credential_custom
#' \item required boolean, Whether this param is required.
#' \item value string, The value you would like to set this param to. Setting this value makes this parameter a fixed param.
#' \item default string, If an argument for this parameter is not defined, it will use this default value. Use true, True, t, y, yes, or 1 for true bool's or false, False, f, n, no, or 0 for false bool's. Cannot be used for parameters that are required or a credential type.
#' \item allowedValues array, The possible values this parameter can take, effectively making this an enumerable parameter. Allowed values is an array of hashes of the following format: `{label: 'Import', 'value': 'import'}`
#' }}
#' \item{arguments}{object, Dictionary of name/value pairs to use to run this script. Only settable if this script has defined params.}
#' \item{isTemplate}{boolean, Whether others scripts use this one as a template.}
#' \item{publishedAsTemplateId}{integer, The ID of the template that this script is backing.}
#' \item{fromTemplateId}{integer, The ID of the template script.}
#' \item{uiReportUrl}{integer, The url of the custom HTML.}
#' \item{uiReportId}{integer, The id of the report with the custom HTML.}
#' \item{uiReportProvideAPIKey}{boolean, Whether the ui report requests an API Key from the report viewer.}
#' \item{templateScriptName}{string, The name of the template script.}
#' \item{templateNote}{string, The template's note.}
#' \item{remoteHostId}{integer, The remote host ID that this script will connect to.}
#' \item{credentialId}{integer, The credential that this script will use.}
#' \item{codePreview}{string, The code that this script will run with arguments inserted.}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{notifications}{object, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{runningAs}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{timeZone}{string, The time zone of this script.}
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{hidden}{boolean, The hidden status of the object.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{targetProjectId}{integer, Target project to which script outputs will be added.}
#' @export
scripts_put_custom <- function(id, name = NULL, parent_id = NULL, arguments = NULL, remote_host_id = NULL, credential_id = NULL, schedule = NULL, notifications = NULL, time_zone = NULL, target_project_id = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/scripts/custom/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(name = name, parentId = parent_id, arguments = arguments, remoteHostId = remote_host_id, credentialId = credential_id, schedule = schedule, notifications = notifications, timeZone = time_zone, targetProjectId = target_project_id)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("put", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update some attributes of this Custom Script
#' @param id integer required. The ID for the script.
#' @param name string optional. The name of the script.
#' @param parent_id integer optional. The ID of the parent job that will trigger this script
#' @param arguments object optional. Dictionary of name/value pairs to use to run this script. Only settable if this script has defined params.
#' @param remote_host_id integer optional. The remote host ID that this script will connect to.
#' @param credential_id integer optional. The credential that this script will use.
#' @param schedule object optional. A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }
#' @param notifications object optional. A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }
#' @param time_zone string optional. The time zone of this script.
#' @param target_project_id integer optional. Target project to which script outputs will be added.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the script.}
#' \item{name}{string, The name of the script.}
#' \item{type}{string, The type of the script (e.g Custom)}
#' \item{createdAt}{string, The time this script was created.}
#' \item{updatedAt}{string, The time the script was last updated.}
#' \item{author}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the script's last run.}
#' \item{finishedAt}{string, The time that the script's last run finished.}
#' \item{category}{string, }
#' \item{projects}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID for the project.
#' \item name string, The name of the project.
#' }}
#' \item{parentId}{integer, The ID of the parent job that will trigger this script}
#' \item{params}{array, An array containing the following fields: 
#' \itemize{
#' \item name string, The variable's name as used within your code.
#' \item label string, The label to present to users when asking them for the value.
#' \item description string, A short sentence or fragment describing this parameter to the end user.
#' \item type string, The type of parameter. Valid options: string, multi_line_string, integer, float, bool, file, database, credential_aws, credential_redshift, or credential_custom
#' \item required boolean, Whether this param is required.
#' \item value string, The value you would like to set this param to. Setting this value makes this parameter a fixed param.
#' \item default string, If an argument for this parameter is not defined, it will use this default value. Use true, True, t, y, yes, or 1 for true bool's or false, False, f, n, no, or 0 for false bool's. Cannot be used for parameters that are required or a credential type.
#' \item allowedValues array, The possible values this parameter can take, effectively making this an enumerable parameter. Allowed values is an array of hashes of the following format: `{label: 'Import', 'value': 'import'}`
#' }}
#' \item{arguments}{object, Dictionary of name/value pairs to use to run this script. Only settable if this script has defined params.}
#' \item{isTemplate}{boolean, Whether others scripts use this one as a template.}
#' \item{publishedAsTemplateId}{integer, The ID of the template that this script is backing.}
#' \item{fromTemplateId}{integer, The ID of the template script.}
#' \item{uiReportUrl}{integer, The url of the custom HTML.}
#' \item{uiReportId}{integer, The id of the report with the custom HTML.}
#' \item{uiReportProvideAPIKey}{boolean, Whether the ui report requests an API Key from the report viewer.}
#' \item{templateScriptName}{string, The name of the template script.}
#' \item{templateNote}{string, The template's note.}
#' \item{remoteHostId}{integer, The remote host ID that this script will connect to.}
#' \item{credentialId}{integer, The credential that this script will use.}
#' \item{codePreview}{string, The code that this script will run with arguments inserted.}
#' \item{schedule}{object, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the object is scheduled
#' \item scheduledDays array, Day based on numeric value starting at 0 for Sunday
#' \item scheduledHours array, Hours of the day it is scheduled on
#' \item scheduledMinutes array, Minutes of the day it is scheduled on
#' \item scheduledRunsPerHour integer, Alternative to scheduled minutes, number of times to run per hour
#' }}
#' \item{notifications}{object, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{runningAs}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{timeZone}{string, The time zone of this script.}
#' \item{lastRun}{object, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{hidden}{boolean, The hidden status of the object.}
#' \item{archived}{string, The archival status of the requested object(s).}
#' \item{targetProjectId}{integer, Target project to which script outputs will be added.}
#' @export
scripts_patch_custom <- function(id, name = NULL, parent_id = NULL, arguments = NULL, remote_host_id = NULL, credential_id = NULL, schedule = NULL, notifications = NULL, time_zone = NULL, target_project_id = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/scripts/custom/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(name = name, parentId = parent_id, arguments = arguments, remoteHostId = remote_host_id, credentialId = credential_id, schedule = schedule, notifications = notifications, timeZone = time_zone, targetProjectId = target_project_id)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("patch", path, path_params, query_params, body_params)

  return(resp)

 }


#' Archive a Custom Script (deprecated, use archiving endpoints instead)
#' @param id integer required. 
#' 
#' @return  An empty HTTP response
#' @export
scripts_delete_custom <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/scripts/custom/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' Start a run
#' @param id integer required. The ID of the sql.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this run.}
#' \item{sqlId}{integer, The ID of this sql.}
#' \item{state}{string, The state of this run.}
#' \item{isCancelRequested}{boolean, True if run cancel requested, else false.}
#' \item{startedAt}{string, The time the last run started.}
#' \item{finishedAt}{string, The time that this run finished.}
#' \item{error}{string, The error message for this run, if present.}
#' \item{output}{array, An array containing the following fields: 
#' \itemize{
#' \item outputName string, The name of the output file.
#' \item fileId integer, The unique ID of the output file.
#' \item path string, The temporary link to download this output file, valid for 36 hours.
#' }}
#' @export
scripts_post_sql_runs <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/scripts/sql/{id}/runs"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' List runs for the given sql
#' @param id integer required. The ID of the sql.
#' @param limit integer optional. Number of results to return. Defaults to 20. Maximum allowed is 100.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to id. Must be one of: id.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of this run.}
#' \item{sqlId}{integer, The ID of this sql.}
#' \item{state}{string, The state of this run.}
#' \item{isCancelRequested}{boolean, True if run cancel requested, else false.}
#' \item{startedAt}{string, The time the last run started.}
#' \item{finishedAt}{string, The time that this run finished.}
#' \item{error}{string, The error message for this run, if present.}
#' \item{output}{array, An array containing the following fields: 
#' \itemize{
#' \item outputName string, The name of the output file.
#' \item fileId integer, The unique ID of the output file.
#' \item path string, The temporary link to download this output file, valid for 36 hours.
#' }}
#' @export
scripts_list_sql_runs <- function(id, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/scripts/sql/{id}/runs"
  path_params  <- list(id = id)
  query_params <- list(limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Check status of a run
#' @param id integer required. The ID of the sql.
#' @param run_id integer required. The ID of the run.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this run.}
#' \item{sqlId}{integer, The ID of this sql.}
#' \item{state}{string, The state of this run.}
#' \item{isCancelRequested}{boolean, True if run cancel requested, else false.}
#' \item{startedAt}{string, The time the last run started.}
#' \item{finishedAt}{string, The time that this run finished.}
#' \item{error}{string, The error message for this run, if present.}
#' \item{output}{array, An array containing the following fields: 
#' \itemize{
#' \item outputName string, The name of the output file.
#' \item fileId integer, The unique ID of the output file.
#' \item path string, The temporary link to download this output file, valid for 36 hours.
#' }}
#' @export
scripts_get_sql_runs <- function(id, run_id) {

  args <- as.list(match.call())[-1]
  path <- "/scripts/sql/{id}/runs/{run_id}"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Cancel a run
#' @param id integer required. The ID of the sql.
#' @param run_id integer required. The ID of the run.
#' 
#' @return  An empty HTTP response
#' @export
scripts_delete_sql_runs <- function(id, run_id) {

  args <- as.list(match.call())[-1]
  path <- "/scripts/sql/{id}/runs/{run_id}"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get the logs for a run
#' @param id integer required. The ID of the sql.
#' @param run_id integer required. The ID of the run.
#' @param last_id integer optional. The ID of the last log message received. Log entries with this ID value or lower will be omitted.Logs are sorted by ID if this value is provided, and are otherwise sorted by createdAt.
#' @param limit integer optional. The maximum number of log messages to return. Default of 10000.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of the log.}
#' \item{createdAt}{string, The time the log was created.}
#' \item{message}{string, The log message.}
#' \item{level}{string, The level of the log. One of unknown,fatal,error,warn,info,debug.}
#' @export
scripts_list_sql_runs_logs <- function(id, run_id, last_id = NULL, limit = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/scripts/sql/{id}/runs/{run_id}/logs"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list(last_id = last_id, limit = limit)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Start a run
#' @param id integer required. The ID of the container.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the run.}
#' \item{containerId}{integer, The ID of the container.}
#' \item{state}{string, The state of the run, one of 'queued' 'running' 'succeeded' 'failed' or 'cancelled'.}
#' \item{isCancelRequested}{boolean, True if run cancel requested, else false.}
#' \item{startedAt}{string, The time the last run started at.}
#' \item{finishedAt}{string, The time the last run completed.}
#' \item{error}{string, The error, if any, returned by the run.}
#' @export
scripts_post_containers_runs <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/scripts/containers/{id}/runs"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' List runs for the given container
#' @param id integer required. The ID of the container.
#' @param limit integer optional. Number of results to return. Defaults to 20. Maximum allowed is 100.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to id. Must be one of: id.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of the run.}
#' \item{containerId}{integer, The ID of the container.}
#' \item{state}{string, The state of the run, one of 'queued' 'running' 'succeeded' 'failed' or 'cancelled'.}
#' \item{isCancelRequested}{boolean, True if run cancel requested, else false.}
#' \item{startedAt}{string, The time the last run started at.}
#' \item{finishedAt}{string, The time the last run completed.}
#' \item{error}{string, The error, if any, returned by the run.}
#' @export
scripts_list_containers_runs <- function(id, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/scripts/containers/{id}/runs"
  path_params  <- list(id = id)
  query_params <- list(limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Check status of a run
#' @param id integer required. The ID of the container.
#' @param run_id integer required. The ID of the run.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the run.}
#' \item{containerId}{integer, The ID of the container.}
#' \item{state}{string, The state of the run, one of 'queued' 'running' 'succeeded' 'failed' or 'cancelled'.}
#' \item{isCancelRequested}{boolean, True if run cancel requested, else false.}
#' \item{startedAt}{string, The time the last run started at.}
#' \item{finishedAt}{string, The time the last run completed.}
#' \item{error}{string, The error, if any, returned by the run.}
#' @export
scripts_get_containers_runs <- function(id, run_id) {

  args <- as.list(match.call())[-1]
  path <- "/scripts/containers/{id}/runs/{run_id}"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("get", path, path_params, query_params, body_params)

  return(resp)

 }


#' Cancel a run
#' @param id integer required. The ID of the container.
#' @param run_id integer required. The ID of the run.
#' 
#' @return  An empty HTTP response
#' @export
scripts_delete_containers_runs <- function(id, run_id) {

  args <- as.list(match.call())[-1]
  path <- "/scripts/containers/{id}/runs/{run_id}"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("delete", path, path_params, query_params, body_params)

  return(resp)

 }


#' Start a run
#' @param id integer required. The ID of the python.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the run.}
#' \item{pythonId}{integer, The ID of the python.}
#' \item{state}{string, The state of the run, one of 'queued' 'running' 'succeeded' 'failed' or 'cancelled'.}
#' \item{isCancelRequested}{boolean, True if run cancel requested, else false.}
#' \item{startedAt}{string, The time the last run started at.}
#' \item{finishedAt}{string, The time the last run completed.}
#' \item{error}{string, The error, if any, returned by the run.}
#' @export
scripts_post_python3_runs <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/scripts/python3/{id}/runs"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("post", path, path_params, query_params, body_params)

  return(resp)

 }


#' List runs for the given python
#' @param id integer required. The ID of the python.
#' @param limit integer optional. Number of results to return. Defaults to 20. Maximum allowed is 100.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to id. Must be one of: id.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of the run.}
#' \item{pythonId}{integer, The ID of the python.}
#' \item{state}{string, The state of the run, one of 'queued' 'running' 'succeeded' 'failed' or 'cancelled'.}
#' \item{isCancelRequested}{boolean, True if run cancel requested, else false.}
#' \item{startedAt}{string, The time the last run started at.}
#' \item{finishedAt}{string, The time the last run completed.}
#' \item{error}{string, The error, if any, returned by the run.}
#' @export
scripts_list_python3_runs <- function(id, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/scripts/python3/{id}/runs"
  path_params  <- list(id = id)
  query_params <- list(limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("ge