#' @title Balance Statistics for `CBPS` Objects
#' 
#' @description
#' Generates balance statistics for `CBPS` and `CBMSM` objects from the \pkg{CBPS} package.
#' 
#' @inheritParams bal.tab
#' @param x a `CBPS` or `CBMSM` object; the output of a call to \pkgfun{CBPS}{CBPS} or \pkgfun{CBPS}{CBMSM}.
#' @param distance an optional formula or data frame containing distance values (e.g., propensity scores) or a character vector containing their names. If a formula or variable names are specified, `bal.tab()` will look in the argument to `data`, if specified. Propensity scores generated by `CBPS()` and `CBMSM()` are automatically included and named `"prop.score"`. For `CBMSM` objects, can be a list of allowable arguments, one for each time point, but each dataset in the list supplied to `distance` must have one row per individual, unlike the data frame in the original call to `CBMSM()`.
#' @param addl an optional formula or data frame containing additional covariates for which to present balance or a character vector containing their names. If a formula or variable names are specified, `bal.tab()` will look in the arguments to the input object, `covs`, and `data`, if specified. For `CBMSM` objects, can be a list of allowable arguments, one for each time point, but each dataset in the list supplied to `addl` must have one row per individual, unlike the data frame in the original call to `CBMSM()`.
#' @param s.d.denom `character`; how the denominator for standardized mean differences should be calculated, if requested. See [col_w_smd()] for allowable options. Abbreviations allowed. If not specified, will be set to `"treated"` if the estimand of the call to `CBPS()` is the ATT and "pooled" if the estimand is the ATE. 
#' @param s.weights Optional; either a vector containing sampling weights for each unit or a string containing the name of the sampling weight variable in `data`. These function like regular weights except that both the adjusted and unadjusted samples will be weighted according to these weights if weights are used. If sampling weights are used in `CBPS()`, it is critical that they are specified in `bal.tab()` as well using the `s.weights` argument.
#' 
#' @returns
#' For point treatments, if clusters are not specified, an object of class `"bal.tab"` containing balance summaries for the `CBPS` object. See [bal.tab()] for details.
#' 
#' If clusters are specified, an object of class `"bal.tab.cluster"` containing balance summaries within each cluster and a summary of balance across clusters. See [`bal.tab.cluster()`][class-bal.tab.cluster] for details.
#' 
#' If `CBPS()` is used with multi-category treatments, an object of class `"bal.tab.multi"` containing balance summaries for each pairwise treatment comparison and a summary of balance across pairwise comparisons. See [`bal.tab.multi()`][class-bal.tab.multi] for details.
#' 
#' If `CBMSM()` is used for longitudinal treatments, an object of class `"bal.tab.msm"` containing balance summaries for each time period and a summary of balance across time periods. See [`bal.tab.msm()`][class-bal.tab.msm] for details.
#' 
#' @details
#' `bal.tab.CBPS()` and `bal.tab.CBMSM()` generate a list of balance summaries for the `CBPS` or `CBMSM` object given and functions similarly to \pkgfun{CBPS}{balance}.
#' 
#' @seealso
#' * [bal.tab()] for details of calculations.
#' * [`bal.tab.cluster()`][class-bal.tab.cluster] for more information on clustered data.
#' * [`bal.tab.multi()`][class-bal.tab.multi] for more information on multi-category treatments.
#' * [`bal.tab.msm()`][class-bal.tab.msm] for more information on longitudinal treatments.
#' 
#' @examplesIf requireNamespace("CBPS", quietly = TRUE)
#' data("lalonde", package = "cobalt")
#' 
#' ## Using CBPS() for generating covariate balancing 
#' ## propensity score weights
#' cbps.out <- CBPS::CBPS(treat ~ age + educ + married + race +
#'                            nodegree + re74 + re75,
#'                        data = lalonde)
#' 
#' bal.tab(cbps.out)

#' @exportS3Method bal.tab CBPS
bal.tab.CBPS <-       function(x,
                               stats, int = FALSE, poly = 1, distance = NULL, addl = NULL, data = NULL, continuous, binary, s.d.denom, thresholds = NULL, weights = NULL, cluster = NULL, imp = NULL, pairwise = TRUE, s.weights = NULL, abs = FALSE, subset = NULL, quick = TRUE,
                               ...) {
    
    tryCatch(args <- c(as.list(environment()), list(...))[-1], error = function(e) .err(conditionMessage(e)))
    
    #Adjustments to arguments
    
    args[vapply(args, rlang::is_missing, logical(1L))] <- NULL
    args[vapply(args, is_null, logical(1L)) & names(args) %nin% names(match.call())[-1]] <- NULL
    
    #Initializing variables
    X <- do.call("x2base", c(list(x), args), quote = TRUE)
    
    args[names(args) %in% names(X)] <- NULL
    
    X <- assign.X.class(X)
    
    out <- do.call("base.bal.tab", c(list(X), args),
                   quote = TRUE)
    return(out)
}

#' @exportS3Method bal.tab CBMSM
bal.tab.CBMSM <- bal.tab.CBPS
