% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/comorbidity.R
\name{comorbidity}
\alias{comorbidity}
\title{Compute comorbidity scores.}
\usage{
comorbidity(x, id, code, score, icd = "icd10", assign0,
  factorise = FALSE, labelled = TRUE, tidy.codes = TRUE)
}
\arguments{
\item{x}{A tidy data frame with one column containing an individual ID and a column containing all diagnostic codes.}

\item{id}{Column of \code{x} containing the individual ID.}

\item{code}{Column of \code{x} containing diagnostic codes. Codes must be in upper case with no punctuation in order to be properly recognised.}

\item{score}{The comorbidity score to compute. Possible choices are the weighted Charlson score (\code{charlson}) and the weighted Elixhauser score (\code{elixhauser}). Values are case-insensitive.}

\item{icd}{The version of ICD coding to use. Possible choices are ICD-9-CM (\code{icd9}) or ICD-10 (\code{icd10}). Defaults to \code{icd10}, and values are case-insensitive.}

\item{assign0}{Apply a hierarchy of comorbidities. If \code{TRUE}, should a comorbidity be present in a patient with different degrees of severity, then the milder form will be assigned to 0 and therefore not counted. By doing this, a type of comorbidity is not counted more than once in each patient. In particular, the comorbidities that are affected by this argument are:
\itemize{
\item "Mild liver disease" (\code{mld}) and "Moderate/severe liver disease" (\code{msld}) for the Charlson score;
\item "Diabetes" (\code{diab}) and "Diabetes with complications" (\code{diabwc}) for the Charlson score;
\item "Cancer" (\code{canc}) and "Metastatic solid tumour" (\code{metacanc}) for the Charlson score;
\item "Hypertension, uncomplicated" (\code{hypunc}) and "Hypertension, complicated" (\code{hypc}) for the Elixhauser score;
\item "Diabetes, uncomplicated" (\code{diabunc}) and "Diabetes, complicated" (\code{diabc}) for the Elixhauser score;
\item "Solid tumour" (\code{solidtum}) and "Metastatic cancer" (\code{metacanc}) for the Elixhauser score.
}}

\item{factorise}{Return comorbidities as factors rather than numeric, where (1 = presence of comorbidity, 0 = otherwise). Defaults to \code{FALSE}.}

\item{labelled}{Attach labels to each comorbidity, compatible with the RStudio viewer via the \code{\link[utils:View]{utils::View()}} function. Defaults to \code{TRUE}.}

\item{tidy.codes}{Tidy diagnostic codes? If \code{TRUE}, all codes are converted to upper case and all non-alphanumeric characters are removed using the regular expression \code{[^[:alnum:]]}. Defaults to \code{TRUE}.}
}
\value{
A data frame with \code{id}, columns relative to each comorbidity domain, comorbidity score, weighted comorbidity score, and categorisations of such scores, with one row per individual.

For the Charlson score, the following variables are included in the dataset:
\itemize{
\item The \code{id} variable as defined by the user;
\item \code{ami}, for acute myocardial infarction;
\item \code{chf}, for congestive heart failure;
\item \code{pvd}, for peripheral vascular disease;
\item \code{cevd}, for cerebrovascular disease;
\item \code{dementia}, for dementia;
\item \code{copd}, chronic obstructive pulmonary disease;
\item \code{rheumd}, for rheumatoid disease;
\item \code{pud}, for peptic ulcer disease;
\item \code{mld}, for mild liver disease;
\item \code{diab}, for diabetes without complications;
\item \code{diabwc}, for diabetes with complications;
\item \code{hp}, for hemiplegia or paraplegia;
\item \code{rend}, for renal disease;
\item \code{canc}, for cancer (any malignancy);
\item \code{msld}, for moderate or severe liver disease;
\item \code{metacanc}, for metastatic solid tumour;
\item \code{aids}, for AIDS/HIV;
\item \code{score}, for the non-weighted version of the Charlson score;
\item \code{index}, for the non-weighted version of the grouped Charlson index;
\item \code{wscore}, for the weighted version of the Charlson score;
\item \code{windex}, for the weighted version of the grouped Charlson index.
}

Conversely, for the Elixhauser score the dataset contains the following variables:
\itemize{
\item The \code{id} variable as defined by the user;
\item \code{chf}, for congestive heart failure;
\item \code{carit}, for cardiac arrhythmias;
\item \code{valv}, for valvular disease;
\item \code{pcd}, for pulmonary circulation disorders;
\item \code{pvd}, for peripheral vascular disorders;
\item \code{hypunc}, for hypertension, uncomplicated;
\item \code{hypc}, for hypertension, complicated;
\item \code{para}, for paralysis;
\item \code{ond}, for other neurological disorders;
\item \code{cpd}, for chronic pulmonary disease;
\item \code{diabunc}, for diabetes, uncomplicated;
\item \code{diabc}, for diabetes, complicated;
\item \code{hypothy}, for hypothyroidism;
\item \code{rf}, for renal failure;
\item \code{ld}, for liver disease;
\item \code{pud}, for peptic ulcer disease, excluding bleeding;
\item \code{aids}, for AIDS/HIV;
\item \code{lymph}, for lymphoma;
\item \code{metacanc}, for metastatic cancer;
\item \code{solidtum}, for solid tumour, without metastasis;
\item \code{rheumd}, for rheumatoid arthritis/collaged vascular disease;
\item \code{coag}, for coagulopathy;
\item \code{obes}, for obesity;
\item \code{wloss}, for weight loss;
\item \code{fed}, for fluid and electrolyte disorders;
\item \code{blane}, for blood loss anaemia;
\item \code{dane}, for deficiency anaemia;
\item \code{alcohol}, for alcohol abuse;
\item \code{drug}, for drug abuse;
\item \code{psycho}, for psychoses;
\item \code{depre}, for depression;
\item \code{score}, for the non-weighted version of the Elixhauser score;
\item \code{index}, for the non-weighted version of the grouped Elixhauser index;
\item \code{wscore_ahrq}, for the weighted version of the Elixhauser score using the AHRQ algorithm (Moore \emph{et al}., 2017);
\item \code{wscore_vw}, for the weighted version of the Elixhauser score using the algorithm in van Walraven \emph{et al}. (2009);
\item \code{windex_ahrq}, for the weighted version of the grouped Elixhauser index using the AHRQ algorithm (Moore \emph{et al}., 2017);
\item \code{windex_vw}, for the weighted version of the grouped Elixhauser index using the algorithm in van Walraven \emph{et al}. (2009).
}

Labels are presented to the user when using the RStudio viewer (e.g. via the \code{\link[utils:View]{utils::View()}} function) for convenience.
}
\description{
Computes comorbidity scores such as the weighted Charlson score and the Elixhauser comorbidity score.
}
\details{
The ICD-10 and ICD-9-CM coding for the Charlson and Elixhauser scores is based on work by Quan \emph{et al}. (2005). Weights for the Charlson score are based on the original formulation by Charlson \emph{et al}. in 1987, while weights for the Elixhauser score are based on work by Moore \emph{et al}. and van Walraven \emph{et al}. Finally, the categorisation of scores and weighted scores is based on work by Menendez \emph{et al}. See \code{vignette("comorbidityscores", package = "comorbidity")} for further details on the comorbidity scores and the weighting algorithm.
ICD-10 and ICD-9 codes must be in upper case and with alphanumeric characters only in order to be properly recognised; set \code{tidy.codes = TRUE} to properly tidy the codes automatically. As a convenience, a message is printed to the R console when non-alphanumeric characters are found.
}
\examples{
set.seed(1)
x <- data.frame(
  id = sample(1:15, size = 200, replace = TRUE),
  code = sample_diag(200),
  stringsAsFactors = FALSE
)

# Charlson score based on ICD-10 diagnostic codes:
comorbidity(x = x, id = "id", code = "code", score = "charlson", assign0 = FALSE)

# Elixhauser score based on ICD-10 diagnostic codes:
comorbidity(x = x, id = "id", code = "code", score = "elixhauser", assign0 = FALSE)
}
\references{
Quan H, Sundararajan V, Halfon P, Fong A, Burnand B, Luthi JC, et al. \emph{Coding algorithms for defining comorbidities in ICD-9-CM and ICD-10 administrative data}. Medical Care 2005; 43(11):1130-1139.

Charlson ME, Pompei P, Ales KL, et al. \emph{A new method of classifying prognostic comorbidity in longitudinal studies: development and validation}. Journal of Chronic Diseases 1987; 40:373-383.

Moore BJ, White S, Washington R, Coenen N, and Elixhauser A. \emph{Identifying increased risk of readmission and in-hospital mortality using hospital administrative data: the AHRQ Elixhauser comorbidity index}. Medical Care 2017; 55(7):698-705.

van Walraven C, Austin PC, Jennings A, Quan H and Forster AJ. \emph{A modification of the Elixhauser comorbidity measures into a point system for hospital death using administrative data}. Medical Care 2009; 47(6):626-633.

Menendez ME, Neuhaus V, van Dijk CN, Ring D. \emph{The Elixhauser comorbidity method outperforms the Charlson index in predicting inpatient death after orthopaedic surgery}. Clinical Orthopaedics and Related Research 2014; 472(9):2878-2886.
}
