#' Write config in a file (JSON/YAML/INI) 
#'
#' @param config.dat a list of config (eg. generated by read.config)
#' @param file.path file to write
#' @param write.type json/ini/yaml
#' @param YAML.indent if write.type is yaml, the YAML file indent will be setted by this 
#' @param encoding, Encoding of filepath parameter, will default to system
#' encoding if not specifield
#' @seealso
#' \code{\link[jsonlite]{toJSON}} convert a list to JSON string, 
#' \code{\link[jsonlite]{prettify}} convert a JSON string to user friendly view, 
#' \code{\link[ini]{write.ini}} write a list in a INI format file, 
#' \code{\link[yaml]{as.yaml}} convert a list to YAML format.
#' @return TRUE/FALSE 
#' @export
#' @examples
#' list.test <- list(a=c(123,456))
#' out.yml <- sprintf('%s/test.yaml', tempdir()) 
#' write.config(list.test, out.yml, write.type = 'yaml')
write.config <- function(config.dat, file.path, write.type = "ini", encoding = getOption("encoding"), 
                         YAML.indent = 2) {
  write.type <- tolower(write.type)
  if (write.type == "ini") {
    ini::write.ini(config.dat, file.path, encoding)
  } else if (write.type == "json") {
    json.dat <- jsonlite::toJSON(config.dat)
    cat(jsonlite::prettify(jsonlite::toJSON(config.dat)), file = file.path)
  } else if (write.type == "yaml") {
    cat(yaml::as.yaml(config.dat, indent = YAML.indent), file = file.path)
  }
  if(file.exists(file.path) && file.size(file.path) > 0) {
    return(TRUE)
  } else {
    return(FALSE)
  }
}
