% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/VaR.R
\name{ES}
\alias{ES}
\title{Compute expected shortfall (ES) of distributions}
\usage{
ES(dist, x = 0.05, dist.type = "qf", qf, ..., intercept = 0,
  slope = 1, control = list())
}
\arguments{
\item{dist}{specifies the distribution whose ES is computed, usually a function or a name
of a function computing quantiles, cdf, pdf, or a random number generator, see
Details.}

\item{x}{level, default is 0.05}

\item{dist.type}{a character string specifying what is computed by \code{dist}, such as
"qf" or "cdf".}

\item{qf}{quantile function, only used if \code{dist.type = "pdf"}.}

\item{...}{passed on to \code{dist}.}

\item{intercept, slope}{requests the ES for the linear transformation \code{intercept +
slope*X}, where \code{X} has distribution specified by \code{dist}, see Details.}

\item{control}{additional control parameters for the numerical integration routine.}
}
\value{
a numeric vector
}
\description{
\code{ES} computes the expected shortfall for distributions specified by the
    arguments. \code{dist} is typically a function (or the name of one). What \code{dist}
    computes is determined by \code{dist.type}, whose default setting is \code{"qf"} (the
    quantile function). Other possible settings of \code{dist.type} include \code{"cdf"}
    and \code{"pdf"}.  Additional arguments for \code{dist} can be given with the
    \code{"..."} arguments.

    Except for the exceptions discussed below, a function computing VaR for the specified
    distribution is constructed and the expected shortfall is computed by numerically
    integrating it. The numerical integration can be fine-tuned with argument
    \code{control}, which should be a named list, see \code{\link{integrate}} for the
    available options.

    If \code{dist.type} is \code{"pdf"}, VaR is not computed, Instead, the partial
    expectation of the lower tail is computed by numerical integration of \code{x *
    pdf(x)}.  Currently the quantile function is required anyway, via argument \code{qf},
    to compute the upper limit of the integral. So, this case is mainly for testing and
    comparison purposes.


    A bunch of expected shortfalls is computed if argument \code{x} or any of the
    arguments in \code{"..."} are of length greater than one. They are recycled to equal
    length, if necessary, using the normal \R recycling rules.

    \code{intercept} and \code{slope} can be used to compute the expected shortfall for
    the location-scale transformation \code{Y = intercept + slope * X}, where the
    distribution of \code{X} is as specified by the other parameters and \code{Y} is the
    variable of interest. The expected shortfall of \code{X} is calculated and then
    transformed to that of \code{Y}. Note that the distribution of \code{X} doesn't need
    to be standardised, although it typically will.

    The \code{intercept} and the \code{slope} can be vectors. Using them may be
    particularly useful for cheap calculations in, for example, forecasting, where the
    predictive distributions are often from the same family, but with different location
    and scale parameters. Conceptually, the described treatment of \code{intercept} and
    \code{slope} is equivalent to recycling them along with the other arguments, but more
    efficiently.

    The names, \code{intercept} and \code{slope}, for the location and scale parameters
    were chosen for their expressiveness and to minimise the possibility for a clash with
    parameters of \code{dist} (e.g., the Gamma distribution has parameter \code{scale}).
}
\examples{
ES(qnorm)

## Gaussian
ES(qnorm, dist.type = "qf")
ES(pnorm, dist.type = "cdf")

## t-dist
ES(qt, dist.type = "qf", df = 4)
ES(pt, dist.type = "cdf", df = 4)

ES(pnorm, x= 0.95, dist.type = "cdf")
ES(qnorm, x= 0.95, dist.type = "qf")
## - VaRES::esnormal(0.95, 0, 1)
## - PerformanceAnalytics::ETL(p=0.05, method = "gaussian", mu = 0,
##                             sigma = 1, weights = 1)             # same

cvar::ES(pnorm, dist.type = "cdf")
cvar::ES(qnorm, dist.type = "qf")
cvar::ES(pnorm, x= 0.05, dist.type = "cdf")
cvar::ES(qnorm, x= 0.05, dist.type = "qf")

## this uses "pdf"
cvar::ES(dnorm, x = 0.05, dist.type = "pdf", qf = qnorm)


## this gives warning (it does more than simply computing ES):
## PerformanceAnalytics::ETL(p=0.95, method = "gaussian", mu = 0, sigma = 1, weights = 1)

## run this if VaRRES is present
\dontrun{
x <- seq(0.01, 0.99, length = 100)
y <- sapply(x, function(p) cvar::ES(qnorm, x = p, dist.type = "qf"))
yS <- sapply(x, function(p) - VaRES::esnormal(p))
plot(x, y)
lines(x, yS, col = "blue")
}

}
