% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/NLR.R
\name{NLR}
\alias{NLR}
\title{DIF statistics based on non-linear regression model.}
\usage{
NLR(Data, group, model, constraints = NULL, type = "both", method = "nls",
match = "zscore", anchor = 1:ncol(Data), start, p.adjust.method = "none", test = "LR",
alpha = 0.05, initboot = T, nrBo = 20)
}
\arguments{
\item{Data}{numeric: either binary data matrix only, or the binary data matrix plus the vector of group .
See \strong{Details}.}

\item{group}{numeric: binary vector of group membership. \code{"0"} for reference group, \code{"1"} for focal group.}

\item{model}{character: generalized logistic regression model to be fitted. See \strong{Details}.}

\item{constraints}{character: which parameters should be the same for both groups. Default value is \code{NULL}. See \strong{Details}.}

\item{type}{character: type of DIF to be tested. Possible values are \code{"both"} (default), \code{"udif"},
\code{"nudif"}, \code{"all"}, or combination of parameters \code{"a"}, \code{"b"}, \code{"c"} and \code{"d"}. See \strong{Details}.}

\item{method}{character: what method should be used for estimation of parameters in \code{model}. The options are
\code{"nls"} for non-linear least squares (default) and \code{"likelihood"} for maximum likelihood method.}

\item{match}{specifies matching criterion. Can be either \code{"zscore"} (default, standardized total score),
\code{"score"} (total test score), or vector of the same length as number of observations in \code{Data}. See \strong{Details}.}

\item{anchor}{a vector of integers specifying which items are currently considered as anchor (DIF free) items. By
default, all items are considered as anchors. Argument is ignored if \code{match} is not \code{"zscore"} or \code{"score"}.
See \strong{Details}.}

\item{start}{numeric: matrix with n rows (where n is the number of items) and 8 columns containing initial
item parameters estimates. See \strong{Details}.}

\item{p.adjust.method}{character: method for multiple comparison correction. See \strong{Details}.}

\item{test}{character: test to be performed for DIF detection (either \code{"LR"} (default), or \code{"F"}).
See \strong{Details}.}

\item{alpha}{numeric: significance level (default is 0.05).}

\item{initboot}{logical: in case of convergence issues, should be starting values recalculated based on
bootstraped samples? (default is \code{TRUE}). See \strong{Details}.}

\item{nrBo}{numeric: the maximal number of iterations for calculation of starting values using
bootstraped samples (default is 20).}
}
\value{
A list with the following arguments:
\describe{
  \item{\code{Sval}}{the values of \code{test} statistics.}
  \item{\code{pval}}{the p-values by \code{test}.}
  \item{\code{adjusted.pval}}{adjusted p-values by \code{p.adjust.method}.}
  \item{\code{df}}{the degress of freedom of \code{test}.}
  \item{\code{test}}{used test.}
  \item{\code{par.m0}}{the matrix of estimated item parameters for m0 model.}
  \item{\code{se.m0}}{the matrix of standard errors of item parameters for m0 model.}
  \item{\code{cov.m0}}{list of covariance matrices of item parameters for m0 model.}
  \item{\code{par.m1}}{the matrix of estimated item parameters for m1 model.}
  \item{\code{se.m1}}{the matrix of standard errors of item parameters for m1 model.}
  \item{\code{cov.m1}}{list of covariance matrices of item parameters for m1 model.}
  \item{\code{conv.fail}}{numeric: number of convergence issues.}
  \item{\code{conv.fail.which}}{the indicators of the items which did not converge.}
  \item{\code{ll.m0}}{log-likelihood of m0 model.}
  \item{\code{ll.m1}}{log-likelihood of m1 model.}
  \item{\code{startBo0}}{the binary matrix. Columns represents iterations of initial values
  recalculations, rows represents items. The value of 0 means no convergence issue in m0 model,
  1 means convergence issue in m0 model.}
  \item{\code{startBo1}}{the binary matrix. Columns represents iterations of initial values
  recalculations, rows represents items. The value of 0 means no convergence issue in m1 model,
  1 means convergence issue in m1 model.}
}
}
\description{
Calculates either DIF likelihood ratio or F statistics for dichotomous data
based on non-linear regression model (generalized logistic regression model).
}
\details{
DIF detection procedure based on Non-Linear Regression is the extension
of Logistic Regression procedure (Swaminathan and Rogers, 1990).

The \code{Data} is a matrix which rows represents examinee scored answers
(1 - correct, 0 - incorrect) and columns correspond to the items.
The \code{group} must be a vector of the same length as \code{nrow(Data)}.

The unconstrained form of 4PL generalized logistic regression model for probability of
correct answer (i.e., y = 1) is

P(y = 1) = (c + cDif*g) + (d + dDif*g - c - cDif*g)/(1 + exp(-(a + aDif*g)*(x - b - bDif*g))),

where x is standardized total score (also called Z-score) and g is group membership.
Parameters a, b, c and d are discrimination, difficulty, guessing and inattention.
Parameters aDif, bDif, cDif and dDif then represent differences between two groups in
discrimination, difficulty, guessing and inattention.

This 4PL model can be further constrained by \code{model} and \code{constraints} arguments.
The arguments \code{model} and \code{constraints} can be also combined.

The \code{model} argument offers several predefined models. The options are as follows:
\code{Rasch} for 1PL model with discrimination parameter fixed on value 1 for both groups,
\code{1PL} for 1PL model with discrimination parameter fixed for both groups,
\code{2PL} for logistic regression model,
\code{3PLcg} for 3PL model with fixed guessing for both groups,
\code{3PLdg} for 3PL model with fixed inattention for both groups,
\code{3PLc} (alternatively also \code{3PL}) for 3PL regression model with guessing parameter,
\code{3PLd} for 3PL model with inattention parameter,
\code{4PLcgdg} for 4PL model with fixed guessing and inattention parameter for both groups,
\code{4PLcgd} (alternatively also \code{4PLd}) for 4PL model with fixed guessing for both groups,
\code{4PLcdg} (alternatively also \code{4PLc}) for 4PL model with fixed inattention for both groups,
or \code{4PL} for 4PL model.

The \code{model} can be specified in more detail with \code{constraints} argument which specifies what
arguments should be fixed for both groups. For example, choice \code{"ad"} means that discrimination (a) and
inattention (d) are fixed for both groups and other parameters (b and c) are not.

The \code{type} corresponds to type of DIF to be tested. Possible values are
\code{"both"} to detect any DIF caused by difference in difficulty or discrimination (i.e., uniform and/or non-uniform),
\code{"udif"} to detect only uniform DIF (i.e., difference in difficulty b),
\code{"nudif"} to detect only non-uniform DIF (i.e., difference in discrimination a), or
\code{"all"} to detect DIF caused by difference caused by any parameter that can differed between groups. The \code{type}
of DIF can be also specified in more detail by using combination of parameters a, b, c and d. For example, with an option
\code{"c"} for 4PL model only the difference in parameter c is tested.

Argument \code{match} represents the matching criterion. It can be either the standardized test score (default, \code{"zscore"}),
total test score (\code{"score"}), or any other continuous or discrete variable of the same length as number of observations
in \code{Data}. Matching criterion is used in \code{NLR} function as a covariate in non-linear regression model.

The \code{start} is a list with as many elements as number of items. Each element is a named numeric
vector representing initial values for parameter estimation. Specifically, parameters
a, b, c, and d are initial values for discrimination, difficulty, guessing and inattention
for reference group. Parameters aDif, bDif, cDif and dDif are then differences in these
parameters between reference and focal group. If not specified, starting
values are calculated with \code{\link[difNLR]{startNLR}} function.

The \code{p.adjust.method} is a character for \code{p.adjust} function from the
\code{stats} package. Possible values are \code{"holm"}, \code{"hochberg"},
\code{"hommel"}, \code{"bonferroni"}, \code{"BH"}, \code{"BY"}, \code{"fdr"},
\code{"none"}.

In case of convergence issues, with an option \code{initboot = TRUE}, the starting values are
re-calculated based on bootstraped samples. Newly calculated initial values are applied only to
items/models with convergence issues.

In case that model considers difference in guessing or inattention parameter, the different parameterization is
used and parameters with standard errors are recalculated by delta method. However, covariance matrices stick with
alternative parameterization.
}
\examples{
\dontrun{
# loading data based on GMAT
data(GMAT)

Data  <- GMAT[, 1:20]
group <- GMAT[, "group"]

# Testing both DIF effects using LR test (default)
# and model with fixed guessing for both groups
NLR(Data, group, model = "3PLcg")

# Using F test
NLR(Data, group, model = "3PLcg", test = "F")

# Testing both DIF effects with Benjamini-Hochberg correction
NLR(Data, group, model = "3PLcg", p.adjust.method = "BH")

# 4PL model with the same guessing and inattention
# to test uniform DIF
NLR(Data, group, model = "4PLcgdg", type = "udif")

# 2PL model to test non-uniform DIF
NLR(Data, group, model = "2PL", type = "nudif")

# 4PL model with fixed a and c parameter
# to test difference in b
NLR(Data, group, model = "4PL", constraints = "ac", type = "b")

# using maximum likelihood estimation method
NLR(Data, group, model = "3PLcg", method = "likelihood")
}

}
\references{
Drabinova, A. & Martinkova P. (2017). Detection of Differential Item Functioning with NonLinear Regression:
Non-IRT Approach Accounting for Guessing. Journal of Educational Measurement, 54(4), 498-517,
\url{https://doi.org/10.1111/jedm.12158}.

Swaminathan, H. & Rogers, H. J. (1990). Detecting Differential Item Functioning Using Logistic Regression Procedures.
Journal of Educational Measurement, 27, 361-370.
}
\seealso{
\code{\link[stats]{p.adjust}}
}
\author{
Adela Hladka (nee Drabinova) \cr
Institute of Computer Science, The Czech Academy of Sciences \cr
Faculty of Mathematics and Physics, Charles University \cr
hladka@cs.cas.cz \cr

Patricia Martinkova \cr
Institute of Computer Science, The Czech Academy of Sciences \cr
martinkova@cs.cas.cz \cr

Karel Zvara \cr
Faculty of Mathematics and Physics, Charles University \cr
}
\keyword{DIF}
