% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/utils.R
\name{apply_dpqr}
\alias{apply_dpqr}
\alias{make_support}
\alias{make_positive_integer}
\title{Utilities for \code{distributions3} objects}
\usage{
apply_dpqr(d, FUN, at, drop = TRUE, type = NULL, ...)

make_support(min, max, d, drop = TRUE)

make_positive_integer(n)
}
\arguments{
\item{d}{A \code{distributions3} object.}

\item{FUN}{Function to be computed. Function should be of type \code{FUN(at, d)}, where
\code{at} is the argument at which the function should be evaluated (e.g., a quantile,
probability, or sample size) and \code{d} is a \code{distributions3} object.}

\item{at}{Specification of values at which \code{FUN} should be evaluated, typically a
numeric vector (e.g., of quantiles, probabilities, etc.) but possibly also a matrix or data
frame.}

\item{drop}{logical. Should the result be simplified to a vector if possible (by
dropping the dimension attribute)? If \code{FALSE} a matrix is always returned.}

\item{type}{Character string used for naming, typically one of \code{"density"}, \code{"logLik"},
\code{"probability"}, \code{"quantile"}, and \code{"random"}. Note that the \code{"random"}
case is processed differently internally in order to vectorize the random number
generation more efficiently.}

\item{...}{Arguments to be passed to  \code{FUN}.}

\item{min, max}{Numeric vectors. Minima and maxima of the supports of a \code{distributions3} object.}

\item{n}{numeric. Number of observations for computing random draws. If \code{length(n) > 1},
the length is taken to be the number required (consistent with base R as, e.g., for \code{rnorm()}).}
}
\description{
Various utility functions to implement methods for distributions with a
unified workflow, in particular to facilitate working with vectorized
\code{distributions3} objects.
These are particularly useful in the computation of densities, probabilities, quantiles,
and random samples when classical d/p/q/r functions are readily available for
the distribution of interest.
}
\examples{


## Implementing a new distribution based on the provided utility functions
## Illustration: Gaussian distribution
## Note: Gaussian() is really just a copy of Normal() with a different class/distribution name


## Generator function for the distribution object.
Gaussian <- function(mu = 0, sigma = 1) {
  stopifnot(
    "parameter lengths do not match (only scalars are allowed to be recycled)" =
      length(mu) == length(sigma) | length(mu) == 1 | length(sigma) == 1
  )
  d <- data.frame(mu = mu, sigma = sigma)
  class(d) <- c("Gaussian", "distribution")
  d
}

## Set up a vector Y containing four Gaussian distributions:
Y <- Gaussian(mu = 1:4, sigma = c(1, 1, 2, 2))
Y

## Extract the underlying parameters:
as.matrix(Y)


## Extractor functions for moments of the distribution include
## mean(), variance(), skewness(), kurtosis().
## These can be typically be defined as functions of the list of parameters.
mean.Gaussian <- function(x, ...) {
  ellipsis::check_dots_used()
  setNames(x$mu, names(x))
}
## Analogously for other moments, see distributions3:::variance.Normal etc.

mean(Y)


## The support() method should return a matrix of "min" and "max" for the
## distribution. The make_support() function helps to set the right names and
## dimension.
support.Gaussian <- function(d, drop = TRUE) {
  stopifnot("d must be a supported distribution object" = is_distribution(d))
  stopifnot(is.logical(drop))

  min <- rep(-Inf, length(d))
  max <- rep(Inf, length(d))

  make_support(min, max, d, drop = drop)
}

support(Y)


## Evaluating certain functions associated with the distribution, e.g.,
## pdf(), log_pdf(), cdf() quantile(), random(), etc. The apply_dpqr()
## function helps to call the typical d/p/q/r functions (like dnorm,
## pnorm, etc.) and set suitable names and dimension.
pdf.Gaussian <- function(d, x, drop = TRUE, ...) {
  FUN <- function(at, d) dnorm(x = at, mean = d$mu, sd = d$sigma, ...)
  apply_dpqr(d = d, FUN = FUN, at = x, type = "density", drop = drop)
}

## Evaluate all densities at the same argument (returns vector):
pdf(Y, 0)

## Evaluate all densities at several arguments (returns matrix):
pdf(Y, c(0, 5))

## Evaluate each density at a different argument (returns vector):
pdf(Y, 4:1)


## Drawing random() samples also uses apply_dpqr() with the argument
## n assured to be a positive integer.
random.Gaussian <- function(x, n = 1L, drop = TRUE, ...) {
  n <- make_positive_integer(n)
  if (n == 0L) {
    return(numeric(0L))
  }
  FUN <- function(at, d) rnorm(n = at, mean = d$mu, sd = d$sigma)
  apply_dpqr(d = x, FUN = FUN, at = n, type = "random", drop = drop)
}

## One random sample for each distribution (returns vector):
random(Y, 1)

## Several random samples for each distribution (returns matrix):
random(Y, 3)


## For further analogous methods see the "Normal" distribution provided
## in distributions3.
methods(class = "Normal")


}
