% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/normality.R
\name{normality}
\alias{normality}
\alias{normality.data.frame}
\title{Performs the Shapiro-Wilk test of normality}
\usage{
normality(.data, ...)

\method{normality}{data.frame}(.data, ..., sample = 5000)
}
\arguments{
\item{.data}{a data.frame or a \code{\link{tbl_df}}.}

\item{...}{one or more unquoted expressions separated by commas.
You can treat variable names like they are positions.
Positive values select variables; negative values to drop variables.
If the first expression is negative, normality() will automatically start with all variables.
These arguments are automatically quoted and evaluated in a context where column names
represent column positions.
They support unquoting and splicing.}

\item{sample}{the number of samples to perform the test.

See vignette("EDA") for an introduction to these concepts.}
}
\value{
An object of the same class as .data.
}
\description{
The normality() performs Shapiro-Wilk test of normality of numerical values.
}
\details{
This function is useful when used with the \code{\link{group_by}}
function of the dplyr package. If you want to test by level of the categorical
data you are interested in, rather than the whole observation,
you can use group_tf as the group_by function.
This function is computed \code{\link{shapiro.test}} function.
}
\section{Normality test information}{

The information derived from the numerical data test is as follows.

\itemize{
\item statistic : the value of the Shapiro-Wilk statistic.
\item p_value : an approximate p-value for the test. This is said in
Roystion(1995) to be adequate for p_value < 0.1.
\item sample : the number of samples to perform the test.
The number of observations supported by the stats::shapiro.test function is 3 to 5000.
}
}

\examples{
\donttest{
# Generate data for the example
carseats <- ISLR::Carseats
carseats[sample(seq(NROW(carseats)), 20), "Income"] <- NA
carseats[sample(seq(NROW(carseats)), 5), "Urban"] <- NA

# Normality test of numerical variables
normality(carseats)

# Select the variable to describe
normality(carseats, Sales, Price)
normality(carseats, -Sales, -Price)
normality(carseats, 1)
normality(carseats, Sales, Price, sample = 300)

# Using dplyr::grouped_dt
library(dplyr)

gdata <- group_by(carseats, ShelveLoc, US)
normality(gdata, "Sales")
normality(gdata, sample = 250)

# Using pipes ---------------------------------
# Normality test of all numerical variables
carseats \%>\%
 normality()

# Positive values select variables
carseats \%>\%
 normality(Sales, Price)

# Positions values select variables
carseats \%>\%
 normality(1)

# Using pipes & dplyr -------------------------
# Test all numerical variables by 'ShelveLoc' and 'US',
# and extract only those with 'ShelveLoc' variable level is "Good".
carseats \%>\%
 group_by(ShelveLoc, US) \%>\%
 normality() \%>\%
 filter(ShelveLoc == "Good")

# extract only those with 'Urban' variable level is "Yes",
# and test 'Sales' by 'ShelveLoc' and 'US'
carseats \%>\%
 filter(Urban == "Yes") \%>\%
 group_by(ShelveLoc, US) \%>\%
 normality(Sales)

# Test log(Income) variables by 'ShelveLoc' and 'US',
# and extract only p.value greater than 0.01.
carseats \%>\%
 mutate(log_income = log(Income)) \%>\%
 group_by(ShelveLoc, US) \%>\%
 normality(log_income) \%>\%
 filter(p_value > 0.01)
}

}
\seealso{
\code{\link{normality.tbl_dbi}}, \code{\link{diagnose_numeric.data.frame}}, \code{\link{describe.data.frame}}, \code{\link{plot_normality.data.frame}}.
}
