#' Contour plot of simulation results of a bio-geochemical variable.
#'
#' @description
#' Contour plot a matrix of values of a bio-geochemical variable, which can be generated through "interpol" function.
#'
#' @param sim a matrix of simulated variables that have been interpolated
#' @param sim.start,sim.end the start and end of the simulation period for the DYRESM-CAEDYM model run of interest. The date format must be "\%Y-\%m-\%d".
#' @param legend.title the legend title of the contour figure.
#' @param min.depth,max.depth,by.value minimum and maximum depth used to be the start of y axis of the contour plot, at the increment of by.value.
#' @param nlevels a set of levels which are used to partition the range of simulation variable.
#'
#' @importFrom grDevices hcl.colors png dev.off
#' @importFrom graphics axis filled.contour mtext par points title
#' @importFrom lubridate year
#' @return This function returns a filled.contour object.
#'
#' @examples
#'  # extract simulated temperature values from DYRESM-CAEDYM simulation file
#'  var.values<-ext_output(dycd.output=system.file("extdata", "dysim.nc", package = "dycdtools"),
#'                        var.extract=c("TEMP"))
#'
#'  for(i in 1:length(var.values)){
#'   expres<-paste0(names(var.values)[i],"<-data.frame(var.values[[",i,"]])")
#'   eval(parse(text=expres))
#'  }
#'
#' # interpolate temperature for depths from 0 to 13 m at increment of 0.5 m
#'   temp.interpolated<-interpol(layerHeights = dyresmLAYER_HTS_Var,
#'                              var = dyresmTEMPTURE_Var,
#'                              min.dept = 0,max.dept = 13,by.value = 0.5)
#'
#' # contour plot of temperature simulations
#'   p <- plot_cont(sim=temp.interpolated,
#'             sim.start="2017-06-06",
#'             sim.end="2017-06-15",
#'             legend.title="T \u00B0C",
#'             min.depth=0,max.depth=13,by.value=0.5,
#'             nlevels=20)
#'
#'   p
#'
#' @export

plot_cont<-function(sim,
                    sim.start,
                    sim.end,
                    legend.title,
                    min.depth,
                    max.depth,
                    by.value,
                    nlevels){

  #---
  # 1. simulation period
  #---
  date<-seq.Date(from = as.Date(sim.start,format="%Y-%m-%d"),
                   to = as.Date(sim.end,format="%Y-%m-%d"),
                   by="day")

  index<-match(seq(lubridate::year(date)[1],lubridate::year(date)[length(date)],by=1),lubridate::year(date))

  #levels<-pretty(range(sim,na.rm = TRUE),nlevels)
  color.palette <- function(n) hcl.colors(n,"RdBu",rev=TRUE)
  #colour<-unlist(lapply(sim,FUN = function(x) color.palette(length(levels)-1)[length(which(levels<x))]))

  #---
  # 2. contour plot the var matrix
  #---

  p <- filled.contour(x=seq(1,ncol(sim),by=1),
                 y=seq(min.depth,max.depth,by=by.value),
                 z=t(sim),
                 ylim = c(max.depth,min.depth),
                 zlim = c(min(sim,na.rm=TRUE),max(sim,na.rm = TRUE)),
                 nlevels = nlevels,
                 color.palette = function(n)hcl.colors(n,"RdBu",rev=TRUE),
                 plot.axes = {
                   axis(3,mgp=c(1,0.25,0),tcl=-0.1, cex.axis=1.8, lwd=0.5,
                        at=seq(1,ncol(sim),by=1)[c(1,index)], labels=lubridate::year(date[c(1,index)]))
                   axis(2,mgp=c(1,0.4,0),tcl=-0.2, cex.axis=1.8, cex.lab=0.8,lwd=0.5)
                   mtext("Depth (m)",side=2,line=1.3,las=0,cex = 1.8)},
                 key.title = {
                   par(cex.main=1.3); title(main=legend.title)
                 })

  return(p)
}
