

#' ee_sensors
#'
#' Returns UC reserve system sensor data
#' @template pages
#' @param  remote_id Need to describe these parameters
#' @param  collection_code Type of collection. Can be \code{CalAcademy}, \code{Private}, \code{VTM}, \code{CDFA}. \code{CalFlora} Others TBA 
#' @param  source  Need to describe these parameters
#' @template dates
#' @template foptions
#' @export
#' @examples  
#' # Currently there are only 40 sensors, so request only needs to be pages 1 and 2.
#' # ee_sensors()
#' # all_sensors <- ee_sensors()
ee_sensors <- function(page = NULL, 
                        page_size = 1000,
						remote_id = NULL, 
						collection_code = NULL, 
						source = NULL, 
						min_date = NULL, 
						max_date = NULL,
						foptions = list()) {
# sensor_url <- "http://ecoengine.berkeley.edu/api/sensors/?format=json"
sensor_url <- paste0(ee_base_url(), "sensors/?format=geojson")

    args <- ee_compact(list(page = page,
                              page_size = page_size,                       
                            remote_id = remote_id, 
                            collection_code = collection_code, 
                            source  = source, 
                            min_date = min_date, 
                            max_date = max_date
   							))
    args$page <- NULL
    sensor_data <- GET(sensor_url, query = args, foptions)
    warn_for_status(sensor_data)
    sensor_results <- content(sensor_data, type = "application/json")
    res <- ldply(sensor_results$features, function(x) {
                             geo_data <- data.frame(t(unlist(x)))
                            })

    res$record <- as.integer(res$properties.record)
    res$geometry.coordinates1 <- as.numeric(res$geometry.coordinates1)
    res$geometry.coordinates2 <- as.numeric(res$geometry.coordinates2)
    res$properties.end_date <- ymd(res$properties.end_date)
    res
}


#' Sensor data 
#'
#' Retrieves data for any sensor returned by \code{\link{ee_list_sensors}}.
#' @param sensor_id The id of the sensor. 
#' @template pages
#' @param  quiet Default is \code{FALSE}. Set to \code{TRUE} to suppress output.
#' @template progress
#' @template foptions
#' @importFrom utils txtProgressBar setTxtProgressBar
#' @export
#' @examples \dontrun{
#' full_sensor_list <- ee_sensors()
#' station <- full_sensor_list$properties.record
#' page_1_data <- ee_sensor_data(sensor_id = station[1], page = 1)
#' page_2_data <- ee_sensor_data(station[1], page = 1:3)
#'}
ee_sensor_data <- function(sensor_id = NULL, page = NULL, page_size = 1000, quiet = FALSE, progress = TRUE, foptions = list()) {
    data_url <- paste0(ee_base_url(), "/sensors/", sensor_id, "/data?format=json")
     
    args <- ee_compact(list(page_size = page_size, sensor_id = sensor_id))
    main_args <- args
    if(is.null(page)) { page <- 1 }
    main_args$page <- page
    temp_data <- GET(data_url, query = args)
    warn_for_status(temp_data)
    sensor_raw <- content(temp_data, type = "application/json")

    required_pages <- ee_paginator(page, sensor_raw$count)
    total_p <- ceiling(sensor_raw$count/page_size)

    if(!quiet) {
    message(sprintf("Search contains %s records (downloading %s page(s) of %s)", sensor_raw$count, length(required_pages), total_p))
    }

    if(progress) pb <- txtProgressBar(min = 0, max = length(required_pages), style = 3)

    results <- list()
    for(i in required_pages) {
        args$page <- i 
        temp_data <- GET(data_url, query = args)
        sensor_iterate <- content(temp_data, type = "application/json")$results
        if(!is.null(sensor_iterate)) {
        raw_data <- do.call(rbind.data.frame, lapply(sensor_iterate, LinearizeNestedList))
        # names(raw_data) <- c("local_date", "value", "data_quality_qualifierid", "data_quality_qualifier_description", "data_quality_valid")
        names(raw_data) <- c("local_date", "value")
        raw_data$local_date <- gsub("T", " ", raw_data$local_date)
        raw_data$local_date <- suppressWarnings(ymd_hms(raw_data$local_date))
        } else {
            raw_data <- NULL
        }
        
        results[[i]] <- raw_data
     if(progress) setTxtProgressBar(pb, i)
     if(i %% 10000 == 0) Sys.sleep(2) 
    }
    # Fix to remove ldply
    results_data <- ldply(ee_compact(results))
    sensor_data <- list(results = sensor_raw$count, call = main_args, type = "sensor", data = results_data)
    class(sensor_data) <- "ecoengine"
    if(progress) close(pb)    
    sensor_data
}




#' Lists subset of the full sensor list
#'
#' @export
#' @examples \dontrun{
#' ee_list_sensors()
#'}
ee_list_sensors <- function() {
full_sensor_list <- ee_sensors()
full_sensor_list[, c("properties.station_name", "properties.units", "properties.variable", "properties.method_name", "record")] 
}


















