% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ego_ergm.R
\name{ego_ergm}
\alias{ego_ergm}
\title{Estimation of ego-Exponential Random Graph Model (ego-ERGM) using Expectation Maximization (EM) as per Salter-Townshend and Murphy (2015).}
\usage{
ego_ergm(net = NULL, form = NULL, core_size = 1, min_size = 5,
  roles = 3, directed = TRUE, edge_covariates = FALSE, seed = 12345,
  steps = 50, tol = 1e-06)
}
\arguments{
\item{net}{The cross-sectional network that an ego-ERGM will be fit on.  Must be presented as a network object.  Any vertex attributes should be attached to networks.  Currently the function does not support comparisons of whole networks.}

\item{form}{The formula comprised of ERGM or TERGM terms used to distinguish between clusters assignments.  Specified as a vector of comma separated terms. No default.}

\item{core_size}{The order of alters to include. The default value of one implies only looking at an ego's alters and the connections among them.}

\item{min_size}{The minimum number of nodes an ego-network must achieve to be included.  Defaults to five.}

\item{roles}{The number of roles that should be fit.  Defaults to 3.}

\item{directed}{Should the longitudinal network be treated as directed? If so, specify as the default TRUE.}

\item{edge_covariates}{Are edge covariates included in the form term? IF so, specify as TRUE.  No default.}

\item{seed}{The seed set to replicate analysis for pseudorandom number generator.}

\item{steps}{The number of default EM steps that should be taken, defaults to 50.}

\item{tol}{The difference in parameter estimates between EM iterations to determine if the algorithm has converged.  Defaults to 1e-6.}
}
\value{
A list of model results, including lambda (the probability of assignments), group.theta (the roles by terms cluster centroids),
        EE.BIC (the Salter-Townshend and Murphy BIC cross-sectional BIC),
       role_assignments (a data frame of the most likely assignments), and reduced_networks (network with excluded ego).
}
\description{
This function estimates an ego-ERGM.  Code taken from Salter-Townshend and Murphy (2015)'s replication archive.
}
\examples{
\donttest{
# Code from xergm.common and their preparation of the Knecht network
library(xergm.common)
set.seed(1)

data("knecht")

for (i in 1:length(friendship)) {
 rownames(friendship[[i]]) <- paste("Student.", 1:nrow(friendship[[i]]), sep="")
 colnames(friendship[[i]]) <- paste("Student.", 1:nrow(friendship[[i]]), sep="")
}
rownames(primary) <- rownames(friendship[[1]])
colnames(primary) <- colnames(friendship[[1]])
sex <- demographics$sex
names(sex) <- rownames(friendship[[1]])
# step 2: imputation of NAs and removal of absent nodes:
friendship <- xergm.common::handleMissings(friendship, na = 10, method = "remove")
friendship <- xergm.common::handleMissings(friendship, na = NA, method = "fillmode")
# step 3: add nodal covariates to the networks
for (i in 1:length(friendship)) {
  s <- xergm.common::adjust(sex, friendship[[i]])
  friendship[[i]] <- network::network(friendship[[i]])
  friendship[[i]] <- network::set.vertex.attribute(friendship[[i]], "sex", s)
  idegsqrt <- sqrt(sna::degree(friendship[[i]], cmode = "indegree"))
  friendship[[i]] <- network::set.vertex.attribute(friendship[[i]],
                                                   "idegsqrt", idegsqrt)
  odegsqrt <- sqrt(sna::degree(friendship[[i]], cmode = "outdegree"))
  friendship[[i]] <- network::set.vertex.attribute(friendship[[i]],
                                                   "odegsqrt", odegsqrt)
}
sapply(friendship, network::network.size)
net <- friendship
rm(list=setdiff(ls(), "net"))

# Reduce down to first time-step
ego_ergm_fit <- ego_ergm(net = net[[1]],
                          form = c("edges", "mutual", "triangle",
                                   "nodeicov('idegsqrt')", "nodeocov('odegsqrt')",
                                   "nodematch('sex')"),
                          core_size = 1,
                          min_size = 5,
                          roles = 3,
                          directed = TRUE,
                          edge_covariates = FALSE,
                          seed = 12345,
                          steps = 50,
                          tol = 1e-06)
}
}
\references{
Salter-Townshend, M. and T.B. Murphy. 2015. Role Analysis in Networks using Mixtures of Exponential Random Graph Models. Journal of Computational and Graphical Statistics, 24(2): 520-538.
}
\keyword{ego-ERGM}
