#' MovingIncubation simulate incubation of a nest with the beginning varying day by day
#' @title Simulate incubation of a nest with the beginning of incubation varying
#' @author Marc Girondot
#' @return A dataframe with informations about thermosensitive period length and incubation length day by day of incubation
#' @param NestsResult A result file generated by searchR
#' @param SexualisationTRN A set of parameters used to model thermal reaction norm during TSP
#' @param temperatures.df A data.frame with 2 or 3 columns: Times, Temperatures and Temperatures.end.incubation (facultative)
#' @param average.incubation.duration The average time to complete incubation
#' @param skip Number of data to skip between two runs
#' @param parameters A set of parameters if result is not provided.
#' @param fixed.parameters Another set of parameters if result is not provided.
#' @param SE Standard error for each parameter if not present in result is not provided (not used still)
#' @param derivate Function used to fit embryo growth: dydt.Gompertz, dydt.exponential or dydt.linear
#' @param test Mean and SD of size of hatchlings as a vector ie test=c(Mean=xx, SD=yy)
#' @param M0 Measure of hatchling size proxi at laying date
#' @param TSP.borders The limits of TSP
#' @param embryo.stages The embryo stages. At least TSP.borders stages must be provided to estimate TSP length
#' @param replicate.CI Number of randomizations to estimate CI
#' @param progress If TRUE, progression bar is shown (desactivate for sweave or knitr)
#' @description Simulate incubation of a nest with the beginning varying day by day\cr
#' Temperatures must be in a data.frame with one column (Time) being the time and the second the temperatures (Temperature). A third columns can indicate the temperature at the end of incubation (Temperature.end.incubation). Do not use FormatNests() for this dataframe.
#' @examples
#' \dontrun{
#' library(embryogrowth)
#' data(resultNest_4p)
#' ti <- seq(from=0, to=(60*24*100), by=60)
#' temp <- rnorm(length(ti), 29, 5)
#' temp <- temp+ti/(60*24*100)/2
#' layout(mat=c(1, 2))
#' parpre <- par(mar=c(4, 4, 1, 1)+0.4)
#' plot(ti/(60*24), temp, type="l", xlab="Days", ylab="Nest temperature", bty="n", las=1)
#' out <- MovingIncubation(NestsResult=resultNest_4p, 
#'      temperatures.df=data.frame(Time=ti, Temperature=temp))
#' with(out, plot(Time/(60*24), Incubation.length.mean/(60*24), xlab="Days along the season", 
#' ylab="Incubation duration", type="l", bty="n", las=1, ylim=c(74, 78)))
#' par(mar=parpre)
#' layout(mat=c(1))
#' }
#' @export


MovingIncubation <-
  function(NestsResult=NULL, 
           temperatures.df=stop("A data.frame with timeseries of temperatures must be provided"),
           average.incubation.duration=60*1440,
           skip = 1, parameters=NULL, fixed.parameters=NULL, SE=NULL, 
           derivate=NULL, test=NULL, M0=NULL, TSP.borders=c(21, 26), 
           embryo.stages="Caretta caretta.SCL", 
           SexualisationTRN=NULL,
           replicate.CI=1, progress=TRUE) {

    # NestsResult=NULL; temperatures.df=NULL; average.incubation.duration=60*1440; skip = 1; parameters=NULL; fixed.parameters=NULL; SE=NULL; derivate=NULL; test=NULL; M0=NULL; TSP.borders=c(21, 26); embryo.stages="Caretta caretta.SCL"; SexualisationTRN=NULL; replicate.CI=1; progress=TRUE

# maintenant il n'est plus possible qu'il n'y ait pas de temperatures
#  
  
	
if (is.null(test)) {
# si tous sont pareils, je reprends les memes
# Correction d'un bug, rajout de [[1]] dans result$test["Mean"][[1]][1] 30/7/2012
	if (all(NestsResult$test["Mean"]==NestsResult$test["Mean"][[1]][1]) & all(NestsResult$test["SD"]==NestsResult$test["SD"][[1]][1])) {
		test <- c(Mean=NestsResult$test["Mean"][[1]][1], SD=NestsResult$test["SD"][[1]][1])
	} else {	
		stop("The size at hatching must be provided.")
	}
}

times <- temperatures.df$Time
temperatures <- temperatures.df$Temperature
if (is.null(temperatures.df$Temperature.end.incubation)) {
  temperatures.end.incubation <- temperatures
} else {
  temperatures.end.incubation <- temperatures.df$Temperature.end.incubation
}


if (is.null(derivate)) derivate <- NestsResult$derivate
if (is.null(M0))	M0 <- NestsResult$M0
if (is.null(fixed.parameters)) fixed.parameters <- NestsResult$fixed.parameters
if (is.null(parameters)) parameters <- NestsResult$par

if (is.null(SexualisationTRN)) SexualisationTRN <- NestsResult$SexualisationTRN

# je peux indiquer des SE en plus de ceux de result
if (is.null(SE)) SE <- NestsResult$SE
if(all(is.na(SE))) replicate.CI <- 1

nbtp <- length(temperatures)

if (progress) pb<-txtProgressBar(min=0, max=(nbtp-3), style=3)

result.out <- data.frame(Time=as.numeric(), TimeWeighted.temperature.mean=as.numeric(), 
                         TimeWeighted.temperature.se=as.numeric(), 
                         TSP.TimeWeighted.temperature.mean=as.numeric(), 
                         TSP.TimeWeighted.temperature.se=as.numeric(), 
                         TSP.MassWeighted.temperature.mean=as.numeric(), 
                         TSP.MassWeighted.temperature.se=as.numeric(), 
                         TSP.STRNWeighted.temperature.mean=as.numeric(), 
                         TSP.STRNWeighted.temperature.se=as.numeric(), 
                         TSP.MassWeighted.STRNWeighted.temperature.mean=as.numeric(), 
                         TSP.MassWeighted.STRNWeighted.temperature.se=as.numeric(), 
                         TSP.length.mean=as.numeric(), 
                         TSP.length.se=as.numeric(), 
                         TSP.begin.mean=as.numeric(), 
                         TSP.begin.se=as.numeric(), 
                         TSP.end.mean=as.numeric(), 
                         TSP.end.se=as.numeric(), 
                         Incubation.length.mean=as.numeric(), 
                         Incubation.length.se=as.numeric(), 
                         Middlethird.begin.mean=as.numeric(), 
                         Middlethird.begin.se=as.numeric(), 
                         Middlethird.end.mean=as.numeric(), 
                         Middlethird.end.se=as.numeric(), 
                         Middlethird.length.mean=as.numeric(), 
                         Middlethird.length.se=as.numeric(), 
                         MiddleThird.TimeWeighted.temperature.mean=as.numeric(), 
                         MiddleThird.TimeWeighted.temperature.se=as.numeric(),
                         MiddleThird.MassWeighted.temperature.mean=as.numeric(), 
                         MiddleThird.MassWeighted.temperature.se=as.numeric())

for (temp in seq(from=1, to=nbtp-2, by=skip)) {
   # print(temp)
  if (progress) 	setTxtProgressBar(pb, (temp-1))
  
    dt <- floor(as.numeric(times[temp:nbtp]-times[temp]))

    tempnid <- temperatures[temp:(temp+length(dt)-1)]
    tempnid2 <- temperatures.end.incubation[temp:(temp+length(dt)-1)]
    # et dans dt j ai le temps
    tempencours <- tempnid + (tempnid2-tempnid)*(dt/average.incubation.duration)
    df <- data.frame(Time=dt, temp=tempencours)
    formated <- FormatNests(df)
    
    out.incubation <- info.nests(parameters=parameters, fixed.parameters = fixed.parameters,
               SE=SE, temperatures = formated, derivate=derivate, test=test, 
               stopattest = TRUE, M0=M0, TSP.borders=TSP.borders, 
               embryo.stages=embryo.stages, 
               SexualisationTRN=SexualisationTRN, progress = FALSE, 
               replicate.CI=replicate.CI, out="summary", warnings=FALSE)$summary
    
    # metric.end.incubation=c(Temp=test$Mean), 
    
    if (identical(out.incubation, FALSE)) {
      break
    } else {
      result.out <- rbind(result.out, cbind(Time=times[temp], out.incubation, row.names=NULL))
    }
}



return(result.out)

}
