/* integration.cpp
 *
 *   Copyright (C) 2012, 2013 David Bolin, Finn Lindgren
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "integration.h"

void shapeInt(vector< map<int,double> > * L, double * a,double * b,int K, const int n,double lim,double * Pv, double * Ev, int max_size, int n_threads){
	//	return [Pv Ev x F]
	
	double *al, *bl,*f,*s,*Li;
	double ** x;
	double fsum,fsum2,Pi,Ei;
	double ai,bi,c,d,rtmp;
	int i,j;

	al = new double[n];
	bl = new double[n];
	f = new double[K];
	s = new double[K];
	Li = new double[n];
	
	x = new double*[n];
	for (i=0; i<n; i++) {
		Li[i] = (*L)[i][i];
		al[i] = Li[i]*a[i];
		bl[i] = Li[i]*b[i];
		x[i] = new double[K];
		//memset(x[i], 0, n);
		for(j=0;j<K;j++){
			x[i][j] = 0.0;
		}
	}
	
	for (i=0; i<K; i++) {
		f[i] = 1.0;
	}

	
	#ifdef _OPENMP
		const int max_nP = omp_get_num_procs();
		int nPtmp;
		if(n_threads == 0){
			nPtmp = max_nP;
		} else {
			nPtmp = min(max_nP, max(n_threads,1));
		}
		const int nP = nPtmp;
		omp_set_num_threads(nP);
	#else
		const int nP = 1;
	#endif
	

	unsigned long m_1 = 4294967087U;
	unsigned long m_2 = 4294944443U;
	unsigned long seed[6];

	/* On Linux, reading from /dev/random is a blocking event if
	 * there is not enough entropy, so we use /dev/urandom
	 * instead.  We are not doing cryptography.
	 * For systems that don't have /dev/urandom, fall back to
	 * seeding based on time(0).
	 * http://man7.org/linux/man-pages/man4/random.4.html
	 * https://developer.apple.com/library/mac/documentation/Darwin/Reference/ManPages/man4/random.4.html
	 */
	ssize_t seed_read = 0;
#if defined (__APPLE__) && defined (__linux__)
	int randomSrc = open("/dev/urandom", O_RDONLY);
	if (randomSrc > 0) {
	   seed_read = read(randomSrc, seed, sizeof(seed));
	   close(randomSrc);
	}
#endif
	if (seed_read != (ssize_t) sizeof(seed)) {
	  srand(time(0));
	  for(i=0;i<6;i++){
	    seed[i] = rand(); 
	  }
	}
    
	seed[0] = seed[0] % m_1;
	seed[1] = seed[1] % m_1;
	seed[2] = seed[2] % m_1;
	seed[3] = seed[3] % m_2;
	seed[4] = seed[4] % m_2;
	seed[5] = seed[5] % m_2;    
    
	RngStream_SetPackageSeed(seed);
	RngStream * RngArray = new RngStream[nP];
	int myrank = 0;
	
	for (i=0; i<nP; i++) {
		RngArray[i] = RngStream_CreateStream("namehere");
	}
	
	for (i=n-1; i>=0; i--) {
		//memcpy(s, zero, sizeof(double)*K);
		for (j=0; j<K; j++) {
			s[j] = 0;
		}
		fsum = 0;
		fsum2 = 0;
		
		for(map<int,double>::iterator iter = ((*L)[i]).begin(); iter != ((*L)[i]).end(); iter++ ) {
			for (j=0; j<K; j++) {
				s[j] += (iter->second)*x[iter->first][j];
			}
		}	
		
		/*
		 #pragma omp parallel for
		 for (int j=0; j<K; j++) {
		 for(map<int,double>::iterator iter = ((*L)[i]).begin(); iter != ((*L)[i]).end(); iter++ ) {
		 s[j] += (iter->second)*x[iter->first][j];	
		 }
		 }	
		 */
		
		#pragma omp parallel private(myrank,ai,bi,c,d,j,rtmp)
		{
			#ifdef _OPENMP
				myrank = omp_get_thread_num();
			#endif
			
			#pragma omp for reduction(+:fsum,fsum2)
			for (j=0; j<K; j++) {
				ai = al[i] + s[j];
				bi = bl[i] + s[j];
				
				if (al[i] == -numeric_limits<double>::infinity()){
					ai = -numeric_limits<double>::infinity();			
				} else {
					ai = al[i] + s[j];
				}
				
				if (bl[i] == numeric_limits<double>::infinity()){
					bi = numeric_limits<double>::infinity();			
				} else {
					bi = bl[i] + s[j];
				}
				
				
				
				if (ai<-9) {
					c = 0;
				}else if(ai>9){
					c = 1;
				}else {
					c = gsl_cdf_ugaussian_P(ai);
				}
				if (bi<-9) {
					d = 0;
				}else if(bi>9){
					d = 1;
				}else {
					d = gsl_cdf_ugaussian_P(bi);
				}
			
				
				f[j] = f[j]*(d-c);
				fsum += f[j];
				fsum2 += f[j]*f[j];
				
				if (d-c<1e-12) { //no weight is given to this sample
					x[i][j] = 0; //just set x to zero
				} else {
					rtmp = c+(d-c)* RngStream_RandU01(RngArray[myrank]);
					x[i][j] = (gsl_cdf_ugaussian_Pinv(rtmp)-s[j])/Li[i];
				}
				
				if (x[i][j] == numeric_limits<double>::infinity()){
					cout << "simulated infinite value, changing to zero" << endl;
					cout << "c=" << c << ", d=" << d << ",d-c=" << d-c << endl;
					x[i][j] = 0;
				}
				
				
				if (x[i][j]!=x[i][j]) {
					cout << j << " x is nan: rtmp=" << rtmp << ", c="<< c;
					cout << ", d="<< d << ", ai="<< ai << ", bi=" << bi <<", s[j]=";
					cout << s[j] << ", Li=" << Li[i]<< endl;
				}
			}
		}
		
		Pi = fsum/K;
		if (Pi!=Pi) {
			cout << i << "Estimated probability is nan, stopping estimation." << endl;
			break;
		}
		Ei = sqrt(max((fsum2-fsum*fsum/K)/K/K,0));

		if (Pi<lim) {
			break;
		}
		
		if(i<n-max_size){
			break;
		}
		
		Pv[i] = Pi;
		Ev[i] = Ei;
	}
	
	delete[] al;
	delete[] bl;
	delete[] f;
	delete[] s;
	delete[] Li;
	for (i=0; i<n; i++) {
		delete[] x[i];
	}
	delete[] x;
	delete[] RngArray;
	
}
