% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calc_daily_stats.R
\name{calc_daily_stats}
\alias{calc_daily_stats}
\title{Calculate daily summary statistics}
\usage{
calc_daily_stats(
  data,
  dates = Date,
  values = Value,
  groups = STATION_NUMBER,
  station_number,
  percentiles = c(5, 25, 75, 95),
  roll_days = 1,
  roll_align = "right",
  water_year_start = 1,
  start_year,
  end_year,
  exclude_years,
  complete_years = FALSE,
  months = 1:12,
  transpose = FALSE,
  ignore_missing = FALSE
)
}
\arguments{
\item{data}{Data frame of daily data that contains columns of dates, flow values, and (optional) groups (e.g. station numbers).
Leave blank if using \code{station_number} argument.}

\item{dates}{Name of column in \code{data} that contains dates formatted YYYY-MM-DD. Only required if dates column name is not 
'Date' (default). Leave blank if using \code{station_number} argument.}

\item{values}{Name of column in \code{data} that contains numeric flow values, in units of cubic metres per second.
Only required if values column name is not 'Value' (default). Leave blank if using \code{station_number} argument.}

\item{groups}{Name of column in \code{data} that contains unique identifiers for different data sets, if applicable. Only required 
if groups column name is not 'STATION_NUMBER'. Function will automatically group by a column named 'STATION_NUMBER' if 
present. Remove the 'STATION_NUMBER' column beforehand to remove this grouping. Leave blank if using \code{station_number} 
argument.}

\item{station_number}{Character string vector of seven digit Water Survey of Canada station numbers (e.g. \code{"08NM116"}) of
which to extract daily streamflow data from a HYDAT database. Requires \code{tidyhydat} package and a HYDAT database.
Leave blank if using \code{data} argument.}

\item{percentiles}{Numeric vector of percentiles to calculate. Set to \code{NA} if none required. Default \code{c(5,25,75,95)}.}

\item{roll_days}{Numeric value of the number of days to apply a rolling mean. Default \code{1}.}

\item{roll_align}{Character string identifying the direction of the rolling mean from the specified date, either by the first 
(\code{'left'}), last (\code{'right'}), or middle (\code{'center'}) day of the rolling n-day group of observations. 
Default \code{'right'}.}

\item{water_year_start}{Numeric value indicating the month (\code{1} through \code{12}) of the start of water year for
analysis. Default \code{1}.}

\item{start_year}{Numeric value of the first year to consider for analysis. Leave blank to use the first year of the source data.}

\item{end_year}{Numeric value of the last year to consider for analysis. Leave blank to use the last year of the source data.}

\item{exclude_years}{Numeric vector of years to exclude from analysis. Leave blank to include all years.}

\item{complete_years}{Logical values indicating whether to include only years with complete data in analysis. Default \code{FALSE}.}

\item{months}{Numeric vector of months to include in analysis (e.g. \code{6:8} for Jun-Aug). Leave blank to summarize 
all months (default \code{1:12}).}

\item{transpose}{Logical value indicating whether to transpose rows and columns of results. Default \code{FALSE}.}

\item{ignore_missing}{Logical value indicating whether dates with missing values should be included in the calculation. If
\code{TRUE} then a statistic will be calculated regardless of missing dates. If \code{FALSE} then only those statistics from
time periods with no missing dates will be returned. Default \code{FALSE}.}
}
\value{
A tibble data frame with the following columns:
  \item{Date}{date (MMM-DD) of daily statistics}
  \item{DayofYear}{day of year of daily statistics}
  \item{Mean}{daily mean of all flows for a given day of the year}
  \item{Median}{daily mean of all flows for a given day of the year}
  \item{Maximum}{daily mean of all flows for a given day of the year}
  \item{Minimum}{daily mean of all flows for a given day of the year}
  \item{P'n'}{each daily n-th percentile selected of all flows for a given day of the year}
  Default percentile columns:
  \item{P5}{daily 5th percentile of all flows for a given day of the year}
  \item{P25}{daily 25th percentile of all flows for a given day of the year}
  \item{P75}{daily 75th percentile of all flows for a given day of the year}
  \item{P95}{daily 95th percentile of all flows for a given day of the year}
  Transposing data creates a column of "Statistics" and subsequent columns for each year selected.
}
\description{
Calculates means, medians, maximums, minimums, and percentiles for each day of the year of flow values 
   from a daily streamflow data set. Can determine statistics of rolling mean days (e.g. 7-day flows) using the \code{roll_days} 
   argument. Note that statistics are based on the numeric days of year (1-365) and not the date of year (Jan 1 - Dec 31).
   Calculates statistics from all values, unless specified. Returns a tibble with statistics.
}
\examples{
# Run if HYDAT database has been downloaded (using tidyhydat::download_hydat())
if (file.exists(tidyhydat::hy_downloaded_db())) {

# Calculate daily statistics using data argument with defaults
flow_data <- tidyhydat::hy_daily_flows(station_number = "08NM116")
calc_daily_stats(data = flow_data,
                 start_year = 1980)

# Calculate daily statistics using station_number argument with defaults
calc_daily_stats(station_number = "08NM116",
                 start_year = 1980)

# Calculate daily statistics regardless if there is missing data for a given day of year
calc_daily_stats(station_number = "08NM116",
                 ignore_missing = TRUE)
                  
# Calculate daily statistics using only years with no missing data
calc_daily_stats(station_number = "08NM116",
                 complete_years = TRUE)

# Calculate daily statistics for water years starting in October between 1980 and 2010
calc_daily_stats(station_number = "08NM116",
                 start_year = 1980,
                 end_year = 2010,
                 water_year_start = 10)
                 
# Calculate daily statistics with custom years and removing certain years
calc_daily_stats(station_number = "08NM116",
                 start_year = 1981,
                 end_year = 2010,
                 exclude_years = c(1991,1993:1995))
                  
# Calculate daily statistics for 7-day flows for July-September months only, 
# with 25 and 75th percentiles starting in 1980
calc_daily_stats(station_number = "08NM116",
                 start_year = 1980,
                 roll_days = 7,
                 months = 7:9,
                 percentiles = c(25,75))       
                 
}
}
