#' @title Predict Method for fastNaiveBayes.gaussian fits
#' @description Uses a fastNaiveBayes.gaussian model and a new data set to create the classifications.
#'     This can either be the raw probabilities generated by the fastNaiveBayes.gaussian model or the classes themselves.
#'
#' @param object A fitted object of class "fastNaiveBayes.gaussian".
#' @param newdata A numeric matrix. A Sparse dgcMatrix is also accepted.
#'     Note that if newdata contains features that were not encountered in the training data, these are omitted from the prediction.
#'     Furthermore, newdata can contain fewer features than encountered in the training data.
#' @param type If "raw", the conditional a-posterior probabilities for each class are returned, and the class with maximal probability else.
#' @param sparse Use a sparse Matrix? If true a sparse matrix will be constructed from x, which can give up to a 40\% speed up.
#'     It's possible to directly feed a sparse dgcMatrix as x, which will set this parameter to TRUE
#' @param threshold A threshold for the minimum probability. For Bernoulli and Multinomial event models Laplace smoothing solves this,
#' but in the case of Gaussian event models, this ensures numerical probabilities
#' @param ... Not used.
#' @return If type = 'class', a factor with classified class levels. If type = 'raw', a matrix with the predicted probabilities of
#'     each class, where each column in the matrix corresponds to a class level.
#'
#' @export
#' @import Matrix
#'
#' @details In the extremely unlikely case that two classes have the exact same estimated probability, the first encountered class
#'     is used as the classification and a warning is issued.
#'
#'     Using a sparse matrix directly can be especially useful if it's necessary to use predict multiple times on the same matrix or
#'     on different subselections of the same initial matrix, see examples for further details.
#' @examples
#' rm(list = ls())
#' library(fastNaiveBayes)
#' cars <- mtcars
#' y <- as.factor(ifelse(cars$mpg > 25, "High", "Low"))
#' x <- cars[, 2:ncol(cars)]
#'
#' dist <- fastNaiveBayes::fastNaiveBayes.detect_distribution(x, nrows = nrow(x))
#'
#' # Gaussian only
#' vars <- c("hp", dist$gaussian)
#' newx <- x[, vars]
#'
#' mod <- fastNaiveBayes.gaussian(newx, y)
#' pred <- predict(mod, newdata = newx)
#' mean(pred != y)
predict.fastNaiveBayes.gaussian <- function(object, newdata, type = c("class", "raw", "rawprob"),
                                            sparse = FALSE, threshold = .Machine$double.eps, ...) {
  type <- match.arg(type)
  if (class(newdata)[1] != "dgCMatrix") {
    if (!is.matrix(newdata)) {
      newdata <- as.matrix(newdata)
    }
    if (sparse) {
      newdata <- Matrix(newdata, sparse = TRUE)
    }
  } else {
    sparse <- TRUE
  }


  names <- intersect(object$names, colnames(newdata))
  if(length(object$names)!=length(names)){
    warning('Columns in test and train set not equal! Only the intersect of the two is used for prediction')
    newdata <- newdata[, names]
  }
  if (length(names) == 1) {
    newdata <- as.matrix(newdata)
  }

  data <- object$probability_table

  probs <- NULL
  newdata <- t(newdata)
  for (j in 1:length(data)) {
    level <- data[[j]]
    level_probs <- NULL

    if(length(names)==1){
      means <- level$means
      stddevs <- level$stddev
    }else{
      means <- level$means[names]
      stddevs <- level$stddev[names]
    }

    level_probs <- colSums((newdata-means)^2/stddevs^2)
    level_probs <- sum(log(1/(sqrt(2*pi*stddevs^2))))-0.5*level_probs

    level_probs[is.infinite(level_probs)] <- max(-100000, log(threshold))

    probs <- cbind(probs, level_probs)

    colnames(probs)[j] <- level$level
  }

  if (type == "rawprob") {
    return(probs)
  }

  priors <- as.vector(object$priors)
  probs <- exp(probs)
  for (i in 1:length(priors)) {
    probs[, i] <- probs[, i] * priors[i]
  }

  denom <- rowSums(probs)
  denom[denom==0] <- 1
  probs <- probs / denom

  if (type == "class") {
    class <- names(object$priors)[max.col(probs, ties.method = "first")]
    return(as.factor(class))
  }
  return(probs)
}
