% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sun.r
\name{sun}
\alias{sun}
\title{Solar radiance and irradiance}
\usage{
sun(
  elevation,
  coeff_bh,
  coeff_dh,
  slope,
  aspect,
  hh,
  horizon_step = 90,
  albedo = 0.2,
  linke = 3,
  day = 1,
  step = 0.5,
  declination = NULL,
  solar_constant = 1367,
  distance_step = 1,
  npartitions = 1,
  beam_rad = TRUE,
  diff_rad = TRUE,
  refl_rad = TRUE,
  glob_rad = TRUE,
  insol_time = TRUE,
  lowMemory = FALSE
)
}
\arguments{
\item{elevation}{A \code{GRaster} with values representing elevation (typically in meters).}

\item{coeff_bh}{A \code{GRaster}: A raster with values of the real-sky beam radiation coefficient. Valid values are between 0 and 1.}

\item{coeff_dh}{A \code{GRaster}: A raster with values of the real-sky diffuse radiation coefficient. Valid values are between 0 and 1.}

\item{slope}{A \code{GRaster}: This is a raster representing topographic slope in radians. It can be generated using \code{\link[=terrain]{terrain()}}.}

\item{aspect}{A \code{GRaster}: This is a raster representing topographic aspect in degrees. It can be generated using \code{\link[=terrain]{terrain()}}. If generated with that function, "east orientation" \emph{must} be used (i.e., argument \code{northIs0} must be \code{FALSE}).}

\item{hh}{A "stack" of \code{GRaster}s: This represents height of the horizon in radians in particular directions. Horizon height can be calculated using \code{\link[=horizonHeight]{horizonHeight()}}. The directions \emph{must} be in "east orientation" (i.e., argument \code{northIs0} in \code{horzionHeight()} must be \code{FALSE}). The directions must correspond with the sequence given by \code{horizon_step} (see next argument). For example, if \code{horizon_step} is 90, then \code{hh} must contain rasters representing horizon height at 0 (east), 90 (north), 180 (west), and 270 (south) aspects.
#'}

\item{horizon_step}{Numeric >0: Difference between angular steps in which horizon height is measured. One horizon height raster will be made per value from 0 to 360 - \code{horizon_step} degrees.}

\item{albedo}{A \code{GRaster} or a numeric value: This is either a raster with values of ground albedo or a numeric value (in which case albedo is assumed to be the same everywhere). Albedo is unit-less, and the default value is 0.2.}

\item{linke}{A \code{GRaster} or a numeric value: This is either a raster with values of the Linke atmospheric turbidity coefficient or a numeric value (in which case the same value is assumed for all locations). The Linke coefficient is unit-less. The default value is 3, but see also the \strong{GRASS} manual page for module \code{r.sun} (\code{grassHelp("r.sun")}).}

\item{day}{Positive integer between 1 to 365, inclusive: Day of year for which to calculate ir/radiation. Default is 1 (January 1st).}

\item{step}{Positive integer between 0 and 24, inclusive. Time step in hours for all-day radiation sums. Decimal values are OK.}

\item{declination}{Numeric or \code{NULL} (default). Declination value. If \code{NULL}, this is calculated automatically.}

\item{solar_constant}{Positive numeric: The solar constant (solar energy hitting the top of the atmosphere). Default is 1367. Units are W / m^2.}

\item{distance_step}{Positive numeric between 0.5 and 1.5, inclusive: Sampling distance coefficient. Default is 1.}

\item{npartitions}{Positive numeric. Number of chunks in which to read input files. Default is 1.}

\item{beam_rad}{Logical: If \code{TRUE} (default), generate a raster with beam irradiation with units of Wh / m^2 / day ("mode 2" of the \code{r.sun} \strong{GRASS} module).}

\item{diff_rad}{Logical: If \code{TRUE} (default), generate a raster representing irradiation in Wh / m^2 /day}

\item{refl_rad}{Logical: If \code{TRUE} (default), generate a raster with ground-reflected irradiation with units of Wh / m^2 / day ("mode 2" of the \code{r.sun} \strong{GRASS} module).}

\item{glob_rad}{Logical:. If \code{TRUE} (default), generate a raster with total irradiance/irradiation with units of Wh / m^2 / day ("mode 2" of the \code{r.sun} \strong{GRASS} module).}

\item{insol_time}{Logical: If \code{TRUE} (default), generate a raster with total insolation time in hours ("mode 2" of the \code{r.sun} \strong{GRASS} module).}

\item{lowMemory}{Logical: If \code{TRUE}, use the low-memory version of the \code{r.sun} \strong{GRASS} module. The default is \code{FALSE}.}
}
\value{
A raster or raster stack stack with the same extent, resolution, and coordinate reference system as \code{elevation}. Assuming all possible rasters are generated they represent:
\itemize{
\item \code{beam_rad}: Beam radiation (Watt-hours/m2/day)
\item \code{diff_rad}: Diffuse radiation (Watt-hours/m2/day)
\item \code{refl_rad}: Reflected radiation (Watt-hours/m2/day)
\item \code{glob_rad}: Global radiation (Watt-hours/m2/day)
\item \code{insol_time}: Insolation duration (hours)
}
}
\description{
The \code{sun()} function calculates beam (direct), diffuse and ground reflected solar irradiation for a given day and set of topographic and atmospheric conditions. The function relies on the \strong{GRASS} module \code{r.sun}, the manual page for which contains a detailed explanation (see \code{grassHelp("r.sun")})
}
\examples{
if (grassStarted()) {

# Setup
library(terra)

# Example data
madElev <- fastData("madElev")

# Convert a SpatRaster to a GRaster
elev <- fast(madElev)

### Calculate input rasters

# Values below are just a guess
coeff_bh <- coeff_dh <- elev
coeff_bh[] <- 0.4
coeff_dh[] <- 0.6

slope <- terrain(elev, "slope")
aspect <- terrain(elev, "aspect", northIs0 = FALSE)

horizon_step <- 90
hh <- horizonHeight(elev, step = horizon_step, northIs0 = FALSE)

### calculate solar ir/radiance

solar <- sun(
	elevation = elev,
	coeff_bh = coeff_bh,
	coeff_dh = coeff_dh,
	slope = slope,
	aspect = aspect,
	hh = hh,
	horizon_step = horizon_step,
	albedo = 0.2,
	linke = 1.5,
	day = 1,
	step = 0.5,
	declination = NULL,
	solar_constant = 1367,
	
	distance_step = 1,
	npartitions = 1,

	beam_rad = TRUE,
	diff_rad = TRUE,
	refl_rad = TRUE,
	glob_rad = TRUE,
	insol_time = TRUE,

	lowMemory = FALSE
)

solar

}
}
\seealso{
\code{\link[=terrain]{terrain()}}, \code{\link[=horizonHeight]{horizonHeight()}}, \strong{GRASS} manual page for module \code{r.sun} (see \code{grassHelp("r.sun")})
}
