% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/final_size.R
\name{final_size}
\alias{final_size}
\title{Final size of an epidemic}
\usage{
final_size(
  r0,
  contact_matrix,
  demography_vector,
  susceptibility,
  p_susceptibility,
  solver = c("iterative", "newton"),
  control = list()
)
}
\arguments{
\item{r0}{The basic reproductive number \eqn{R_0} of the disease.}

\item{contact_matrix}{Social contact matrix. Entry \eqn{m_{ij}} gives average
number of contacts in group \eqn{i} reported by participants in group \eqn{j}}

\item{demography_vector}{Demography vector. Entry \eqn{v_{i}} gives
proportion of total population in group \eqn{i} (model will normalise
if needed).}

\item{susceptibility}{A matrix giving the susceptibility of individuals in
demographic group \eqn{i} and risk group \eqn{k}.}

\item{p_susceptibility}{A matrix giving the probability that an individual
in demography group \eqn{i} is in risk (or susceptibility) group \eqn{k}.
Each row represents the overall distribution of individuals in demographic
group \eqn{i} across risk groups, and each row \emph{must sum to 1.0}.}

\item{solver}{Which solver to use. Options are "iterative" (default) or
"newton", for the iterative solver, or the Newton solver, respectively.
Special conditions apply when using the Newton solver, see the \code{control}
argument.}

\item{control}{A list of named solver options, see \emph{Solver options}.}
}
\value{
A data.frame of the proportion of infected individuals, within each
demography group and susceptibility group combination.
If the demography groups and susceptibility groups are named, these
names are added to relevant columns. If the groups are not named, synthetic
names are added of the form \verb{demo_grp_<i>}, for each demographic group
\eqn{i}.
}
\description{
\code{final_size} calculates the final size of an epidemic outbreak
in a population with heterogeneous mixing, and with heterogeneous
susceptibility to infection such as that conferred by an immunisation
programme.
}
\section{Solver options}{
The \code{control} argument accepts a list of solver options, with the iterative
solver taking two extra arguments than the Newton solver. This is an optional
argument, and default options are used within the solver functions if an
argument is missing. Arguments provided override the solver defaults.
\subsection{Common options}{
\enumerate{
\item \code{iterations}: The number of iterations over which to solve for the final
size, unless the error is below the solver tolerance. Default = 10000.
\item \code{tolerance}: The solver tolerance; solving for final size ends when the
error drops below this tolerance. Defaults to set \code{1e-6}. Larger tolerance
values are likely to lead to inaccurate final size estimates.
}
}

\subsection{Iterative solver options}{
\enumerate{
\item \code{step_rate}: The solver step rate. Defaults to 1.9 as a value found to
work well.
\item \code{adapt_step}: Boolean, whether the solver step rate should be changed
based on the solver error. Defaults to TRUE.
}
}
}

\examples{
# load example POLYMOD data included in the package
data(polymod_uk)
r0 <- 2.0
contact_matrix <- polymod_uk$contact_matrix
demography_vector <- polymod_uk$demography_vector

# define the number of age and susceptibility groups
n_demo_grps <- length(demography_vector)
n_risk_grps <- 3

# In this example, all risk groups from all age groups are fully
# susceptible
susceptibility <- matrix(
  data = 1, nrow = n_demo_grps, ncol = n_risk_grps
)

p_susceptibility <- matrix(
  data = 1, nrow = n_demo_grps, ncol = n_risk_grps
)
# p_susceptibility rows must sum to 1.0
p_susceptibility <- p_susceptibility / rowSums(p_susceptibility)

# using default arguments for `solver` and `control`
final_size(
  r0 = r0,
  contact_matrix = contact_matrix,
  demography_vector = demography_vector,
  susceptibility = susceptibility,
  p_susceptibility = p_susceptibility
)

# using manually specified solver settings for the iterative solver
control <- list(
  iterations = 100,
  tolerance = 1e-3,
  step_rate = 1.9,
  adapt_step = TRUE
)

final_size(
  r0 = r0,
  contact_matrix = contact_matrix,
  demography_vector = demography_vector,
  susceptibility = susceptibility,
  p_susceptibility = p_susceptibility,
  solver = "iterative",
  control = control
)

# manual settings for the newton solver
control <- list(
  iterations = 100,
  tolerance = 1e-3
)

final_size(
  r0 = r0,
  contact_matrix = contact_matrix,
  demography_vector = demography_vector,
  susceptibility = susceptibility,
  p_susceptibility = p_susceptibility,
  solver = "newton",
  control = control
)
}
\keyword{epidemic}
\keyword{model}
