% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/factor_number.R
\name{factor.number}
\alias{factor.number}
\title{Factor number selection methods}
\usage{
factor.number(
  x,
  fm.restricted = FALSE,
  method = c("ic", "er"),
  q.max = NULL,
  do.plot = FALSE,
  center = TRUE
)
}
\arguments{
\item{x}{input time series matrix, with each row representing a variable}

\item{fm.restricted}{whether to estimate the number of restricted or unrestricted factors}

\item{method}{A string specifying the factor number selection method; possible values are:
\itemize{
   \item{\code{"ic"}}{ information criteria-based methods of Alessi, Barigozzi & Capasso (2010) when \code{fm.restricted = TRUE} or Hallin and Liška (2007) when \code{fm.restricted = FALSE}}
   \item{\code{"er"}}{ eigenvalue ratio of Ahn and Horenstein (2013)}
}}

\item{q.max}{maximum number of factors; if \code{q.max = NULL}, a default value is selected as \code{min(50, floor(sqrt(min(dim(x)[2] - 1, dim(x)[1]))))}}

\item{do.plot}{whether to plot the information criteria values}

\item{center}{whether to de-mean the input \code{x} row-wise}
}
\value{
if \code{method = "ic"}, a vector containing minimisers of the six information criteria, otherwise, the maximiser of the eigenvalue ratio
}
\description{
Methods to estimate the number of factor.
When \code{method = 'er'}, the factor number is estimated by maximising the ration of successive eigenvalues.
When \code{method = 'ic'}, the information criterion-methods discussed in Hallin and Liška (2007) (when \code{fm.restricted = FALSE})
and Alessi, Barigozzi and Capasso (2010) (when \code{fm.restricted = TRUE}) are implemented, with the information criterion called by \code{ic.op = 5} recommended by default.
}
\details{
For further details, see references.
}
\examples{
library(fnets)

set.seed(123)
n <- 500
p <- 50
common <- sim.unrestricted(n, p)
idio <- sim.var(n, p)
x <- common$data * apply(idio$data, 1, sd) / apply(common$data, 1, sd) + idio$data

hl <- factor.number(x, fm.restricted = FALSE, do.plot = TRUE)
hl
library(fnets)

set.seed(123)
n <- 500
p <- 50
common <- sim.restricted(n, p)
idio <- sim.var(n, p)
x <- common$data * apply(idio$data, 1, sd) / apply(common$data, 1, sd) + idio$data

abc <- factor.number(x, fm.restricted = TRUE, do.plot = TRUE)
abc

er <- factor.number(x, method = "er", fm.restricted = TRUE, do.plot = TRUE)
er
}
\references{
Ahn, S. C. & Horenstein, A. R. (2013) Eigenvalue ratio test for the number of factors. Econometrica, 81(3), 1203--1227.

Alessi, L., Barigozzi, M., and Capasso, M. (2010) Improved penalization for determining the number of factors in approximate factor models. Statistics & Probability Letters, 80(23-24):1806–1813.

Bai, J. & Ng, S. (2002) Determining the number of factors in approximate factor models. Econometrica. 70: 191-221.

Hallin, M. & Liška, R. (2007) Determining the number of factors in the general dynamic factor model. Journal of the American Statistical Association, 102(478), 603--617.

Owens, D., Cho, H. & Barigozzi, M. (2022) fnets: An R Package for Network Estimation and Forecasting via Factor-Adjusted VAR Modelling. arXiv preprint arXiv:2301.11675.
}
