% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{predCoef}
\alias{predCoef}
\title{\eqn{h}-step Prediction coefficients}
\arguments{
\item{X}{the data \eqn{X_1, \ldots, X_T}}

\item{P}{the maximum order of coefficients to be computed; has to be a positive integer}

\item{H}{the maximum lead time; has to be a positive integer}

\item{t}{a vector of values \eqn{t}; the elements have to satisfy
\code{max(t) <= length(X)} and  \code{min(t) >= min(max(N[N != 0]),p)}.}

\item{N}{a vector of values \eqn{N}; the elements have to satisfy
\code{max(N[N != 0]) <= min(t)} and \code{min(N[N != 0]) >= 1 + P}.
\eqn{N = 0} corresponds to the case where all data is taken into account.}
}
\value{
Returns a named list with elements \code{coef}, \code{t}, and \code{N},
        where \code{coef} is an array of dimension
        \code{P} \eqn{\times} \code{P} \eqn{\times} \code{H} \eqn{\times}
        \code{length(t)} \eqn{\times} \code{length(N)}, and
        \code{t}, and \code{N} are the parameters provided on the call of the
        function. See the example on how to access the vector
        \eqn{\hat v_{N,T}^{(p,h)}(t)}.
}
\description{
This function computes the localised and iterated Yule-Walker coefficients
for h-step ahead forecasting of \eqn{X_{t+h}} from \eqn{X_{t}, ..., X_{t-p+1}},
where \eqn{h = 1, \ldots,} \code{H} and \eqn{p = 1, \ldots,} \code{P}.
}
\details{
For every \eqn{t \in} \code{t} and every \eqn{N \in} \code{N} the (iterated) Yule-Walker
estimates \eqn{\hat v_{N,T}^{(p,h)}(t)} are computed. They are defined as
\deqn{\hat v_{N,T}^{(p,h)}(t) :=  e'_1 \big( e_1 \big( \hat a_{N,T}^{(p)}(t) \big)' + H \big)^h, \quad N \geq 1,}
and
\deqn{\hat v_{0,T}^{(p,h)}(t) :=  \hat v_{t,T}^{(p,h)}(t),}
with
\deqn{ e_1 := \left(\begin{array}{c} 1 \\ 0 \\ \vdots \\ 0 \end{array} \right), \quad H := \left( \begin{array}{ccccc} 0 & 0 & \cdots & 0 & 0 \\ 1 & 0 & \cdots & 0 & 0 \\ 0 & 1 & \cdots & 0 & 0 \\ \vdots & \ddots & \cdots & 0 & 0 \\ 0 & 0 & \cdots & 1 & 0 \end{array} \right)}
and
\deqn{ \hat a_{N,T}^{(p)}(t) := \big( \hat\Gamma_{N,T}^{(p)}(t) \big)^{-1} \hat\gamma_{N,T}^{(p)}(t),}
where
\deqn{\hat\Gamma_{N,T}^{(p)}(t) := \big[ \hat \gamma_{i-j;N,T}(t) \big]_{i,j = 1, \ldots, p}, \quad \hat \gamma_{N,T}^{(p)}(t) := \big( \hat \gamma_{1;N,T}(t), \ldots, \hat \gamma_{p;N,T}(t) \big)'}
and
\deqn{\hat \gamma_{k;N,T}(t) := \frac{1}{N-|k|} \sum_{\ell=t-N+|k|+1}^{t} X_{\ell-|k|,T} X_{\ell,T}}
is the usual lag-\eqn{k} autocovariance estimator (without mean adjustment),
computed from the observations \eqn{X_{t-N+1}, \ldots, X_{t}}.

The Durbin-Levinson Algorithm is used to successively compute the solutions to the
Yule-Walker equations (cf. Brockwell/Davis (1991), Proposition 5.2.1).
To compute the \eqn{h}-step ahead coefficients we use the recursive relationship
\deqn{\hat v_{i,N,T}^{(p)}(t,h) = \hat a_{i,N,T}^{(p)}(t) \hat v_{1,N,T}^{(p,h-1)}(t) + \hat v_{i+1,N,T}^{(p,h-1)}(t) I\{i \leq p-1\},}
(cf. the proof of Lemma C.3 in Kley et al (2017)).
}
\examples{
T <- 100
X <- rnorm(T)

P <- 5
H <- 1
m <- 20

Nmin <- 25
pcoef <- predCoef(X, P, H, (T - m - H + 1):T, c(0, seq(Nmin, T - m - H, 1)))

## Access the prediction vector for p = 2, h = 1, t = 95, N = 25
p <- 2
h <- 1
t <- 95
N <- 35
res <- pcoef$coef[p, 1:p, h, pcoef$t == t, pcoef$N == N]
}
\references{
Brockwell, P. J. & Davis, R. A. (1991).
Time Series: Theory and Methods. Springer, New York.

Kley, T., Preuss, P. & Fryzlewicz, P. (2016).
Predictive model choice in time series analysis under stationarity and non-stationarity.
[cf. \url{personal.lse.ac.uk/kley/forecastLSTS.pdf}]
}

