#' Plot the co-occurrence profile of a species
#'
#' \code{species_profile_plot()} accepts a species and list of inventories like
#' those generated by
#' \code{\link[=assessment_list_inventory]{assessment_list_inventory()}} and
#' generates a histogram of the co-occurrence profile of that species. Repeated
#' co-occurrences across multiple assessments are included in summary
#' calculations but self co-occurrences are not.
#'
#' @param species The scientific name of the target plant species
#' @param inventory_list A list of site inventories having the format of
#' \code{\link[=assessment_list_inventory]{assessment_list_inventory()}}
#' @param native Logical indicating whether only native co-occurrences should be
#'   considered.
#'
#'
#' @import dplyr
#' @importFrom ggplot2 aes ggplot geom_col scale_x_continuous labs geom_vline
#'   theme_minimal
#' @importFrom rlang .data
#' @importFrom stats sd
#'
#' @examples
#' # species_profile_plot() is best used in combination with
#' # download_assessment_list() and assessment_list_inventory().
#'
#' \donttest{
#' ontario <- download_assessment_list(database = 2)
#' ontario_invs <- assessment_list_inventory(ontario)
#' species_profile_plot("Aster lateriflorus", ontario_invs, native = TRUE)
#' }
#'
#' @export


species_profile_plot <-
  function(species, inventory_list, native = FALSE) {
    if (!is_inventory_list(inventory_list)) {
      message(
        "assessment_list must be a list of dataframes obtained from universalFQA.org. Type ?download_assessment_list for help.")
      return(invisible(NULL))
    }

    included <- vector("logical")
    for (inventory in seq_along(inventory_list)) {
      included[inventory] <-
        (species %in% inventory_list[[inventory]]$scientific_name)
    } # gives a logical vector indicating which inventories include the given species
    # currently ignores the possibility that a species is included twice with different C

    if (sum(included) == 0) {
      stop("Species does not appear in any assessment. No profile plot generated.",
           call. = FALSE)
    }

    short_list <-
      inventory_list[included] # all of these should include the species now

    cooccur_df <- do.call(rbind, short_list)

    species_only <- dplyr::filter(cooccur_df,
                                  .data$scientific_name == species)
    target_c <- species_only$c[1] # record target species c-value.

    cooccur_df <- dplyr::filter(cooccur_df,
                                .data$scientific_name != species) # to ignore self-cooccurrence

    if (native == TRUE) {
      cooccur_df <- dplyr::filter(cooccur_df,
                                  .data$nativity == "native")
    }

    cooccur <- dplyr::mutate(cooccur_df,
                             as.factor(.data$c))

    c_counts <- cooccur |>
      dplyr::group_by(c) |>
      dplyr::count()

    missing_fill <- data.frame(c = 0:10, n = 0)
    missing_fill <- dplyr::anti_join(missing_fill,
                                     c_counts,
                                     by = "c")
    c_counts <- rbind(c_counts, missing_fill)
    c_counts <- c_counts |>
      dplyr::arrange(c) |>
      dplyr::ungroup() |>
      dplyr::mutate(species, target_c) |>
      dplyr::select(species,
                    target_c,
                    cospecies_c = "c",
                    cospecies_n = "n")

    if (native == TRUE){
      plot_title <- "native co-occurrence profile"} else {
        plot_title <- "co-occurrence profile"
      }

    ggplot2::ggplot(c_counts, ggplot2::aes(x = .data$cospecies_c,
                                           y = .data$cospecies_n)) +
      ggplot2::geom_col() +
      ggplot2::scale_x_continuous(breaks = seq(from = 0,
                                               to = 11,
                                               by = 2)) +
      ggplot2::labs(
        x = "Co-occurring species C values",
        y = "Frequency",
        title = paste(species, plot_title)
      ) +
      ggplot2::geom_vline(xintercept = target_c,
                          linetype = "dashed") +
      ggplot2::theme_minimal()
  }
