#' Download official data of disaster risk areas as an sf object.
#'
#' This function reads the the official data of disaster risk areas in Brazil. It specifically focuses
#' on geodynamic and hydro-meteorological disasters capable of triggering landslides and floods. The
#' data set covers the whole country. Each risk area polygon (known as 'BATER') has unique code id (column 'geo_bater').
#' The data set brings information on the extent to which the risk area polygons overlap with census
#' tracts and block faces (column "acuracia") and number of ris areas within each risk area (column 'num').
#' Orignal data were generated by IBGE and CEMADEN. For more information about the methodology, see deails
#' at https://www.ibge.gov.br/geociencias/organizacao-do-territorio/tipologias-do-territorio/21538-populacao-em-areas-de-risco-no-brasil.html
#'
#' @param year A year number in YYYY format.
#' @param tp Whether the function returns the 'original' dataset with high resolution or a dataset with 'simplified' borders (Default)#' @export
#' @export
#' @examples \donttest{
#'
#' library(geobr)
#'
#' # Read all disaster risk areas in an specific year
#'   d <- read_disaster_risk_area(year=2010)
#'
#' }
#'
#'

read_disaster_risk_area <- function(year, tp="simplified"){

  # Get metadata with data addresses
  metadata <- download_metadata()

  # Select geo
  temp_meta <- subset(metadata, geo=="disaster_risk_area")

  # Select data type
  temp_meta <- select_data_type(temp_meta, tp)

  # Verify year input
  if(is.null(year)){ stop(paste0("Error: Invalid Value to argument 'year'. It must be one of the following: ",
                                 paste(unique(temp_meta$year),collapse = " ")))

  } else if (year %in% temp_meta$year){ temp_meta <- temp_meta[temp_meta[,2] == year, ]

  } else { stop(paste0("Error: Invalid Value to argument 'year'. It must be one of the following: ",
                       paste(unique(temp_meta$year),collapse = " ")))
  }


  # list paths of files to download
  filesD <- as.character(temp_meta$download_path)

  # download files
  temps <- download_gpkg(filesD)

  # read sf
  temp_sf <- sf::st_read(temps, quiet=T)
  return(temp_sf)
}
